local BaikCore = LibStub("AceAddon-3.0"):GetAddon("BaikCore")
local BaikActionbar = BaikCore.BaikActionbar
local Assert = BaikCore.Assert
local Database = BaikCore.Database
local Log = BaikCore.Log
local Option = BaikCore.Option
local Ui = BaikCore.Ui

-- Add Module
local Border = BaikActionbar:NewModule("Border")
BaikActionbar.Border = Border

Border.DEFAULT = false

-- Private Variables
local borderFunc = {}

local OPTION_NAME = "border_style"
local GLOBAL_BORDER_DB = "global_border"

-- Private Methods
local function HideShowFrame(frame, hide)
    if Database:Get(GLOBAL_BORDER_DB) then
        hide = true
    end

    if hide then
        Ui:SetHideFinal(frame)
    else
        Ui:UndoSetHideFinal(frame)
    end
end

local function SyncGlobal()
    local hide = Database:Get(GLOBAL_BORDER_DB)

    for dbValue, func in pairs(borderFunc) do
        func(hide and true or Database:Get(dbValue))
    end
end

local function GetOpts()
    return {
        name = "Border",
        type = "group",
        args = {
            hide = {
                name = "Remove",
                type = "toggle",
                get = function(...)
                    return Database:Get(GLOBAL_BORDER_DB)
                end,
                set = function(self, value, ...)
                    Database:Set(GLOBAL_BORDER_DB, value)
                    SyncGlobal()
                end,
            }
        }
    }
end

-- Border API
function Border:RemoveButton(btn, hide)
    Assert:Table(btn)
    Assert:Boolean(hide)

    HideShowFrame(btn.NormalTexture, hide)
end

function Border:RemoveButtons(btnTable, hide)
    Assert:Table(btnTable)
    Assert:Boolean(hide)

    for _, btn in pairs(btnTable) do
        self:RemoveButton(btn, hide)
    end
end

function Border:RemoveFrame(frame, hide, force)
    Assert:Table(frame)
    Assert:Boolean(hide)

    HideShowFrame(frame, hide)

    if force then
        if hide then
            frame:Hide()
        else
            frame:Show()
        end
    end
end

function Border:RemoveFrames(frameTbl, hide, force)
    Assert:Table(frameTbl)
    Assert:Boolean(hide)

    for _, frame in pairs(frameTbl) do
        self:RemoveFrame(frame, hide, force)
    end
end

function Border:RemoveAlpha(frame, hide)
    Assert:Table(frame)
    Assert:Boolean(hide)

    if Database:Get(GLOBAL_BORDER_DB) then
        hide = true
    end

    local alpha = hide and 0 or 1
    frame:SetAlpha(alpha)
end

function Border:RemoveAlphas(frameTbl, hide)
    Assert:Table(frameTbl)
    Assert:Boolean(hide)

    for _, frame in pairs(frameTbl) do
        self:RemoveAlpha(frame, hide)
    end
end

function Border:HideFrame(frame, hide)
    Assert:Table(frame)
    Assert:Boolean(hide)

    if Database:Get(GLOBAL_BORDER_DB) then
        hide = true
    end

    if hide then
        frame:Hide()
    else
        frame:Show()
    end
end

function Border:HideFrames(frameTbl, hide)
    Assert:Table(frameTbl)
    Assert:Boolean(hide)

    for _, frame in pairs(frameTbl) do
        self:HideFrame(frame, hide)
    end
end

function Border:GetOpts(dbValue, SetBorderFunc)
    borderFunc[dbValue] = SetBorderFunc
    return {
        name = "Remove Border",
        type = "toggle",
        disabled = function(...)
            return Database:Get(GLOBAL_BORDER_DB)
        end,
        get = function(...)
            return Database:Get(dbValue)
        end,
        set = function(self, value, ...)
            Database:Set(dbValue, value)
            SetBorderFunc(value)
        end
    }
end

-- Ace3 Callbacks
function Border:OnInitialize()
    Log:v("Border_OnInitialize")
    Database:InitIfEmpty(GLOBAL_BORDER_DB, Border.DEFAULT)
    BaikActionbar:AddOption(OPTION_NAME, GetOpts())
end

function Border:OnEnable()
    Log:v("Border_OnEnable")
    SyncGlobal()
end

function Border:OnDisable()
    Log:v("Border_OnDisable")
end
