
--[[--------------------------------------------------------------------
	Auc-Stat-TheUndermineJournal - The Undermine Journal price statistics module
	Copyright (c) 2011, 2012 Johnny C. Lam
	All rights reserved.
	
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:
	
	1. Redistributions of source code must retain the above copyright
		notice, this list of conditions and the following disclaimer.
	2. Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
		
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
	TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
	PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
	CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
	SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
	INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
	POSSIBILITY OF SUCH DAMAGE.
--]]--------------------------------------------------------------------

--[[
	This addon is a continuation of the original project created by Johnny C. Lam.
	Development resumed in 2015 under Heavy Stone Crab.
--]]

if not AucAdvanced then return end

local libraryName = "The Undermine Journal"
local libraryType = "Stat"
local library, parent, private = AucAdvanced.NewModule(libraryType, libraryName)

if not library then return end

local auctionPrint, decode, _, _, replicate, empty, get, set, default, debugPrint, fill, _TRANS = AucAdvanced.GetModuleLocals()
local bellCurve = AucAdvanced.API.GenerateBellCurve()
local resources = AucAdvanced.Resources
local resolveServerKey = AucAdvanced.ResolveServerKey
local wipe = wipe
local math_huge = math.huge
local math_floor = math.floor
local secondsDay = 24 * 60 * 60
local seconds3Days = 3 * secondsDay

local R_1 = 0.4 -- 102
local G_1 = 1.0 -- 255
local B_1 = 0.4 -- 102

-- TUJ colors.
local R_2 = 0.9
local G_2 = 0.8
local B_2 = 0.5

-- TUJ data table for market data from most-recently requested item.
local TUJdata = {}
local customPrice = 0

library.Processors = {
	config = function(callbackType, ...)
			private.SetupConfigGui(...)
		end,
	load = function(callbackType, addon)
			if private.OnLoad then
				private.OnLoad(addon)
			end
		end,
	itemtooltip = function(callbackType, ...)
			private.ProcessTooltip(...)
		end,
	battlepettooltip = function(callbackType, ...)
			private.ProcessTooltip(...)
		end
}

-- library.GetPrice()
-- (optional) Returns the estimated price for an item link.
function library.GetPrice(hyperlink, serverKey)
	if not get("stat.TUJ.enable") then return end
	if not private.GetInfo(hyperlink, serverKey) then return end
	
	return TUJdata.recent, TUJdata.market, TUJdata.stddev, TUJdata.globalMedian, TUJdata.globalMean,
		TUJdata.globalStdDev, TUJdata.age, TUJdata.days
end

-- library.GetPriceColumns()
-- (optional) Returns the column names for GetPrice.
function library.GetPriceColumns()
	return "Local 3-Day Mean", "Local 14-Day Mean", "Local 14-Day Std Dev", "Global Median", "Global Mean",
		"Global Std Dev", "Data Age", "Last Seen Locally"
end

-- library.GetPriceArray()
-- Returns pricing and other statistical info in an array.
function library.GetPriceArray(hyperlink, serverKey)
	if not get("stat.TUJ.enable") then return end
	if not private.GetInfo(hyperlink, serverKey) then return end
	
	-- customPrice is the lowest of all selected price values, or 0 if the item has been filtered out.
	customPrice = math_huge
	
	-- Determine lowest price.
	if get("stat.TUJ.includeRecent") and TUJdata.recent then
		if TUJdata.recent < customPrice then
			local debounce = true
			
			if get("stat.TUJ.limitRecentDays") and TUJdata.days then
				if TUJdata.days > 3 then
					debounce = false
				end
			end
			
			if debounce then
				customPrice = TUJdata.recent
			end
		end
	end
	
	if get("stat.TUJ.includeMarket") and TUJdata.market then
		if TUJdata.market < customPrice then
			local debounce = true
			
			if get("stat.TUJ.limitMarketDays") and TUJdata.days then
				if TUJdata.days > 14 then
					debounce = false
				end
			end
			
			if debounce then
				customPrice = TUJdata.market
			end
		end
	end
	
	if get("stat.TUJ.includeGlobalMedian") and TUJdata.globalMedian then
		if TUJdata.globalMedian < customPrice then
			customPrice = TUJdata.globalMedian
		end
	end
	
	if get("stat.TUJ.includeGlobalMean") and TUJdata.globalMean then
		if TUJdata.globalMean < customPrice then
			customPrice = TUJdata.globalMean
		end
	end
	
	-- Filter out unwanted values.
	if get("stat.TUJ.includeStdDev") and TUJdata.stddev and TUJdata.market then
		local debounce = true
		
		if get("stat.TUJ.limitMarketDays") and TUJdata.days then
			if TUJdata.days > 14 then
				debounce = false
			end
		end
		
		if debounce and TUJdata.stddev > TUJdata.market * get("stat.TUJ.thresholdStdDev") / 100 then
			customPrice = 0
		end
	end
	
	if get("stat.TUJ.includeGlobalStdDev") and TUJdata.globalStdDev and TUJdata.globalMedian then
		if TUJdata.globalStdDev > TUJdata.globalMedian * get("stat.TUJ.thresholdGlobalStdDev") / 100 then
			customPrice = 0
		end
	end
	
	if get("stat.TUJ.excludeVendorItems") and TUJdata.days then
		if TUJdata.days == 252 then
			customPrice = 0
		end
	end
	
	-- No usable price sources.
	if customPrice == math_huge then
		customPrice = 0
	end
	
	local priceArray = {}
	
	priceArray.price = customPrice
	priceArray.seen = 0
	
	-- (optional) Values that other modules may be looking for.
	priceArray.recent = TUJdata.recent -- local 3-day mean
	priceArray.market = TUJdata.market -- local 14-day mean
	priceArray.stddev = TUJdata.stddev -- local 14-day standard deviation
	priceArray.globalMedian = TUJdata.globalMedian -- global mean
	priceArray.globalMean = TUJdata.globalMean -- global median 
	priceArray.globalStdDev = TUJdata.globalStdDev -- global standard deviation
	priceArray.age = TUJdata.age -- seconds since TUJ data was compiled
	priceArray.days = TUJdata.days -- days since item was last seen locally
	
	return priceArray
end

-- library.GetItemPDF()
-- Returns the Probability Density Function for an item link.
function library.GetItemPDF(hyperlink, serverKey)
	if not get("stat.TUJ.enable") then return end
	if not private.GetInfo(hyperlink, serverKey) then return end
	
	local mean
	local standardDeviation
	
	--[[
		Prioritize local information over global information.
		
		There is no "right" way to return data with this function. Unlike
		GetPriceArray, GetItemPDF only returns one primary value - the bell curve
		- and the information from TUJ can produce two bell curves.
		
		A compelling change to this function may be to prioritize returning values
		that are used to determine customPrice, but such a change has no effect on
		this module, and is therefore beyond its scope.
	--]]
	if TUJdata.market and TUJdata.stddev then
		mean = TUJdata.market
		standardDeviation = TUJdata.stddev
	else
		if TUJdata.globalMean and TUJdata.globalStdDev then
			mean = TUJdata.globalMean
			standardDeviation = TUJdata.globalStdDev
		end
	end
	
	-- No available data.
	if not mean then return end
	
	if standardDeviation then
		if standardDeviation == 0 then
			return
		end
	else
		return
	end
	
	-- Bound standardDeviation to avoid extreme values, which can cause problems for GetMarketValue.
	if standardDeviation > mean then
		standardDeviation = mean
	elseif standardDeviation < mean * 0.01 then
		standardDeviation = mean * 0.01
	end
	
	-- Calculate the lower and upper bounds as +/- 3 standard deviations.
	local bottom = mean - 3 * standardDeviation
	local top = mean + 3 * standardDeviation
	
	bellCurve:SetParameters(mean, standardDeviation)
	
	return bellCurve, bottom, top
end

function private.OnLoad(addon)
	default("stat.TUJ.disableOriginalTooltip", true)
	default("stat.TUJ.enable", true)
	
	-- Tooltip variables.
	default("stat.TUJ.tooltip", true)
	default("stat.TUJ.multiplyStackSize", false)
	default("stat.TUJ.custom", false)
	default("stat.TUJ.recent", true)
	default("stat.TUJ.market", true)
	default("stat.TUJ.stddev", true)
	default("stat.TUJ.globalMedian", true)
	default("stat.TUJ.globalMean", true)
	default("stat.TUJ.globalStdDev", true)
	default("stat.TUJ.days", true)
	
	-- Price source variables.
	default("stat.TUJ.includeRecent", false)
	default("stat.TUJ.includeMarket", true)
	default("stat.TUJ.includeGlobalMedian", true)
	default("stat.TUJ.includeGlobalMean", false)
	
	-- Filter variables.
	default("stat.TUJ.limitRecentDays", true)
	default("stat.TUJ.limitMarketDays", true)
	default("stat.TUJ.includeStdDev", false)
	default("stat.TUJ.thresholdStdDev", 50)
	default("stat.TUJ.includeGlobalStdDev", false)
	default("stat.TUJ.thresholdGlobalStdDev", 50)
	default("stat.TUJ.excludeVendorItems", true)
	
	-- Only run this function once.
	private.OnLoad = nil
end

-- private.GetInfo(hyperlink, serverKey)
-- Returns the market info for the requested item in the TUJdata table.
function private.GetInfo(hyperlink, serverKey)
	-- TheUndermineJournal addon doesn't support cross-server pricing.
	if resolveServerKey(serverKey) ~= resources.ServerKey then return end
	
	local linkType, itemID, suffix, factor = decode(hyperlink)
	
	if linkType ~= "item" and linkType ~= "battlepet" then return end
	
	wipe(TUJdata)
	
	if TUJMarketInfo then
		TUJMarketInfo(hyperlink, TUJdata)
	else
		return
	end
	
	if TUJdata.itemid then
		return itemID
	end
	
	if TUJdata.species then
		return TUJdata.species
	end
end

function private.SetupConfigGui(gui)
	local tab = gui:AddTab(library.libName, library.libType .. " Modules")
	
	gui:AddHelp(tab, "TODO",
		_TRANS("Author's Note"),
		_TRANS("This help section is somewhat out of date, but I am too lazy to update it. No one reads this anyways. Mouse over the options for clarification about what they do.")
	) gui:AddHelp(tab, "what TUJ",
		_TRANS("TUJ_Help_Question1"),
		_TRANS("TUJ_Help_Answer1")
	) gui:AddHelp(tab, "what auc stat TUJ",
		_TRANS("TUJ_Help_Question2"),
		_TRANS("TUJ_Help_Answer2")
	) gui:AddHelp(tab, "which setup",
		_TRANS("TUJ_Help_Question3"),
		_TRANS("TUJ_Help_Answer3")
	) gui:AddHelp(tab, "where setup",
		_TRANS("TUJ_Help_Question4"),
		_TRANS("TUJ_Help_Answer4")
	)
	
	-- All options here will be duplicated in the tooltip frame.
	function private.addTooltipControls(id)
		gui:MakeScrollable(id)
		
		gui:AddControl(id, "Header", 0, _TRANS("TUJ_Interface_Options"))
		gui:AddControl(id, "Note", 0, 1, nil, nil, " ")
		
		gui:AddControl(id, "Checkbox", 0, 1, "stat.TUJ.disableOriginalTooltip", _TRANS("TUJ_Interface_DisableOriginalTooltip"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_DisableOriginalTooltip"))
		gui:AddControl(id, "Note", 0, 1, nil, nil, " ")
		
		gui:AddControl(id, "Checkbox", 0, 1, "stat.TUJ.enable", _TRANS("TUJ_Interface_Enable"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_Enable"))
		
		gui:AddControl(id, "Subhead", 0, _TRANS("TUJ_Interface_Tooltip"))
		gui:AddControl(id, "Checkbox", 0, 4, "stat.TUJ.tooltip", _TRANS("TUJ_Interface_ToggleTooltip"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleTooltip"))
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.multiplyStackSize", _TRANS("TUJ_Interface_ToggleMultiplyStackSize"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleMultiplyStackSize"))
		gui:AddControl(id, "Note", 0, 1, nil, nil, " ")
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.custom", _TRANS("TUJ_Interface_ToggleCustom"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleCustom"))
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.recent", _TRANS("TUJ_Interface_ToggleRecent"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleRecent"))
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.market", _TRANS("TUJ_Interface_ToggleMarket"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleMarket"))
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.stddev", _TRANS("TUJ_Interface_ToggleStdDev"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleStdDev"))
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.globalMedian", _TRANS("TUJ_Interface_ToggleGlobalMedian"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleGlobalMedian"))
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.globalMean", _TRANS("TUJ_Interface_ToggleGlobalMean"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleGlobalMean"))
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.globalStdDev", _TRANS("TUJ_Interface_ToggleGlobalStdDev"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleGlobalStdDev"))
		gui:AddControl(id, "Checkbox", 0, 7, "stat.TUJ.days", _TRANS("TUJ_Interface_ToggleDays"))
		gui:AddTip(id, _TRANS("TUJ_HelpTooltip_ToggleDays"))
	end
	
	private.addTooltipControls(tab)
	
	gui:AddControl(tab, "Subhead", 0, _TRANS("TUJ_Interface_Custom"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.includeRecent", _TRANS("TUJ_Interface_IncludeRecent"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_IncludeRecent"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.includeMarket", _TRANS("TUJ_Interface_IncludeMarket"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_IncludeMarket"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.includeGlobalMedian", _TRANS("TUJ_Interface_IncludeGlobalMedian"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_IncludeGlobalMedian"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.includeGlobalMean", _TRANS("TUJ_Interface_IncludeGlobalMean"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_IncludeGlobalMean"))
	
	gui:AddControl(tab, "Subhead", 0, _TRANS("TUJ_Interface_Filters"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.limitRecentDays", _TRANS("TUJ_Interface_LimitRecentDays"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_LimitRecentDays"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.limitMarketDays", _TRANS("TUJ_Interface_LimitMarketDays"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_LimitMarketDays"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.includeStdDev", _TRANS("TUJ_Interface_IncludeStdDev"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_IncludeStdDev"))
	gui:AddControl(tab, "WideSlider", 0, 6, "stat.TUJ.thresholdStdDev", 0, 100, 1, _TRANS("TUJ_Interface_ThresholdStdDev"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.includeGlobalStdDev", _TRANS("TUJ_Interface_IncludeGlobalStdDev"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_IncludeGlobalStdDev"))
	gui:AddControl(tab, "WideSlider", 0, 6, "stat.TUJ.thresholdGlobalStdDev", 0, 100, 1, _TRANS("TUJ_Interface_ThresholdGlobalStdDev"))
	gui:AddControl(tab, "Checkbox", 0, 4, "stat.TUJ.excludeVendorItems", _TRANS("TUJ_Interface_ExcludeVendorItems"))
	gui:AddTip(tab, _TRANS("TUJ_HelpTooltip_ExcludeVendorItems"))
	gui:AddControl(tab, "Note", 0, 1, nil, nil, " ")
	
	local tooltipID = AucAdvanced.Settings.Gui.tooltipID
	
	if tooltipID then
		private.addTooltipControls(tooltipID)
		gui:AddControl(tooltipID, "Note", 0, 1, nil, nil, " ")
	end
end

function private.ProcessTooltip(tooltip, hyperlink, serverKey, quantity, decoded, additional, order)
	if TUJTooltip then
		if get("stat.TUJ.disableOriginalTooltip") then
			TUJTooltip(false)
		else
			TUJTooltip(true)
		end
	end
	
	if not get("stat.TUJ.enable") then return end
	if not get("stat.TUJ.tooltip") then return end
	if not private.GetInfo(hyperlink, serverKey) then return end
	
	if get("stat.TUJ.multiplyStackSize") then
		if not quantity or quantity < 1 then
			quantity = 1
		end
		
		tooltip:AddLine(_TRANS("TUJ_Tooltip_StackSize"):format(quantity), R_1, G_1, B_1)
	else
		quantity = 1
	end
	
	if TUJdata.age then
		if TUJdata.age > seconds3Days then
			local age1 = TUJdata.age / secondsDay
			local age2 = math.floor((TUJdata.age / secondsDay - math_floor(TUJdata.age / secondsDay)) * 24)
			
			if age2 ~= 1 then
				if age2 ~= 0 then
					tooltip:AddLine(_TRANS("TUJ_Tooltip_Age1"):format(age1, age2), R_1, G_1, B_1)
				else
					tooltip:AddLine(_TRANS("TUJ_Tooltip_Age3"):format(age1), R_1, G_1, B_1)
				end
			else
				tooltip:AddLine(_TRANS("TUJ_Tooltip_Age2"):format(age1, age2), R_1, G_1, B_1)
			end
		end
	end
	
	if get("stat.TUJ.custom") and library.GetPriceArray(hyperlink, serverKey) then
		tooltip:AddLine(_TRANS("TUJ_Tooltip_Custom"), customPrice * quantity)
	end
	
	if get("stat.TUJ.recent") and TUJdata.recent then
		tooltip:AddLine(_TRANS("TUJ_Tooltip_Recent"), TUJdata.recent * quantity)
	end
	
	if get("stat.TUJ.market") and TUJdata.market then
		tooltip:AddLine(_TRANS("TUJ_Tooltip_Market"), TUJdata.market * quantity)
	end
	
	if get("stat.TUJ.stddev") and TUJdata.stddev then
		tooltip:AddLine(_TRANS("TUJ_Tooltip_StdDev"), TUJdata.stddev * quantity, R_2, G_2, B_2)
	end
	
	if get("stat.TUJ.globalMedian") and TUJdata.globalMedian then
		tooltip:AddLine(_TRANS("TUJ_Tooltip_GlobalMedian"), TUJdata.globalMedian * quantity)
	end
	
	if get("stat.TUJ.globalMean") and TUJdata.globalMean then
		tooltip:AddLine(_TRANS("TUJ_Tooltip_GlobalMean"), TUJdata.globalMean * quantity)
	end
	
	if get("stat.TUJ.globalStdDev") and TUJdata.globalStdDev then
		tooltip:AddLine(_TRANS("TUJ_Tooltip_GlobalStdDev"), TUJdata.globalStdDev * quantity, R_2, G_2, B_2)
	end
	
	if get("stat.TUJ.days") and TUJdata.days then
		if TUJdata.days == 255 then
			tooltip:AddLine(_TRANS("TUJ_Tooltip_NotSeen"), R_1, G_1, B_1)
		elseif TUJdata.days == 252 then
			tooltip:AddLine(_TRANS("TUJ_Tooltip_VendorItems"), R_1, G_1, B_1)
		elseif TUJdata.days > 250 then
			tooltip:AddLine(_TRANS("TUJ_Tooltip_LastSeen250"), R_1, G_1, B_1)
		elseif TUJdata.days > 1 then
			tooltip:AddLine(_TRANS("TUJ_Tooltip_Days"):format(TUJdata.days), R_1, G_1, B_1)
		elseif TUJdata.days == 1 then
			tooltip:AddLine(_TRANS("TUJ_Tooltip_Day"), R_1, G_1, B_1)
		end
	end
end
