﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

-------------------------------------------------------------------------------

Unit Skinning Options: Role

class options =
{
	string orient				-- location of the first role icon on the bar: LEFT (grows right), RIGHT (grows left), TOP (grows down), BOTTOM (grows up)
	string strata				-- the frame strata to use (BACKGROUND, LOW, HIGH, etc.) or nil to use the parent frame's strata
	float size					-- height and width of each role icon
	int level					-- the frame level to use (1-9) or nil to use the parent frame's level+1
	
	class border				-- sets the frame's border and background or nil for no border/background at all
	{
		class backdrop			-- description of the backdrop itself
		{
			string bgFile		-- path to the frame background texture or nil for no background
			string edgeFile		-- path to the frame edge texture file or nil for no frame edge
			boolean tile		-- true if the background texture should be tiled to fill the frame
			float tileSize		-- size of each background tile
			float edgeSize		-- thickness of the edge texture around the frame
			
			class insets = 
			{	
				int left		-- inset of background texture from the outside edges of the frame
				int right
				int top
				int bottom
			};
		};
		
		class color
		{
			class border = { r, g, b, a }	-- color and transparency of the edge texture
			class backdrop = { r, g, b, a }	-- color and transparency of the background texture
		};
	};
}

--]]---------------------------------------------------------------------------

if not nUI then nUI = {}; end
if not nUI_Unit then nUI_Unit = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local GetTime   = GetTime;

-------------------------------------------------------------------------------
-- unit role event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

local frame = CreateFrame( "Frame", "$parent_Role", nUI_Unit.Drivers )

local RoleCallbacks    = {};
local RoleUnits        = {};
local LeaderStatus     = {};
local RaidRoleStatus   = {};
local MLStatus         = {};

nUI_Unit.Drivers.Role  = frame;

-------------------------------------------------------------------------------

local function onRoleEvent()
	
	nUI_Unit:refreshRoleCallbacks();

end

frame:SetScript( "OnEvent", onRoleEvent );
frame:RegisterEvent( "PLAYER_ENTERING_WORLD" );

-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for unit role changes GUID

frame.newUnitInfo = function( list_unit, unit_info )

	local new_data  = nUI_Unit:updateRoleInfo( list_unit, unit_info );
	local callbacks = RoleCallbacks;
	local unitlist  = RoleUnits;
	
	nUI_Unit:notifyCallbacks( nUI_L["unit role"], callbacks, unitlist, unit_info, list_unit, new_data );
	
end

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of unit role listeners we manage
--
-- calling this method will return the current unit_info structure for this 
-- unit if it exists or nil if the unit does not exist at this time
--
-- Note: these callbacks will be notified both when the underlying GUID for the
--		 unit changes or when the role info of the underlying GUID to the
--		 player changes. If the underlying unit does not exist, the callback
--		 will be passed a nil unit_info structure

function nUI_Unit:registerRoleCallback( unit_id, callback )
	
	local unit_info = nil;
	
	if unit_id and callback then
		
		-- get the list of callbacks for this unit id and add this callback
		
		local list = RoleCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );
		
		-- if this is a new unit id, add it to the callback list
		
		if not RoleCallbacks[unit_id] then
			RoleCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for the unit id, then register our
		-- event driver to receive notice when the GUID changes on this id
		
		if #list == 1 then
			nUI_Unit:registerStatusCallback( unit_id, nUI_Unit.Drivers.Role );
		end
		
		-- collect role information for this unit as we know it at this time
	
		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			nUI_Unit:updateRoleInfo( unit_id, unit_info );
		end
	end
	
	return unit_info;
	
end

function nUI_Unit:unregisterRoleCallback( unit_id, callback )
	
	if unit_id and callback then
		
		-- get the list of current callbacks for this unit ud and remove this callback
		
		local list = RoleCallbacks[unit_id] or {};
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that's the last callback in the list, then remove our event handler of
		-- the list of unit change callbacks for that unit it
		
		if #list == 0 then
			nUI_Unit:unregisterStatusCallback( unit_id, nUI_Unit.Drivers.Role );
			nUI_Unit:unregisterRaidGroupCallback( unit_id, nUI_Unit.Drivers.Role );
		end
	end
end

-------------------------------------------------------------------------------
-- update the role information for this unit
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current GUID
-- if the data has changed, otherwise returns nil if nothing changed

function nUI_Unit:updateRoleInfo( unit_id, unit_info )

	local modified  = false;
	
--	nUI:debug( "nUI_UnitRole: checking "..unit_id..", unit_info = "..( unit_info and unit_info.name or "<nil>"), 1 );
	
	if unit_info then

		-- if the unit is not in a party or a raid, then there is no role
		
		if not unit_info.in_party and not unit_info.in_raid then
			
			if unit_info.role_info then
					
--				nUI:debug( "nUI_UnitRole: "..unit_id.." is no longer in a party or raid", 1 );
				
				modified = true;
				unit_info.modified = true;
				unit_info.last_change = GetTime();
				unit_info.role_info   = nil;
			end
			
		-- otherwise see what the role is, if any
		
		else

			local role;
			local role_info  = unit_info.role_info or {};
			local raid_info  = unit_info.raid_info;
			local in_raid    = unit_info.in_raid and raid_info or false;
			local is_leader  = unit_info.status_info and unit_info.status_info.is_leader or false;
			local is_assist  = in_raid and raid_info.rank == 1 or false;
			local is_tank    = in_raid and raid_info.role == "maintank" or false;
			local is_offtank = in_raid and raid_info.role == "mainassist" or false;
			local is_ml      = in_raid and raid_info.is_ml or false;
			
			if is_leader then
				role = ("%s%s%s"):format( role or "", role and "," or "", unit_info.in_raid and nUI_L["Raid Leader"] or nUI_L["Party Leader"] );
			end
			
			if is_assist then
				role = ("%s%s%s"):format( role or "", role and "," or "", nUI_L["Raid Assistant"] );
			end
			
			if is_tank then
				role = ("%s%s%s"):format( role or "", role and "," or "", nUI_L["Main Tank"] );
			end
			
			if is_offtank then
				role = ("%s%s%s"):format( role or "", role and "," or "", nUI_L["Off-Tank"] );
			end
			
			if is_ml then
				role = ("%s%s%s"):format( role or "", role and "," or "", nUI_L["Master Looter"] );
			end
			
			
--			nUI:debug( "nUI_UnitRole: "..unit_id.." role status check: leader="..(is_leader and "true" or "false")..", assist="..(is_assist and "true" or "false")..", tank="..(is_tank and "true" or "false")..", offtank="..(is_offtank and "true" or "false")..", ml="..(is_ml and "true" or "false"), 1 );
			
			if role_info.is_leader  ~= is_leader
			or role_info.is_assist  ~= is_assist
			or role_info.is_tank    ~= is_tank
			or role_info.is_offtank ~= is_offtank
			or role_info.is_ml      ~= is_ml
			or role_info.role       ~= role
			then
				
				role_info.role       = role;
				role_info.is_leader  = is_leader;
				role_info.is_assist  = is_assist;
				role_info.is_tank    = is_tank;
				role_info.is_offtank = is_offtank;
				role_info.is_ml      = is_ml;
				
				modified              = true;
				unit_info.modified    = true;
				unit_info.last_change = GetTime();
				unit_info.role_info   = role_info;
	
			end
		end
	end
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all of the registered unit role listeners, even if there's no 
-- change in data... typically used when  entering the world

function nUI_Unit:refreshRoleCallbacks()

	nUI_Unit:refreshCallbacks( 
	
		nUI_L["unit role"], RoleCallbacks, RoleUnits, 
	
		function( list_unit, unit_info ) 
			nUI_Unit:updateRoleInfo( list_unit, unit_info ); 
		end 
	);

end

-------------------------------------------------------------------------------
-- create a new unit role frame

function nUI_Unit:createRoleFrame( parent, unit_id, id, options, clickable )

	local frame   = nUI_Unit:createFrame( "$parent_Role"..(id or ""), parent, unit_id, false );	
	frame.leader  = frame:CreateTexture( "$parentLeader" );
	frame.assist  = frame:CreateTexture( "$parentAssistant" );
	frame.tank    = frame:CreateTexture( "$parentTank" );
	frame.offtank = frame:CreateTexture( "$parentOffTank" );
	frame.ml      = frame:CreateTexture( "$parentMasterLooter" );
	frame.Super   = {};
	
	frame.leader:SetTexture( "Interface\\GroupFrame\\UI-Group-LeaderIcon" );
	frame.leader:SetTexCoord( 0, 0, 0, 1, 1, 0, 1, 1 );
	frame.leader:SetAlpha( 0 );
	
	frame.assist:SetTexture( "Interface\\GroupFrame\\UI-GROUP-ASSISTANTICON" );
	frame.assist:SetTexCoord( 0, 0, 0, 1, 1, 0, 1, 1 );
	frame.assist:SetAlpha( 0 );
	
	frame.tank:SetTexture( "Interface\\GroupFrame\\UI-GROUP-MAINTANKICON" );
	frame.tank:SetTexCoord( 0, 0, 0, 1, 1, 0, 1, 1 );
	frame.tank:SetAlpha( 0 );
	
	frame.offtank:SetTexture( "Interface\\GroupFrame\\UI-GROUP-MAINASSISTICON" );
	frame.offtank:SetTexCoord( 0, 0, 0, 1, 1, 0, 1, 1 );
	frame.offtank:SetAlpha( 0 );
	
	frame.ml:SetTexture( "Interface\\GroupFrame\\UI-Group-MasterLooter" );
	frame.ml:SetTexCoord( 0, 0, 0, 1, 1, 0, 1, 1 );
	frame.ml:SetAlpha( 0 );
		
	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated

	frame.Super.newUnitInfo = frame.newUnitInfo;
	frame.newUnitInfo       = function( list_unit, unit_info )
		
		frame.Super.newUnitInfo( list_unit, unit_info );
		
		if frame.enabled then
			nUI_Unit:updateRoleFrame( frame );
		end
		
	end
	
	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled

	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
		local prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				frame.unit_info = nUI_Unit:registerRoleCallback( frame.unit, frame );
				nUI_Unit:updateRoleFrame( frame );
			else
				nUI_Unit:unregisterRoleCallback( frame.unit, frame );
			end
		end
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame.

	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

		frame.Super.applyScale( scale );

		local icon_size = frame.options.icon_size * frame.scale * nUI.scale;
		
		if frame.leader.icon_size ~= icon_size 
		or frame.leader.inset    ~= frame.inset
		then
			
			frame.leader.icon_size = icon_size;
			frame.leader.inset     = frame.inset;
			
			frame.leader:SetWidth( icon_size - frame.inset );
			frame.leader:SetHeight( icon_size - frame.inset );
			
			frame.assist:SetWidth( icon_size - frame.inset );
			frame.assist:SetHeight( icon_size - frame.inset );
			
			frame.tank:SetWidth( icon_size - frame.inset );
			frame.tank:SetHeight( icon_size - frame.inset );
			
			frame.offtank:SetWidth( icon_size - frame.inset );
			frame.offtank:SetHeight( icon_size - frame.inset );
			
			frame.ml:SetWidth( icon_size - frame.inset );
			frame.ml:SetHeight( icon_size - frame.inset );
			
		end		

		-- determine how many icons are active at this time
		
		local icons  = 0;
		
		if frame.leader.active  then icons = icons+1; end			
		if frame.assist.active  then icons = icons+1; end
		if frame.tank.active    then icons = icons+1; end
		if frame.offtank.active then icons = icons+1; end
		if frame.ml.active      then icons = icons+1; end

		-- resize the main frame
			
		local length = icon_size * icons;
			
		if frame.horizontal then				
			frame:SetWidth( length );
		else
			frame:SetHeight( length );
		end
		
	end
	
	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.
	
	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

		-- set up the frame orientation
		
		if options.orient == "TOP" then
			frame.anchor_pt1 = "TOP";
			frame.anchor_pt2 = "BOTTOM";
			frame.horizontal = false;
		elseif options.orient == "BOTTOM" then
			frame.anchor_pt1 = "BOTTOM";
			frame.anchor_pt2 = "TOP";
			frame.horizontal = false;
		elseif options.orient == "RIGHT" then
			frame.anchor_pt1 = "RIGHT";
			frame.anchor_pt2 = "LEFT";
			frame.horizontal = true;
		else
			frame.anchor_pt1 = "LEFT";
			frame.anchor_pt2 = "RIGHT";
			frame.horizontal = true;
		end

--		nUI:debug( "nUI_UnitRole: set anchor_pt1 = "..frame.anchor_pt1..", anchor_pt = "..frame.anchor_pt2, 1 );

		frame.Super.applyOptions( options );

		-- and refresh the frame
		
		nUI_Unit:updateRoleFrame( frame );
		
	end

	-- initiate the frame
	
	frame.unit_info = nUI_Unit:registerRoleCallback( frame.unit, frame );
	
	frame.applyOptions( options );
	
	return frame;
	
end

-------------------------------------------------------------------------------
-- remove a unit role frame

function nUI_Unit:deleteRoleFrame( frame )

	nUI_Unit:unregisterRoleCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
end

-------------------------------------------------------------------------------
-- display the appropriate icon for the unit's role
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updateRoleFrame( frame )
	
	local unit_info = frame.unit_info;
	
	-- if there is no unit or we don't know it's role, then hide the icon
	
	if not frame.options or not unit_info or not unit_info.role_info then
		
		if frame.active then

--			nUI:debug( "hiding role panel for "..frame.unit.." -- no unit or no role", 1 );
			
			frame.active = false;
			frame:SetAlpha( 0 );
			frame:EnableMouse( false );
			frame:SetScript( "OnEnter", nil );
			frame:SetScript( "OnLeave", nil );
			
		end
	
	-- otherwise, show the icons as required
	
	else

		-- which icons are we going to show?
		
		local role_info  = unit_info.role_info;
		local role       = role_info.role;
		local is_leader  = role_info.is_leader;
		local is_assist  = role_info.is_assist;
		local is_tank    = role_info.is_tank;
		local is_offtank = role_info.is_offtank;
		local is_ml      = role_info.is_ml;
		local shown      = {};
		local hidden     = {};
		
		-- if there has been a change in role, then we need to update the role icons
		
		if frame.is_leader  ~= is_leader
		or frame.is_assist  ~= is_assist
		or frame.is_tank    ~= is_tank
		or frame.is_offtank ~= is_offtank
		or frame.is_ml      ~= is_ml
		or frame.role       ~= role
		then

			frame.is_leader  = is_leader;
			frame.is_assist  = is_assist;
			frame.is_tank    = is_tank;
			frame.is_offtank = is_offtank;
			frame.is_ml      = is_ml;
			frame.role       = role;
			
			if is_leader then
				nUI:TableInsertByValue( shown, frame.leader );
			else
				nUI:TableInsertByValue( hidden, frame.leader );
			end
			
			if is_assist then
				nUI:TableInsertByValue( shown, frame.assist );
			else
				nUI:TableInsertByValue( hidden, frame.assist );
			end
			
			if is_tank then
				nUI:TableInsertByValue( shown, frame.tank );
			else
				nUI:TableInsertByValue( hidden, frame.tank );
			end
			
			if is_offtank then
				nUI:TableInsertByValue( shown, frame.offtank );
			else
				nUI:TableInsertByValue( hidden, frame.offtank );
			end
			
			if is_ml then
				nUI:TableInsertByValue( shown, frame.ml );
			else
				nUI:TableInsertByValue( hidden, frame.ml );
			end
			
			-- make sure all of the inactive icons are hidden
			
			for i=1,#hidden do
				
				local texture = hidden[i];
				
				if texture.active then
					
					texture.active = false;
					texture:ClearAllPoints();
					texture:SetAlpha( 0 );
					
				end
			end
			
			-- if there are active icons, then make sure the frame is visible and
			-- make sure we have tooltip support enabled and size the frame
			
			if #shown > 0 then
				
				-- make sure all active icons are shown
			
				local last_icon;
			
				for i=1,#shown do
				
					local texture = shown[i];
				
					if not texture.active then
					
						texture.active = true;
						texture:SetAlpha( 1 );
					
					end
				
					local xOfs = last_icon and horizontal and (frame.size * 0.1 * (frame.options.orient == "LEFT" and 1 or -1)) or 0;
					local yOfs = last_icon and vertical and (frame.size * 0.1 * (frame.options.orient == "TOP" and -1 or 1)) or 0;
				
--					nUI:debug( "nUI_UnitRole: setting "..texture:GetName().." icon #"..i.." anchor for "..frame.unit.." at ( "..(frame.anchor_pt1 or "<nil>")..", "..(last_icon and last_icon:GetName() or frame:GetName())..", "..(frame.anchor_pt2 or frame.anchor_pt1 or "<nil>")..", "..xOfs..", "..yOfs.." )", 1 );
				
					texture:ClearAllPoints();
					texture:SetPoint( frame.anchor_pt1, last_icon or frame, last_icon and frame.anchor_pt2 or frame.anchor_pt1, xOfs, yOfs );
				
				end

				if not frame.active then
					
					frame.active = true;
					frame:SetAlpha( 1 );
					frame:EnableMouse( true );
					
					frame:SetScript( "OnEnter",					
						function()
							GameTooltip:SetOwner( frame );
							GameTooltip:SetText( (unit_info.in_raid and nUI_L["Raid Role: |cFF00FFFF%s|r"] or nUI_L["Party Role: |cFF00FFFF%s|r"]):format( frame.role ), 1, 1, 1 );
							GameTooltip:Show();
						end
					);
					
					frame:SetScript( "OnLeave",
						function()
							GameTooltip:Hide();
						end
					);
					
				end
				
				if frame.horizontal then
					frame:SetWidth( frame.size * #shown );
				else
					frame:SetHeight( frame.size * #shown );
				end
					
			-- otherwise, make sure the frame is hidden and there are no tooltips
			
			elseif frame.active then

--				nUI:debug( "nUI_UnitRole: hiding role icons -- none apply for "..frame.unit, 1 );
				
				frame.active = false;
				frame:EnableMouse( false );
				frame:SetAlpha( 0 );
				frame:SetScript( "OnEnter", nil );
				frame:SetScript( "OnLeave", nil );
				
			end
		end		
	end
end
