﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

-------------------------------------------------------------------------------

Unit Skinning Options: PvP

class options =
{
	string strata				-- the frame strata to use (BACKGROUND, LOW, HIGH, etc.) or nil to use the parent frame's strata
	float size					-- height and width of the frame
	int level					-- the frame level to use for the aura icons (1-9) or nil to use the parent frame's level+1
	
	class border				-- sets the frame's border and background or nil for no border/background at all
	{
		class backdrop			-- description of the backdrop itself
		{
			string bgFile		-- path to the frame background texture or nil for no background
			string edgeFile		-- path to the frame edge texture file or nil for no frame edge
			boolean tile		-- true if the background texture should be tiled to fill the frame
			float tileSize		-- size of each background tile
			float edgeSize		-- thickness of the edge texture around the frame
			
			class insets = 
			{	
				int left		-- inset of background texture from the outside edges of the frame
				int right
				int top
				int bottom
			};
		};
		
		class color
		{
			class border = { r, g, b, a }	-- color and transparency of the edge texture
			class backdrop = { r, g, b, a }	-- color and transparency of the background texture
		};
	};
}

--]]---------------------------------------------------------------------------

if not nUI_Unit then nUI_Unit = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local GetTime   = GetTime;

-------------------------------------------------------------------------------
-- unit PvP event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

local frame = CreateFrame( "Frame", "$parent_PvP", nUI_Unit.Drivers )

local PvPCallbacks    = {};
local PvPUnits        = {};

nUI_Unit.Drivers.PvP  = frame;

-------------------------------------------------------------------------------

local function onPvPEvent()
	
	-- for these events, we don't know which units are affected, so
	-- we span the list of all known interested units to see who is watching
		
	if event == "PLAYER_ENTERING_WORLD" then
		
		nUI_Unit:refreshPvPCallbacks();

	end
end

frame:SetScript( "OnEvent", onPvPEvent );
frame:RegisterEvent( "PLAYER_ENTERING_WORLD" );

-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for PvP status changes

frame.newUnitInfo = function( list_unit, unit_info )

	local new_data  = nUI_Unit:updatePvPInfo( list_unit, unit_info );
	local callbacks = PvPCallbacks;
	local unitlist  = PvPUnits;
	
	nUI_Unit:notifyCallbacks( nUI_L["unit PvP"], callbacks, unitlist, unit_info, list_unit, new_data );
	
end

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of PvP status listeners we manage
--
-- calling this method will return the current unit_info structure for this 
-- unit if it exists or nil if the unit does not exist at this time
--
-- Note: these callbacks will be notified both when the underlying GUID for the
--		 unit changes or when the PvP status info of the underlying GUID to the
--		 player changes. If the underlying unit does not exist, the callback
--		 will be passed a nil unit_info structure

function nUI_Unit:registerPvPCallback( unit_id, callback )
	
	local unit_info = nil;
	
	if unit_id and callback then
		
		-- get the list of callbacks for this unit id and add this callback
		
		local list = PvPCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );
		
		-- if this is a new unit id, add it to the callback list
		
		if not PvPCallbacks[unit_id] then
			PvPCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for the unit id, then register our
		-- event driver to receive notice when the GUID changes on this id
		-- or the unit reaction updates (flags PvP or announces faction)
		
		if #list == 1 then
			nUI_Unit:registerReactionCallback( unit_id, nUI_Unit.Drivers.PvP );
		end
		
		-- collect PvP status information for this unit as we know it at this time
	
		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			nUI_Unit:updatePvPInfo( unit_id, unit_info );
		end
	end
	
	return unit_info;
	
end

function nUI_Unit:unregisterPvPCallback( unit_id, callback )
	
	if unit_id and callback then
		
		-- get the list of current callbacks for this unit ud and remove this callback
		
		local list = PvPCallbacks[unit_id] or {};
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that's the last callback in the list, then remove our event handler of
		-- the list of unit change callbacks for that unit it
		
		if #list == 0 then
			nUI_Unit:unregisterReactionCallback( unit_id, nUI_Unit.Drivers.PvP );
		end
	end
end

-------------------------------------------------------------------------------
-- update the PvP status information for this unit
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current GUID
-- if the data has changed, otherwise returns nil if nothing changed

function nUI_Unit:updatePvPInfo( unit_id, unit_info )

	local modified  = false;
	
	if unit_info then
		
		local pvp_icon;
		local pvp_tt_title;
		local pvp_tt_text;

		-- determine what the PvP icon and game tooltip should be
		
		if unit_info.is_ffa then
			
			pvp_icon     = "Interface\\TargetingFrame\\UI-PVP-FFA";
			pvp_tt_title = PVPFFA;
			pvp_tt_text  = NEWBIE_TOOLTIP_PVPFFA;
			
		elseif unit_info.is_pvp and unit_info.faction then
			
			local group   = unit_info.faction;
			local faction = unit_info.faction_name;
			
			pvp_icon     = "Interface\\TargetingFrame\\UI-PVP-"..unit_info.faction;
			pvp_tt_title = faction;
			pvp_tt_text  = getglobal( "NEWBIE_TOOLTIP_"..strupper(unit_info.faction) );
			
		end
		
		-- if it has changed from what we know, then update the cache
		
		if unit_info.pvp_icon     ~= pvp_icon
		or unit_info.pvp_tt_title ~= pvp_tt_title
		or unit_info.pvp_tt_text  ~= pvp_tt_text
		then
			
			modified               = true;
			unit_info.modified     = true;
			unit_info.last_change  = GetTime();
			unit_info.pvp_icon     = pvp_icon;
			unit_info.pvp_tt_title = pvp_tt_title;
			unit_info.pvp_tt_text  = pvp_tt_text;
		end
	end
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all of the registered PvP status listeners, even if there's no 
-- change in data... typically used when entering the world

function nUI_Unit:refreshPvPCallbacks()

	nUI_Unit:refreshCallbacks( 
	
		nUI_L["unit PvP"], PvPCallbacks, PvPUnits, 
	
		function( list_unit, unit_info ) 
			nUI_Unit:updatePvPInfo( list_unit, unit_info ); 
		end 
	);

end

-------------------------------------------------------------------------------
-- create a new PvP status frame

function nUI_Unit:createPvPFrame( parent, unit_id, id, options )

	local frame   = nUI_Unit:createFrame( "$parent_PvP"..(id or ""), parent, unit_id, false );
	frame.texture = frame:CreateTexture( "$parentTexture" );
	frame.Super   = {};
	
	frame.texture:SetPoint( "CENTER", frame, "CENTER", 0, 0 );

	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated

	frame.Super.newUnitInfo = frame.newUnitInfo;
	frame.newUnitInfo       = function( list_unit, unit_info )
		
		frame.Super.newUnitInfo( list_unit, unit_info );
		
		if frame.enabled then
			nUI_Unit:updatePvPFrame( frame );
		end
		
	end
	
	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled
	
	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
		local prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				frame.unit_info = nUI_Unit:registerPvPCallback( frame.unit, frame );
				nUI_Unit:updatePvPFrame( frame );
			else
				nUI_Unit:unregisterPvPCallback( frame.unit, frame );
			end
		end
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame. Is also recreates
	-- the font to present clear, sharp, readable text versus the blurred text you get
	-- as a result of frame:SetScale() or text:SetTextHeight()

	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

		frame.Super.applyScale( scale );
		if frame.texture.size   ~= frame.size 
		or frame.texture.width  ~= frame.width
		or frame.texture.height ~= frame.height 
		or frame.texture.inset  ~= frame.inset
		then
			
			frame.texture.size   = frame.size;
			frame.texture.width  = frame.width;
			frame.texture.height = frame.height;
			frame.texture.inset  = frame.inset;
			
			frame.texture:SetWidth( (frame.size or frame.width) - frame.inset );
			frame.texture:SetHeight( (frame.size or frame.height) - frame.inset );

		end				
	end
	
	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.

	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

		frame.Super.applyOptions( options );
		nUI_Unit:updatePvPFrame( frame );
		
	end

	-- initiate the frame
	
	frame.unit_info = nUI_Unit:registerPvPCallback( frame.unit, frame );
		
	frame.applyOptions( options );
	
	return frame;
	
end

-------------------------------------------------------------------------------
-- remove a PvP status frame

function nUI_Unit:deletePvPFrame( frame )

	nUI_Unit:unregisterPvPCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
end

-------------------------------------------------------------------------------
-- display the appropriate icon for the unit's PvP status
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updatePvPFrame( frame )
	
	local unit_info = frame.unit_info;
	
	-- if there is no unit or the unit is not flagged PvP, then hide the icon
	
	if not unit_info or (not unit_info.is_pvp and not unit_info.is_ffa) then
		
		if frame.active then
			
			frame.active = false;
			frame.texture:SetAlpha( 0 );
			
			frame:EnableMouse( false );
			frame:SetScript( "OnEnter", nil );
			frame:SetScript( "OnLeave", nil );
			
		end
	
	-- otherwise, show the icon and clip the appropriate region
	
	else

		-- if the PvP status is hidden, show it
		
		if not frame.active then
			
			frame.active = true;
			frame.texture:SetAlpha( 1 );
			
			frame:EnableMouse( true );
			
			frame:SetScript( "OnEnter",
				function()
					if unit_info.pvp_tt_title and unit_info.pvp_tt_text then
						GameTooltip_AddNewbieTip( unit_info.pvp_tt_title, 1.0, 1.0, 1.0, unit_info.pvp_tt_text, 1);					
					end
				end
			);
			
			frame:SetScript( "OnLeave", 
				function()
					GameTooltip:Hide();
				end
			);
			
		end

		-- if the PvP status has changed from what we last knew, then update 
		
		if frame.pvp_icon ~= unit_info.pvp_icon then
			
			frame.pvp_icon = unit_info.pvp_icon;
			
			frame.texture:SetTexture( frame.pvp_icon );
			frame.texture:SetTexCoord( 0.03125, 0, 0.03125, 0.59375, 0.625, 0, 0.625, 0.59375 );
			
		end
	end
end
