﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

-------------------------------------------------------------------------------

Unit Skinning Options: Health

class options =
{
	string orient				-- location of the zero value on the bar: LEFT (grows right), RIGHT (grows left), TOP (grows down), BOTTOM (grows up)
	string overlay				-- path to the overlay texture (tranparency) to use on the bar or nil for a flat bar
	string strata				-- the frame strata to use (BACKGROUND, LOW, HIGH, etc.) or nil to use the parent frame's strata
	float height				-- height of the bar
	float width					-- width of the bar
	int level					-- the frame level to use (1-9) or nil to use the parent frame's level+1
	boolean show_bar			-- displays the graphic bar when true -- setting false allows for text only displays
	
	class cur_health			-- enables current health as text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor_pt		-- anchor point on the text to use as the origin: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		string relative_to		-- frame to anchor the text relative to (default is this frame)
		string relative_pt		-- anchor point on the anchor frame to use as the origin: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class max_health			-- enables maximum health as text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor_pt		-- anchor point on the text to use as the origin: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		string relative_to		-- frame to anchor the text relative to (default is this frame)
		string relative_pt		-- anchor point on the anchor frame to use as the origin: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class pct_health			-- enables current health as a percent of maximum health text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor_pt		-- anchor point on the text to use as the origin: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		string relative_to		-- frame to anchor the text relative to (default is this frame)
		string relative_pt		-- anchor point on the anchor frame to use as the origin: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class mix_health			-- enables combines "current / maximum" health as a text display when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 70%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor_pt		-- anchor point on the text to use as the origin: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		string relative_to		-- frame to anchor the text relative to (default is this frame)
		string relative_pt		-- anchor point on the anchor frame to use as the origin: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class border				-- sets the frame's border and background or nil for no border/background at all
	{
		class backdrop			-- description of the backdrop itself
		{
			string bgFile		-- path to the frame background texture or nil for no background
			string edgeFile		-- path to the frame edge texture file or nil for no frame edge
			boolean tile		-- true if the background texture should be tiled to fill the frame
			float tileSize		-- size of each background tile
			float edgeSize		-- thickness of the edge texture around the frame
			
			class insets = 
			{	
				int left		-- inset of background texture from the outside edges of the frame
				int right
				int top
				int bottom
			};
		};
		
		class color
		{
			class border = { r, g, b, a }	-- color and transparency of the edge texture
			class backdrop = { r, g, b, a }	-- color and transparency of the background texture
		};
	};
}

--]]---------------------------------------------------------------------------

if not ClickCastFrames then	ClickCastFrames = {};end
if not nUI then nUI = {}; end
if not nUI_Unit then nUI_Unit = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end
if not nUI_DefaultConfig.BarColors then nUI_DefaultConfig.BarColors = {}; end

local CreateFrame    = CreateFrame;
local IsAddOnLoaded  = IsAddOnLoaded;
local UnitHealth     = UnitHealth;
local UnitHealthMax  = UnitHealthMax;

nUI_UnitOptions.BarColors = nUI_DefaultConfig.BarColors;

-------------------------------------------------------------------------------
-- default colors for unit health bars

nUI_DefaultConfig.BarColors.Health =
{
	["min"] = { r = 1, g = 0, b = 0, a = 1 },	-- empty bar color
	["mid"] = { r = 1, g = 1, b = 0, a = 1 },	-- bar color at 50%
	["max"] = { r = 0, g = 1, b = 0, a = 1 },	-- full bar color
};

-------------------------------------------------------------------------------
-- unit health event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

local frame = CreateFrame( "Frame", "$parent_Health", nUI_Unit.Drivers )

local HealthCallbacks    = {};
local HealthUnits        = {};
local HelperHealthMethod = nil;
local health_timer       = 0;

nUI_Unit.Drivers.Health  = frame;

-------------------------------------------------------------------------------
-- default function to fetch the health and maximum health for a unit using
-- Bliz's rather uninformative health data

local function GetBlizHealth( unit_id, unit_info )
	
	local cur_health;
	local max_health;

	if unit_info.is_visible then
		
		cur_health = UnitHealth( unit_id );
		max_health = UnitHealthMax( unit_id );

	end
	
	return cur_health, max_health;
	
end

local select_health = true;
local GetHealth     = GetBlizHealth;

-------------------------------------------------------------------------------
-- function for interfacing to the MobInfo2 advanced mob health data AddOn

local function GetMI2Health( unit_id, unit_info )
	
	local cur_health;
	local max_health;
	local mi2data;

	-- if the unit is in visible range and not a player controlled unit
	-- then hit MI2 up for the mob's health data
	
    if unit_info.is_visible then
		
		if not unit_info.is_controlled then
			
			mi2data = HelperHealthMethod( unit_info.name, unit_info.level, unit_id );
			
		end
	
		-- if we have data from MI2, use that
		
		if mi2data then
		
			cur_health = mi2data.healthCur or UnitHealth( unit_id );
			max_health = mi2data.healthMax or UnitHealthMax( unit_id );
	
		-- otherwise, get the health data from Bliz
		
		else
		
			cur_health = UnitHealth( unit_id );
			max_health = UnitHealthMax( unit_id );
	
		end
	end
	
	return cur_health, max_health;	
	
end

-------------------------------------------------------------------------------

local function onHealthEvent()
	
	if event == "VARIABLES_LOADED" then
		
		nUI:patchConfig();
		nUI:registerSkinnedFrame( frame );
		
	elseif event == "PLAYER_ENTERING_WORLD" then

		-- if we haven't yet, pick a means of getting mob health data
		
		if select_health then
				
			local got_healper = false;
			local mi2         = IsAddOnLoaded( "MobInfo2" );					
			select_health     = false;
			
			-- check to see if MobInfo2 is available
			
			if not got_helper then
			
				-- if it isn't loaded yet, it may be LOD, see if we can get it loaded
				
				if not mi2 then
						
					LoadAddOn( "MobInfo2" );
					
					mi2 = IsAddOnLoaded( "MobInfo2" );
					
				end
				
				-- if it is loaded, then we'll use it to get our mob health
				
				if mi2 then
				
--					nUI:debug( "nUI_UnitHealth: using MobInfo2 for unit health", 1 );
					
					got_helper         = true;
					HelperHealthMethod = MI2_GetMobData;
					GetHealth          = GetMI2Health;
					
				end
			end
			
			-- if we didn't find any mob health helper addons, then we'll have
			-- to use Bliz's default methods
	
			if not got_helper then

--				nUI:debug( "nUI_UnitHealth: defaulting to Bliz health methods", 1 );
				
				GetHealth = GetBlizHealth;
				
			end

		end
		
		-- span the list of interested listeners to update health data when leaving instances
		
		nUI_Unit:refreshHealthCallbacks();
		
	-- change in health or max health
	
	elseif HealthCallbacks[arg1] and #HealthCallbacks[arg1] > 0 then
		
		frame.newUnitInfo( arg1, nUI_Unit:getUnitInfo( arg1 ) );
		
	end
end

frame:SetScript( "OnEvent", onHealthEvent );
frame:RegisterEvent( "VARIABLES_LOADED" );
frame:RegisterEvent( "PLAYER_ENTERING_WORLD" );
frame:RegisterEvent( "UNIT_HEALTH" );
frame:RegisterEvent( "UNIT_MAXHEALTH" );

-------------------------------------------------------------------------------

local function onHealthUpdate( who, elapsed )
	
	health_timer = health_timer + elapsed;
	
	if health_timer > 0.08 then -- check at 12.5fps
	
		health_timer = 0;
		
		-- we're only going to bother looking at the lists that have at least one listener
			
		for list_unit in pairs( HealthCallbacks ) do			
			if #HealthCallbacks[list_unit] > 0 then				
				frame.newUnitInfo( list_unit, nUI_Unit:getUnitInfo( list_unit ) );
			end
		end				
	end
end

--frame:SetScript( "OnUpdate", onHealthUpdate );

-------------------------------------------------------------------------------
-- unless the skin spefically defined bar colors, we'll default to the nUI set

frame.applySkin = function( skin )

	local skin = skin and skin.BarColors or nUI_DefaultConfig.BarColors;
	
	nUI_UnitOptions.BarColors = skin;
	
end
	
-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for unit health changes GUID or fires an event we care about

frame.newUnitInfo = function( list_unit, unit_info )

	local new_data  = nUI_Unit:updateHealthInfo( list_unit, unit_info );
	local callbacks = HealthCallbacks;
	local unitlist  = HealthUnits;
	
	nUI_Unit:notifyCallbacks( nUI_L["unit health"], callbacks, unitlist, unit_info, list_unit, new_data );
	
end

-------------------------------------------------------------------------------
-- initialize configuration for the unit health bar colors
-- 
-- this method is called when the mod's saved variables have been loaded by Bliz and
-- may be called again whenever the unit bar color configuration has been changed
-- by the player or programmatically. Passing true or a non-nil value for "use_default"
-- will cause the player's current health color configuration to be replaced with
-- the default settings defined at the top of this file (which cannot be undone!)

function nUI_Unit:configHealth( use_default )
	
	if not nUI_UnitOptions then nUI_UnitOptions = {}; end
	if not nUI_UnitOptions.BarColors then nUI_UnitOptions.BarColors = {}; end
	if not nUI_UnitOptions.BarColors.Health then nUI_UnitOptions.BarColors.Health = {}; end
	
	local config  = nUI_UnitOptions.BarColors.Health;
	local default = nUI_DefaultConfig.BarColors.Health;
	
	for range in pairs( default ) do

		local source = default[range];
		local target = config[range] or {};
		
		if use_default then
			
			target.r = source.r;
			target.g = source.g;
			target.b = source.b;
			target.a = source.b;

		else
			
			target.r = tonumber( target.r or source.r );
			target.g = tonumber( target.g or source.g );
			target.b = tonumber( target.b or source.b );
			target.a = tonumber( target.a or source.a );
			
		end
		
		nUI_UnitOptions.BarColors.Health[range] = target;
		
	end
	
	nUI_Unit:refreshHealthCallbacks();
	
end

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of unit health listeners we manage
--
-- calling this method will return the current unit_info structure for this 
-- unit if it exists or nil if the unit does not exist at this time
--
-- Note: these callbacks will be notified both when the underlying GUID for the
--		 unit changes or when the health info of the underlying GUID changes. 
--       If the underlying unit does not exist, the callback will be passed a 
--       nil unit_info structure

function nUI_Unit:registerHealthCallback( unit_id, callback )
	
	local unit_info = nil;
	
	if unit_id and callback then
		
		-- get the list of callbacks for this unit id and add this callback
		
		local list = HealthCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );
		
		-- if this is a new unit id, add it to the callback list
		
		if not HealthCallbacks[unit_id] then
			HealthCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for the unit id, then register our
		-- event driver to receive notice when the GUID changes on this id
		
		if #list == 1 then
			nUI_Unit:registerUnitChangeCallback( unit_id, nUI_Unit.Drivers.Health );
		end
		
		-- collect health information for this unit as we know it at this time
	
		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			nUI_Unit:updateHealthInfo( unit_id, unit_info );
		end
	end
	
	return unit_info;
	
end

function nUI_Unit:unregisterHealthCallback( unit_id, callback )
	
	if unit_id and callback then
		
		-- get the list of current callbacks for this unit ud and remove this callback
		
		local list = HealthCallbacks[unit_id] or {};
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that's the last callback in the list, then remove our event handler of
		-- the list of unit change callbacks for that unit it
		
		if #list == 0 then
			nUI_Unit:unregisterUnitChangeCallback( unit_id, nUI_Unit.Drivers.Health );
		end
	end
end

-------------------------------------------------------------------------------
-- update the health information for this unit
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current GUID
-- if the data has changed, otherwise returns nil if nothing changed

function nUI_Unit:updateHealthInfo( unit_id, unit_info )

	local modified  = false;
	
	if unit_info then

		-- check the unit's current health data
		
		local cur_health, max_health = GetHealth( unit_id, unit_info );
		local pct_health;
		local color;
		
		if cur_health and max_health then
			pct_health = max( 0, min( cur_health / max_health, 1 ));
		end
		
		-- if it has changed, then update the cache
		
		if unit_info.cur_health  ~= cur_health
		or unit_info.max_health  ~= max_health
		or unit_info.pct_health  ~= pct_health
		then
			
			modified              = true;
			unit_info.modified    = true;
			unit_info.last_change = GetTime();
			unit_info.cur_health  = cur_health;
			unit_info.max_health  = max_health;
			unit_info.pct_health  = pct_health;
		end
		
		-- if we don't know the unit's health status, there should be no health bar

		local r, g, b, a;
		
		if not cur_health
		or not max_health
		or not pct_health
		then
		
			r = 0;
			g = 0;
			b = 0;
			a = 0;

		-- otherwise, select a color for the health bar based on the current health level
		
		else
			
			local range, color1, color2;
			
			-- select the color set we are going to use
			
			if pct_health > 0.5 then
				
				range  = (pct_health - 0.5) * 2;
				color1 = nUI_UnitOptions.BarColors.Health["mid"] or nUI_DefaultConfig.BarColors.Health["mid"];
				color2 = nUI_UnitOptions.BarColors.Health["max"] or nUI_DefaultConfig.BarColors.Health["max"];
				
			else
				
				range  = pct_health * 2;
				color1 = nUI_UnitOptions.BarColors.Health["min"];
				color2 = nUI_UnitOptions.BarColors.Health["mid"];
				
			end

			r = (color2.r - color1.r) * range + color1.r;
			g = (color2.g - color1.g) * range + color1.g;
			b = (color2.b - color1.b) * range + color1.b;
			a = (color2.a - color1.a) * range + color1.a;
			
		end

		-- if the color for the health bar has changed, update the cache
		
		if not unit_info.health_color
		or unit_info.health_color.r ~= r
		or unit_info.health_color.g ~= g
		or unit_info.health_color.b ~= b
		or unit_info.health_color.a ~= a
		then
			
			modified                 = true;
			unit_info.modified       = true;
			unit_info.last_change    = GetTime();
			unit_info.health_color   = {};
			unit_info.health_color.r = r;
			unit_info.health_color.g = g;
			unit_info.health_color.b = b;
			unit_info.health_color.a = a;
			
		end
	end
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all of the registered unit health listeners, even if there's no 
-- change in data... typically used when the health color options change
-- or entering the world

function nUI_Unit:refreshHealthCallbacks()

	nUI_Unit:refreshCallbacks( 
	
		nUI_L["unit health"], HealthCallbacks, HealthUnits, 
	
		function( list_unit, unit_info ) 
			nUI_Unit:updateHealthInfo( list_unit, unit_info ); 
		end 
	);
	
end

-------------------------------------------------------------------------------
-- create a new unit health frame

function nUI_Unit:createHealthFrame( parent, unit_id, id, options, clickable )

	local frame  = nUI_Unit:createFrame( "$parent_Health"..(id or ""), parent, unit_id, clickable );	
	frame.bar    = nUI_Bars:createStatusBar( "$parentBar", frame );	
	frame.cur    = frame:CreateFontString( "$parentCurrent", "OVERLAY" );	-- shows current health as text
	frame.max    = frame:CreateFontString( "$parentMaximum", "OVERLAY" );	-- shows maximum health as text
	frame.pct    = frame:CreateFontString( "$parentPercent", "OVERLAY" );	-- shows percent health as text
	frame.mix    = frame:CreateFontString( "$parentMixed", "OVERLAY" );		-- shows current and maximum health as text
	frame.Super  = {};

	frame.bar:SetPoint( "CENTER", frame, "CENTER", 0, 0 );
	
	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated
	
	frame.Super.newUnitInfo = frame.newUnitInfo;
	frame.newUnitInfo       = function( list_unit, unit_info )
		
		frame.Super.newUnitInfo( list_unit, unit_info );
		
		if frame.enabled then
			nUI_Unit:updateHealthFrame( frame );
		end
		
	end;
	
	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled

	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
		local prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				frame.unit_info = nUI_Unit:registerHealthCallback( frame.unit, frame );
				frame.bar.setEnabled( frame.options.bar and frame.options.bar.enabled or false );
				nUI_Unit:updateHealthFrame( frame );
			else
				nUI_Unit:unregisterHealthCallback( frame.unit, frame );
				frame.bar.setEnabled( false );
			end
		end
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame.

	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

		frame.Super.applyScale( scale );
		
		frame.configText( frame.cur, frame.options.cur_health );
		frame.configText( frame.max, frame.options.max_health );
		frame.configText( frame.pct, frame.options.pct_health );
		frame.configText( frame.mix, frame.options.mix_health );

		if frame.bar.size   ~= frame.size 
		or frame.bar.width  ~= frame.width
		or frame.bar.height ~= frame.height 
		or frame.bar.inset  ~= frame.inset
		then
			
			frame.bar.size   = frame.size;
			frame.bar.width  = frame.width;
			frame.bar.height = frame.height;
			frame.bar.inset  = frame.inset;
			
			frame.bar:SetWidth( (frame.size or frame.width) - frame.inset );
			frame.bar:SetHeight( (frame.size or frame.height) - frame.inset );

		end		
	end
	
	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.
	
	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

		frame.Super.applyOptions( options );
		
		-- extra frame level work
		
		frame.bar:SetFrameStrata( frame:GetFrameStrata() );
		frame.bar:SetFrameLevel( frame:GetFrameLevel() );
		
		-- enable or disable the display of a graphic bar
		
		frame.bar.setEnabled( options.bar and options.bar.enabled or false );

		if not frame.bar.enabled then
			
			frame.bar:SetAlpha( 0 );
			
		else
			
			frame.bar:SetAlpha( 1 );
			frame.bar.setOrientation( options.bar.orient or "LEFT" );
			frame.bar.setBar( options.bar.texture, options.bar.min_offset, options.bar.max_offset );
			frame.bar.setOverlay( options.bar.overlay );
			
		end

		-- special options for allowing the use of bar colors for text colors
		
		frame.max.barcolor = options.max_health and options.max_health.barcolor;
		frame.max.maxcolor = options.max_health and options.max_health.maxcolor;
		
		frame.cur.barcolor = options.cur_health and options.cur_health.barcolor;
		frame.cur.maxcolor = options.cur_health and options.cur_health.maxcolor;
		
		frame.pct.barcolor = options.pct_health and options.pct_health.barcolor;
		frame.pct.maxcolor = options.pct_health and options.pct_health.maxcolor;
		
		frame.mix.barcolor = options.mix_health and options.mix_health.barcolor;
		frame.mix.maxcolor = options.mix_health and options.mix_health.maxcolor;
		
		-- and refresh the frame
		
		nUI_Unit:updateHealthFrame( frame );
		
	end
	
	-- initiate the frame
	
	frame.unit_info = nUI_Unit:registerHealthCallback( frame.unit, frame );
		
	frame.applyOptions( options );
	
	return frame;
	
end

-------------------------------------------------------------------------------
-- remove a unit health frame

function nUI_Unit:deleteHealthFrame( frame )

	frame.bar.deleteBar();
	
	nUI_Unit:unregisterHealthCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
end

-------------------------------------------------------------------------------
-- update the health text and health bar as required
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updateHealthFrame( frame )
	
	local unit_info = frame.unit_info;
	
	-- if there is no unit or we don't know it's health, then hide the bar elements
	
	if not unit_info or 
	not unit_info.pct_health 
	or unit_info.max_health <= 0 then
		
		if frame.active then
			
			frame.active = false;
			
			frame.bar:SetAlpha( 0 );
			frame.cur:SetAlpha( 0 );
			frame.max:SetAlpha( 0 );
			frame.pct:SetAlpha( 0 );
			frame.mix:SetAlpha( 0 );
			
		end
	
	-- otherwise, update the health bar elements
	
	else

		local pct   = ("%0.1f%%"):format( unit_info.pct_health * 100 );
		local max   = ("%0.0f"):format( unit_info.max_health );
		local cur   = ("%0.0f"):format( unit_info.cur_health );
		local mix   = cur.." / "..max;
		local txt_color;
		
		-- if the health bar elements are hidden, show them
		
		if not frame.active then
			
			frame.active = true;
			
			frame.bar:SetAlpha( frame.bar.enabled and 1 or 0 );
			frame.cur:SetAlpha( frame.cur.enabled and 1 or 0 );
			frame.max:SetAlpha( frame.max.enabled and 1 or 0 );
			frame.pct:SetAlpha( frame.pct.enabled and 1 or 0 );
			frame.mix:SetAlpha( frame.mix.enabled and 1 or 0 );
			
		end

		-- if the health bar is active, update it
		
		if frame.bar.enabled then

			frame.bar.updateBar( unit_info.pct_health, unit_info.health_color );
			
		end
		
		-- if we're show current health text, update it
		
		if frame.cur.enabled then
			
			if frame.cur.value ~= cur then
				
				frame.cur.value = cur;
				frame.cur:SetText( cur );
				
			end
	
			if frame.cur.maxcolor then txt_color = nUI_UnitOptions.BarColors.Health.max;
			elseif frame.cur.barcolor then txt_color = unit_info.health_color;
			else txt_color = nil;
			end
			
			if txt_color 
			and (frame.cur.r ~= txt_color.r or frame.cur.g ~= txt_color.g or frame.cur.b ~= txt_color.b)
			then
				
				frame.cur.r = txt_color.r;
				frame.cur.g = txt_color.g;
				frame.cur.b = txt_color.b;
				
				frame.cur:SetTextColor( txt_color.r, txt_color.g, txt_color.b, 1 );
				
			end
		end
		
		-- if we're show maximum health text, update it
		
		if frame.max.enabled then
			
			if frame.max.value ~= max then
			
				frame.max.value = max;
				frame.max:SetText( max );
				
			end
	
			if frame.max.maxcolor then txt_color = nUI_UnitOptions.BarColors.Health.max;
			elseif frame.max.barcolor then txt_color = unit_info.health_color;
			else txt_color = nil;
			end
			
			if txt_color 
			and (frame.max.r ~= txt_color.r or frame.max.g ~= txt_color.g or frame.max.b ~= txt_color.b)
			then
				
				frame.max.r = txt_color.r;
				frame.max.g = txt_color.g;
				frame.max.b = txt_color.b;
				
				frame.max:SetTextColor( txt_color.r, txt_color.g, txt_color.b, 1 );
				
			end
		end
		
		-- if we're show percent health text, update it
		
		if frame.pct.enabled then
			
			if frame.pct.value ~= pct then
				
				frame.pct.value = pct;
				frame.pct:SetText( pct );
				
			end
	
			if frame.pct.maxcolor then txt_color = nUI_UnitOptions.BarColors.Health.max;
			elseif frame.pct.barcolor then txt_color = unit_info.health_color;
			else txt_color = nil;
			end
			
			if txt_color 
			and (frame.pct.r ~= txt_color.r or frame.pct.g ~= txt_color.g or frame.pct.b ~= txt_color.b)
			then
				
				frame.pct.r = txt_color.r;
				frame.pct.g = txt_color.g;
				frame.pct.b = txt_color.b;
				
				frame.pct:SetTextColor( txt_color.r, txt_color.g, txt_color.b, 1 );
				
			end
		end
		
		-- if we're show mixed current/maximum health text, update it
		
		if frame.mix.enabled then
			
			if frame.mix.value ~= mix then
				
				frame.mix.value = mix;
				frame.mix:SetText( mix );
				
			end		
	
			if frame.mix.maxcolor then txt_color = nUI_UnitOptions.BarColors.Health.max;
			elseif frame.mix.barcolor then txt_color = unit_info.health_color;
			else txt_color = nil;
			end
			
			if txt_color 
			and (frame.mix.r ~= txt_color.r or frame.mix.g ~= txt_color.g or frame.mix.b ~= txt_color.b)
			then
				
				frame.mix.r = txt_color.r;
				frame.mix.g = txt_color.g;
				frame.mix.b = txt_color.b;
				
				frame.mix:SetTextColor( txt_color.r, txt_color.g, txt_color.b, 1 );
				
			end
		end
	end
end
