﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--]]---------------------------------------------------------------------------

if not nUI_Unit then nUI_Unit = {}; end
if not nUI_Unit.Drivers then nUI_Unit.Drivers = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end
if not ClickCastFrames then	ClickCastFrames = {};end

local CreateFrame        = CreateFrame;
local ToggleDropDownMenu = ToggleDropDownMenu;
local UnitName           = UnitName;

-------------------------------------------------------------------------------

local ClickableFrames = {};

-------------------------------------------------------------------------------
-- toggle click-casting support on/off for known clickable unit frames

function nUI_Unit:toggleClickCasting()
	
	local click_cast = strlower( nUI_UnitOptions.click_cast ) == "no";

	DEFAULT_CHAT_FRAME:AddMessage( 
		nUI_L["nUI: click-casting registration is %s"]:format( (click_cast and nUI_L["|cFF00FF00ENABLED|r"] or nUI_L["|cFFFF0000DISABLED|r"]) ), 
		1, 0.83, 0 
	);

	nUI_UnitOptions.click_cast = click_cast and "yes" or "no";
	
	for i,frame in ipairs( ClickableFrames ) do
		
		frame.click_cast       = click_cast;
		ClickCastFrames[frame] = click_cast or nil;
		
	end
end

-------------------------------------------------------------------------------
-- create a new base unit frame (not the same as creating a unit itself!)

function nUI_Unit:createFrame( name, parent, unit_id, clickable )
	
	if not parent then
		
		DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: must pass a valid parent frame to nUI_Unit:createFrame() for unit id [%s (%s)]"]:format( unit_id, name ), 1, 0.5, 0.5 ); 
	
	else
			
		local frame      = CreateFrame( "Button", name, parent, clickable and "SecureUnitButtonTemplate" or nil );		
		frame.background = CreateFrame( "Frame", name.."_Background", parent );		
		frame.parent     = parent;
		frame.click_cast = strlower( nUI_UnitOptions.click_cast or "yes" ) ~= "no";
		frame.clickable  = clickable;
		frame.enabled    = true;
		frame.active     = true;
		frame.unit       = unit_id;
		frame.Labels     = {};
		
		frame.background:SetAllPoints( frame );
	
		-- set attributes for secure frames and downline driver use
		
		frame:SetAttribute( "unit", unit_id );	
	
		-- if the frame's not clickable, then do away with the mouse functionality
		
		if not frame.clickable then
	
			frame:RegisterForClicks();
			frame:EnableMouse( false );
			frame:SetScript( "OnClick", nil );
			frame:SetScript( "OnEnter", nil );
			frame:SetScript( "OnLeave", nil );
			
		-- implement unit frame click functionality		

		else
			
			-- add this frame to the list of clickable unit frames
			
			table.insert( ClickableFrames, frame );
			
			-- set up click-casting support if it is enabled
			
			if frame.click_cast then
				ClickCastFrames[frame] = true;
			end
			
			-- enable click targeting and right-click menu
			
			frame:SetAttribute( "*type1", "target" );
			frame:SetAttribute( "type2",  "menu" );
			
			frame:EnableMouse( true );
			frame:RegisterForClicks( "LeftButtonUp", "RightButtonUp" );	
	
			frame.menu = function()
	
				local popup = frame.popup or frame.parent.popup;
				
				if popup then
						
					local anchor = frame.popup and frame or frame.parent;
					this.unit    = unit_id;
					this.id      = id;
					this.name,
					this.server  = UnitName( unit_id );
								
					ToggleDropDownMenu( 1, nil, popup, frame.popup and frame or frame.parent, anchor:GetWidth()/2, anchor:GetHeight()/2 );
					
				end			
			end		
	
			-- enable mouseover tooltips
			
			frame:SetScript( "OnEnter", 
			
				function() 
			
					local unit_info = frame.unit_info or frame.parent.unit_info;
					local role      = unit_info and unit_info.role_info and unit_info.role_info.role or nil;
					local range     = unit_info and unit_info.range_info and unit_info.range_info.text or nil;
					local build     = unit_info and unit_info.build or nil;
					
					if unit_spec == "normal" then unit_spec = nil; end
					
					GameTooltip:SetOwner( frame );
					GameTooltip:SetUnit( unit_id ); 
					
					if role then
						local fmt;
						
						if unit_info.in_raid then fmt = nUI_L["Raid Role: |cFF00FFFF%s|r"];
						else fmt = nUI_L["Party Role: |cFF00FFFF%s|r"];
						end
						
						GameTooltip:AddLine( fmt:format( role ), 1, 1, 1, 1 );
					end
					
					if build and build.name and build.points then 
						GameTooltip:AddLine( nUI_L["Talent Build: <build name> (<talent points>)"]:format( build.name, build.points ), 1, 0.83, 0 ); 
					end 

					if range then
						local color = unit_info.range_info.color;
						GameTooltip:AddLine( range, color.r, color.g, color.b, 1 );
					end
					
					GameTooltip:Show();
				end 
			);
	
			frame:SetScript( "OnLeave", 
				function() 
					GameTooltip:Hide(); 
				end 
			);
			
		end
		
		-- generic "I've got new data" method
		
		frame.newUnitInfo = function( unit_id, unit_info )
		
			if not unit_id then 				
--				DEFAULT_CHAT_FRAME:AddMessage( frame:GetName().." called with new unit info and a <nil> unit id" );				
			elseif UnitIsUnit( unit_id, frame.unit ) then	
				frame.unit_info = unit_info;				
			end
		end
		
		-- method for sizing and anchoring text labels
		
		frame.configText = function( text, config, scale )
	
			local scale       = (scale or frame.scale or 1) * nUI.scale;
			local font_size   = (config and config.fontsize or 12) * scale * 1.75;
			local justifyH    = config and config.justifyH or "CENTER";
			local justifyV    = config and config.justifyV or "MIDDLE";
			local r           = config and config.color and config.color.r or 1;
			local g           = config and config.color and config.color.g or 1;
			local b           = config and config.color and config.color.b or 1;
			local a           = config and config.color and config.color.a or 1;			

			frame.Labels[text] = 
			{
				config = config,
				scale  = scale,
			};
			
			-- set the text font size
						
			if text.font_size ~= font_size
			then
				
				-- first time here?
				
				if not text.font_size then 
					text.active  = true;
				end
		
				text.font_size = font_size;
				text:SetFont( nUI_L["font1"], font_size, "OUTLINE" );
	
			end

			-- show or hide the text based on whether or not there is a config for it
			
			text.enabled = config and config.enabled or false;
			
			if not config and text.active then

				text.active = false;
				text.value  = nil;				
				text:SetAlpha( 0 );
				text:SetText( "" );				
				
			elseif config then

				text.active = true;
				text:SetAlpha( 1 );
				
			end
			
			-- set text justification
			
			if text.justifyH ~= justifyH then
				text.justifyH = justifyH;
				text:SetJustifyH( justifyH );
			end
			
			if text.justigyV ~= justifyV then
				text.justifyV = justifyV;
				text:SetJustifyV( justifyV );
			end
			
			-- set text color
			
			if text.r ~= r
			or text.g ~= g
			or text.b ~= b
			or text.a ~= a
			then
				
				text.r = r;
				text.g = g;
				text.b = b;
				text.a = a;
				
				text:SetTextColor( r, g, b, a );
				
			end
			
			-- if the text has a fixed value then set it
			
			if config and config.label and text.value ~= config.label then
				text.value = config.label;
				text:SetText( config.label );
			end
			
		end
		
		-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
		-- this method actually recalculates the size of the frame and uses frame:SetHeight()
		-- and frame:SetWidth() to reflect the actual size of the frame. Is also recreates
		-- the font to present clear, sharp, readable text versus the blurred text you get
		-- as a result of frame:SetScale() or text:SetTextHeight()
		
		frame.applyScale = function( scale )
			
			local anchor = scale and frame.anchor or nil;
			local scale  = scale or frame.scale or 1;			
			frame.scale  = scale;
			
			if frame.options then
					
				local size   = frame.options.size and (frame.options.size * scale * nUI.scale) or nil;
				local height = not size and (frame.options.height * scale * nUI.scale) or nil;
				local width  = not size and (frame.options.width * scale * nUI.scale) or nil;
				local inset  = (frame.options.inset or 0) * scale * nUI.scale;
				
				if frame.size   ~= size
				or frame.height ~= height
				or frame.width  ~= width
				or frame.inset  ~= inset
				then
					
					frame.height = height;
					frame.width  = width;
					frame.size   = size;
					frame.inset  = inset;
					
					frame:SetHeight( size or height );
					frame:SetWidth( size or width );
					
				end
				
				if anchor then frame.applyAnchor(); end
				
			end
		end
	
		-- this method applies the anchor point of the frame. As with all else, the
		-- frame's anchor is only moved if the point defined is different than the
		-- point that is already known
		
		frame.applyAnchor = function( anchor )
		
			local anchor      = anchor or frame.anchor or {};
			local scale       = (frame.scale or 1) * nUI.scale;
			local anchor_pt   = anchor.anchor_pt or "CENTER";
			local relative_to = anchor.relative_to or frame.parent:GetName();
			local relative_pt = anchor.relative_pt or anchor_pt;
			local xOfs        = (anchor.xOfs or 0) * scale;
			local yOfs        = (anchor.yOfs or 0) * scale;
	
			frame.anchor = anchor;
			
			if frame.anchor_pt   ~= anchor_pt
			or frame.relative_to ~= relative_to
			or frame.relative_pt ~= relative_pt
			or frame.xOfs        ~= xOfs
			or frame.yOfs        ~= yOfs
			then
				
				frame.anchor_pt   = anchor_pt;
				frame.relative_to = relative_to;
				frame.relative_pt = relative_pt;
				frame.xOfs        = xOfs;
				frame.yOfs        = yOfs;
				
				frame:ClearAllPoints();
				frame:SetPoint( anchor_pt, relative_to:gsub( "$parent", frame.parent:GetName() ), relative_pt, xOfs, yOfs );
				
			end
			
			-- and anchor the text

			for text in pairs( frame.Labels ) do

				local config      = frame.Labels[text].config;
				local scale       = frame.Labels[text].scale;
				
				anchor_pt   = (config and config.anchor_pt or "CENTER");
				relative_to = (config and config.relative_to or frame:GetName());
				relative_pt = (config and config.relative_pt or anchor_pt);
				xOfs        = (config and config.xOfs or 0) * scale;
				yOfs        = (config and config.yOfs or 0) * scale;
				
				if text.scale       ~= scale
				or text.xOfs        ~= xOfs
				or text.yOfs        ~= yOfs
				or text.anchor_pt   ~= anchor_pt
				or text.relative_to ~= relative_to
				or text.relative_pt ~= relative_pt
				then
					
					text.scale       = scale;
					text.xOfs        = xOfs;
					text.yOfs        = yOfs;
					text.anchor_pt   = (config and config.anchor_pt or "CENTER");
					text.relative_to = (config and config.relative_to or frame:GetName());
					text.relative_pt = (config and config.relative_pt or "CENTER");
	
					text:ClearAllPoints();
					text:SetPoint( anchor_pt, relative_to:gsub( "$parent", frame.parent:GetName() ), relative_pt, xOfs, yOfs );
					
				end
			end
		end
		
		-- toggles a frame on or off based on the passed state and whether or not
		-- the frame has an options set and, if it does, is or is not enabled there
		-- returns true if the frame is enabled after the call, false if not
		
		frame.setEnabled = function( enabled )

			if frame.options and not frame.options.enabled then
				frame.enabled = false;
			else
				frame.enabled = enabled;
			end
			
		end
		
		-- applies the set of frame options to this frame. Typically called when the frame 
		-- is first created or when the user changes options via config.
		
		frame.applyOptions = function( options )
			
			frame.options = options;
			
			-- set up frame layering
			
			frame:SetFrameStrata( options.strata or frame.parent:GetFrameStrata() );
			frame:SetFrameLevel( frame.parent:GetFrameLevel() + (options.level or 1) );
			frame.background:SetFrameLevel( frame:GetFrameLevel() - 1 );

			-- set a party or raid id if needed
			
			frame.id = (options.raid_id or options.party_id) or parent.id or nil;
			
			-- if there's a frame border, set it (frame border hides when the frame hides
			
			if options.border then
					
				local border_color = options.border.color.border;
				local backdrop_color = options.border.color.backdrop;
				
				frame:SetBackdrop( options.border.backdrop );
				frame:SetBackdropBorderColor( border_color.r, border_color.g, border_color.b, border_color.a );
				frame:SetBackdropColor( backdrop_color.r, backdrop_color.g, backdrop_color.b, backdrop_color.a );
		
			else 
				
				frame:SetBackdrop( nil );
				
			end
			
			-- if there's a frame background, set it (background does not hide when the frame hides
			
			if options.background then
					
				local border_color = options.background.color.border;
				local backdrop_color = options.background.color.backdrop;
				
				frame.background:SetBackdrop( options.border.backdrop );
				frame.background:SetBackdropBorderColor( border_color.r, border_color.g, border_color.b, border_color.a );
				frame.background:SetBackdropColor( backdrop_color.r, backdrop_color.g, backdrop_color.b, backdrop_color.a );
		
			else 
				
				frame.background:SetBackdrop( nil );
				
			end
	
			-- size and anchor the frame
			
			frame.applyScale( parent.scale or 1 );

			-- set frame's active state
			
			frame.setEnabled( options.enabled );
			
			if frame.enabled and not frame:IsShown() then frame:Show();
			elseif not frame.enabled and frame:IsShown() then frame:Hide();
			end			
			
		end
		
		-- register the frame for scaling
		
		nUI:registerScalableFrame( frame );
		
		return frame;
		
	end	
	
end

-------------------------------------------------------------------------------

function nUI_Unit:deleteFrame( frame )

	nUI:unregisterScaleableFrame( frame );

	-- disable the frame (makes sure that chilren can clean up too)
	
	frame.setEnabled( false );
	
	-- if this frame is registered for click-casting, unregister it
	
	if frame.click_cast then
		ClickCastFrames[frame] = nil;
	end
	
	-- if this is a clickable frame, then unregister its state headers

	frame:SetAttribute( "unit", ATTRIBUTE_NOOP );
	
	if frame.clickable then

		nUI:TableRemoveByValue( ClickableFrames, frame );
			
		frame:SetAttribute( "*type1", ATTRIBUTE_NOOP );
		frame:SetAttribute( "type2",  ATTRIBUTE_NOOP );
		
		frame:RegisterForClicks();
		frame:EnableMouse( false );
	
		frame:SetScript( "OnEnter", nil );
		frame:SetScript( "OnLeave", nil );
		
		frame.menu = nil;

	end
	
	-- hide the frame
	
	frame:SetParent( nil );	
	frame:ClearAllPoints();
	frame:Hide();	
	
end
