﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

-------------------------------------------------------------------------------

Unit Skinning Options: Feedback

class options =
{
	string strata				-- the frame strata to use (BACKGROUND, LOW, HIGH, etc.) or nil to use the parent frame's strata
	float height				-- height of the frame
	float width					-- width of the frame
	int level					-- the frame level to use for the aura icons (1-9) or nil to use the parent frame's level+1
	boolean show_agro			-- enable frame background color when the unit has the last hostile target's agro
	boolean show_hits			-- enable frame bordder highlights when unit is hit or healed
	boolean show_curses			-- enable frame background color when the unit does not have agro and is cursed, poisoned, diseased or charmed (magic)
	
	class msg_label				-- enables display of the text values of the combat feedback when not nil
	{
		int scale				-- adjusts the size of the text relative to the bar's narrow dimension: 1 = normal or 75%
		string justifyH			-- horiztontal text justification: LEFT, RIGHT or CENTER
		string justifyV			-- vertical text justification: TOP, BOTTOM or MIDDLE
		string anchor			-- where to anchor the text relative to the feedback frame: LEFT, RIGHT, CENTER, TOPRIGHT, etc.
		float xOfs				-- horizontal distance between "anchor" on the text and "anchor" on the bar
		float yOfs				-- vertical distance between text and the bar
		
		class color { r,g,b,a }	-- text color to use
	};
	
	class border				-- sets the frame's border and background or nil for no border/background at all
	{
		class backdrop			-- description of the backdrop itself
		{
			string bgFile		-- path to the frame background texture or nil for no background
			string edgeFile		-- path to the frame edge texture file or nil for no frame edge
			boolean tile		-- true if the background texture should be tiled to fill the frame
			float tileSize		-- size of each background tile
			float edgeSize		-- thickness of the edge texture around the frame
			
			class insets = 
			{	
				int left		-- inset of background texture from the outside edges of the frame
				int right
				int top
				int bottom
			};
		};
		
		class color
		{
			class border = { r, g, b, a }	-- color and transparency of the edge texture
			class backdrop = { r, g, b, a }	-- color and transparency of the background texture
		};
	};
}

--]]---------------------------------------------------------------------------

if not nUI then nUI = {}; end
if not nUI_Unit then nUI_Unit = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local CreateFrame       = CreateFrame;
local GetTime           = GetTime;
local UnitCanAttack     = UnitCanAttack;
local UnitExists        = UnitExists;
local UnitIsDeadOrGhost = UnitIsDeadOrGhost;
local UnitIsUnit        = UnitIsUnit;

-------------------------------------------------------------------------------
-- default options for the unit feedback colors

nUI_DefaultConfig.FeedbackColors =
{
	-- backdrop colors
	
	["bgAgro"]     = { r = 1, g = 0, b = 0, },
	["bgMagic"]    = DebuffTypeColor["Magic"],
	["bgCurse"]    = DebuffTypeColor["Curse"],
	["bgDisease"]  = DebuffTypeColor["Disease"],
	["bgPoison"]   = DebuffTypeColor["Poison"],
	
	-- highlight colors
	
	["hitHeal"]     = { r = 0, g = 1, b = 0, },
	["hitPhysical"] = { r = 1, g = 0, b = 0, },
	["hitHoly"]     = { r = 1, g = 0, b = 0, },
	["hitFire"]     = { r = 1, g = 0, b = 0, },
	["hitNature"]   = { r = 1, g = 0, b = 0, },
	["hitFrost"]    = { r = 1, g = 0, b = 0, },
	["hitShadow"]   = { r = 1, g = 0, b = 0, },
	["hitArcane"]   = { r = 1, g = 0, b = 0, },
	
	-- text colors
	
	["txtNormal"]   = { r = 1, g = 1, b = 1, },
	["txtHeal"]     = { r = 0, g = 0, b = 0, },
	["txtGlancing"] = { r = 0.75, g = 0.75, b = 0.75, },
	["txtCritical"] = { r = 1, g = 0.83, b = 0, },
	["txtCrushing"] = { r = 1, g = 0.25, b = 0.25, },
	
};

local HitColorMap =
{
	[1]  = "hitPhysical",
	[2]  = "hitHoly",
	[4]  = "hitFire",
	[8]  = "hitNature",
	[16] = "hitFrost",
	[32] = "hitShadow",
	[65] = "hitArcane",
};
-------------------------------------------------------------------------------
-- unit feedback event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

local frame = CreateFrame( "Frame", "$parent_Feedback", nUI_Unit.Drivers )

local HostileName;
local TargetInfo;
local ToTInfo;
local FeedbackTimer        = 0;
local FeedbackCallbacks    = {};
local FeedbackUnits        = {};

nUI_Unit.Drivers.Feedback  = frame;

-------------------------------------------------------------------------------

local registered = false;

local function onFeedbackEvent()
	
	if event == "VARIABLES_LOADED" then
		
		nUI:patchConfig();
		nUI_Unit:configFeedback();
		
	-- for these events, we don't know which units are affected, so
	-- we span the list of all known interested units to see who is watching
		
	elseif event == "PLAYER_ENTERING_WORLD" then
		
		-- register to receive notices when the target changes so we can find out
		-- who has the new target's agro when required. We acutally use the 
		-- unit reaction callback instead of unit change because we also need to
		-- know if we can attack the unit or if it is friendly to us

		if not registered then
			
			registered = true;
			nUI_Unit:registerReactionCallback( "target", nUI_UnitDrivers.Feedback );
			nUI_Unit:registerReactionCallback( "targettarget", nUI_UnitDrivers.Feedback );
			
		else
			
			nUI_Unit:refreshFeedbackCallbacks()

		end
		
	-- a combat event
	
	elseif event == "UNIT_COMBAT" then
		
		nUI_Unit:updateFeedbackInfo( arg1, nUI_Unit:getUnitInfo( arg1 ), arg2, arg3, arg4, arg5 );
		
	end
end

frame:SetScript( "OnEvent", onFeedbackEvent );
frame:RegisterEvent( "VARIABLES_LOADED" );
frame:RegisterEvent( "PLAYER_ENTERING_WORLD" );
frame:RegisterEvent( "UNIT_COMBAT" );

-------------------------------------------------------------------------------
-- feedback update event handler

local function onFeedbackUpdate( who, elapsed )
	
	FeedbackTimer = FeedbackTimer + elapsed;
	
	if FeedbackTimer > 0.08 then -- 12.5fps check rate on taget agro
	
		FeedbackTimer = 0;

		-- our consideration of who the agro target of our target is, is based on
		-- who the last hostile target was we looked at. So, if we see a new hostile
		-- target, we need to remember who it was
	
		if  nUI_Unit.TargetInfo
		and nUI_Unit.TargetInfo ~= TargetInfo
		and nUI_Unit.TargetInfo.attackable
		then
			
			HostileName = nUI_Unit.TargetInfo.name;
		
		-- otherwise, if we have a hostile target name and that target
		-- no longer exists or is dead, then we can have no target agro
		
		elseif HostileName 
		and (not UnitExists( HostileName ) or UnitIsDeadOrGhost( HostileName ))
		then

			HostileName = nil;
			
		end
		
		-- who is target of our hostile target?
		
		local tot_info = HostileName and nUI_Unit:getUnitInfo( HostileName.."-target" ) or nil;
		
		-- even if we do have a ToT, it is only an agro target if the last hostile
		-- target we saw is able to attack that ToT (it could be healing a friendly or
		-- itself or targeting no one now
		
		if tot_info and not UnitCanAttack( HostileName, HostileName.."-target" ) then
			tot_info = nil;
		end
		
		-- if the ToT has changed from what we last knew, then we need to update
		-- our record and refresh all of the intersted listeners 
		
		if tot_info ~= ToTInfo then
			TotInfo = tot_info;
			nUI_Unit:refreshFeedbackCallbacks();
		end
	end
end

frame:SetScript( "OnUpdate", onFeedbackUpdate );

-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for unit feedback changes GUID

frame.newUnitInfo = function( list_unit, unit_info )

	local new_data  = nUI_Unit:updateFeedbackInfo( list_unit, unit_info );
	local callbacks = FeedbackCallbacks;
	local unitlist  = FeedbackUnits;
	local new_agro  = false;
	
	nUI_Unit:notifyCallbacks( nUI_L["unit feedback"], callbacks, unitlist, unit_info, list_unit, new_data );
	
end

-------------------------------------------------------------------------------
-- initialize configuration for the unit feedback color indicators
-- 
-- this method is called when the mod's saved variables have been loaded by Bliz and
-- may be called again whenever the unit feedback configuration has been changed
-- by the player or programmatically. Passing true or a non-nil value for "use_default"
-- will cause the player's current feedback color configuration to be replaced with
-- the default settings defined at the top of this file (which cannot be undone!)

function nUI_Unit:configFeedback( use_default )
	
	if not nUI_UnitOptions then nUI_UnitOptions = {}; end
	if not nUI_UnitOptions.FeedbackColors then nUI_UnitOptions.FeedbackColors = {}; end
	
	for feedback in pairs( nUI_DefaultConfig.FeedbackColors ) do
		nUI_Unit:configFeedbackColor( feedback, use_default );
	end
end

function nUI_Unit:configFeedbackColor( feedback, use_default )
	
	local config  = nUI_UnitOptions.FeedbackColors[feedback] or {};
	local default = nUI_DefaultConfig.FeedbackColors[feedback] or {};
	
	if use_default then
			
		config.r = default.r;
		config.g = default.g;
		config.b = default.b;

	else
			
		config.r = tonumber( config.r or default.r );
		config.g = tonumber( config.g or default.g );
		config.b = tonumber( config.b or default.b );

	end
	
	nUI_UnitOptions.FeedbackColors[feedback] = config;
	nUI_Unit:refreshFeedbackCallbacks();
	
end

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of unit feedback listeners we manage
--
-- calling this method will return the current unit_info structure for this 
-- unit if it exists or nil if the unit does not exist at this time
--
-- Note: these callbacks will be notified both when the underlying GUID for the
--		 unit changes or when the feedback info of the underlying GUID to the
--		 player changes. If the underlying unit does not exist, the callback
--		 will be passed a nil unit_info structure

function nUI_Unit:registerFeedbackCallback( unit_id, callback )
	
	local unit_info = nil;
	
	if unit_id and callback then
		
		-- get the list of callbacks for this unit id and add this callback
		
		local list = FeedbackCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );
		
		-- if this is a new unit id, add it to the callback list
		
		if not FeedbackCallbacks[unit_id] then
			FeedbackCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for the unit id, then register our
		-- event driver to receive notice when the GUID changes on this id
		-- We use the aura callback system for this as we need both to know
		-- that the unit changed and to know that its auras have been 
		-- updated (which would cause or cure curse, poison, etc.)
		
		if #list == 1 then
			nUI_Unit:registerAuraCallback( unit_id, nUI_Unit.Drivers.Feedback );
		end
		
		-- collect feedback information for this unit as we know it at this time
	
		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			nUI_Unit:updateFeedbackInfo( unit_id, unit_info );
		end
	end
	
	return unit_info;
	
end

function nUI_Unit:unregisterFeedbackCallback( unit_id, callback )
	
	if unit_id and callback then
		
		-- get the list of current callbacks for this unit ud and remove this callback
		
		local list = FeedbackCallbacks[unit_id] or {};
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that's the last callback in the list, then remove our event handler of
		-- the list of unit change callbacks for that unit it
		
		if #list == 0 then
			nUI_Unit:unregisterAuraCallback( unit_id, nUI_Unit.Drivers.Feedback );
		end
	end
end

-------------------------------------------------------------------------------
-- update the feedback information for this unit
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current GUID
-- if the data has changed, otherwise returns nil if nothing changed

function nUI_Unit:updateFeedbackInfo( unit_id, unit_info, action, effect, hitValue, hitType )

	local modified  = false;
	
	if unit_info then
		
		local feedback = {};
		local cached   = unit_info.feedback or {};

		-- if we have an action, then set the feedback to that action
		
		if action then
			
			feedback.start_time = GetTime();
			feedback.end_time   = feedback.start_time + COMBATFEEDBACK_FADEINTIME + COMBATFEEDBACK_HOLDTIME + COMBATFEEDBACK_FADEOUTTIME;
			feedback.action     = action;
			feedback.effect     = effect;
			feedback.hitValue   = hitValue;
			feedback.hitType    = hitType;

			if action == "HEAL" then feedback.hitColor = nUI_UnitOptions.FeedbackColors["hitHeal"];
			else feedback.hitColor = nUI_UnitOptions.FeedbackColors[HitColorMap[hitType or 1]];
			end
				
			if     action == "HEAL"     then feedback.txtColor = nUI_UnitOptions.FeedbackColors["txtHeal"];
			elseif effect == "GLANCING" then feedback.txtColor = nUI_UnitOptions.FeedbackColors["txtGlancing"];
			elseif effect == "CRITICAL" then feedback.txtColor = nUI_UnitOptions.FeedbackColors["txtCritical"];
			elseif effect == "CRUSHING" then feedback.txtColor = nUI_UnitOptions.FeedbackColors["txtCrushing"];
			else                             feedback.txtColor = nUI_UnitOptions.FeedbackColors["txtNormal"];
			end
			
		-- otherwise, the feedback info stays the same
		
		else
			
			feedback.start_time = cached.start_time;
			feedback.end_time   = cached.end_time;
			feedback.action     = cached.action;
			feedback.effect     = cached.effect;
			feedback.hitValue   = cached.hitValue;
			feedback.hitType    = cached.hitType;
			feedback.hitColor   = cached.hitColor or {};
			feedback.txtColor   = cached.txtColor or {};
			
		end

		-- does this unit have the last hostile target's agro?
		
		feedback.has_agro = unit_info == TotInfo;
		feedback.bg_color = feedback.has_agro and nUI_UnitOptions.FeedbackColors["bgAgro"] or feedback.bgColor or {};
		
		-- does this unit have any curses on it?
		
		if unit_info.aura_info 
		and unit_info.aura_info.debuff 
		then
			
			for debuff in pairs( unit_info.aura_info.debuff ) do
		
				if debuff.type == "Magic" then
					feedback.has_magic = true;
					feedback.bgColor   = feedback.has_agro and feedback.bgColor or nUI_UnitOptions.FeedbackColors["bgMagic"];
				elseif debuff.type == "Curse" then
					feedback.has_curse = true;
					feedback.bgColor   = feedback.has_agro and feedback.bgColor or nUI_UnitOptions.FeedbackColors["bgCurse"];
				elseif debuff.type == "Poison" then
					feedback.has_poison = true;
					feedback.bgColor   = feedback.has_agro and feedback.bgColor or nUI_UnitOptions.FeedbackColors["bgPoison"];
				elseif debuff.type == "Disease" then
					feedback.has_disease = true;
					feedback.bgColor   = feedback.has_agro and feedback.bgColor or nUI_UnitOptions.FeedbackColors["bgDisease"];
				end
				
			end
		end
		
		-- has anything changed?
		
		if not unit_info.feedback
		or not unit_info.bgColor
		or not unit_info.txtColor
		or not unit_info.hitColor
		or cached.start_time    ~= feedback.start_time
		or cached.has_agro      ~= feedback.has_agro
		or cached.has_magic     ~= feedback.has_magic
		or cached.has_curse     ~= feedback.has_curse
		or cached.has_poison    ~= feedback.has_poison
		or cached.has_disease   ~= feedback.has_disease
		or unit_info.bgColor.r  ~= feedback.bgColor.r
		or unit_info.bgColor.g  ~= feedback.bgColor.g
		or unit_info.bgColor.b  ~= feedback.bgColor.b
		or unit_info.hitColor.r ~= feedback.hitColor.r
		or unit_info.hitColor.g ~= feedback.hitColor.g
		or unit_info.hitColor.b ~= feedback.hitColor.b
		or unit_info.txtColor.r ~= feedback.txtColor.r
		or unit_info.txtColor.g ~= feedback.txtColor.g
		or unit_info.txtColor.b ~= feedback.txtColor.b
		then
			
			modified              = true;
			unit_info.modified    = true;
			unit_info.last_change = GetTime();
			unit_info.feedback    = feedback;
			
		end
	end
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all of the registered unit feedback listeners, even if there's no 
-- change in data... typically used when the feedback color options change
-- or entering the world

function nUI_Unit:refreshFeedbackCallbacks()

	nUI_Unit:refreshCallbacks( 
	
		nUI_L["unit feedback"], FeedbackCallbacks, FeedbackUnits, 
	
		function( list_unit, unit_info ) 
			nUI_Unit:updateFeedbackInfo( list_unit, unit_info ); 
		end 
	);

end

-------------------------------------------------------------------------------
-- create a new unit feedback frame

function nUI_Unit:createFeedbackFrame( parent, unit_id, id, options )

	local frame            = nUI_Unit:createFrame( "$parent_Feedback"..(id or ""), parent, unit_id, false );
	frame.texture          = frame:CreateTexture( "$parentTexture", "BACKGROUND" );
	frame.highlight        = frame:CreateTexture( "$parentHighlight", "ARTWORK" );
	frame.text             = frame:CreateFontString( "$parentText", "OVERLAY" );
	frame.texture.active   = true;
	frame.highlight.active = true;
	frame.Super            = {};

	frame.texture:SetAlpha( 0 );
	frame.highlight:SetAlpha( 0 );
	frame.texture:SetPoint( "CENTER", frame, "CENTER", 0, 0 );
	frame.texture:SetTexture( "Interface\\Tooltips\\UI-Tooltip-Background" );	
	frame.highlight:SetPoint( "CENTER", frame, "CENTER", 0, 0 );
	frame.highlight:SetTexture( "Interface\\AddOns\\nUI\\Units\\Art\\nUI_UnitFeedback" );

	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated
	
	frame.Super.newUnitInfo = frame.newUnitInfo;
	frame.newUnitInfo       = function( list_unit, unit_info )
		
		frame.Super.newUnitInfo( list_unit, unit_info );
		
		if frame.enabled then
			nUI_Unit:updateFeedbackFrame( frame );
		end
		
	end
	
	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled
	
	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
		local prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				frame.unit_info = nUI_Unit:registerFeedbackCallback( frame.unit, frame );
				nUI_Unit:updateFeedbackFrame( frame );
			else
				nUI_Unit:unregisterFeedbackCallback( frame.unit, frame );
			end
		end
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame.

	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

		frame.Super.applyScale( scale );		
		
		frame.configText( frame.text, frame.options.text );
		
		if frame.texture.size   ~= frame.size 
		or frame.texture.width  ~= frame.width
		or frame.texture.height ~= frame.height 
		or frame.texture.inset  ~= frame.inset
		then
			
			frame.texture.size   = frame.size;
			frame.texture.width  = frame.width;
			frame.texture.height = frame.height;
			frame.texture.inset  = frame.inset;
			
			frame.texture:SetWidth( (frame.size or frame.width) - frame.inset );
			frame.texture:SetHeight( (frame.size or frame.height) - frame.inset );

			frame.highlight:SetWidth( (frame.size or frame.width) - frame.inset );
			frame.highlight:SetHeight( (frame.size or frame.height) - frame.inset );
			
		end				
	end
	
	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.

	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

		frame.Super.applyOptions( options );
		nUI_Unit:updateFeedbackFrame( frame );
		
	end

	-- initiate the frame
	
	frame.unit_info = nUI_Unit:registerFeedbackCallback( frame.unit, frame );
	
	frame.applyOptions( options );
	
	return frame;
	
end

-------------------------------------------------------------------------------
-- remove a unit feedback frame

function nUI_Unit:deleteFeedbackFrame( frame )

	nUI_Unit:unregisterFeedbackCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
end

-------------------------------------------------------------------------------
-- display the appropriate icon for the unit's feedback
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updateFeedbackFrame( frame )
	
	local unit_info = frame.unit_info;
	
	-- if there is no unit or we don't know it's feedback, then hide the icon
	
	if not unit_info or not unit_info.feedback then
		
		if frame.active then
			
			if frame.texture.active then frame.texture:SetAlpha( 0 ); end
			if frame.highlight.active then frame.highlight:SetAlpha( 0 ); end
			
			frame.active           = false;
			frame.texture.active   = false;
			frame.highlight.active = false;
			
		end
	
	-- otherwise, show the icon and clip the appropriate region
	
	else

		frame.active = true;			
		frame.texture.active = true;
		frame.highlight.active = true;
	end
end
