﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--]]---------------------------------------------------------------------------

if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local GetRestState    = GetRestState;
local GetXPExhaustion = GetXPExhaustion;
local IsResting       = IsResting;
local UnitLevel       = UnitLevel;
local UnitXP          = UnitXP;
local UnitXPMax       = UnitXPMax;

-------------------------------------------------------------------------------

nUI_DefaultConfig.XPBar =
{
	enabled = true;
	
	anchor =
	{
		anchor_pt   = "TOPRIGHT",
		relative_to = "nUI_Dashboard",
		relative_pt = "CENTER",
		xOfs        = -685,
		yOfs        = -181,
	},
	
	options =
	{
		enabled  = true;
		strata   = nil,
		level    = 3,
		height   = 15,
		width    = 585,
		inset    = 0,
		
		bar =
		{
			enabled     = true,
			orient      = "LEFT",
			tick_height = 35,
			tick_width  = 40,
			overlay     = "Interface\\AddOns\\nUI\\Bars\\Art\\nUI_XPRepOverlay",
			rested_tick = "Interface\\AddOns\\nUI\\Bars\\Art\\nUI_XPRestedTick",
			
			colors =
			{
				rested = { r = 0.25, g = 1, b = 0.25 },
				xp     = { r = 0.25, g = 0.5, b = 1 },
			},
		},
	},
};

-------------------------------------------------------------------------------

-- note the last value in this table is a dummy value

local XpPerLevel = 
{
	   400,    900,   1400,   2100,   2800,   3600,   4400,   5400,   6500,    7600,	-- 1 to 10
	  8700,   9800,  11000,  12300,  13600,  15000,  16400,  17800,  19300,   20800,	-- 11 to 20
	 22400,  24000,  25500,  27200,  28900,  30500,  32200,  33900,  36300,   38800, 	-- 21 to 30
	 41600,  44600,  48000,  51400,  55000,  58700,  62400,  66200,  70200,   74300,	-- 31 to 40
	 78500,  82800,  87100,  91600,  95300, 101000, 105800, 110700, 115700,  120900,	-- 41 to 50
	126100, 131500, 137000, 142500, 148200, 154000, 159900, 165800, 172000,  494000,	-- 51 to 60
	574700, 614400, 650300, 682300, 710200, 734100, 753700, 768900, 779700,  810888 	-- 61 to 70
};

-------------------------------------------------------------------------------

local background    = CreateFrame( "Frame", "nUI_XPBarBackground", nUI_Dashboard.Anchor );
local frame         = nUI_Bars:createStatusBar( "nUI_XPBar", nUI_Dashboard.Anchor );
frame.text          = frame:CreateFontString( "$parentLabel", "OVERLAY" );
frame.rested        = frame:CreateTexture( "$parentRestedBar", "BORDER" );
frame.tick          = frame:CreateTexture( "$parentRestedTick", "ARTWORK" );
frame.rested.active = true;
frame.tick.active   = true;
frame.Super         = {};

frame.rested:SetTexture( 1, 1, 1, 1 );
frame.updateBar( nil, { 0, 0, 0, 0 } );
background:SetAllPoints( frame );

-------------------------------------------------------------------------------

local function UpdateXP( player_lvl )
	
	if player_lvl == MAX_PLAYER_LEVEL or not frame.options then
		
		if frame.showing then 
			frame.setEnabled( false );
			frame.showing = false;
			frame.bar:SetAlpha( 0 ); 
			frame.rested:SetAlpha( 0 );
			frame.tick:SetAlpha( 0 );
			frame:SetScript( "OnEnter", nil );
			frame:SetScript( "OnLeave", nil );
			frame:EnableMouse( false );
		end
		
	else

		local is_rested, name, mult  = GetRestState();
		local resting    = IsResting();
		local rested_xp  = GetXPExhaustion();
		local xp         = UnitXP( "player" );
		local xp_max     = UnitXPMax( "player" );
		local label      = frame.text;
		local xp_pct     = xp / xp_max;
		local txt_color;

		if frame.xp        ~= xp
		or frame.xp_max    ~= xp_max
		or frame.xp_pct    ~= xp_pct
		or frame.rested_xp ~= rested_xp
		or frame.level     ~= player_lvl
		then
			
			frame.xp        = xp;
			frame.xp_max    = xp_max;
			frame.xp_pct    = xp_pct;
			frame.rested_xp = rested_xp;
			frame.level     = player_lvl;
				
			if not frame.showing then
				
				frame.showing = true;
				frame.bar:SetAlpha( 1 ); 
				frame.rested:SetAlpha( 1 );
				frame.tick:SetAlpha( 1 );
				frame:EnableMouse( true );
				
				frame:SetScript( "OnEnter", 
				
					function()
						
						GameTooltip:SetOwner( frame );
						GameTooltip:SetText( nUI_L["Current level: <level>"]:format( frame.level ), 1, 0.83, 0 );
						
						if frame.xp then
							GameTooltip:AddLine( nUI_L["Current XP: <experience points>"]:format( frame.xp ), 1, 0.83, 0 );
						end
						
						if frame.xp_max then
							GameTooltip:AddLine( nUI_L["Required XP: <XP required to reach next level>"]:format( frame.xp_max ), 1, 0.83, 0 );
						end
						
						if frame.xp and frame.xp_max then
							GameTooltip:AddLine( nUI_L["Remaining XP: <XP remaining to level>"]:format( frame.xp_max - frame.xp ), 1, 0.83, 0 );
						end
						
						if frame.xp_pct then
							GameTooltip:AddLine( nUI_L["Percent complete: <current XP / required XP>"]:format( frame.xp_pct * 100 ), 1, 0.83, 0 );
						end
						
						if frame.rested_xp and frame.xp_max then
							GameTooltip:AddLine( nUI_L["Rested XP: <total rested experience> (percent)"]:format( frame.rested_xp, frame.rested_xp / frame.xp_max * 100 ), 1, 0.83, 0 );
						end

						if frame.levels and frame.levels > 0 then
							GameTooltip:AddLine( nUI_L["Rested Levels: <levels>"]:format( frame.levels ), 1, 0.83, 0 );
						end
						
						GameTooltip:Show();
						
					end
				);
				
				frame:SetScript( "OnLeave", function() GameTooltip:Hide(); end );
				
			end
			
			frame.updateBar( xp_pct, frame.options.bar.colors.xp );
			
			-- if we have some rested XP, then place a marker in the appropriate place
			
			if rested_xp and rested_xp > 0 then
	
				txt_color = frame.text.enabled and frame.options.label.color.rested;
	
				local offset = xp + rested_xp;
				local level  = player_lvl;
				local dX     = offset / xp_max;
	
				while offset > XpPerLevel[level] do 
					offset = offset - XpPerLevel[level]; 
					dX     = offset / XpPerLevel[level+1];
					level  = level+1; 
				end

				frame.levels = level - player_lvl + offset / XpPerLevel[level];
				
				-- we only display the rested XP end marker if the marker is on the 
				-- current level as the player or the player is less than lvl 69. Otherwise,
				-- there is no such thing as wraparound rested XP for a lvl69 player.
				
				if not (level > player_lvl and player_lvl+1 == MAX_PLAYER_LEVEL) then
						
					local orient = frame.options.bar.orient;
								
					if not frame.tick.active then 
						frame.tick.active = true;
						frame.tick:SetAlpha( 1 );
					end
					
					frame.tick:ClearAllPoints();
					
					if orient == "RIGHT" then
						frame.tick:SetPoint( "CENTER", frame, "RIGHT", -dX * frame:GetWidth(), -1.5 );
					elseif orient == "TOP" then				
						frame.tick:SetPoint( "CENTER", frame, "TOP", 0, -dX * frame:GetHeight() );
					elseif orient == "BOTTOM" then
						frame.tick:SetPoint( "CENTER", frame, "BOTTOM", 0, dX * frame:GetWidth() );
					else
						frame.tick:SetPoint( "CENTER", frame, "LEFT", dX * frame:GetWidth(), -1.5 );
					end
					
				elseif frame.tick.active then
	
					frame.tick.active = false;
					frame.tick:SetAlpha( 0 );
					
				end
				
				-- draw a rested XP bar behind the XP bar
				
				if xp + rested_xp > xp_max then
	
					if frame.options.bar.orient == "TOP" or frame.options.bar.orient == "BOTTOM" then
						frame.rested:SetHeight( frame:GetHeight() * (1 - xp_pct ) );
					else
						frame.rested:SetWidth( frame:GetWidth() * (1 - xp_pct ) );
					end
					
				-- draw a partial bar from the end of the current XP to 
				-- XP + rested XP
				
				else
	
					local rested_pct = rested_xp / xp_max;
					
					if frame.options.bar.orient == "TOP" or frame.options.bar.orient == "BOTTOM" then
						frame.rested:SetHeight( frame:GetHeight() * rested_pct );
					else
						frame.rested:SetWidth( frame:GetWidth() * rested_pct );
					end
				end
				
			else
				
				text_color = frame.text.enabled and frame.options.label.color.normal;
				
				if frame.tick.active then
				
					frame.tick.active = false;
					frame.tick:SetAlpha( 0 );
					
				end
				
				if frame.rested.active then
					
					frame.rested.active = false;
					frame.rested:SetAlpha( 0 );
					
				end
			end
		
			if label.enabled then
				
				label:SetText( nUI_L["Level <player level>: <experience> of <max experience> (<percent of total>), <rested xp> rested XP"]:format( player_lvl, xp, xp_max, xp_pct * 100, rested_xp or 0 ) );
				label:SetTextColor( txt_color.r, txt_color.g, txt_color.b, 1 );
			
			end
		end
	end
end

-------------------------------------------------------------------------------

local function onXPBarEvent()
	
	if event == "VARIABLES_LOADED" then
		
		nUI:registerScalableFrame( frame );
		nUI:registerSkinnedFrame( frame );
		
	elseif event == "PLAYER_ENTERING_WORLD" then 
	
		UpdateXP( nUI_Unit.PlayerInfo and nUI_Unit.PlayerInfo.level or UnitLevel( "player" ) );
		
	elseif event == "PLAYER_LEVEL_UP" then 
	
		UpdateXP( tonumber( arg1 ) );
		
	else
	
		UpdateXP( nUI_Unit.PlayerInfo.level );
		
	end
end

background:SetScript( "OnEvent", onXPBarEvent );
background:RegisterEvent( "VARIABLES_LOADED" );
background:RegisterEvent( "PLAYER_ENTERING_WORLD" );
background:RegisterEvent( "PLAYER_XP_UPDATE" );
background:RegisterEvent( "UPDATE_EXHAUSTION" );
background:RegisterEvent( "PLAYER_LEVEL_UP" );
background:RegisterEvent( "PLAYER_UPDATE_RESTING" );
background:RegisterEvent( "PLAYER_ENTERING_WORLD" );

-------------------------------------------------------------------------------

frame.Super.onSizeChanged = frame.onSizeChanged;
frame.onSizeChanged       = function()

	frame.Super.onSizeChanged();
	
	if frame.bar.horizontal then
		frame.rested:SetHeight( frame.bar:GetHeight() );
	else
		frame.rested:SetWidth( frame.bar:GetWidth() );
	end
end

-------------------------------------------------------------------------------

frame.Super.setOrientation = frame.setOrientation;
frame.setOrientation       = function( orient )
	
	frame.Super.setOrientation( orient );
	frame.rested:ClearAllPoints();
	
	if orient == "RIGHT" then
		frame.rested:SetPoint( "TOPRIGHT", frame.bar, "TOPLEFT", 0, 0 );
		frame.rested:SetPoint( "BOTTOMRIGHT", frame.bar, "BOTTOMLEFT", 0, 0 );
	elseif orient == "TOP" then
		frame.rested:SetPoint( "TOPLEFT", frame.bar, "BOTTOMLEFT", 0, 0 );
		frame.rested:SetPoint( "TOPRIGHT", frame.bar, "BOTTOMRIGHT", 0, 0 );
	elseif orient == "BOTTOM" then
		frame.rested:SetPoint( "BOTTOMLEFT", frame.bar, "TOPLEFT", 0, 0 );
		frame.rested:SetPoint( "BOTTOMRIGHT", frame.bar, "TOPRIGHT", 0, 0 );
	else
		frame.rested:SetPoint( "TOPLEFT", frame.bar, "TOPRIGHT", 0, 0 );
		frame.rested:SetPoint( "BOTTOMLEFT", frame.bar, "BOTTOMRIGHT", 0, 0 );
	end
end
		
-------------------------------------------------------------------------------

frame.applyScale = function( scale )
	
	local anchor  = scale and frame.anchor or nil;
	local scale   = scale or frame.scale or 1;
	local options = frame.options;
	
	frame.scale = scale;
	
	if options then
		
		local width       = options.width * scale * nUI.scale;
		local height      = options.height * scale * nUI.scale;
		local tick_width  = (options.bar and options.bar.tick_width or 0) * scale * nUI.scale;
		local tick_height = (options.bar and options.bar.tick_height or 0) * scale * nUI.scale;
		local text        = frame.text;
		local font_size   = (options.label and options.label.fontsize or 12) * scale * 1.75 * nUI.scale;
		local justifyH    = options.label  and options.label.justifyH or "CENTER";
		local justifyV    = options.label  and options.label.justifyV or "MIDDLE";
	
		-- set the bar size
		
		if frame.width  ~= width
		or frame.height ~= height
		then
			
			frame.width  = width;
			frame.height = height;
			
			frame:SetHeight( height );
			frame:SetWidth( width );

			frame.onSizeChanged();
			
		end
		
		if frame.tick.width ~= tick_width
		or frame.tick.height ~= tick_height
		then
			
			frame.tick.width  = tick_width;
			frame.tick.height = tick_height;
			
			frame.tick:SetWidth( tick_width );
			frame.tick:SetHeight( tick_height );
		
		end
		
		-- set the text font size
					
		if text.font_size ~= font_size
		then
			
			-- first time here?
			
			if not text.font_size then 
				text.active  = true;
			end
	
			text.font_size = font_size;
			text:SetFont( nUI_L["font1"], font_size, "OUTLINE" );

		end
		
		-- show or hide the text based on whether or not there is a config for it
		
		text.enabled = options.label and options.label.enabled or false;
		
		if not text.enabled and text.active then

			text.active = false;
			text.value  = nil;				
			text:SetAlpha( 0 );
			text:SetText( "" );				
			
		elseif text.enabled then

			text.active = true;
			text:SetAlpha( 1 );
			
		end
		
		-- set text justification
		
		if text.justifyH ~= justifyH then
			text.justifyH = justifyH;
			text:SetJustifyH( justifyH );
		end
		
		if text.justigyV ~= justifyV then
			text.justifyV = justifyV;
			text:SetJustifyV( justifyV );
		end
	end
	
	if anchor then frame.applyAnchor( anchor ); end
	
end

-------------------------------------------------------------------------------

frame.applyAnchor = function( anchor )
	
	local anchor      = anchor or frame.anchor or {};
	local anchor_pt   = anchor.anchor_pt or "CENTER";
	local relative_to = anchor.relative_to or frame:GetParent():GetName();
	local relative_pt = anchor.relative_pt or anchor_pt;
	local xOfs        = (anchor.xOfs or 0) * nUI.scale;
	local yOfs        = (anchor.yOfs or 0) * nUI.scale;
	
	if frame.anchor_pt   ~= anchor_pt
	or frame.relative_to ~= relative_to
	or frame.relative_pt ~= relative_pt
	or frame.xOfs        ~= xOfs
	or frame.yOfs        ~= yOfs
	then
		
		frame.anchor_pt = anchor_pt;
		frame.relative_to = relative_to;
		frame.relative_pt = relative_pt;
		frame.xOfs        = xOfs;
		frame.yOfs        = yOfs;
		
		frame:ClearAllPoints();
		frame:SetPoint( anchor_pt, relative_to, relative_pt, xOfs, yOfs );
		
	end
	
	if frame.options and frame.text.enabled then
		
		local text  = frame.text;
		local label = frame.options.label;

		anchor_pt   = label.anchor_pt or "CENTER";
		relative_to = label.relative_to or frame:GetName();
		relative_pt = label.relative_pt or anchor_pt;
		xOfs        = (label.xOfs or 0) * frame.scale * nUI.scale;
		yOfs        = (label.yOfs or 0) * frame.scale * nUI.scale;
		
		if text.anchor_pt ~= anchor_pt
		or text.relative_to ~= relative_to
		or text.relative_pt ~= relative_pt
		or text.xOfs        ~= xOfs
		or text.yOfs        ~= yOfs
		then
			
			text.anchor_pt = anchor_pt;
			text.relative_to = relative_to;
			text.relative_pt = relative_pt;
			text.xOfs        = xOfs;
			text.yOfs        = yOfs;
			
			text:SetPoint( anchor_pt, relative_to, relative_pt, xOfs, yOfs );
			
		end
	end
end

-------------------------------------------------------------------------------

frame.applyOptions = function( options )
	
	frame.options = options;
	
	if not options or not options.enabled then

		frame.setEnabled( false );
		frame:Hide();
		
	else
		
		frame.setEnabled( true );
		frame:Show();
	
		background:SetFrameStrata( options.strata or background:GetParent():GetFrameStrata() );
		background:SetFrameLevel( background:GetFrameLevel() + (options.level or 1) );
		
		frame:SetFrameStrata( background:GetFrameStrata() );
		frame:SetFrameLevel( background:GetFrameLevel()+1 );
		
		-- set up the bar
		
		if not options.bar or not options.bar.enabled then
			
			frame.bar.enabled = false;
			frame.bar:SetAlpha( 0 );
			frame.rested:SetAlpha( 0 );
			frame.tick:SetAlpha( 0 );
			
		else
			
			frame.bar.enabled = true;
			frame.bar:SetAlpha( 1 );
			frame.rested:SetAlpha( 1 );
			frame.tick:SetAlpha( 1 );
			
			frame.tick:SetTexture( options.bar.rested_tick );
			frame.tick:SetTexCoord( 0, 0, 0, 1, 1, 0, 1, 1 );
			frame.setOverlay( options.bar.overlay );
			frame.setBar( nil, 0, 1 );
			frame.setOrientation( options.bar.orient );

			frame.rested:SetVertexColor( options.bar.colors.rested.r, options.bar.colors.rested.g, options.bar.colors.rested.b, 1 );
			
		end
		
		-- if there's a border, set it
		
		if options.border then
				
			local border_color = options.border.color.border;
			local backdrop_color = options.border.color.backdrop;
			
			frame:SetBackdrop( options.border.backdrop );
			frame:SetBackdropBorderColor( border_color.r, border_color.g, border_color.b, border_color.a );
			frame:SetBackdropColor( backdrop_color.r, backdrop_color.g, backdrop_color.b, backdrop_color.a );
	
		else 
			
			frame:SetBackdrop( nil );
			
		end
		
		-- if there's a background, set it
		
		if options.background then
				
			local border_color = options.background.color.border;
			local backdrop_color = options.background.color.backdrop;
			
			background:SetAlpha( 1 );
			background:SetBackdrop( options.background.backdrop );
			background:SetBackdropBorderColor( border_color.r, border_color.g, border_color.b, border_color.a );
			background:SetBackdropColor( backdrop_color.r, backdrop_color.g, backdrop_color.b, backdrop_color.a );
	
		else 
			
			background:SetAlpha( 0 );
			background:SetBackdrop( nil );
			
		end
		
		frame.applyScale( options.scale or frame.scale or 1 );
		
	end
end

-------------------------------------------------------------------------------

frame.applySkin = function( skin )
	
	local skin = skin and skin.XPBar or nUI_DefaultConfig.XPBar;
	
	if not skin or not skin.enabled then
		
		frame.setEnabled( false );
		frame:Hide();
		background:Hide();
		
	else
		
		frame.setEnabled( true );
		frame:Show();
		background:Show();
		
		frame.applyOptions( skin.options );
		frame.applyAnchor( skin.anchor );
		
	end
end
