﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--]]---------------------------------------------------------------------------

if not nUI then nUI = {}; end
if not nUI_Unit then nUI_Unit = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local CreateFrame          = CreateFrame;
local GetContainerItemLink = GetContainerItemLink;
local GetContainerNumSlots = GetContainerNumSlots;
local GetItemCount         = GetItemCount;
local GetItemInfo          = GetItemInfo;
local GetNumSpellTabs      = GetNumSpellTabs;
local GetPetFoodTypes      = GetPetFoodTypes;
local GetSpellName         = GetSpellName;
local GetSpellTabInfo      = GetSpellTabInfo;
local GetSpellTexture      = GetSpellTexture;
local GetTime              = GetTime;
local InCombatLockdown     = InCombatLockdown;

-------------------------------------------------------------------------------

nUI_DefaultConfig.PetFeeder =
{
	size = 50,
	cols = 6,
	gap  = 2,
};

-------------------------------------------------------------------------------
-- create the pet feeder 

nUI_PetFeederEvents   = CreateFrame( "Frame", "nUI_PetFeederEvents", WorldFrame );
nUI_PetFeeder         = CreateFrame( "Frame", "nUI_PetFeeder", nUI_Dashboard.Anchor );
nUI_PetFeeder.Options = CreateFrame( "Frame", "$parentOptions", nUI_PetFeeder );
nUI_PetFeeder.Label   = nUI_PetFeeder:CreateFontString();
nUI_PetFeeder.Buttons = {};

nUI_PetFeeder.Label:SetPoint( "TOP", nUI_PetFeeder, "TOP", 0, -5 );
nUI_PetFeeder.Label:SetJustifyH( "CENTER" );
nUI_PetFeeder.Label:SetJustifyV( "TOP" );
nUI_PetFeeder.Label:SetTextColor( 1, 0.83, 0, 1 );

nUI_PetFeeder:SetBackdrop(
	{
		bgFile   = "Interface\\Tooltips\\UI-Tooltip-Background", 
		edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", 
		tile     = true, 
		tileSize = 1, 
		edgeSize = 10, 
		insets   = {left = 2, right = 2, top = 2, bottom = 2},
	}
);

nUI_PetFeeder:SetFrameStrata( "HIGH" );
nUI_PetFeeder:SetFrameLevel( 0 );
nUI_PetFeeder:Hide();
nUI_PetFeeder:SetBackdropColor( 0, 0, 0, 0.75 );

nUI_PetFeeder.Options:SetFrameStrata( nUI_PetFeeder:GetFrameStrata() );
nUI_PetFeeder.Options:SetFrameLevel( nUI_PetFeeder:GetFrameLevel()+1 );

-------------------------------------------------------------------------------
-- ensures any buttons we have defined will not reappear the next time we go
-- to feed the pet unless they are needed at that time

nUI_PetFeeder:SetScript( "OnHide",

	function()		
		if nUI_PetFeeder.ValidFoods then
			for i in pairs( nUI_PetFeeder.Buttons ) do
				nUI_PetFeeder.Buttons[i]:Hide();
			end
		end
	end
);

-------------------------------------------------------------------------------
-- extracts the Bliz item ID from the Bliz item link

function nUI_GetItemIdFromLink( itemLink )

	local itemId = nil;
	
	if itemLink then
		
		local _, _, id  = string.find( itemLink, "item:(%d+)" );
	
		itemId = tonumber( id );

	end
	
	return itemId;
end

-------------------------------------------------------------------------------
-- sorts a list of foods by descending food level

local function nUI_SortFoodList( left, right )
	
	if left.itemLevel > right.itemLevel then
		return true;
		
	elseif left.itemLevel == right.itemLevel then
		return left.itemName < right.itemName;
		
	end
	
	return false;
	
end

-------------------------------------------------------------------------------
-- given the current pet, what list of foods is that pet capable of eating?

local function UpdateFoodList( pet_info )

	-- if we don't already know it, discover the name of the feed pet spell
	
	if not nUI_PetFeeder.feedPetSpell then
			
		for i = 1, GetNumSpellTabs() do
			
			local name, icon, offset, count = GetSpellTabInfo( i );
			
			if icon == "Interface\\Icons\\Ability_Hunter_BeastTaming" then
				
				for id = offset + 1, offset + count do
					
					icon = GetSpellTexture( id, BOOKTYPE_SPELL );
					
					if icon == "Interface\\Icons\\Ability_Hunter_BeastTraining" then
						nUI_PetFeeder.feedPetSpell = GetSpellName( id, BOOKTYPE_SPELL );
--						nUI:debug( ("nUI_PetFeeder: found the player's feed pet spell at %d -- [%s]"):format( id, nUI_PetFeeder.feedPetSpell or "<nil>" ), 1 );
					end
				end
			end
		end
	end
	
	-- determine what type of foods this pet eats if we haven't yet
	
	if not pet_info.pet_foods then
		pet_info.modified    = true;
		pet_info.last_change = GetTime();
		pet_info.pet_foods   = { GetPetFoodTypes() };
		
		if #pet_info.pet_foods == 0 then
			pet_info.pet_foods = nil;
--			nUI:debug( "nUI_PetFeeder: GetPetFoodTypes() returned an empty food list for "..pet_info.name, 1 );
		else
--			for value in pairs( pet_info.pet_foods ) do
--				nUI:debug( ("nUI_PetFeeder: %s can eat %s"):format( pet_info.name, pet_info.pet_foods[value] ), 1 );
--			end
		end
	end

	-- build a list of foods this pet can and will eat
	
	local found_food = false;
	local foodCache  = nUI_PetFeeder.FoodList;
	local petFoods   = pet_info.pet_foods;
	local validFoods = {};
	
	if petFoods and #petFoods > 0 then

		local petLevel = pet_info.level;

		-- for each type of food this pet will eat...
		
		for i, foodType in ipairs( petFoods ) do
	
--			nUI:debug( "nUI_PetFeeder: processing food list for "..foodType.."...", 1 );
			
			local foodList = foodCache[strlower(foodType)] or {};
		
			-- for each food item in the local cache...

			for itemId in pairs( foodList ) do
				
				local food = foodList[itemId];

				-- if we don't know anything about the food item yet, get
				-- its details from the user's local data cache
				
				if not food.itemLink then

					-- fetch info about the item
					
					food.itemName, food.itemLink, food.itemRarity, food.itemLevel, 
					food.itemMinLevel, food.itemType, food.itemSubType, food.itemCount, 
					food.itemEquipLoc, food.itemTexture = GetItemInfo( itemId );

					if food.itemName and not food.feedButton then
			
--						nUI:debug( "nUI_PetFeeder: creating a pet feeder button for "..food.itemName, 1 );
						
						-- create a button to use when feeding that item
						
						local name   = nUI_PetFeeder:GetName().."_Button_"..itemId;
						
						button = CreateFrame( "CheckButton", name, nUI_PetFeeder, "ActionBarButtonTemplate" );
												
						button:SetNormalTexture( "" );
						button:SetFrameStrata( nUI_PetFeeder.Options:GetFrameStrata() );
						button:SetFrameLevel( nUI_PetFeeder.Options:GetFrameLevel()+1 );

						button.SetNormalTexture = function() end;
						button.layers           = {};
						button.layers.icon      = _G[name.."Icon"];
						button.layers.count     = _G[name.."Count"];

						button.layers.icon:SetTexture( food.itemTexture );
						button.layers.icon.SetTexture = function() end;
						
						-- record the data and save the button
						
						food.itemId     = itemId;
						food.foodType   = foodType;						
						food.feedButton = button;	
						
					end					
				end			
				
				-- if the user knows about this food item and the item's level is valid
				-- for this pet, then add it to the list of valid foods

				if not food.itemId then
				
--					nUI:debug( "nUI_PetFeeder: food item "..itemId.." has no data and cannot be added to the food list (uncached?)", 1 );
					
				elseif food.itemLevel < petLevel-30 then
				
--					nUI:debug( "nUI_PetFeeder: food item "..food.itemName.." is too low level for "..pet_info.name, 1 );
					
--				elseif food.itemMinLevel ~= 0 and food.itemMinLevel > petLevel then
					
--					nUI:debug( "nUI_PetFeeder: food item "..food.itemName.." is too high level for "..pet_info.name, 1 );
					
				else

--					nUI:debug( "nUI_PetFeeder: adding food item "..food.itemName.." to the list of valid foods", 1 );
					validFoods[food.itemId] = food;
					found_food = true;
					
				end
			end
		end		
	end	
	
	-- if the food list has changed, update the unit_info
	
	local modified = false;
	
	if not found_food then
		
--		nUI:debug( "nUI_PetFeeder: "..pet_info.name.."'s valid food list is empty", 1 );
		
		if pet_info.valid_foods then
			pet_info.modified    = true;
			pet_info.last_change = GetTime();
			pet_info.valid_foods = nil;
		end
		
	elseif not pet_info.valid_foods then
		
		modified = true;
		
	else
		
		for id in pairs( pet_info.valid_foods ) do
			if not validFoods[id]
			or pet_info.valid_foods[id] ~= validFoods[id] then
				modified = true;
				break;
			end
		end
		
		if not modified then
			for id in pairs( validFoods ) do
				if not pet_info.valid_foods[id]
				or pet_info.valid_foods[id] ~= validFoods[id] then
					modified = true;
					break;
				end
			end
		end
	end
	
	if modified then 
		
--		nUI:debug( "nUI_PetFeeder: "..pet_info.name.." currently has valid foods in the food list", 1 );
		
		pet_info.modified    = true;
		pet_info.last_change = GetTime();
		pet_info.valid_foods = validFoods;
		
	end	
end

-------------------------------------------------------------------------------
-- feeder event management

local function onPetFeederEvent()

	if event == "VARIABLES_LOADED" then

		-- initialize the pet feeder configuration
		
		nUI:patchConfig();
		nUI_PetFeeder:configOptions( use_default );
		
		-- let us know when the player's pet changes or when the pet's level changes
		
--		nUI_Unit:registerUnitChangeCallback( "pet", nUI_PetFeeder );
--		nUI_Unit:registerLevelCallback( "pet", nUI_PetFeeder );
		
	end
end

nUI_PetFeederEvents:RegisterEvent( "VARIABLES_LOADED" );
nUI_PetFeederEvents:SetScript( "OnEvent", onPetFeederEvent );

-------------------------------------------------------------------------------
-- this callback method is executed when the player's pet changes

nUI_PetFeeder.newUnitInfo = function( list_unit, unit_info )

	if unit_info and unit_info.is_hunter_pet then
		UpdateFoodList( unit_info );
	end
end

-------------------------------------------------------------------------------

function nUI_PetFeeder:configOptions( use_default )
	
	if not nUI_PetFeederOptions then nUI_PetFeederOptions = {}; end
	
	local config  = nUI_PetFeederOptions;
	local default = nUI_DefaultConfig.PetFeeder;
	
	if use_default then
		
		config.size = default.size;
		config.cols = default.cols;
		config.gap  = default.gap;
		
	else
		
		config.size = tonumber( config.size or default.size );
		config.cols = tonumber( config.cols or default.cols );
		config.gap  = tonumber( config.gap or default.gap );
	
	end
end

-------------------------------------------------------------------------------

function nUI_PetFeeder:FeedPet( caller )

	local pet_info = nUI_Unit.PetInfo;
	
	if not pet_info then
			
		DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: I looked everywhere, but I couldn't find a pet to feed. Perhaps he's in your backpack?"], 1, 0.83, 0 );
		
	elseif not pet_info.is_hunter_pet then
		
		DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: You know, I could be wrong, but I don't think feeding %s is a very good idea... it doesn't look to me like what you have in your bags is what %s is thinking about eating."]:format( pet_info.name, pet_info.name ), 1, 0.83, 0 );
		
	elseif InCombatLockdown() then
			
		DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: It looks to me like you're a little busy... maybe you should try feeding %s AFTER you leave combat?"]:format( pet_info.name ), 1, 0.83, 0 );
		
	else

		-- toggle the pet feeder off if it is currently active
		
		if nUI_PetFeeder:IsShown() then
			
			nUI_PetFeeder:Hide();

		-- otherwise, prepare to show the feeder
		
		else
				
			local validFoods = pet_info.valid_foods;
			local foods = {};
	
			-- sometimes we can't get the food list when the player first logs in
			-- or otherwise zones. If we don't have a food list already, try to
			-- get one now
			
			if not validFoods or not #validFoods then
				
				UpdateFoodList( pet_info );				
				validFoods = pet_info.valid_foods;
				
			end
			
			if validFoods then
					
				-- build a list of foods in inventory for the current pet
				
				for i=0,4 do
					
					for j=0, GetContainerNumSlots( i ) do
						
						local itemLink = GetContainerItemLink( i, j );
		
						if itemLink then
							
							local itemId   = nUI_GetItemIdFromLink( itemLink );
							
							if itemId and validFoods[itemId] then
								nUI:TableInsertByValue( foods, validFoods[itemId] );
								validFoods[itemId].itemBag  = i;
								validFoods[itemId].itemSlot = j;
							end				
						end
					end
				end
		
				-- sort the list descending by food level
				
				table.sort( foods, nUI_SortFoodList );
	
--				nUI:debug( ("nUI_PetFeeder: found %d item%s in your bags to feed %s"):format( #foods, #foods == 1 and "" or "s", pet_info.name ), 1 );
				
				-- label the feeder
				
				nUI_PetFeeder.Label:SetFont( nUI_L["font1"], 22 * nUI.scale, "OUTLINE" );
				nUI_PetFeeder.Label:SetText( nUI_L["Click to feed %s"]:format( pet_info.name ) );
					
				-- make sure all of the existing buttons in the feeder frame are hidden
				
	
				if nUI_PetFeeder.Buttons then
					for id in pairs( nUI_PetFeeder.Buttons ) do
						nUI_PetFeeder.Buttons[id]:ClearAllPoints();
						nUI_PetFeeder.Buttons[id]:Hide();
					end
				end
				
				-- start a new button list
				
				nUI_PetFeeder.Buttons = {};
				
				-- lay the buttons out in the feed frame
				
				local width       = 0;
				local height      = 0;
				local labelWidth  = nUI_PetFeeder.Label:GetStringWidth();
				local labelHeight = nUI_PetFeeder.Label:GetStringHeight();
				local size        = nUI_PetFeederOptions.size * nUI.scale;
				local gap         = nUI_PetFeederOptions.gap * nUI.scale;
				local got_food    = false;
				local last_row    = nil;
				local last_col    = nil;
				local buttonId    = 1;
				
				for i in pairs( foods ) do
					
					local foodItem = foods[i];
					local button   = foodItem.feedButton;
	
					nUI_PetFeeder.Buttons[buttonId] = button;
					buttonId = buttonId+1;
									
					button:ClearAllPoints();				
					button:SetScale( size / button:GetHeight() );
					button:RegisterForClicks( "LeftButtonUp" );
					button:SetAttribute( "type1",       "spell");
					button:SetAttribute( "spell1",      nUI_PetFeeder.feedPetSpell );
					button:SetAttribute( "target-bag",  foodItem.itemBag );
					button:SetAttribute( "target-slot", foodItem.itemSlot );
					button:Show();
	
					button:SetScript( "PostClick", 
						function() 
							nUI_PetFeeder:Hide(); 
						end 
					);
					
					button:SetScript( "OnEnter", 
						function() 
							GameTooltip:SetOwner( button );
							GameTooltip:SetBagItem( foodItem.itemBag, foodItem.itemSlot );
							GameTooltip:Show();
						end 
					);
					
					button:SetScript( "OnLeave", 
						function() 
							GameTooltip:Hide(); 
						end 
					);
	
					button.layers.count:SetText( GetItemCount( foodItem.itemLink, false ) );
					
					if not got_food then
						
						button:SetPoint( "TOPLEFT", nUI_PetFeeder.Options, "TOPLEFT", 0, 0 );
						height   = size;
						width    = size;
						last_row = button;
						got_food = true;
					
					elseif (i % nUI_PetFeederOptions.cols) == 0 then
						
						button:SetPoint( "TOPLEFT", last_row, "BOTTOMLEFT", 0, -gap );
						height   = height + size + gap;
						last_row = button;
						
					else
						
						button:SetPoint( "TOPLEFT", last_col, "TOPRIGHT", gap, 0 );
						
						if i <= nUI_PetFeederOptions.cols then
							width = width + size + gap;
						end
					end
					
					last_col = button;
	
				end
	
				nUI_PetFeeder.Options:ClearAllPoints();
				nUI_PetFeeder.Options:SetWidth( width );
				nUI_PetFeeder.Options:SetHeight( height );
				nUI_PetFeeder.Options:SetPoint( "BOTTOM", nUI_PetFeeder, "BOTTOM", 0, 5 );
	
				nUI_PetFeeder:ClearAllPoints();
				nUI_PetFeeder:SetHeight( height + labelHeight + 15 );
				nUI_PetFeeder:SetWidth( max( labelWidth, width ) + 10 );
				
				if caller then nUI_PetFeeder:SetPoint( "BOTTOMLEFT", caller, "TOPLEFT", 0, 0 );
				else nUI_PetFeeder:SetPoint( "CENTER", WorldFrame, "CENTER", 0, 0 );
				end
				
				nUI_PetFeeder:Show();			
			end			
		end
	end
end

-------------------------------------------------------------------------------

function nUI_PetFeeder:PrintFoodTypes( r, g, b )

	local pet_info = nUI_Unit.PetInfo;
	
	if pet_info and pet_info.is_hunter_pet then
			
		local PetFoods = pet_info.pet_foods;
		
		if PetFoods and #PetFoods > 0 then
			
			local foodList = "";
			local lastFood;
			
			for i, type in ipairs( nUI_PetFeeder.PetFoods ) do
	
				local item = "|cFF00FFFF"..type.."|r";
				
				if i == 1 then
					foodList = item;
				elseif i == #PetFoods then  
					last_food = item;
				elseif i > 1 then
					foodList = foodList..", "..item;
				end
			end
			
			DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: %s can eat %s%s%s"]:format( pet_info.name or nUI_L["Your pet"], foodList, nUI_L[" or "], lastFood ), r, g, b );
			
		end
	end
end

-------------------------------------------------------------------------------

function nUI_PetFeeder:PrintPetFoods()

	local pet_info = nUI_Unit.PetInfo;
	
	if pet_info and pet_info.is_hunter_pet then

		local validFoods = pet_info.valid_foods;
		local foodList = {};
		
		for i=0,4 do
			
			for j=0, GetContainerNumSlots( i ) do
				
				local itemLink = GetContainerItemLink( i, j );

				if itemLink then
					
					local itemId = nUI_GetItemIdFromLink( itemLink );
						
					if itemId and validFoods[itemId] then
						foodList[itemId] = validFoods[itemId];
					end				
				end
			end
		end
							
		nUI_PetFeeder:PrintFoodList( foodList );
	end	
end

-------------------------------------------------------------------------------

function nUI_PetFeeder:PrintFoodList( foodList, r, g, b )

	local pet_info = nUI_Unit.PetInfo;
	
	r = r or 1;
	g = g or 0.83;
	b = b or 0;
	
	if pet_info and pet_info.is_hunter_pet then
		
		if not foodList then
			
			foodList = pet_info.valid_foods;
			
		end
	
		if foodList then
	
			local foods = {};
			
			for itemId in pairs( foodList ) do
				nUI:TableInsertByValue( foods, foodList[itemId] );
			end
			
			table.sort( foods, nUI_SortFoodList );
			
			local got_food = false;
					
			for i in pairs( foods ) do
				
				local foodItem = foods[i];
	
				if not got_food then
					DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: You can feed %s the following...\n"]:format( pet_info.name ), r, g, b );
					got_food = true;
				end
	
				DEFAULT_CHAT_FRAME:AddMessage( "    "..foodItem.itemName.." -- "..foodItem.itemLevel.." "..foodItem.foodType.." ("..GetItemCount( foodItem.itemLink, false )..")", r, g, b );
	
			end
				
			if not got_food then
				
				DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: You have nothing you can feed %s in your current inventory"]:format( pet_info.name ), r, g, b );
	
			else
				
				DEFAULT_CHAT_FRAME:AddMessage( " ", r, g, b );
				DEFAULT_CHAT_FRAME:AddMessage( nUI_L["    Click the pet happiness indictor to feed your pet now"], r, g, b );
				
			end
		end
	else
		
		DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: You don't have a pet!"], r, g, b );
		
	end
end

-------------------------------------------------------------------------------
-- This list of known food types has been unabashadly lifted from Fizzwidget 
-- Feed-o-Matic and all credit for researching the food types, etc., goes to
-- it's author(s). That said, my implementation of the list is entirely unique.

nUI_PetFeeder.FoodList =
{
	[nUI_FOOD_FUNGUS] = 
	{
		[28112] = {},	-- Underspore Pod
		[24539] = {},	-- Marsh Lichen		
		[27859] = {},	-- Zangar Caps		
		[4607]  = {},	-- Delicious Cave Mold		
		[8948]  = {},	-- Dried King Bolete		
		[4604]  = {},	-- Forest Mushroom Cap		
		[4608]  = {},	-- Raw Black Truffle		
		[4605]  = {},	-- Red-speckled Mushroom		
		[3448]  = {},	-- Senggin Root		
		[4606]  = {},	-- Spongy Morel		
	},					
	[nUI_FOOD_FISH] = 
	{
		[5527]  = {},	-- Goblin Deviled Clams
		[6316]  = {},	-- Loch Frenzy Delight
		[12238] = {},	-- Darkshore Grouper
		[13759] = {},	-- Raw Nightfin Snapper
		[16766] = {},	-- Undermine Clam Chowder
		[21071] = {},	-- Raw Sagefish
		[21552] = {},	-- Striped Yellowtail
		[21153] = {},	-- Raw Greater Sagefish
		[27665] = {},	-- Poached Bluefish		
		[27437] = {},	-- Icefin Bluefish		
		[27667] = {},	-- Spicy Crawdad
		[27439] = {},	-- Furious Crawdad
		[27438] = {},	-- Golden Darter
		[27435] = {},	-- Figluster's Mudfish
		[27664] = {},	-- Grilled Mudfish
		[27663] = {},	-- Blackened Sporefish
		[27429] = {},	-- Zangarian Sporefish
		[27662] = {},	-- Feltail Delight
		[27425] = {},	-- Spotted Feltail
		[27422] = {},	-- Barbed Gill Trout
		[27661] = {},	-- Blackened Trout
		[30155] = {},	-- Clam Bar
		[24477] = {},	-- Jaggal Clam Meat
		[27858] = {},	-- Sunspring Carp
		[29452] = {},	-- Zangar Trout
		[13889] = {},	-- Raw Whitescale Salmon		
		[19996] = {},	-- Harvest Fish
		[13933] = {},  -- Lobster Stew
		[13888] = {},  -- Darkclaw Lobster
		[13935] = {},	-- Baked Salmon		
		[13546] = {},	-- Bloodbelly Fish		
		[6290]  = {},	-- Brilliant Smallfish		
		[4593]  = {},	-- Bristle Whisker Catfish		
		[5503]  = {},	-- Clam Meat		
		[16971] = {},	-- Clamlette Surprise		
		[2682]  = {},	-- Cooked Crab Claw		
		[13927] = {},	-- Cooked Glossy Mightfish		
		[2675]  = {},	-- Crawler Claw		
		[13930] = {},	-- Filet of Redgill		
		[5476]  = {},	-- Fillet of Frenzy		
		[4655]  = {},	-- Giant Clam Meat		
		[6038]  = {},	-- Giant Clam Scorcho		
		[13928] = {},	-- Grilled Squid		
		[13929] = {},	-- Hot Smoked Bass		
		[13893] = {},	-- Large Raw Mightfish		
		[4592]  = {},	-- Longjaw Mud Snapper		
		[8364]  = {},	-- Mithril Head Trout		
		[13932] = {},	-- Poached Sunscale Salmon		
		[5095]  = {},	-- Rainbow Fin Albacore		
		[6291]  = {},	-- Raw Brilliant Smallfish		
		[6308]  = {},	-- Raw Bristle Whisker Catfish		
		[13754] = {},	-- Raw Glossy Mightfish		
		[6317]  = {},	-- Raw Loch Frenzy		
		[6289]  = {},	-- Raw Longjaw Mud Snapper		
		[8365]  = {},	-- Raw Mithril Head Trout		
		[6361]  = {},	-- Raw Rainbow Fin Albacore		
		[13758] = {},	-- Raw Redgill		
		[6362]  = {},	-- Raw Rockscale Cod		
		[6303]  = {},	-- Raw Slitherskin Mackerel		
		[8959]  = {},	-- Raw Spinefin Halibut		
		[4603]  = {},	-- Raw Spotted Yellowtail		
		[13756] = {},	-- Raw Summer Bass		
		[13760] = {},	-- Raw Sunscale Salmon		
		[4594]  = {},	-- Rockscale Cod		
		[787]   = {},	-- Slitherskin Mackerel		
		[5468]  = {},	-- Soft Frenzy Flesh		
		[15924] = {},	-- Soft-shelled Clam Meat		
		[12216] = {},	-- Spiced Chili Crab		
		[8957]  = {},	-- Spinefin Halibut		
		[6887]  = {},	-- Spotted Yellowtail		
		[5504]  = {},	-- Tangy Clam Meat		
		[12206] = {},	-- Tender Crab Meat		
		[13755] = {},	-- Winter Squid		
		[7974]  = {},	-- Zesty Clam Meat		
	},					
	[nUI_FOOD_MEAT] = 
	{
		[729]   = {},	-- Stringy Vulture Meat
		[1080]  = {},	-- Tough Condor Meat
		[2886]  = {},	-- Crag Boar Rib
		[7097]  = {},	-- Leg Meat
		[19304] = {},	-- Spiced Beef Jerky
		[19305] = {},	-- Pickled Kodo Foot
		[19306] = {},	-- Crunchy Frog
		[19223] = {},	-- Darkmoon Dog
		[20424] = {},	-- Sandworm Meat
		[20074] = {},	-- Heavy Crocolisk Stew
		[21023] = {},	-- Dirge's Kickin' Chimaerok Chops
		[23495] = {},	-- Springpaw Appetizer
		[27668] = {},	-- Lynx Meat
		[27635] = {},	-- Lynx Steak
		[27669] = {},	-- Bat Flesh
		[27636] = {},	-- Bat Bites
		[22644] = {},	-- Crunchy Spider Leg
		[22645] = {},	-- Crunchy Spider Surprise
		[24105] = {},	-- Roasted Moongraze Tenderloin
		[23676] = {},	-- Moongraze Stag Tenderloin
		[29292] = {},	-- Helboar Bacon
		[27677] = {},	-- Chunk o' Basilisk
		[27678] = {},	-- Clefthoof Meat
		[27658] = {},	-- Roasted Clefthoof
		[27659] = {},	-- Warp Burger
		[27651] = {},	-- Buzzard Bites
		[27674] = {},	-- Ravager Flesh
		[27681] = {},	-- Warped Flesh
		[27671] = {},	-- Buzzard Meat
		[27660] = {},	-- Talbuk Steak
		[27682] = {},	-- Talbuk Venison
		[27657] = {},	-- Blackened Basilisk
		[27854] = {},	-- Smoked Talbuk Venison
		[29451] = {},	-- Clefthoof Ribs
		[21235] = {},	-- Winter Veil Roast
		[19995] = {},	-- Harvest Boar				
		[4457]  = {},	-- Barbecued Buzzard Wing		
		[3173]  = {},	-- Bear Meat		
		[2888]  = {},	-- Beer Basted Boar Ribs		
		[3730]  = {},	-- Big Bear Meat		
		[3726]  = {},	-- Big Bear Steak		
		[3220]  = {},	-- Blood Sausage		
		[2677]  = {},	-- Boar Ribs		
		[3404]  = {},	-- Buzzard Wing		
		[12213] = {},	-- Carrion Surprise		
		[2679]  = {},	-- Charred Wolf Meat		
		[769]   = {},	-- Chunk of Boar Meat		
		[2673]  = {},	-- Coyote Meat		
		[2684]  = {},	-- Coyote Steak		
		[1081]  = {},	-- Crisp Spider Meat		
		[12224] = {},	-- Crispy Bat Wing		
		[5479]  = {},	-- Crispy Lizard Tail		
		[2924]  = {},	-- Crocolisk Meat		
		[3662]  = {},	-- Crocolisk Steak		
		[4599]  = {},	-- Cured Ham Steak		
		[17119] = {},	-- Deeprun Rat Kabob		
		[5478]  = {},	-- Dig Rat Stew		
		[5051]  = {},	-- Dig Rat		
		[12217] = {},	-- Dragonbreath Chili		
		[2687]  = {},	-- Dry Pork Ribs		
		[9681]  = {},	-- Grilled King Crawler Legs		
		[2287]  = {},	-- Haunch of Meat		
		[12204] = {},	-- Heavy Kodo Meat		
		[3727]  = {},	-- Hot Lion Chops		
		[13851] = {},	-- Hot Wolf Ribs		
		[12212] = {},	-- Jungle Stew		
		[5472]  = {},	-- Kaldorei Spider Kabob		
		[5467]  = {},	-- Kodo Meat		
		[5480]  = {},	-- Lean Venison		
		[1015]  = {},	-- Lean Wolf Flank		
		[12209] = {},	-- Lean Wolf Steak		
		[3731]  = {},	-- Lion Meat		
		[12223] = {},	-- Meaty Bat Wing		
		[3770]  = {},	-- Mutton Chop		
		[12037] = {},	-- Mystery Meat		
		[4739]  = {},	-- Plainstrider Meat		
		[12184] = {},	-- Raptor Flesh		
		[12203] = {},	-- Red Wolf Meat		
		[12210] = {},	-- Roast Raptor		
		[2681]  = {},	-- Roasted Boar Meat		
		[5474]  = {},	-- Roasted Kodo Meat		
		[8952]  = {},	-- Roasted Quail		
		[1017]  = {},	-- Seasoned Wolf Kabob		
		[5465]  = {},	-- Small Spider Leg		
		[6890]  = {},	-- Smoked Bear Meat		
		[3729]  = {},	-- Soothing Turtle Bisque		
		[2680]  = {},	-- Spiced Wolf Meat		
		[17222] = {},	-- Spider Sausage		
		[5471]  = {},	-- Stag Meat		
		[5469]  = {},	-- Strider Meat		
		[5477]  = {},	-- Strider Stew		
		[2672]  = {},	-- Stringy Wolf Meat		
		[2685]  = {},	-- Succulent Pork Ribs		
		[3728]  = {},	-- Tasty Lion Steak		
		[3667]  = {},	-- Tender Crocolisk Meat		
		[12208] = {},	-- Tender Wolf Meat		
		[18045] = {},	-- Tender Wolf Steak		
		[5470]  = {},	-- Thunder Lizard Tail		
		[12202] = {},	-- Tiger Meat		
		[117]   = {},	-- Tough Jerky		
		[3712]  = {},	-- Turtle Meat		
		[12205] = {},	-- White Spider Meat		
		[3771]  = {},	-- Wild Hog Shank		
	},					
	[nUI_FOOD_BREAD] = 
	{
		[1113]  = {},	-- Conjured Bread		
		[5349]  = {},	-- Conjured Muffin		
		[1487]  = {},	-- Conjured Pumpernickel		
		[1114]  = {},	-- Conjured Rye		
		[8075]  = {},	-- Conjured Sourdough		
		[8076]  = {},	-- Conjured Sweet Roll	
		[22895] = {},	-- Conjured Cinnamon Roll
		[22019] = {},	-- Conjured Croissant
		
		[23160] = {},	-- Friendship Bread
		[30817] = {},	-- Spice Bread
		[29394] = {},	-- Lyribread
		[28486] = {},	-- Moser's Magnificent Muffin		
		[27855] = {},	-- Mag'har Grainbread
		[29449] = {},	-- Bladespire Bagel
		[19696] = {},	-- Harvest Bread	
		[21254] = {},	-- Winter Veil Cookie
		[2683]  = {},	-- Crab Cake		
		[4541]  = {},	-- Freshly Baked Bread		
		[17197] = {},	-- Gingerbread Cookie		
		[3666]  = {},	-- Gooey Spider Cake		
		[8950]  = {},	-- Homemade Cherry Pie		
		[4542]  = {},	-- Moist Cornbread		
		[4544]  = {},	-- Mulgore Spice Bread		
		[4601]  = {},	-- Soft Banana Bread		
		[4540]  = {},	-- Tough Hunk of Bread		
		[16169] = {},	-- Wild Ricecake		
	},					
	[nUI_FOOD_CHEESE] = 
	{
		[3665]  = {},	-- Curiously Tasty Omelet
		[12218] = {},	-- Monster Omelet
		[17406] = {},	-- Holiday Cheesewheel
		[30458] = {},	-- Stromgarde Muenster
		[29448] = {},	-- Mag'har Mild Cheese	        
		[27857] = {},	-- Garadar Sharp		
		[8932]  = {},	-- Alterac Swiss		
		[414]   = {},	-- Dalaran Sharp		
		[2070]  = {},	-- Darnassian Bleu		
		[422]   = {},	-- Dwarven Mild		
		[3927]  = {},	-- Fine Aged Cheddar	        
		[1707]  = {},	-- Stormwind Brie		
	},					
	[nUI_FOOD_FRUIT] = 
	{
		[4656]  = {},	-- Small Pumpkin
		[21030] = {},	-- Darnassus Kimchi Pie
		[21031] = {},	-- Cabbage Kimchi
		[21033] = {},	-- Radish Kimchi
		[29393] = {},	-- Diamond Berries
		[27856] = {},	-- Skethyl Berries
		[29450] = {},	-- Telaari Grapes
		[19994] = {},	-- Harvest Fruit			
		[8953]  = {},	-- Deep Fried Plantains		
		[4539]  = {},	-- Goldenbark Apple		
		[16168] = {},	-- Heaven Peach		
		[4602]  = {},	-- Moon Harvest Pumpkin		
		[4536]  = {},	-- Shiny Red Apple		
		[4538]  = {},	-- Snapvine Watermelon		
		[4537]  = {},	-- Tel'Abim Banana		
		[11950] = {},	-- Windblossom Berries		
	},					
};						
