﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

    nUI is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with nUI.  If not, see <http://www.gnu.org/licenses/>.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--]]---------------------------------------------------------------------------

if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end

local CreateFrame = CreateFrame;

-------------------------------------------------------------------------------

local BUTTONBAG_COLS = 7;

local ButtonBagConfig = 
{	
	-- transient minimap buttons we want on the top row
	
	TopRow =
	{
		["MiniMapMeetingStoneFrame"] = 1,
		["MiniMapVoiceChatFrame"] = 1,
		["MiniMapRecordingButton"] = 1,
	},
	
	-- buttons we want to force ignore of
	
	Excludes = 
	{		
		["MiniMapMailFrame"] = 1,
		["MiniMapBattlefieldFrame"] = 1,
		["MinimapBackdrop"] = 1,
		["MiniMapPing"] = 1,
		["MiniMapCompassRing"] = 1,
		["MinimapZoomIn"] = 1,
		["MinimapZoomOut"] = 1,
		["MiniMapTracking"] = 1,
		["MiniMapWorldMapButton"] = 1,
		["GatherMiniNoteUpdateFrame"] = 1,
		["TimeManagerClockButton"] = 1,
	},	
	
	-- buttons we want to force inclusion of
	
	Includes = 
	{		
		["WIM_IconFrame"] = 1,
		["CTMod2_MinimapButton"] = 1,
		["PoisonerMinimapButton"] = 1,
		["GameTimeFrame"] = 1,
		["MobMapMinimapButtonFrame"] = 1,
		["BaudGearMinimapButton"] = 1,
	},
	
	-- button name patterns to be excluded (mostly Minimap POI's)
	
	ExcludePatterns =
	{
		[1] = "GatherNote.",
		[2] = "CartographerNotes.",
		[3] = "GatherMatePin.",
		[4] = "FishingExtravaganza.",
		[5] = "RecipeRadarMinimapIcon.",
	},
	
	-- for a table of unnamed buttons we have to locate by object instance
	
	Unnamed =
	{
	},
	
	-- fixed button sizes for buttons that misreport their width (for scaling)
	
	ButtonSize =
	{
		["Enchantrix"] = 36,
	},
};

-------------------------------------------------------------------------------

local name             = "nUI_ButtonBag";
local anchor           = CreateFrame( "Frame", name.."Events", WorldFrame );
local frame            = CreateFrame( "Frame", name, nUI_Dashboard.Anchor );
local bag_button       = CreateFrame( "CheckButton", name.."Button", nUI_Dashboard.Anchor, "SecureAnchorButtonTemplate" );
local bag_list         = {};

frame.close_btn = CreateFrame( "Button", "$parent_CloseButton", frame, "UIPanelCloseButton" );
frame.label     = frame:CreateFontString( "$parent_Label", "ARTWORK" );
frame.top       = frame:CreateTexture( "$parent_Top", "BORDER" );
frame.middle    = frame:CreateTexture( "$parent_Middle", "BACKGROUND" );
frame.bottom    = frame:CreateTexture( "$parent_Bottom", "BORDER" );
frame.buttons   = {};

frame.label:SetJustifyH( "LEFT" );
frame.label:SetJustifyV( "MIDDLE" );
frame.label:SetFont( nUI_L["font2"], 10, "OUTLINE" );
frame.label:SetPoint( "TOPLEFT", frame, "TOPLEFT", 47, -10 );
frame.label:SetText( nUI_L["Minimap Button Bag"] );

frame.close_btn:SetPoint( "TOPRIGHT", frame, "TOPRIGHT", 0, -1 );
frame.close_btn:SetScript( "OnClick", function() bag_button:Click(); end );

frame.top:SetPoint( "TOPLEFT", frame, "TOPLEFT", 0, 0 );
frame.top:SetPoint( "TOPRIGHT", frame, "TOPRIGHT", 0, 0 );
frame.top:SetTexture( "Interface\\AddOns\\nUI\\AddOns\\Art\\nUI_ButtonBag_Top" );
frame.top:SetTexCoord( 0.25, 0, 0.25, 1, 1, 0, 1, 1 );
frame.top:SetWidth( CONTAINER_WIDTH );
frame.top:SetHeight( CONTAINER_WIDTH / 192 * 64 );

frame.bottom:SetPoint( "BOTTOMLEFT", frame, "BOTTOMLEFT", 0, 0 );
frame.bottom:SetPoint( "BOTTOMRIGHT", frame, "BOTTOMRIGHT", 0, 0 );
frame.bottom:SetTexture( "Interface\\AddOns\\nUI\\AddOns\\Art\\nUI_ButtonBag_Bottom" );
frame.bottom:SetTexCoord( 0.25, 0, 0.25, 1, 1, 0, 1, 1 );
frame.bottom:SetWidth( CONTAINER_WIDTH );
frame.bottom:SetHeight( CONTAINER_WIDTH / 192 * 16 );

frame.middle:SetPoint( "TOPLEFT", frame.top, "BOTTOMLEFT", 0, 0.5 );
frame.middle:SetPoint( "BOTTOMRIGHT", frame.bottom, "TOPRIGHT", 0, -0.5 );
frame.middle:SetTexture( "Interface\\AddOns\\nUI\\AddOns\\Art\\nUI_ButtonBag_Middle" );
frame.middle:SetWidth( CONTAINER_WIDTH );
frame.middle:SetHeight( 6 );
frame.middle:SetTexCoord( 0.25, 0, 0.25, 6/256, 1, 0, 1, 6/256 );

frame.box_height = frame.top:GetHeight() + frame.bottom:GetHeight() + 1;

bag_button:SetHeight( 64 );
bag_button:SetWidth( 64 );
bag_button:SetNormalTexture( "Interface\\AddOns\\nUI\\AddOns\\Art\\nUI_ButtonBag_Normal" );

bag_button.checked_texture = bag_button:CreateTexture();
bag_button.checked_texture:SetAllPoints( bag_button );
bag_button.checked_texture:SetTexture( "Interface\\AddOns\\nUI\\AddOns\\Art\\nUI_ButtonBag_Checked" );
bag_button:SetCheckedTexture( bag_button.checked_texture );
bag_button:SetAttribute( "newstate", "on,off" );

frame:SetHeight( frame.box_height );
frame:SetWidth( CONTAINER_WIDTH );
frame:SetScript( "OnShow", function() frame.shown = true; updateContainerFrameAnchors(); end );
frame:SetScript( "OnHide", function() frame.shown = false; updateContainerFrameAnchors(); end );
frame:SetAttribute( "showstates", "on" );

-------------------------------------------------------------------------------

bag_button:SetScript( "OnEnter",

	function()
		
		GameTooltip:SetOwner( bag_button );
		GameTooltip:SetText( nUI_L["Minimap Button Bag"] );
		GameTooltip:Show();
		
	end
);

bag_button:SetScript( "OnLeave", function() GameTooltip:Hide(); end );

-------------------------------------------------------------------------------

local function testMinimapButton( button )

	local result = nil;
	
	if button then
			
		local button_name  = button:GetName();	
		local hasClick     = false;
		local hasMouseUp   = false;
		local hasMouseDown = false;
		
		-- test the current button to see if it is one we can/should/will move
		
		if button_name then

			-- the Gatherer and Cartographer mod minimap notes are misinterpreted as 
			-- minimap buttons. This forces us to ignore all their note nodes
			
			for i in pairs( ButtonBagConfig.ExcludePatterns ) do
				
				if button_name:match( ButtonBagConfig.ExcludePatterns[i] ) then
					return nil;
				end
			end
			
			-- otherwise, determine what to do with the button
			
			if ButtonBagConfig.Excludes[button_name] then
				
				nUI_DebugLog["ButtonBag"][button_name] = "skipped forced ignore button";
				return nil;
				
			elseif ButtonBagConfig.Includes[button_name] then
				
				nUI_DebugLog["ButtonBag"][button_name] = "moved forced include button";
				return button;
				
			elseif button.HasScript then
	
				if button:HasScript( "OnClick" ) and button:GetScript( "OnClick" ) then 
					hasClick = true;
				end
				
				if button:HasScript( "OnMouseUp" ) and button:GetScript( "OnMouseUp" ) then
					hasMouseUp = true;
				end
				
				if button:HasScript( "OnMouseDown" ) and button:GetScript( "OnMouseDown" ) then
					hasMouseDown = true; 
				end
	
				if hasClick or hasMouseUp or hasMouseDown then
				
					nUI_DebugLog["ButtonBag"][button_name] = "autodetected and moved button";
					return button;
					
				else
							
					nUI_DebugLog["ButtonBag"][button_name] = "does not appear to be a button";
					
				end				
				
			else				
				nUI_DebugLog["ButtonBag"][button_name] = "does not support \"HasScript()\"";
			end					
		end
		
		-- if we didn't find a button, try diving the children of this frame
		-- recursion ftw
		
		if not result then
			for _,child in ipairs( { button:GetChildren() } ) do
				
				result = testMinimapButton( child );
				
				if result then 
					child.is_embedded = true;
					break; 
				end
			end
		end
	end
	
	return result;
end

-------------------------------------------------------------------------------

local function compressButtonBag()
	
	local new_bag = {};
	local row     = 1;
	local col     = 1;
	local old_bag = frame.buttons;
	
	if old_bag then
		
		for i=1,(frame.last_row or 0) do
			
			if old_bag[i] then
					
				for j=1,BUTTONBAG_COLS do
					
					if old_bag[i][j] then
						
						local button = old_bag[i][j];
						
						if not new_bag[row] then 
							new_bag[row] = {}; 
						end
			
						new_bag[row][col] = button;
						button.row = row;
						button.col = col;

						if col == BUTTONBAG_COLS then 
							col = 1;
							row = row+1;
						else
							col = col+1;
						end
					end
				end
			end
		end
	end
	
	frame.buttons = new_bag;
	
end

-------------------------------------------------------------------------------

local function buttonBagInsert( bag, button, row, col )

	button.row = row;
	button.col = col;
	
	if not bag[row] then bag[row] = {}; end
	
	bag[row][col] = button;
	
	if not frame.last_row 
	or row > frame.last_row then
		frame.last_row = row;
	end
	
end

-------------------------------------------------------------------------------
-- this routine adds a button to the button bag. If the new button does not
-- have a row or a column, then we look for an empty spot in the table to
-- locate it

local function addButtonBagButton( new_button )

	local row = new_button.row;
	local col = new_button.col;
	local bag = frame.buttons;

	-- make sure the column is a valid number
	
	if col and (col < 1 or col > BUTTNBAG_COLS) then col = nil; end
	
	-- if the button has a row and a column then see if that spot is
	-- still open in the button bag. If not.. then bump the button
	
	if row and col then
		
		if not bag[row] then 
			buttonBagInsert( bag, new_button, row, col );
		elseif not bag[row][col] then 
			buttonBagInsert( bag, new_button, row, col );
		elseif bag[row][col].name == new_button.name then
			buttonBagInsert( bag, new_button, row, col );
		else
			row = nil;
		end		
	end

	-- if the new button has not been placed, then look for
	-- the first unused spot in the bag to place it in
	
	if not row or not col then
		
		local new_row = 1;
		local new_col = 1;
		
		while bag[new_row] do
						
			while bag[new_row][new_col] and new_col <= BUTTONBAG_COLS do
				new_col = new_col+1;
			end

			if new_col <= BUTTONBAG_COLS then
				break;
			end			
			
			new_row = new_row+1;
			new_col = 1;
		end
		
		buttonBagInsert( bag, new_button, new_row, new_col );
	end
end

-------------------------------------------------------------------------------

local set_top_row  = false;
local first_button = true;

function buttonBagFix()

	-- special processing... the GroupCalendar mod hijacks the minimap time of day
	-- indicator which we normally hide... so... if GroupCalendar is loaded, then we
	-- need to unhide that button and move it.
	
	if IsAddOnLoaded( "GroupCalendar" ) and not GameTimeFrame:IsShown() then
		GameTimeFrame:Show();
	end
	
	-- build a list of children in the minimap
	
	local children    = { Minimap:GetChildren() };
	local children2   = { MinimapBackdrop:GetChildren() };
	local new_buttons = {};
	local top_row     = {};

	for _, child in ipairs( children2 ) do 
		table.insert(children, child); 
	end

	-- check the buttons we are forcing into the top row

	if not set_top_row then
		
		for i in pairs( ButtonBagConfig.TopRow ) do

			local move_button = _G[i];
			
			if move_button then
	
				local button_name = move_button:GetName();
	
				if not bag_list[button_name] then
				
					-- make sure we don't pick this same button up in another scan
					
					move_button:SetParent( frame );
					move_button.cached_SetParent = move_button.SetParent;
					move_button.SetParent        = function() end;
					
					-- moving a new button
					
					top_row[button_name] = 
					{
						name  = button_name,
						row   = nil,
						col   = nil,
						frame = move_button,
					}				
					
				end
			end
		end
	end
	
	-- check the buttons that are unnamed, where we have to look for the objects

	ButtonBagConfig.Unnamed["Enchantrix"] = Enchantrix and Enchantrix.MiniIcon or nil;
	
	for i in pairs( ButtonBagConfig.Unnamed ) do
		
		local button_name = i;
		local move_button = ButtonBagConfig.Unnamed[i];
		
		if move_button and move_button:IsShown() then
			
			if not bag_list[button_name] then
			
				-- make sure we don't pick this same button up in another scan
				
				move_button:SetParent( frame );
				move_button.cached_SetParent = move_button.SetParent;
				move_button.SetParent        = function() end;
				
				new_buttons[button_name] = 
				{
					name  = button_name,
					row   = nil,
					col   = nil,
					frame = move_button,
				}				
				
			end
		end			
	end
	
	-- check the buttons we are forcing a move for
	
	for i in pairs( ButtonBagConfig.Includes ) do
		
		local move_button = testMinimapButton( _G[i] );
		
		if move_button and move_button:IsShown() then

			local button_name = move_button:GetName();

			if not bag_list[button_name] then
			
				-- make sure we don't pick this same button up in another scan
				
				move_button:SetParent( frame );
				move_button.cached_SetParent = move_button.SetParent;
				move_button.SetParent        = function() end;
				
				-- moving a new button
				
				new_buttons[button_name] = 
				{
					name  = button_name,
					row   = nil,
					col   = nil,
					frame = move_button,
				}				
				
			end
		end
	end
	
	-- parse the list of minimap children and see who might be a button
	-- so we can prepare to move it
	
	for _, child in ipairs( children ) do
		
		local move_button = testMinimapButton( child );

		if move_button and move_button:IsShown() then
		
			local button_name = move_button:GetName();
	
			if not bag_list[button_name] then
					
				-- make sure we don't pick this same button up in another scan
				
				move_button:SetParent( frame );
				move_button.cached_SetParent = move_button.SetParent;
				move_button.SetParent        = function() end;
				
				-- moving a new button
				
				new_buttons[button_name] = 
				{
					name  = button_name,
					row   = nil,
					col   = nil,
					frame = move_button,
				}				
					
			end
		end
	end

	-- now add all of the unknown buttons in the button bag
	
	for i in pairs( new_buttons ) do
	
		addButtonBagButton( new_buttons[i] );
		bag_list[i] = true;
		
	end
	
	-- now compress the button bag and lay out the buttons
	
	compressButtonBag();
	
	local bag           = frame.buttons;
	local button_size   = (CONTAINER_WIDTH - 32) / BUTTONBAG_COLS;
	local middle_height = (#bag-1) * button_size + 6;
	local y2            = middle_height / 256;
	
	frame.containers = {};
	frame:SetHeight( middle_height + frame.box_height );
	frame.middle:SetHeight( middle_height );
	frame.middle:SetTexCoord( 0.25, 0, 0.25, y2, 1, 0, 1, y2 );

	for i=1,#bag do
		
		frame.containers[i] = {};
		
		for j=1, #bag[i] do
			
			local button    = bag[i][j];
			local scale     = button_size / (ButtonBagConfig.ButtonSize[button.name] or button.frame:GetWidth());
			local container = button.container or CreateFrame( "Button", "$parent_Button_"..i.."_"..j, frame );			

			button.container       = container;
			frame.containers[i][j] = container;
			
			container:SetWidth( button_size );
			container:SetHeight( button_size );
			container:SetNormalTexture( "Interface\\Icons\\Buttons\\UI-PageButton-Background" );
			
			if i == 1 and j == 1 then
				container:SetPoint( "LEFT", frame.middle, "TOPLEFT", 20, -4 );
			elseif j == 1 then
				container:SetPoint( "TOPLEFT", frame.containers[i-1][1], "BOTTOMLEFT", 0, 0 );
			else
				container:SetPoint( "LEFT", frame.containers[i][j-1], "RIGHT", 0, 0 );
			end
			
			button.frame:SetScale( scale );
			button.frame:ClearAllPoints();
			button.frame:SetPoint( "CENTER", container, "CENTER", 0, 0 );
			button.frame:RegisterForDrag();
			
		end
	end	
	
	-- set the top row buttons

	if not set_top_row then
		
		set_top_rows = true;
			
		for i in pairs( top_row ) do
			
			local button    = top_row[i];
			local scale     = button_size / (ButtonBagConfig.ButtonSize[button.name] or button.frame:GetWidth());
			local container = button.container or CreateFrame( "Button", "$parent_Button_"..i, frame );			
	
			button.container = container;
			
			container:SetWidth( button_size );
			container:SetHeight( button_size );
			container:SetNormalTexture( "Interface\\Icons\\Buttons\\UI-PageButton-Background" );
			
			button.frame:SetScale( scale );
			button.frame:ClearAllPoints();
			button.frame:SetPoint( "CENTER", container, "CENTER", 0, 0 );
			button.frame:RegisterForDrag();
			
		end
		
		local fix_buttons = {};
		
		if top_row["MiniMapMailFrame"] then table.insert( fix_buttons, top_row["MiniMapMailFrame"] ); end
		if top_row["MiniMapBattlefieldFrame"] then table.insert( fix_buttons, top_row["MiniMapBattlefieldFrame"] ); end
		if top_row["MiniMapMeetingStoneFrame"] then table.insert( fix_buttons, top_row["MiniMapMeetingStoneFrame"] ); end
		if top_row["MiniMapVoiceChatFrame"] then table.insert( fix_buttons, top_row["MiniMapVoiceChatFrame"] ); end
		if top_row["MiniMapRecordingButton"] then table.insert( fix_buttons, top_row["MiniMapRecordingButton"] ); end

		for i=1,#fix_buttons do
			
			local button = fix_buttons[i];
			
			if i == 1 then button.container:SetPoint( "LEFT", frame.middle, "TOPLEFT", 20 + button_size * 2, button_size - 4 );
			else button.container:SetPoint( "LEFT", fix_buttons[i-1].container, "RIGHT", 0, 0 );
			end
			
		end
	end	
end		

-------------------------------------------------------------------------------

local timer      = 6;
local loop_count = 0;

local function onButtonBagUpdate( who, elapsed )
	
	-- check for new minimap buttons every five seconds for the first thirty seconds
	-- as a new load and then every 30 seconds thereafter... unfortunately, we have
	-- no way of knowing when mods post their minimap buttons or LOD mods add a new
	-- button.
	
	timer = timer - elapsed;
		
	if timer <= 0 then
		
		if loop_count < 6 then 
			timer      = 6;
			loop_count = loop_count+1; 
		else
			timer = 30;
		end;
		
		buttonBagFix();
		
	end	
end

-------------------------------------------------------------------------------

local function onButtonBagEvent()

	if event == "VARIABLES_LOADED" then

		if not nUI_DebugLog then nUI_DebugLog = {}; end
		nUI_DebugLog["ButtonBag"] = {};

		nUI_BagBar:SetAttribute( "addchild", bag_button );
		nUI_BagBar:SetAttribute( "addchild", frame );
		nUI_BagBar:SetAttribute( "state", "off" );

		-- request to toggle the minimap button bag
		
		option = nUI_SlashCommands[nUI_SLASHCMD_BUTTONBAG];
		
		nUI_SlashCommands:setHandler( option.command, function() nUI_ButtonBagButton:Click(); end );
		

	elseif event == "PLAYER_LOGIN" then
		
		anchor:SetScript( "OnUpdate", onButtonBagUpdate );
		
	end
end

anchor:SetScript( "OnEvent", onButtonBagEvent );
anchor:RegisterEvent( "VARIABLES_LOADED" );
anchor:RegisterEvent( "PLAYER_LOGIN" );


