﻿--[[---------------------------------------------------------------------------

Copyright (c) 2008, 2009 by K. Scott Piel 
All Rights Reserved

E-mail: < kscottpiel@gmail.com >
Web:    < http://www.scottpiel.com >

This file is part of nUI.

	The copyright for all material provided within the nUI software package 
	("nUI") is held by Kenneth Scott Piel. Except as stated herein, none of 
	the material may be copied, reproduced, distributed, republished, 
	downloaded, displayed, posted or transmitted in any form or by any means, 
	including, but not limited to, electronic, mechanical, photocopying, 
	recording, or otherwise, without the prior written permission of 
	the copyright holder. Permission is granted to display, copy, distribute 
	and download the materials on this Site for personal, non-commercial use 
	only provided you do not modify the materials and that you retain all 
	copyright and other proprietary notices contained in the materials. You 
	also may not, without the copyright holder's permission, "mirror" any 
	material contained in nUI on any other server. This permission terminates 
	automatically if you breach any of these terms or conditions. Upon 
	termination, you will immediately destroy any downloaded and printed 
	materials. Any unauthorized use of any material contained in nUI may 
	violate copyright laws, trademark laws, the laws of privacy and publicity, 
	and communications regulations and statutes.
	
	nUI is packaged in four distributable versions known as "nUI Release",
	"nUI+", "nUI Development" and "nUI PTR Beta" -- Redistribution for these
	versions is governed by the following terms...
	
	1) Redistribution of the nUI Release (aka nUI Lite) version is permitted under 
	   the following terms... Permission is hereby	granted for unlimited free 
	   and open distribution of "nUI Release" / "nUI Lite" by anyone in any 
	   form and by any means provided the nUI Release distribution contents 
	   are not altered in any way, are distributed in full with all copyright 
	   statements and licensing terms included and intact and that any 
	   interface the end user is provided for the purpose of downloading nUI 
	   includes a plainly visible and functioning link to nUI's official web 
	   site at http://www.nUIaddon.com and a plainly visible notice that nUI 
	   accepts user donations with a working link to nUI's donation page at 
	   http://www.nUIaddon.com/donate.html
	   
	2) Permission is hereby granted for distribution of the "nUI+", "nUI+
	   Development" and "nUI+ PTR Beta" versions of nUI via the online download
	   service at http://www.wowinterface.com and the copyright holder's own
	   web site http://www.nuiaddon.com -- The end user is granted permission
	   to download any nUI package from these two web sites for their personal
	   use under the same terms and conditions as nUI Release but are prohibited
	   from sharing the contents of these packages via any means in any form
	   with anyone other than via direct transfer with immediate friends and
	   family members. Distribution of nUI+, nUI+ Development or nUI+ PTR Beta
	   via any other means by any other entity in any other form is strictly 
	   prohibited without the copyright holder's express written permission
	   explicitly granting such distribution rights specifically to that entity.
	   
	3) Deep-linking and leeching of nUI distributions is strictly prohibited. 
	   Any individual or entity who wishes to offer downloads of nUI 
	   distributions must either host the legal and unmodified distribution 
	   on their own servers to be distributed at their own expense using their 
	   own bandwidth or they must link the user back to the official download 
	   page on the third party provider's servers from which the user can 
	   initiate the download. Use of any download link or mechanism which 
	   initiates a download of any nUI distribution from a third party 
	   distribtion site that bypasses the official content and download pages 
	   or advertisements of that third party site is strictly prohibited
       without the express written consent of that site.
       
    See the included files "nUI_RELEASE_LICENSE.txt" and "nUI_PLUS_LICENSE.txt"
    for the complete terms of nUI's licensing terms.
	   
    nUI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    enclosed license for more details.
	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--]]---------------------------------------------------------------------------

if not nUI then nUI = {}; end
if not nUI_Unit then nUI_Unit = {}; end
if not nUI_UnitOptions then nUI_UnitOptions = {}; end
if not nUI_DefaultConfig then nUI_DefaultConfig = {}; end
if not nUI_Profile then nUI_Profile = {}; end;

local GetTime           = GetTime;
local GetPetHappiness   = GetPetHappiness;
local UnitExists        = UnitExists;
local UnitIsUnit        = UnitIsUnit;
local InCombatLockdown  = InCombatLockdown;

-------------------------------------------------------------------------------
-- default options for the pet happiness indicator

nUI_DefaultConfig.PetHappiness =
{
	chat_enabled  = "yes",	-- enable chat frame messages when happiness changes
	show_happy    = "yes",	-- show/hide the happiness indicator when pet is happy
	unhappy_flash = "no",	-- flash the happiness indicator when the pet is unhappy
	unhappy_scale = 1.25,	-- modify the size of the indicator when the pet is unhappy
	angry_flash   = "yes",	-- flash the happiness indicator when the pet is angry
	angry_scale   = 1.5,	-- modify the size of the indicator when the pet is angry
	flash_rate    = 2,      -- the time in seconds for each flash cycle when enabled
};

-------------------------------------------------------------------------------
-- locals

local happiness_flash = 0;
local frame_alpha     = 1;
local flash_rate      = nUI_DefaultConfig.PetHappiness.flash_rate;
local angry_flash     = nUI_DefaultConfig.PetHappiness.angry_flash ~= "no";
local angry_scale     = nUI_DefaultConfig.PetHappiness.angry_scale;
local unhappy_flash   = nUI_DefaultConfig.PetHappiness.unhappy_flash == "yes";
local unhappy_scale   = nUI_DefaultConfig.PetHappiness.unhappy_scale;
local show_happy      = nUI_DefaultConfig.PetHappiness.show_happy ~= "no";
local chat_enabled    = nUI_DefaultConfig.PetHappiness.chat_enabled ~= "no";

-------------------------------------------------------------------------------
-- pet happiness event management

if not nUI_Unit.Drivers then 
	nUI_Unit.Drivers = CreateFrame( "Frame", "nUI_UnitDrivers", WorldFrame ); 
end

nUI_Profile.nUI_UnitHappiness       = {};
nUI_Profile.nUI_UnitHappiness.Frame = {};

local ProfileCounter      = nUI_Profile.nUI_UnitHappiness;
local FrameProfileCounter = nUI_Profile.nUI_UnitHappiness.Frame;

local frame = CreateFrame( "Frame", "$parent_Happiness", nUI_Unit.Drivers )

local HappinessCallbacks    = {};
local HappinessUnits        = {};
local HappinessFrames       = {};
local NewUnitInfo           = {};
local UpdateQueue           = {};
local queue_timer           = 1 / nUI_DEFAULT_FRAME_RATE;

nUI_Unit.Drivers.Happiness  = frame;

-------------------------------------------------------------------------------
-- variables used in methods within this module are declared here to eliminate
-- the use of dynamic memory and the garbage collector

local unit_id;
local unit_info;
local list;
local i;
local hFrame;
local modified;
local old_loyalty;
local old_happiness;
local name;
local pet_happiness, pet_damage, pet_loyalty;
local has_pet_ui, is_hunter_pet;
local prior_state;
local happiness;
local loyalty;
local damage;
local text;
local x1, y1, x2, y2;
local frame_scale;

-------------------------------------------------------------------------------
	
local function onHappinessEvent()
	
--	nUI_ProfileStart( ProfileCounter, "onHappinessEvent", event );
	
	-- if we've just loaded our configuration options, then enable or disable support
	-- for pet happiness indicator flashing accordingly
	
	if event == "ADDON_LOADED" then

		if arg1 == "nUI" then
			nUI:patchConfig();
			nUI_Unit:configHappiness();
		end

	else

		UpdateQueue["pet"] = true;
		NewUnitInfo["pet"] = nUI_Unit:getUnitInfo( "pet" );
		
	end

--	nUI_ProfileStop();
	
end

frame:SetScript( "OnEvent", onHappinessEvent );
frame:RegisterEvent( "ADDON_LOADED" );
frame:RegisterEvent( "PLAYER_ENTERING_WORLD" );
frame:RegisterEvent( "UNIT_HAPPINESS" );
frame:RegisterEvent( "UNIT_MAXHAPPINESS" );
frame:RegisterEvent( "PET_UI_UPDATE" );
frame:RegisterEvent( "PET_RENAMEABLE" );

-------------------------------------------------------------------------------
-- handles flashing the pet happiness indicator when required

local function onHappinessUpdate( who, elapsed )

--	nUI_ProfileStart( ProfileCounter, "onHappinessUpdate" );
	
	queue_timer = queue_timer - elapsed;
	
	if queue_timer <= 0 then -- process the update queue at the user selected frame rate
	
		queue_timer = nUI_Unit.frame_rate;

		for unit_id in pairs( UpdateQueue ) do
		
			if UpdateQueue[unit_id] then
			
				UpdateQueue[unit_id] = false;
				unit_info = NewUnitInfo[unit_id];
								
				if HappinessCallbacks[unit_id] and #HappinessCallbacks[unit_id] > 0 then
					nUI_Unit:notifyCallbacks( 
						nUI_L["pet happiness"], HappinessCallbacks, HappinessUnits, 
						unit_info, unit_id, nUI_Unit:updateHappinessInfo( unit_id, unit_info ) 
					);
				end
			end
		end
	end	

	if frame.flash_enabled then
			
		happiness_flash = (happiness_flash + elapsed) % flash_rate;
		
		-- calculate an alpha value for the frames so they all flash together
		-- if and when they are flashing
		
		frame_alpha = happiness_flash / flash_rate;
		
		if frame_alpha > 0.5 then	frame_alpha = (frame_alpha - 0.5) * 2;	-- fading in
		else frame_alpha = 1.0 - frame_alpha * 2; -- fading out
		end
		
		-- set alpha for any active frames that are flashing
		
		for i in pairs( HappinessFrames ) do		
			hFrame = HappinessFrames[i];
			if hFrame.active and hFrame.flashing then 
				hFrame.alpha = frame_alpha;
				hFrame:SetAlpha( frame_alpha ); 
			end
		end
	end
	
--	nUI_ProfileStop();
	
end

frame:SetScript( "OnUpdate", onHappinessUpdate );

-------------------------------------------------------------------------------
-- this callback method is called when one of the unit IDs we are monitoring
-- for pet happiness changes GUID

frame.newUnitInfo = function( unit_id, unit_info )

--	nUI_ProfileStart( ProfileCounter, "newUnitInfo" );
	
	UpdateQueue[unit_id] = true;
	NewUnitInfo[unit_id] = unit_info;

--	nUI_ProfileStop();
	
end

-------------------------------------------------------------------------------
-- initialize configuration for the pet happiness indicators
-- 
-- this method is called when the mod's saved variables have been loaded by Bliz and
-- may be called again whenever the pet happiness configuration has been changed
-- by the player or programmatically. Passing true or a non-nil value for "use_default"
-- will cause the player's current pet happiness configuration to be replaced with
-- the default settings defined at the top of this file (which cannot be undone!)

function nUI_Unit:configHappiness( use_default )

--	nUI_ProfileStart( ProfileCounter, "configHappiness" );
	
	if not nUI_UnitOptions then nUI_UnitOptions = {}; end
	if not nUI_UnitOptions.PetHappiness then nUI_UnitOptions.PetHappiness = {}; end
	
	local config  = nUI_UnitOptions.PetHappiness;
	local default = nUI_DefaultConfig.PetHappiness;
	
	if use_default then

		config.chat_enabled  = default.chat_enabled;
		config.show_happy    = default.show_happy;
		config.unhappy_flash = default.unhappy_flash;
		config.unhappy_scale = default.unhappy_scale;
		config.angry_flash   = default.angry_flash;
		config.flash_rate    = default.flash_rate;
		
	else
		
		config.chat_enabled  = strlower( config.chat_enabled or default.chat_enabled );
		config.show_happy    = strlower( config.show_happy or default.show_happy );
		config.unhappy_flash = strlower( config.unhappy_flash or default.unhappy_flash );
		config.unhappy_scale = tonumber( config.unhappy_scale or default.unhappy_scale );
		config.angry_flash   = strlower( config.angry_flash or default.angry_flash );
		config.flash_rate    = tonumber( config.flash_rate or default.flash_rate );
		
	end	
	
	-- in the event we have frames already registered, update them according
	-- to the potentially new set of options
		
	flash_rate      = config.flash_rate;
	angry_flash     = config.angry_flash ~= "no";
	angry_scale     = config.angry_scale;
	unhappy_flash   = config.unhappy_flash == "yes";
	unhappy_scale   = config.unhappy_scale;
	show_happy      = config.show_happy ~= "no";
	chat_enabled    = config.chat_enabled ~= "no";
	
	-- indicator flash enabled
	
	if unhappy_flash 
	or angry_flash 
	then

		frame.flash_enabled = true;
		
	-- indicator flash disabled
	
	else
		
		for frame in pairs( HappinessFrames ) do
			if frame.active and frame.flashing then
				frame.flashing = false;
				frame:SetAlpha( 1 );
			end
		end

		frame.flash_enabled = false;
		
	end	
	
	nUI_Unit:refreshHappinessInfo();
	
--	nUI_ProfileStop();
	
end

-------------------------------------------------------------------------------
-- add and remove callbacks from the list of happiness indicators we manage
--
-- calling this method will return the current pet unit_info structure
-- if there is a hunter pet, otherwise it will return nil
--
-- Note: these callbacks will be notified when the player gains a hunter pet, 
--       loses a hunter pet or when a hunter pet's happiness changes. These
--		 callbacks will also be notified if the unit changes from the hunter
--       pet to some other GUID or from another GUID to the hunter pet.

function nUI_Unit:registerHappinessCallback( unit_id, callback )

--	nUI_ProfileStart( ProfileCounter, "registerHappinessCallback" );
	
	unit_info = nil;
	
	if unit_id and callback then

		-- get the list of callbacks for this unit id and add the new callback to it
		
		list = HappinessCallbacks[unit_id] or {};
		
		nUI:TableInsertByValue( list, callback );

		-- if this is a new unit id, add it to the callback table
		
		if not HappinessCallbacks[unit_id] then
			HappinessCallbacks[unit_id] = list;
		end
		
		-- if this is the first callback for this unit id, add our event handler
		-- the the unit change callback list for this unit id
		
		if #list == 1 then
			nUI_Unit:registerUnitChangeCallback( unit_id, nUI_Unit.Drivers.Happiness );
		end				
		
		-- collect pet happiness information as we know it at this time

		unit_info = nUI_Unit:getUnitInfo( unit_id );
		
		if unit_info then
			
			nUI_Unit:updateHappinessInfo( unit_info );
			
			if not unit_info.is_hunter_pet then
				unit_info = nil;
			end
		end
	end
	
--	nUI_ProfileStop();
	
	return unit_info;
	
end

function nUI_Unit:unregisterHappinessCallback( unit_id, callback )
	
--	nUI_ProfileStart( ProfileCounter, "unregisterHappinessCallback" );
	
	if unit_id and callback then

		-- remove the callback from this unit id's callback list
		
		list = HappinessCallbacks[unit_id] or {};		
		
		nUI:TableRemoveByValue( list, callback );
		
		-- if that was the last callback for that unit id, then we don't need
		-- to know if there were any changes to it anymore
		
		if #list == 0 then
			nUI_Unit:unregisterUnitChangeCallback( unit_id, nUI_Unit.Drivers.Happiness );
		end				
	end

--	nUI_ProfileStop();
	
end

-------------------------------------------------------------------------------
-- refresh the happiness information for the player's current pet if it applies
--
-- note: it is the caller's responsibility to ensure that the unit_info being
--       passed belongs to the unit_id that is passed. Generally third party
--       consumers of unit_info should not call this method, rather they 
--       should use the callback registration system to get change notices
--       and let the nUI unit driver engine do the updating. If you MUST call
--       this method, you should first test that the following condition 
--       evaluates as true: UnitGUID( unit_id ) == unit_info.guid
--
-- returns the updated unit information structure for the current hunter pet
-- if the pet exists and the data has changed, otherwise returns nil if the
-- passed unit_info is not a pet, the pet is not a hunter pet or nothing changed

function nUI_Unit:updateHappinessInfo( unit_id, unit_info )

--	nUI_ProfileStart( ProfileCounter, "updateHappinessInfo" );
	
	modified = false;
	
	if unit_info then

		old_loyalty   = unit_info.pet_loyalty;
		old_happiness = unit_info.pet_happiness;
		name          = unit_info and unit_info.name or nUI_L["Your pet"];
		
		pet_happiness = nil;
		pet_damage    = nil;
		pet_loyalty   = nil;
		has_pet_ui    = nil;
		is_hunter_pet = nil;
		
		-- if the player's pet exists, then update happiness information about it
		
		if unit_info == nUI_Unit.PetInfo then
			unit_info.is_pet = true;
		end
		
		if unit_info.is_pet then
	
			has_pet_ui, is_hunter_pet = HasPetUI();
			
		else
			
--			nUI:debug( "nUI_UnitHappiness: "..unit_id.." is not a pet" );
			
		end
		
		if unit_info.has_pet_ui    ~= has_pet_ui
		or unit_info.is_hunter_pet ~= is_hunter_pet
		then
			
--			nUI:debug( "nUI_UnitHappiness: "..unit_id.." has_pet_ui = "..(has_pet_ui and "true" or "false")..", is_hunter_pet = "..(is_hunter_pet and "true" or "false"), 1 );
			
			modified                = true;
			unit_info.modified      = true;
			unit_info.last_change   = GetTime();
			unit_info.has_pet_ui    = has_pet_ui;
			unit_info.is_hunter_pet = is_hunter_pet;
			
		end
			
		if unit_info.is_hunter_pet then 
			
			pet_happiness, pet_damage, pet_loyalty = GetPetHappiness();
		
		end
		
		if unit_info.pet_happiness ~= pet_happiness
		or unit_info.pet_damage    ~= pet_damage
		or unit_info.pet_loyalty   ~= pet_loyalty
		then

--			nUI:debug( "nUI_UnitHappiness: ".."updating happiness data for "..unit_id.." -- happiness = "..(pet_happiness or "<nil>")..", damage = "..(pet_damage or "<nil>")..", loyalty = "..(pet_loyalty or "<nil>"), 1 );
			
			modified                = true;
			unit_info.last_change   = GetTime();
			unit_info.modified      = true;
			unit_info.pet_happiness = pet_happiness;
			unit_info.pet_damage    = pet_damage;
			unit_info.pet_loyalty   = pet_loyalty;
		
			if chat_enabled then
				
				-- if information about the pet has changed, then notify the player
				
				if pet_happiness
				and old_happiness ~= pet_happiness 
				then
		
					-- pet happiness
			
					if pet_happiness == 1 then	-- angry
						
						DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: Warning! %s is |cFFFFFFFFNOT|r happy! Better feed soon."]:format( name ), 1, 0.5, 0.5 );
						
					elseif pet_happiness == 2 then -- unhappy
			
						DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: %s is unhappy... time to feed!"]:format( name ), 1, 0.83, 0 );
						
					elseif pet_happiness == 3 then -- happy
			
						DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: %s is happy."]:format( name ), 0.5, 1, 0.5 );
						
					end		
				end
				
				if pet_loyalty
				and old_loyalty ~= pet_loyalty 
				then
					
					-- pet loyalty
					
					if pet_loyalty < 0 then -- losing loyalty
						
						if pet_loyalty <= -29 then
							DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: Warning... %s is %s losing loyalty "]:format( name, nUI_L["quickly"] ), 1, 0.5, 0.5 );
						else
							DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: Warning... %s is %s losing loyalty "]:format( name, nUI_L["slowly"] ), 1, 0.83, 0 );
						end
						
					elseif pet_loyalty > 0 then

						if pet_loyalty >= 20 then
							DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: %s is %s gaining loyalty"]:format( name, nUI_L["quickly"] ), 0.5, 1, 0.5 );
						else
							DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: %s is %s gaining loyalty"]:format( name, nUI_L["slowly"] ), 1, 1, 1 );
						end
						
					elseif old_loyalty and old_loyalty < 0 then
						
						DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: %s has stopped losing loyalty"]:formae( name ), 1, 1, 1 );
						
					elseif old_loyalty and old_loyalty > 0 then
						
						DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI: %s has stopped gaining loyalty"]:format( name ), 1, 1, 1 );
						
					end
				end
			end
		end
	end
	
	-- return the updated pet information or nil if we don't have a hunter pet
	
--	nUI_ProfileStop();
	
	return modified and unit_info or nil;
	
end

-------------------------------------------------------------------------------
-- update all interested callbacks if information about their pet has changed

function nUI_Unit:refreshHappinessInfo()
	
--	nUI_ProfileStart( ProfileCounter, "refreshHappinessInfo" );
	
	for unit_id in pairs( HappinessCallbacks ) do
		if HappinessCallbacks[unit_id] and #HappinessCallbacks[unit_id] > 0 then
			UpdateQueue[unit_id] = true;
			NewUnitInfo[unit_id] = nUI_Unit:getUnitInfo( unit_id );
		end
	end

--	nUI_ProfileStop();
	
end

-------------------------------------------------------------------------------
-- create a new pet happiness indicator frame

function nUI_Unit:createHappinessFrame( parent, unit_id, id, options )
	
--	nUI_ProfileStart( ProfileCounter, "createHappinessFrame" );
	
	local frame     = nUI_Unit:createFrame( "$parent_Happiness"..(id or ""), parent, unit_id, false );	
	frame.texture   = frame:CreateTexture( "$parentTexture" );
	frame.flashing  = false;
	frame.Super     = {};
	
	frame.texture:SetPoint( "CENTER", frame, "CENTER", 0, 0 );
	frame.texture:SetTexture( "Interface\\PetPaperDollFrame\\UI-PetHappiness" );

	nUI:TableInsertByValue( HappinessFrames, frame );
	
	-- called when the underlying GUID for the unit changes or when the
	-- content of the GUID is updated
	
	frame.Super.newUnitInfo = frame.newUnitInfo;
	frame.newUnitInfo       = function( list_unit, unit_info )
		
--		nUI_ProfileStart( FrameProfileCounter, "newUnitInfo" );
		
		frame.Super.newUnitInfo( list_unit, unit_info );
		
--		nUI:debug( "nUI_UnitHappiness: got new pet info for "..list_unit.." in "..frame:GetName().." -- unit_info = "..(unit_info and unit_info.name or "<nil>") );
		
		if frame.enabled then
			nUI_Unit:updateHappinessFrame( frame );
		end
		
--		nUI_ProfileStop();
		
	end

	-- setting enabled to false will prevent the frame from updating when new
	-- unit information is received (saves framerate). Setting enabled true will
	-- call the frame to immediately update if its content has changed since it
	-- was disabled

	frame.Super.setEnabled = frame.setEnabled;	
	frame.setEnabled       = function( enabled )
		
--		nUI_ProfileStart( FrameProfileCounter, "setEnabled" );
		
		prior_state = frame.enabled;
		
		frame.Super.setEnabled( enabled );
		
		if frame.enabled ~= prior_state then
		
			if frame.enabled then
				frame.unit_info = nUI_Unit:registerHappinessCallback( frame.unit, frame );
				nUI_Unit:updateHappinessFrame( frame );
			else
				nUI_Unit:unregisterHappinessCallback( frame.unit, frame );
			end
		end

--		nUI_ProfileStop();
		
	end
	
	-- used to change the scale of the frame... rather than the Bliz widget frame:SetScale()
	-- this method actually recalculates the size of the frame and uses frame:SetHeight()
	-- and frame:SetWidth() to reflect the actual size of the frame. Is also recreates
	-- the font to present clear, sharp, readable text versus the blurred text you get
	-- as a result of frame:SetScale() or text:SetTextHeight()

	frame.Super.applyScale = frame.applyScale;
	frame.applyScale       = function( scale )

--		nUI_ProfileStart( FrameProfileCounter, "applyScale" );
		
		frame.Super.applyScale( scale );
		
		if frame.texture.hSize  ~= frame.hSize 
		or frame.texture.vSize  ~= frame.vSize 
		or frame.texture.width  ~= frame.width
		or frame.texture.height ~= frame.height 
		or frame.texture.hInset ~= frame.hInset
		or frame.texture.vInset ~= frame.vInset
		then
			
			frame.texture.hSize  = frame.hSize;
			frame.texture.vSize  = frame.vSize;
			frame.texture.width  = frame.width;
			frame.texture.height = frame.height;
			frame.texture.hInset = frame.hInset;
			frame.texture.vInset = frame.vInset;
			
			frame.texture:SetWidth( (frame.hSize or frame.width) - frame.hInset );
			frame.texture:SetHeight( (frame.vSize or frame.height) - frame.vInset );

		end				

--		nUI_ProfileStop();
		
	end
	
	-- applies the set of frame options to this frame. Typically called when the frame 
	-- is first created or when the user changes options via config.

	frame.Super.applyOptions = frame.applyOptions;
	frame.applyOptions       = function( options )

--		nUI_ProfileStart( FrameProfileCounter, "applyOptions" );
		
		frame.Super.applyOptions( options );		
		nUI_Unit:updateHappinessFrame( frame );		
		
--		nUI_ProfileStop();
		
	end

	-- method for toggling click-to-feed functionality on and off
	
	frame.enableClickToFeed = function( enabled )
		
--		nUI_ProfileStart( FrameProfileCounter, "enableClickToFeed" );
		
		if frame.click_to_feed ~= enabled then
			
			frame.click_to_feed = enabled;
	
			if not enabled then
							
				frame:EnableMouse( false );
				frame:RegisterForClicks();
				frame:SetScript( "OnClick", nil );
				frame:SetScript( "OnEnter", nil );
				frame:SetScript( "OnLeave", nil );
				
			else
	
				local unit_info = frame.unit_info;
				
				frame:EnableMouse( true );
				frame:RegisterForClicks( "AnyUp" );
				
				frame:SetScript( "OnClick", 
					function() 
						GameTooltip:Hide();
						nUI_PetFeeder:FeedPet( frame ); 
					end 
				);
				
				frame:SetScript( "OnEnter", 
					function() 
						
						happiness = frame.unit_info.pet_happiness;
						loyalty   = frame.unit_info.pet_loyalty;
						damage    = frame.unit_info.pet_damage;
						
						GameTooltip:SetOwner( frame );
						
						if happiness == 3 then GameTooltip:SetText( nUI_L["nUI: %s is happy."]:format( frame.unit_info.name ), 0, 1, 0 );
						elseif happiness == 2 then GameTooltip:SetText( nUI_L["nUI: %s is unhappy... time to feed!"]:format( frame.unit_info.name ), 1, 0.83, 0 );
						else GameTooltip:SetText( nUI_L["nUI: Warning! %s is |cFFFFFFFFNOT|r happy! Better feed soon."]:format( frame.unit_info.name ), 1, 0.5, 0.5 );
						end
						
						if loyalty then
							if loyalty <= -29 then GameTooltip:AddLine( nUI_L["nUI: Warning... %s is %s losing loyalty "]:format( nUI_L["Your pet"], nUI_L["quickly"] ), 1, 0.5, 0.5 );
							elseif loyalty < 0 then GameTooltip:AddLine( nUI_L["nUI: Warning... %s is %s losing loyalty "]:format( nUI_L["Your pet"], nUI_L["slowly"] ), 1, 0.83, 0 );
							elseif loyalty > 0 and loyalty < 20 then GameTooltip:AddLine( nUI_L["nUI: %s is %s gaining loyalty"]:format( nUI_L["Your pet"], nUI_L["slowly"] ), 1, 1, 1 );
							elseif loyalty > 0 then GameTooltip:AddLine( nUI_L["nUI: %s is %s gaining loyalty"]:format( nUI_L["Your pet"], nUI_L["quickly"] ), 0, 1, 0 );
							end
						end

						if damage then
							if damage < 100 then GameTooltip:AddLine( nUI_L["Your pet's current damage penalty is %d%%"]:format( damage - 100 ), 1, 0.5, 0.5 );
							elseif damage > 100 then GameTooltip:AddLine( nUI_L["Your pet's current damage bonus is %d%%"]:format( damage - 100 ), 0, 1, 0 );
							end
						end
						
						if happiness ~= 3 then
							
							if not nUI_PetFeeder:IsShown() then
								text = nUI_L["Click to feed %s"]:format( unit_info.name );
							else
								text = nUI_L["Click to cancel feeding"];
							end
							
							GameTooltip:AddLine( "<"..text..">", 0, 1, 1 );
							
						end
						
						GameTooltip:Show();
					end 
				);
				
				frame:SetScript( "OnLeave", 
					function() 
						GameTooltip:Hide(); 
					end 
				);
				
			end
		end

--		nUI_ProfileStop();
		
	end
	
	-- initiate the frame
	
	frame.unit_info = nUI_Unit:registerHappinessCallback( frame.unit, frame );
	
	frame.applyOptions( options );
	
--	nUI_ProfileStop();
	
	return frame;
	
end

-------------------------------------------------------------------------------
-- remove a happiness indicator frame

function nUI_Unit:deleteHappinessFrame( frame )

--	nUI_ProfileStart( ProfileCounter, "deleteHappinessFrame" );
	
	nUI:TableRemoveByValue( HappinessFrames, frame );
	nUI_Unit:unregisterHappinessCallback( frame.unit, frame );
	nUI_Unit:deleteFrame( frame );
	
--	nUI_ProfileStop();
	
end

-------------------------------------------------------------------------------
-- set the pet happiness indicator according to the current values
--
-- note: this method expends extra energy in state management... as in knowing
--       exactly what state it is currently in and only updating the frame text,
--       content, colors, alphas, etc. when a change in state occurs. The extra
--       effort is spent on state management in order to reduce impact to the
--       graphis engine so as to preserve frame rate. It costs far less to check
--		 a memory value that and burn through the graphics geometry. It does not
--       matter how many times the unit changes GUID or how many times this 
--       method will call, it will only alter the graphics elements when its
--       relative state changes.

function nUI_Unit:updateHappinessFrame( frame )
	
--	nUI_ProfileStart( ProfileCounter, "updateHappinessFrame" );
	
	unit_info = frame.unit_info;
	unit_id = frame.unit;

--	nUI:debug( "nUI_UnitHappiness: ".."updating pet happiness button for "..unit_id, 1 );
	
	-- if there's no info for this frame, then the associated unit
	-- doesn't exist at this time and the frame should be hidden
	-- likewise, even if there is unit info, if this isn't a
	-- hunter pet then happiness does not apply to it
	
	if not unit_info 
	or not unit_info.is_hunter_pet 
	then
		
		if frame.active then

--			nUI:debug( "nUI_UnitHappiness: ".."disabling pet happiness for "..unit_id.." -- "..(unit_info and "not a hunter pet" or "unit does not exist"), 1 );
			
			frame.active = false;
			frame:SetAlpha( 0 );
			
			frame.enableClickToFeed( false );
			
		end

	elseif not frame.options.show_happy
	and unit_info.pet_happiness == 3
	then
		
		if frame.active then

--			nUI:debug( "nUI_UnitHappiness: ".."disabling pet happiness for "..unit_id.." -- happiness indicator is disabled for happy pets", 1 );
			
			frame.active = false;
			frame:SetAlpha( 0 );
			
			frame.enableClickToFeed( false );
			
		end

	else

		local happiness = unit_info.pet_happiness;

		-- update the texture if it needs changing
		
		if frame.happiness ~= happiness then
		
			-- angry pet
						
			if happiness == 1 then
				
				frame.flashing = frame.options.angry_flash;
				frame_scale    = frame.options.angry_scale or 1;
				x1, y1, x2, y2 = 0.375, 0.5625, 0, 0.359375;
				
			-- unhappy pet
			
			elseif happiness == 2 then
	
				frame.flashing = frame.options.unhappy_flash;
				frame_scale    = frame.options.unhappy_scale or 1;
				x1, y1, x2, y2 = 0.1875, 0.375, 0, 0.359375;
				
			-- happy pet
			
			elseif happiness == 3 then
	
				frame.flashing = false;
				frame_scale    = 1;
				x1, y1, x2, y2 = 0, 0.1875, 0, 0.359375;
							
			-- this should never happen, but could if Bliz changes it's code
			
			else 
			
				DEFAULT_CHAT_FRAME:AddMessage( nUI_L["nUI_UnitHappiness.lua: unhandled pet happiness value (%s)"]:format( happiness or "<nil>" ), 1, 0.5, 0.5 );
				happiness = frame.happiness;
				
			end		

			-- make sure the alpha is full on the happiness icon if we're not flashing
			
			if not frame.flashing and frame.alpha ~= 1 then
				frame.alpha = 1;
				frame:SetAlpha( 1 );
			end
			
--			nUI:debug( "nUI_UnitHappiness: ".."pet happiness changed from "..(frame.happiness or "<nil>").." to "..(happiness or "<nil>").." for "..unit_id, 1 );
			
			frame.happiness = happiness;
			frame.texture:SetTexCoord( x1, y1, x2, y2 );
			frame:SetScale( frame_scale );

		end
		
		-- make sure the click-to-feed function is currently enabled
		-- and the happiness indicator is visible
		
		if not frame.active or not frame.click_to_feed then
				
			frame.active = true;
			frame:SetAlpha( frame.flashing and frame_alpha or 1 );
			frame.enableClickToFeed( true );
			
		end		
	end	
	
--	nUI_ProfileStop();
	
end
