
--- Returns a class color format string.
local function _GetClassTypeColorHex(classType)
	local color = RAID_CLASS_COLORS[classType]
	return string.format("|cff%02x%02x%02x",
		color.r * 255, color.g * 255, color.b * 255)
end

--- Returns a message type color format string.
local function _GetMessageTypeColorHex(messageType)
	local r, g, b, _ = GetMessageTypeColor(messageType)
	return string.format("|cff%02x%02x%02x", r * 255, g * 255, b * 255)
end



--- Returns a prefix for NPC emotes.
-- The prefix includes only the format string to style text like a typical
-- /emote message.
local function _CreateNpcEmotePrefix()
	local textColor = _GetMessageTypeColorHex("CHAT_MSG_EMOTE")
	return textColor
end

--- Returns an appropriate emote prefix.
-- @return The emote prefix.
local function _CreateEmotePrefix(isPlayer)
	return _CreateNpcEmotePrefix()
end



--- Returns a prefix for NPC gossip.
-- The prefix is of the form "arthas says:", styled to look like ai typical NPC
-- /say message.
-- @return The gossip prefix.
local function _CreateNpcGossipPrefix()
	local textColor = _GetMessageTypeColorHex("CHAT_MSG_MONSTER_SAY")
	local guid = UnitGUID("npc")
	if not guid then
		return "... "
	end

	local name = UnitName("npc")
	return string.format("%s|Hunit:%s:%s|h%s|h says: ",
		textColor, guid, name, name)
end

--- Returns a prefix for player gossip.
-- The prefix is of the form "hascat says:", styled to look like a typical /say
-- message.
-- @return The gossip prefix.
local function _CreatePlayerGossipPrefix()
	local textColor, _ = _GetMessageTypeColorHex("CHAT_MSG_SAY")
	local name = UnitName("player")
	local _, classType = UnitClass("player")
	local classColor = _GetClassTypeColorHex(classType)
	return string.format("%s|Hplayer:%s|h[%s]|h%s says: ",
		classColor, name, name, textColor)
end

--- Returns an appropriate gossip prefix.
-- @param isPlayer true if the prefix should be for the player; False if the
-- prefix should be for the target NPC.
-- @return
local function _CreateGossipPrefix(isPlayer)
	if isPlayer then
		return _CreatePlayerGossipPrefix()
	else
		return _CreateNpcGossipPrefix()
	end
end



--- Returns true if the player should accept a quest.
-- This returns true if room is available in the quest log to accept a quest.
-- This returns false if the quest log is full. If the isDaily parameter is
-- true, this returns false if the maximum number of daily quests have already
-- been completed.
-- @param isDaily True to test if a daily quest should be accepted.
-- @param True if the quest being checked is a daily quest.
local function _ShouldAcceptQuest(isDaily)
	local _, numQuests = GetNumQuestLogEntries()
	if numQuests >= MAX_QUESTS then
		return false
	end
	
	return true
end

--- Finds the first acceptable quest in a list of available quests.
-- The available quests are given as a flattened list of 6-tuples including the
-- following attributes per quest: name, level, isTrivial, isDaily, 
-- isRepeatable, and isLegendary. The index of the first acceptable quest is
-- returned.
-- @return The index of the first acceptable quest.
local function _FindAcceptQuest(...)
	local index = 1
	for i = 1, select("#", ...), 6 do
		local isDaily = select(i + 3, ...)
		if _ShouldAcceptQuest(isDaily) then
			return index
		end
		index = index + 1
	end
	
	return 0
end

--- Finds the first complete quest in a list of active quests.
-- The active quests are given as a flattened list of 5-tuples including the
-- following attributes per quest: name, level, isTrivial, isComplete, and
-- isLegendary. The index of the first quest for which isComplete is true is
-- returned.
-- @return The index of the first complete quest.
local function _FindCompleteQuest(...)
	local index = 1
	for i = 1, select("#", ...), 5 do
		local isComplete = select(i + 3, ...)
		if isComplete then
			return index
		end
		index = index + 1
	end
	
	return 0
end



--- Returns the vendor price of a given item.
-- @param item an item id, item name, or item link
-- @return The vendor price, in copper.
local function _GetItemVendorPrice(item)
	local _, _, _, _, _, _, _, _, _, _, vendorPrice = GetItemInfo(item)
	return vendorPrice
end

--- Returns the quest item choice with the highest vendor value.
-- @return The choice index of the highest vendor value item.
local function _GetBestQuestItemChoice()
	local bestChoice = nil
	local bestPrice = nil
	for choice = 1, GetNumQuestChoices() do
		local item = GetQuestItemLink("choice", choice)
		local price = _GetItemVendorPrice(item)
		if (not bestPrice) or (bestPrice < price) then
			bestChoice = choice
			bestPrice = price
		end
	end
	return bestChoice
end

--- Returns true if the message looks like an emote.
-- NPC emotes are typically enclosed in angle brackets.
local function _IsEmote(message)
	return string.match(message, "^<(.+)>$")
end

--- Tests if automatic interaction is enabled.
-- When key is pressed, all interaction with the target NPC will be automatic
-- except for the choice of quest rewards.
-- @return True if interaction should be automatic.
local function _IsGossipEnabled()
	return IsShiftKeyDown()
end

--- Tests whether or not the target NPC is a merchant.
local function _IsMerchant()
	return GetMerchantNumItems() > 0 or CanMerchantRepair()
end

--- Tests if automatic quest rewards are enabled.
-- When CTRL is pressed, quest rewards will be automatically chosen based
-- on their vendor value.
-- @return True if quest rewards should be automatically chosen.
local function _IsRewardEnabled()
	return IsControlKeyDown()
end

--- Prints styled gossip text to a given frame.
-- This is given a list of messages, each to be printed individually. The
-- messages are given a prefix and style appropriate to the type of message
-- (emote or text) and the source or the message (player or NPC).
-- @param isPlayer True if the gossip text is for the player; false if the text
-- is for an NPC. 
-- @param frame The chat frame to which the gossip should be written.
local function _PrintGossipItems(isPlayer, frame, ...)
	local emotePrefix = _CreateEmotePrefix(isPlayer)
	local gossipPrefix = _CreateGossipPrefix(isPlayer)
	for i = 1, select('#', ...) do
		local message = strtrim(select(i, ...))
		if strlen(message) > 0 then
			if _IsEmote(message) then
				frame:AddMessage(emotePrefix .. message)
			else
				frame:AddMessage(gossipPrefix .. message)
			end
		end
	end
end

--- Prints gossip text to the default chat frame.
-- If the gossip text contains multiple lines, each line will be output
-- individually. Each line of text is formatted to appear like normal 
-- creature speech messages.
-- @param player true if the gossip text is spoken by the player.
-- @param ... strings to be written.
local function _PrintGossip(player, ...)
	for i = 1, select("#", ...) do
		local message = select(i, ...)
		if type(message) == "string" then
			_PrintGossipItems(player, DEFAULT_CHAT_FRAME, strsplit("\n", message))
		end
	end
end



-- Construct the addon.
local addon = CreateFrame("Frame", "hcGossipBotFrame")

--- Dispatches events to this addon by calling the method with the 
-- same name as the given event.
function addon:OnEvent(event, ...)
	if self[event] then
		self[event](self, event, ...)
	end
end

-- Register the event handler and initial events.
addon:SetScript("OnEvent", addon.OnEvent)

--- Fires when an NPC gossip interaction begins.
-- This is fired for most initial NPC interactions. 
function addon:GOSSIP_SHOW(event)
	if not _IsGossipEnabled() then
		return
	end
	
	-- If any active quests have been completed, turn in the first one 
	-- the active quest list.
	if GetNumGossipActiveQuests() > 0 then
		local index = _FindCompleteQuest(GetGossipActiveQuests())
		if index > 0 then
			SelectGossipActiveQuest(index)
			return
		end
	end
	
	-- If any quests are available, accept the first one in the available
	-- quest list.
	if GetNumGossipAvailableQuests() > 0 then
		local index = _FindAcceptQuest(GetGossipAvailableQuests())
		if index > 0 then
			SelectGossipAvailableQuest(index)
			return
		end
	end

	-- If only one gossip option is available, select it.
	if GetNumGossipOptions() == 1 then
		_PrintGossip(true, (GetGossipOptions()))
		_PrintGossip(false, GetGossipText())
		SelectGossipOption(1)
		return
	end
	
	local option_count = GetNumGossipOptions() + 
		GetNumActiveQuests()

	if (option_count == 0) then
		-- This seems to fire for merchants
		if not _IsMerchant() then
			_PrintGossip(false, GetGossipText())
		end
		CloseGossip()
		return
	end
end

---	Fires when escort quests are started by a party member.
-- @param name the name of the user who started the quest.
-- @param quest te name of the quest that was started.
function addon:QUEST_ACCEPT_CONFIRM(event, name, quest)
	if not _IsGossipEnabled() then
		return
	end
	
	if _ShouldAcceptQuest() then
		ConfirmAcceptQuest()
	end
end

---	Fires when a quest is automatically completed.
-- These types of quests may be handed in remotely, and will pop up 
-- with a completion indicator in the objective list.
-- @param questId the id of the quest completed.
function addon:QUEST_AUTOCOMPLETE(event, questId)
	if not _IsGossipEnabled() then
		return
	end
	
	ShowQuestComplete(questId)
end

---	Fires when the player is looking at the Complete page for a quest.
-- This may be at a questgiver, or may be via remote completion.
function addon:QUEST_COMPLETE(event)
	if not _IsGossipEnabled() then
		return
	end
	
	local choice = nil
	
	if GetNumQuestChoices() > 0 then
		if GetNumQuestChoices() > 1 then
			if not _IsRewardEnabled() then
				return
			end
			choice = _GetBestQuestItemChoice()
		else
			choice = 1
		end
	end
	
	_PrintGossip(false, GetProgressText())
	GetQuestReward(choice)
end

--- Fires when details of an available quest are presented by a questgiver.
function addon:QUEST_DETAIL(event)
	if not _IsGossipEnabled() then
		return
	end
	
	if not _ShouldAcceptQuest(QuestIsDaily()) then
		DeclineQuest()
		return
	end

	_PrintGossip(false, GetQuestText(), GetObjectiveText())
	AcceptQuest()
end

--- Fires when interacting with a questgiver about an active quest.
function addon:QUEST_PROGRESS(event)
	if not _IsGossipEnabled() then
		return
	end
	
	_PrintGossip(false, GetProgressText())
	
	CompleteQuest()
end

--- Registers gossip events once the addon is loaded.
-- The ADDON_LOADED event is unregistered to prevent redundant calls to this
-- handler.
function addon:ADDON_LOADED(event, name)
	if name ~= "hcGossipBot" then
		return
	end

	self:UnregisterAllEvents()

	self:RegisterEvent("GOSSIP_SHOW")
	self:RegisterEvent("QUEST_ACCEPT_CONFIRM")
	self:RegisterEvent("QUEST_AUTOCOMPLETE")
	self:RegisterEvent("QUEST_COMPLETE")
	self:RegisterEvent("QUEST_DETAIL")
	self:RegisterEvent("QUEST_PROGRESS")
end

addon:RegisterEvent("ADDON_LOADED")

