local L = LibStub("AceLocale-3.0"):GetLocale("WorldBossStatus")
WorldBossStatus = LibStub("AceAddon-3.0"):NewAddon("WorldBossStatus", "AceConsole-3.0", "AceEvent-3.0" );

local trackableCurrencies = {
	{ id = 395 },
	{ id = 697 },
	{ id = 777 },
	{ id = 752 },
	{ id = 738 },
	{ id = 776 }
}

for key,value in pairs(trackableCurrencies) do
	value.name, _, value.texture = GetCurrencyInfo(value.id)
end

local textures = {}

textures.worldBossStatus = "Interface\\TARGETINGFRAME\\UI-RaidTargetingIcon_8.png"
textures.question = "|TInterface\\Minimap\\OBJECTICONS:20:20:0:0:256:192:32:64:20:48|t"
textures.alliance = "|TInterface\\WorldStateFrame\\AllianceIcon:18|t"
textures.horde = "|TInterface\\WorldStateFrame\\HordeIcon:18|t"
textures.bossDefeated = "|TInterface\\WorldMap\\Skull_64Red:18|t"
textures.bossAvailable = "|TInterface\\WorldMap\\Skull_64Grey:18|t"
textures.quest = "|TInterface\\Minimap\\OBJECTICONS:20:20:0:0:256:192:32:64:20:48|t"
textures.questCompleted = "|TInterface\\BUTTONS\\UI-CheckBox-Check:18|t"

local addonName = "WordBossStatus";
local LDB = LibStub("LibDataBroker-1.1", true)
local LDBIcon = LDB and LibStub("LibDBIcon-1.0", true)
local LibQTip = LibStub('LibQTip-1.0')

--statuses = {}
--statuses[0] = L["Available"]
--statuses[1] = L["Defeated"]

--colors = {"FF00FF00", "FFFF0000"}
--colors[0] = "FF00FF00"
--colors[1] = "FFFF0000"

local red = { r = 1.0, g = 0.2, b = 0.2 }
local blue = { r = 0.4, g = 0.4, b = 1.0 }
local green = { r = 0.2, g = 1.0, b = 0.2 }
local yellow = { r = 1.0, g = 1.0, b = 0.2 }
local gray = { r = 0.5, g = 0.5, b = 0.5 }
local black = { r = 0.0, g = 0.0, b = 0.0 }
local white = { r = 1.0, g = 1.0, b = 1.0 }
local frame

local WORLD_BOSSES = { WORLD_BOSS_ORDOS, WORLD_BOSS_FOUR_CELESTIALS, WORLD_BOSS_OONDASTA, WORLD_BOSS_NALAK, 
					   WORLD_BOSS_GALLEON, WORLD_BOSS_SHA_OF_ANGER}


local WorldBossStatusLauncher = LDB:NewDataObject(addonName, {
		type = "data source",
		text = L["World Boss Status"],
			--"launcher",
		icon = textures.worldBossStatus,
		OnClick = function(clickedframe, button)
			WorldBossStatus:ShowOptions() 
		end,
		OnEnter = function(self)
			frame = self
			RequestRaidInfo()
			WorldBossStatus:ShowToolTip()
		end,
	})
	
local defaults = {
	realm = {
		characters = {
			},
	},
	global = {
		realms = {
			},
		MinimapButton = {
			hide = false,
		}, 
		characterOptions = {
			levelRestriction = true,
			minimumLevel = 90,
			removeInactive = true,
			inactivityThreshold = 28,
			include = 3,
		},
		bossOptions = {
			hideBoss = {
			},			
		},
		bonusRollOptions = {
			trackWeeklyQuest = true,
			trackedCurrencies = { 
				[738] = true, 
				[776] = true, 
			},
		},
	},
};

local options = {
    handler = WorldBossStatus,
    type = "group",
    args = {
		features = {
			handler = WorldBossStatus,
			type = 'group',
			name = L["General Options"],
			desc = "",
			order = 10,
			args = {			
				showMiniMapButton = {
					type = "toggle",
					name = L["Minimap Button"],
					desc = L["Toggles the display of the minimap button."],
					get = "IsShowMinimapButton",
					set = "ToggleMinimapButton",
					order=1,
				},
			},
		},
		characterOptions = {
			handler = WorldBossStatus,
			type = 'group',
			name = L["Character Options"],
			desc = "",
			order = 20,
			args = {	
				inlcudeCharactersOptions = {
					type = 'group',
					inline = true,
					name = L["Show Characters"],
					desc = "",
					order = 1,
					args = {					
						realmOption = {
							type = "toggle",
							name = L["On this realm"],
							desc = L["Show characters on this realm."],
							get = function(info)
								return WorldBossStatus.db.global.characterOptions.include == 2
							end,
							set = function(info, value)
								if value then 
									WorldBossStatus.db.global.characterOptions.include = 2
								else
									WorldBossStatus.db.global.characterOptions.include = 1
								end
							end,
							order=1,
						},
						accountOption = {
							type = "toggle",
							name = L["On this account"],
							desc = L["Show characters on this WoW account."],
							get = function(info)
								return WorldBossStatus.db.global.characterOptions.include == 3
							end,
							set = function(info, value)
								if value then 
									WorldBossStatus.db.global.characterOptions.include = 3
								else
									WorldBossStatus.db.global.characterOptions.include = 1
								end
							end,
							order=2,
						},
					},
				},
				characterLevelOptions = {
					type= "group",
					inline = true,
					name = L["Level Restriction"],
					desc = "",
					order=5,
					args = {
						enableLevelRestriction = {
							type = "toggle",
							name = L["Enable"],
							desc = L["Enable level restriction."],
							get = function(info)
								return WorldBossStatus.db.global.characterOptions.levelRestriction
							end,
							set = function(info, value)
								WorldBossStatus.db.global.characterOptions.levelRestriction = value
							end,
							order=1,
						},
						minimumLevelOption = {
							type = "range",
							name = L["Minimum Level"],
							desc = L["Show characters this level and higher."],
							step = 1, min = 1, max = 90,
							order = 2,
							get = function(info)
								return WorldBossStatus.db.global.characterOptions.minimumLevel
							end,
							set = function(info, value)
								WorldBossStatus.db.global.characterOptions.minimumLevel = value
							end,
							disabled = function()
								return not WorldBossStatus.db.global.characterOptions.levelRestriction
							end,
						},
					},
 				},
				hideInactiveOptions = {
					type= "group",
					inline = true,
					name = L["Hide Inactive Characters"],
					desc = "",
					order=6,
					args = {
						purgeInactiveCharacters = {
							type = "toggle",
							name = L["Enable"],
							desc = L["Enable hiding inactive characters."],
							get = function(info)
								return WorldBossStatus.db.global.characterOptions.removeInactive
							end,
							set = function(info, value)
								WorldBossStatus.db.global.characterOptions.removeInactive = value
							end,
							order=1,
						},
						inactivityThresholdOption = {
							type = "range",
							name = L["Inactivity Threshold (days)"],
							desc = L["Hide characters that have been inactive for this many days."],
							step = 1, min = 14, max = 42,
							order = 2,
							get = function(info)
								return WorldBossStatus.db.global.characterOptions.inactivityThreshold
							end,
							set = function(info, value)
								WorldBossStatus.db.global.characterOptions.inactivityThreshold = value
							end,
							disabled = function()
								return not WorldBossStatus.db.global.characterOptions.removeInactive
							end,
						},
					},
				},
				trackedCharactersOption = {
					type = "group",
					inline = true,
					name = L["Remove Tracked Characters"],
					desc = "",
					order = 7,
					args = {
						realmSelect = {
							type = "select",
							name = L["Realm"],
							desc = L["Select a realm to remove a tracked character from."],
							order = 1,
							values = function()
										local realmList = {}

										for realm in pairs(WorldBossStatus.db.global.realms) do
											realmList[realm] = realm
										end

										return realmList
									 end,
							get = function(info)
									return selectedRealm
								  end,
							set = function(info, value)
									selectedRealm = value
									selectedCharacter = nil
								  end,
						},
						characterSelect = {
							type = "select",
							name = L["Character"],
							desc = L["Select the tracked character to remove."],
							order = 2,
							disabled = function()
										  return selectedRealm == nil
									   end,
							values = function()
										local list = {}
										local realmInfo = WorldBossStatus.db.global.realms[selectedRealm]
										if realmInfo then
											local characters = realmInfo.characters
	
											for key,value in pairs(characters) do
												list[key] = key
											end
										end
										return list
									 end,
							get = function(info)
									return selectedCharacter
								  end,
							set = function(info, value)
									selectedCharacter = value
								  end,
						},
						removeAction = {
							type = "execute",							
							name = L["Remove"],
							desc = L["Click to remove the selected tracked character."],
							order = 3,
							disabled = function()
										  return selectedRealm == nil or selectedCharacter == nil
									   end,
							func = function()

								local realmInfo = WorldBossStatus.db.global.realms[selectedRealm]
								local characterInfo = realmInfo.characters[selectedCharacter]
								local count = 0

								if not realmInfo then
									return
								end

								if characterInfo then 
									realmInfo.characters[selectedCharacter] = nil								
								end
								
								for key,value in pairs(realmInfo.characters) do 
									count = count + 1
								end
								
								if count == 0 then 
									WorldBossStatus.db.global.realms[selectedRealm] = nil
								end
							end,
						},
					},
				},
			}		
		},
		bossTracking = {
			type = "group",
			name = L["Boss Options"],
			desc = "",
			order = 30,
			args = {
				trackedBosses = {
					type = "multiselect",
					name = L["Tracked Bosses"],
					desc = L["Select the world bosses you would like to track."],
					width = "full",
					values = WORLD_BOSSES,
					get = function(info, key)
							return not WorldBossStatus.db.global.bossOptions.hideBoss[WORLD_BOSSES[key]]
						end,
					set = function(info, key, value)
							WorldBossStatus.db.global.bossOptions.hideBoss[WORLD_BOSSES[key]] = not value
						end,
				},
			}
		},
		bonusRollTracking = {
			type = "group",
			--inline = true,
			handler = WorldBossStatus,
			name = L["Bonus Roll Options"],
			desc = "",
			order = 40,
			args = {
				enabled = {
					type = "toggle",
					name = L["Track weekly quest"],
					desc = L["Enable tracking the weekly bonus roll currency quest."],
					width = "full",
					get = function(info)
							return WorldBossStatus.db.global.bonusRollOptions.trackWeeklyQuest
						end,
					set = function(info, value)
							WorldBossStatus.db.global.bonusRollOptions.trackWeeklyQuest = value
						end,
					order=1,
				},
				trackedCurrencies = {
					type = "multiselect",
					name = L["Tracked Currencies"],
					desc = L["Select the currencies you would like to track."],
					width = "full",
					values = "GetCurrencyOptions",
					get = function(info, key)
							return WorldBossStatus.db.global.bonusRollOptions.trackedCurrencies[trackableCurrencies[key].id]
						end,
					set = function(info, key, value)
							WorldBossStatus.db.global.bonusRollOptions.trackedCurrencies[trackableCurrencies[key].id] = value
						end,
				},
			}			
		}
	}
}

function WorldBossStatus:GetCurrencyOptions()
    local itemsList = {}
	
    for key,value in pairs(trackableCurrencies) do
		itemsList[key] = "|T"..value.texture..":0|t "..value.name
    end

    return itemsList
end

--function WorldBossStatus:GetTrackedCharacters()
--	local characterList = {}

--	for realm in pairs(WorldBossStatus.db.global.realms) do
--		local realmInfo = self.db.global.realms[realm]
--		local characters = nil
		
--		if realmInfo then
--			local characters = realmInfo.characters
	
--			for key,value in pairs(characters) do
--				local name = key.."-"..realm
--				characterList[name] = name
--			end
			
--		end
--	end

--	return characterList
--end

local function CleanupCharacters()
	local threshold = WorldBossStatus.db.global.characterOptions.inactivityThreshold * (24 * 60 * 60)	
	
	if not WorldBossStatus.db.global.characterOptions.removeInactive or threshold == 0 then
		return
	 end
	

	for realm in pairs(WorldBossStatus.db.global.realms) do
		local realmInfo = self.db.global.realms[realm]
		local characters = nil
		
		if realmInfo then
			local characters = realmInfo.characters
	
			for key,value in pairs(characters) do
				if value.lastUpdate and value.lastUpdate < time() - threshold then
					value = nil
				end
			end
			
		end
	end
	
end

--function WorldBossStatus:GetOtherCharacters(realm)
--	characters = {}

--	return characters
--end 

function WorldBossStatus:DisplayCharacterInTooltip(characterName, characterInfo)
	local tooltip = WorldBossStatus.tooltip
	local line = tooltip:AddLine()
	local factionIcon = ""
	local coins = 0
	local seals = 0

	if characterInfo.faction and characterInfo.faction == "Alliance" then
		factionIcon = textures.alliance
	elseif characterInfo.faction and characterInfo.faction == "Horde" then
		factionIcon = textures.horde
	end

	tooltip:SetCell(line, 2, factionIcon.." "..characterName)

	if (characterInfo.bonusRolls) then
		coins = characterInfo.bonusRolls.questCurrency		
		seals = characterInfo.bonusRolls.rolls
	end

	column = 2
	for key,value in pairs(trackableCurrencies) do
		if WorldBossStatus.db.global.bonusRollOptions.trackedCurrencies[value.id] then
			column = column + 1
			if (characterInfo.bonusRolls and characterInfo.bonusRolls.currency and characterInfo.bonusRolls.currency[value.id]) then
				tooltip:SetCell(line, column, characterInfo.bonusRolls.currency[value.id], nil, "RIGHT")
			end
		end
	
	end

	if (WorldBossStatus.db.global.bonusRollOptions.trackWeeklyQuest) then
		column = column +1
	
		local questCompleted = characterInfo.bonusRolls and characterInfo.bonusRolls.questCompleted and time() < characterInfo.bonusRolls.questReset

		if questCompleted then
			tooltip:SetCell(line, column, textures.questCompleted, nil, "CENTER")
		end
	
	end


	column = column + 1
	for i = 1, 6  do
		if not WorldBossStatus.db.global.bossOptions.hideBoss[WORLD_BOSSES[i]] then		
			local boss = WORLD_BOSSES[i]
			local defeated = (characterInfo.bossKills[boss] and characterInfo.bossKills[boss] > time())

			if defeated then 
				tooltip:SetCell(line, column, textures.bossDefeated)
			else
				tooltip:SetCell(line, column, textures.bossAvailable )
			end

			if characterInfo.class then
				local color = RAID_CLASS_COLORS[characterInfo.class]
				tooltip:SetCellTextColor(line, 2, color.r, color.g, color.b)
			end

			column = column+1
		end
	end
end


function WorldBossStatus:IsShowMinimapButton(info)
	return not self.db.global.MinimapButton.hide
end

function WorldBossStatus:ToggleMinimapButton(info, value)
	self.db.global.MinimapButton.hide = not value

	if self.db.global.MinimapButton.hide then
		LDBIcon:Hide(addonName)
	else
		LDBIcon:Show(addonName)
	end

	LDBIcon:Refresh(addonName)
	LDBIcon:Refresh(addonName)
end

--function WorldBossStatus:IsShowObjectives(info)
--	return not self.db.global.Objectives.hide
--end

--function WorldBossStatus:ToggleObjectives(info, value)
--	self.db.global.Objectives = not value	
--end

function WorldBossStatus:ShowOptions()
	InterfaceOptionsFrame_OpenToCategory(self.optionsFrame)
	InterfaceOptionsFrame_OpenToCategory(self.optionsFrame)
end

function WorldBossStatus:OnInitialize()	
	self.db = LibStub("AceDB-3.0"):New("WorldBossStatusDB", defaults, true)

	LDBIcon:Register(addonName, WorldBossStatusLauncher, self.db.global.MinimapButton)
	--WorldBossStatus:RegisterChatCommand("wbs", "ChatCommandHandler")


	local wbscfg = LibStub("AceConfig-3.0")
	wbscfg:RegisterOptionsTable("World Boss Status", options)
	wbscfg:RegisterOptionsTable("World Boss Status Features", options.args.features)
	wbscfg:RegisterOptionsTable("World Boss Status Characters", options.args.characterOptions)
	wbscfg:RegisterOptionsTable("World Boss Status Bosses", options.args.bossTracking)
	wbscfg:RegisterOptionsTable("World Boss Status Bonus Rolls", options.args.bonusRollTracking)


	local wbsdia = LibStub("AceConfigDialog-3.0")

	self.optionsFrame =  wbsdia:AddToBlizOptions("World Boss Status Features", L["World Boss Status"])
	wbsdia:AddToBlizOptions("World Boss Status Characters", L["Characters"], L["World Boss Status"])
	wbsdia:AddToBlizOptions("World Boss Status Bosses", L["Bosses"], L["World Boss Status"])
	wbsdia:AddToBlizOptions("World Boss Status Bonus Rolls", L["Bonus Rolls"], L["World Boss Status"])

	RequestRaidInfo()
end

--function WorldBossStatus:ChatCommandHandler(input)
--  if not input or input:trim() == "" then
--		RequestRaidInfo()
--		WorldBossStatus:ShowStatusInChatWindow()  
--    else 
--        WorldBossStatus:ShowOptions()
--    end
--end

--function WorldBossStatus:ShowStatusInChatWindow()
--	local statuses = {};

--	for x = 1, 6 do
--		statuses[x] = L["Available"]
--	end

--	local  num = GetNumSavedWorldBosses()
--	for i = 1, num do
--		name, worldBossID, reset = GetSavedWorldBossInfo(i)
--		statuses[worldBossID] = L["Defeated"]
--	end

--	for x = 1, 6 do
--		self:Print(WORLD_BOSSES[x].name.." "..statuses[x])
--	end
--end

local function ShowHeader(tooltip, marker, headerName)
	line = tooltip:AddHeader()

	if (marker) then
		tooltip:SetCell(line, 1, marker)
	end
	
	tooltip:SetCell(line, 2, headerName, nil, nil, nil, nil, nil, 50)
	tooltip:SetCellTextColor(line, 2, yellow.r, yellow.g, yellow.b)

	column = 2
	for key,value in pairs(trackableCurrencies) do
		if WorldBossStatus.db.global.bonusRollOptions.trackedCurrencies[value.id] then
			column = column + 1
			tooltip:SetCell(line, column, "|T"..value.texture..":0|t", nil, "RIGHT")
		end
	
	end

	if (WorldBossStatus.db.global.bonusRollOptions.trackWeeklyQuest) then
		column = column + 1
		tooltip:SetCell(line, column, textures.quest, nil, "CENTER")
	end

	column = column + 1
	for i = 1, 6  do				
		if not WorldBossStatus.db.global.bossOptions.hideBoss[WORLD_BOSSES[i]] then
			tooltip:SetCell(line, column, WORLD_BOSSES[i], "CENTER")
			tooltip:SetCellTextColor(line, column, yellow.r, yellow.g, yellow.b)
			column = column+1
		end
	end

	return line
end 

function WorldBossStatus:DisplayRealmInToolip(realmName)
	local realmInfo = self.db.global.realms[realmName]
	local characters = nil
	local collapsed = false
	local epoch = time() - (WorldBossStatus.db.global.characterOptions.inactivityThreshold * 24 * 60 * 60)

	if realmInfo then
		characters = realmInfo.characters
		collapsed = realmInfo.collapsed
	end

	local characterNames = {}
	local currentCharacterName = UnitName("player")
	local currentRealmName = GetRealmName()
	local tooltip = WorldBossStatus.tooltip
	local levelRestriction = WorldBossStatus.db.global.characterOptions.levelRestruction or false;
	local minimumLevel = 1

	if WorldBossStatus.db.global.characterOptions.levelRestriction then
		minimumLevel = WorldBossStatus.db.global.characterOptions.minimumLevel		
		if not minimumLevel then minimumLevel = 90 end
	end	
		
	if not characters then
		return 
	end

	for k,v in pairs(characters) do
		local inlcude = true
		if (realmName ~= currentRealmName or k ~= currentCharacterName) and 
		   (not WorldBossStatus.db.global.characterOptions.removeInactive or v.lastUpdate > epoch)  and
   		   (v.level >= minimumLevel) then
				table.insert(characterNames, k);
		end
	end

	if (table.getn(characterNames) == 0) then
		return
	end
			   
	table.sort(characterNames)

	tooltip:AddSeparator(2,0,0,0,0)

	if not collapsed then
		line = ShowHeader(tooltip, "|TInterface\\Buttons\\UI-MinusButton-Up:16|t", realmName)

		tooltip:AddSeparator(3,0,0,0,0)

		for k,v in pairs(characterNames) do
			WorldBossStatus:DisplayCharacterInTooltip(v, characters[v])
		end

		tooltip:AddSeparator(1, 1, 1, 1, 1.0)
	else
		line = ShowHeader(tooltip, "|TInterface\\Buttons\\UI-PlusButton-Up:16|t", realmName)
	end

	tooltip:SetCellTextColor(line, 2, yellow.r, yellow.g, yellow.b)	
	tooltip:SetCellScript(line, 1, "OnMouseUp", RealmOnClick, realmName)
end

function RealmOnClick(cell, realmName)
	WorldBossStatus.db.global.realms[realmName].collapsed = not WorldBossStatus.db.global.realms[realmName].collapsed
	WorldBossStatus:ShowToolTip()
end

function WorldBossStatus:ShowToolTip()
	local tooltip = WorldBossStatus.tooltip
	local characterName = UnitName("player")
	local bossKills = WorldBossStatus:GetWorldBossKills()
	local characters = WorldBossStatus.db.realm.characters
	local class, className = UnitClass("player")
	local includeCharacters = WorldBossStatus.db.global.characterOptions.include or 3

	if LibQTip:IsAcquired("WorldBossStatusTooltip") and tooltip then
		tooltip:Clear()
	else
		bossCount = 0
		
		for i = 1, 6  do				
			if not WorldBossStatus.db.global.bossOptions.hideBoss[WORLD_BOSSES[i]] then
				bossCount = bossCount + 1
			end
		end

		currencyCount = 0

		for key,value in pairs(trackableCurrencies) do
			if WorldBossStatus.db.global.bonusRollOptions.trackedCurrencies[value.id] then
				currencyCount = currencyCount + 1
			end
		end

		questCount = 0

		if WorldBossStatus.db.global.bonusRollOptions.trackWeeklyQuest then
			questCount = 1
		end

		tooltip = LibQTip:Acquire("WorldBossStatusTooltip", 3 + currencyCount + bossCount + questCount, "CENTER", "LEFT", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER", "CENTER")
		WorldBossStatus.tooltip = tooltip 
	end

	line = tooltip:AddHeader(" ")
	tooltip:SetCell(1, 1, "|TInterface\\TARGETINGFRAME\\UI-RaidTargetingIcon_8:16|t "..L["World Boss Status"], nil, "LEFT", tooltip:GetColumnCount())
	tooltip:AddSeparator(6,0,0,0,0)
	ShowHeader(tooltip, nil, L["Character"])
	tooltip:AddSeparator(6,0,0,0,0)
			
	local info = WorldBossStatus:GetCharacterInfo()
	WorldBossStatus:DisplayCharacterInTooltip(characterName, info)
	tooltip:AddSeparator(6,0,0,0,0)
	tooltip:AddSeparator(1, 1, 1, 1, 1.0)

	if includeCharacters > 1 then
		WorldBossStatus:DisplayRealmInToolip(GetRealmName())
	end
			
	if includeCharacters == 3 then
		realmNames = {}
				
		for k,v in pairs(WorldBossStatus.db.global.realms) do
			if (k ~= GetRealmName()) then
				table.insert(realmNames, k);
			end
		end
				
		for k,v in pairs(realmNames) do
			WorldBossStatus:DisplayRealmInToolip(v)
		end
	end

	line = tooltip:AddLine(" ")
	tooltip:SetCell(tooltip:GetLineCount(), 1, L["Click to open the options menu"], nil, "LEFT", tooltip:GetColumnCount())

	if (frame) then
		tooltip:SetAutoHideDelay(0.01, frame)
		tooltip:SmartAnchorTo(frame)
	end 

	tooltip:UpdateScrolling()
	tooltip:Show()
end

function WorldBossStatus:SaveCharacterInfo()
	local characterName = UnitName("player")
	local realmName = GetRealmName()
	
	if not self.db.global.realms then
		self.db.global.realms = {}
	end

	local realmInfo = self.db.global.realms[realmName]

	if not realmInfo then
		realmInfo = {}
		realmInfo.characters = {}
	end
	
	realmInfo.characters[characterName]  = WorldBossStatus:GetCharacterInfo()


	self.db.global.realms[realmName] = realmInfo
end

function WorldBossStatus:GetCharacterInfo()
	local characterInfo = {}
	local class, className = UnitClass("player")
	local level = UnitLevel("player")
	local englishFaction, localizedFaction = UnitFactionGroup("player")

	characterInfo.bossKills = WorldBossStatus:GetWorldBossKills()
	characterInfo.lastUpdate = time()
	characterInfo.class = className
	characterInfo.level = level
	characterInfo.faction = englishFaction
	characterInfo.bonusRolls = WorldBossStatus:GetBonusRollsStatus()

	return characterInfo
end

function WorldBossStatus:GetBonusRollCurrency()
	name, amount, texturePath, earnedThisWeek, weeklyMax, totalMax, isDiscovered = GetCurrencyInfo(738)

	return amount
end

function WorldBossStatus:GetWeeklyQuestResetTime()
	local TUESDAY = 2
	local WEDNESDAY = 3
	local THURSDAY = 4
	local regionResets = {["US"] = TUESDAY, ["EU"] = WEDNESDAY, ["CN"] = THURSDAY, ["KR"] = THURSDAY, ["TW"] = THURSDAY};

	local region = GetCVar("realmList"):match("^(%a+)%."):upper()
	local resetWeekDay = regionResets[region]
	local offset = WorldBossStatus:GetRealmOffset()

	local nextReset = time() + GetQuestResetTime()

	if not resetWeekDay then
		return nil
	end

	while tonumber(date("%w", nextReset + offset)) ~= resetWeekDay do
		nextReset = nextReset + (24 * 60 *60)
	end

	return nextReset
end
 
function WorldBossStatus:GetRealmOffset()
       local localDate = date("*t", time())
	   local realmDate = date("*t", time())

	   realmDate.wday, realmDate.month, realmDate.day, realmDate.year = CalendarGetDate();
	   realmDate.hour, realmDate.min = GetGameTime()

	   local localTime = time{year=localDate.year, month=localDate.month, day=localDate.day, hour=localDate.hour}
	   local realmTime = time{year=realmDate.year, month=realmDate.month, day=realmDate.day, hour=realmDate.hour}
	   local offset = realmTime - localTime

       return offset
end

function WorldBossStatus:GetBonusRollsStatus()
	local bonusRolls = {}
	
	bonusRolls.currency = {}
	
	for key,value in pairs(trackableCurrencies) do
		_, balance = GetCurrencyInfo(value.id)
		bonusRolls.currency[value.id] = balance	
	end

	bonusRolls.questCompleted = IsQuestFlaggedCompleted(33134) 
	bonusRolls.questReset = WorldBossStatus:GetWeeklyQuestResetTime()

	return bonusRolls
end

function WorldBossStatus:GetWorldBossKills()
	local bossKills = {}
	local num = GetNumSavedWorldBosses()

	for i = 1, num do
		name, worldBossID, reset = GetSavedWorldBossInfo(i)
		expires = time() + reset

		bossKills[name] = expires
	end

	return bossKills
end

function WorldBossStatus:UPDATE_INSTANCE_INFO()
	WorldBossStatus:SaveCharacterInfo()
	if LibQTip:IsAcquired("WorldBossStatusTooltip") and WorldBossStatus.tooltip then
		WorldBossStatus:ShowToolTip()
	end
end

oldLogout = Logout;
oldQuit = Quit;
function Quit()
	RequestRaidInfo();
	oldQuit();
end
function Logout()
	RequestRaidInfo();
	oldLogout();
end

function WorldBossStatus:OnEnable()
	--CleanupCharacters()
	self:RegisterEvent("UPDATE_INSTANCE_INFO");
end

function WorldBossStatus:OnDisable()
    -- Called when the addon is disabled
end