﻿-- TODO: Other languages

Wishlist = AceLibrary('AceAddon-2.0'):new('AceConsole-2.0', 'AceEvent-2.0', 'AceDB-2.0', 'FuBarPlugin-2.0')
local self = Wishlist
WishlistData = WishlistData or { Factions = {}, Predefined = {} }
local data = WishlistData
local L = AceLibrary('AceLocale-2.2'):new('Wishlist')

-----------------------------
--- AceLibrary References ---
-----------------------------

local dewdrop = AceLibrary('Dewdrop-2.0')
local tablet = AceLibrary('Tablet-2.0')

-- FuBar Options
Wishlist.name = 'Wishlist'
Wishlist.title = 'Wishlist'
Wishlist.hasIcon = 'Interface/PvPRankBadges/PvPRank' .. UnitFactionGroup('player')
Wishlist.clickableTooltip = true

----------------------
--- FuBar Handlers ---
----------------------

function Wishlist:OnInitialize()
	-- Money formatting
	local abacus = LibStub:GetLibrary('LibAbacus-3.0', true)
	if abacus then
		self.FormatMoney = abacus.FormatMoneyCondensed
	else
		self.FormatMoney = self.FormatMoneySimple
	end

	self.ShortNames = {
		['Warsong Gulch Mark of Honor'] = 'WSG Mark(s)',
		['Arathi Basin Mark of Honor'] = 'AB Mark(s)',
		['Alterac Valley Mark of Honor'] = 'AV Mark(s)',
		['Eye of the Storm Mark of Honor'] = 'EotS Mark(s)',
	}

	self.Textures = {
		Honor = 'Interface/PvPRankBadges/PvPRank' .. UnitFactionGroup('player'),
		Money = 'Interface/MoneyFrame/UI-MoneyIcons', -- TODO: Better icon
		ArenaPoints = 'Interface/PVPFrame/PVP-ArenaPoints-Icon',
		['Warsong Gulch Mark of Honor'] = 'Interface/Icons/INV_Misc_Rune_07',
		['Arathi Basin Mark of Honor'] = 'Interface/Icons/INV_Jewelry_Amulet_07',
		['Alterac Valley Mark of Honor'] = 'Interface/Icons/INV_Jewelry_Necklace_21',
		['Eye of the Storm Mark of Honor'] = 'Interface/Icons/Spell_Nature_EyeOfTheStorm',
		['Badge of Justice'] = 'Interface/Icons/Spell_Holy_ChampionsBond',
		['Halaa Battle Token'] = 'Interface/Icons/INV_Misc_Rune_08',
		['Halaa Research Token'] = 'Interface/Icons/INV_Misc_Rune_09',
	}

	self.Factions = { 'Cenarion Expedition', 'Lower City', "The Sha'tar", 'Keepers of Time', 'The Aldor', 'The Scryers', 'Ashtongue Deathsworn', "Sha'tari Skyguard", 'Shattered Sun Offensive' }

	if UnitFactionGroup('player') == 'Horde' then
		-- Battleground Factions
		table.insert(self.Factions, 'Frostwolf Clan')
		table.insert(self.Factions, 'The Defilers')
		table.insert(self.Factions, 'Warsong Outriders')

		-- BC Factions
		table.insert(self.Factions, 'Thrallmar')
		table.insert(self.Factions, "The Mag'har")
	else
		-- Battleground Factions
		table.insert(self.Factions, 'Silverwing Sentinels')
		table.insert(self.Factions, 'Stormpike Guard')
		table.insert(self.Factions, 'The League of Arathor')

		-- BC Factions
		table.insert(self.Factions, 'Honor Hold')
		table.insert(self.Factions, 'Kurenai')
	end

	-- Sort all factions
	table.sort(self.Factions)

	-- AtlasLoot Hook
	local atlasData = AtlasLoot_Data
	if atlasData then
		local pvp = atlasData.AtlasLootGeneralPvPItems
		if not pvp then
			LoadAddOn("AtlasLoot_SetsandPvP")
			pvp = atlasData.AtlasLootGeneralPvPItems
		end

		if pvp then
			self:BuildFromAtlasLoot(pvp)
		end
	end
end

function Wishlist:OnEnable()
	-- Create data variables
	self.Data = {
		Factions = {},
		Honor = GetHonorCurrency(),
		ArenaPoints = GetArenaCurrency(),
		Items = {
			['Warsong Gulch Mark of Honor'] = 0,
			['Arathi Basin Mark of Honor'] = 0,
			['Alterac Valley Mark of Honor'] = 0,
			['Eye of the Storm Mark of Honor'] = 0,
			['Badge of Justice'] = 0,
			['Halaa Battle Token'] = 0,
			['Halaa Research Token'] = 0,
		},
		Money = GetMoney(),
	}

	-- Create database
	self:RegisterDB("WishlistDB", "WishlistCharDB")
	self:RegisterDefaults("profile", {
		Settings = {
			ShowMoneyChanges = false,
			ShowHonorChanges = false,
			ShowItemsChanges = false,
		},
		Text = {
			ShowMoney = true,
		},
	})
	self:RegisterDefaults("char", {
		Items = {},
	})

	self:UpdateItems()

	-- Keep honor up-to-date
	self:RegisterEvent("HONOR_CURRENCY_UPDATE")

	-- Keep reputation up-to-date
	self:RegisterEvent("UPDATE_FACTION")

	-- Keep money up-to-date
	self:RegisterEvent("PLAYER_MONEY")

	self:RegisterEvent("BAG_UPDATE")

	self:UpdateTotals()
	self:Update()

	self:RegisterMenu()
end

function Wishlist:OnDisable()
	self:UnregisterAllEvents()
end

function Wishlist:OnDataUpdate()
	self.Result, self.Failed = self:CheckItem(self.Totals)

	self.ItemResults = {}
	local _, item
	for _, item in ipairs(self.db.char.Items) do
		local result, failed = self:CheckItem(item)
		table.insert(self.ItemResults, { ItemId = item.ItemId, Result = result, Failed = failed, Count = #item.Requirements })
	end
end

function Wishlist:OnTextUpdate()
	local text = ''
	if self.db.profile.Text.ShowMoney then
		text = text .. self:GetMoneyText()
	end
	self:SetText(text)
end

function Wishlist:OnTooltipUpdate()
	local cat
	if self.Result then
		-- Nothing missing, user got all items (or none :P)
		cat = tablet:AddCategory('columns', 1, 'showWithoutChildren', true)
		if #self.db.char.Items == 0 then
			cat:AddLine({ text = L['|cffffffffAdd items first'], justify = 'CENTER' })
		else
			local index, item
			for index, item in ipairs(self.db.char.Items) do
				local name, link = GetItemInfo(item.ItemId)
				cat:AddLine({ text = link, justify = 'CENTER', func = function() self:Print((L['%s, you got everything!']):format(link)) end })
			end
			cat:AddLine({ text = L['|cffffffffGot everything!'], justify = 'CENTER' })
		end
	else
		-- Something is missing, show totals
		cat = tablet:AddCategory('columns', 3, 'showWithoutChildren', true)
		cat:AddLine({
			text = '', justify = 'CENTER',
			text2 = L['Need'], justify2 = 'CENTER',
			text3 = L['Done'],	justify3 = 'CENTER',
		})

		-- Show totals of requirements
		local index, req
		local total = 0
		for index, req in ipairs(self.Failed) do
			local txt, txt2, done = self:RequirementToTooltip(req)
			cat:AddLine({ text = txt, text2 = txt2, text3 = self:PercentageToColoredText(done) })
			total = total + done
		end
		total = total / #self.Failed

		cat:AddLine({ text = L['|cffffffffTotal'], justify = 'CENTER', text2 = '', text3 = self:PercentageToColoredText(total) })

		-- Show percentage for each item
		cat = tablet:AddCategory('text', L['Options Items'], 'columns', 2)
		local index, item
		for index, item in ipairs(self.ItemResults) do
			local name, link = GetItemInfo(item.ItemId)
			local done
			local func
			if item.Result then
				done = self:PercentageToColoredText(1)
				func = function() self:Print((L['%s, you got everything!']):format(link)) end
			else
				local total = 0
				local count = item.Count
				local need = ''
				for index, req in ipairs(item.Failed) do
					local _, _, done = self:RequirementToTooltip(req)
					if strlen(need) > 0 then
					  need = need .. ', '
					end
					need = need .. self:RequirementToNeeded(req)
					total = total + done
				end
				total = (total + count - #item.Failed) / count
				if total == 1 then
				  total = 0.99 -- Just to never show 100%
				end
				done = self:PercentageToColoredText(total)
				func = function() self:Print((L['%s, you need %s.']):format(link, need)) end
			end
			cat:AddLine({ text = link, text2 = done, func = func })
		end
	end
end

------------------------
--- Validate Methods ---
------------------------

function Wishlist:CheckItem(item)
	local result = true
	local failed = {}

	local index, req
	for index, req in ipairs(item.Requirements) do
		if not self:ValidateRequirement(req) then
			table.insert(failed, req)
			result = false
		end
	end

	return result, failed
end

function Wishlist:ValidateRequirement(req)
	if req then
		if req.Type == 'Honor' then
			-- Check if we have enough honor
			return self.Data.Honor >= req.Honor
		elseif req.Type == 'Money' then
			-- Check if we have enough money
			return self.Data.Money >= req.Money
		elseif req.Type == 'Arena' then
			-- Check if we have enough arena points
			return self.Data.ArenaPoints >= req.ArenaPoints
		elseif req.Type == 'Faction' then
			-- Check if we have enough reputation
			return self:GetFactionStanding(req.Faction) >= req.Standing
		elseif req.Type == 'Item' then
			-- Check if we have enough of the needed item
			return self:GetItemCount(req.Name) >= req.Required
		end
	end
	return false
end

----------------------
--- Helper Methods ---
----------------------

function Wishlist:AddItem(item)
	local name, link = GetItemInfo(item)
	if not name then return end
	local _, _, item = string.find(link, "item:(%-?%d+):")
	item = tonumber(item)

	-- Add item to character's watched items with no requirements
	self:AddNewItem({ ItemId = item, Requirements = {} }, link)
end

function Wishlist:AddNewItem(item, link)
	table.insert(self.db.char.Items, item)

	self:UpdateTotals()
	self:Update()
	self:UpdateItemsMenu()

	self:Print((L['Added %s to watch list']):format(link))
end

function Wishlist:AddRequirements(item, requirements)
  -- Combine new requirements with current ones
	self:CombineRequirements(item.Requirements, requirements)

	self:UpdateTotals()
	self:Update()
	self:UpdateItemsMenu()
end

function Wishlist:ColorToHex(color)
	return ("|cff%02x%02x%02x"):format(color.r * 255, color.g * 255, color.b * 255)
end

function Wishlist:CombineRequirements(original, extra)
	local index, req
	for index, req in ipairs(extra) do
		local existing = self:FindMatchingRequirement(original, req)
		if existing then
			if existing.Type == 'Honor' then
				existing.Honor = existing.Honor + req.Honor
			elseif existing.Type == 'Money' then
				existing.Money = existing.Money + req.Money
			elseif existing.Type == 'Arena' then
				existing.ArenaPoints = existing.ArenaPoints + req.ArenaPoints
			elseif existing.Type == 'Item' then
				existing.Required = existing.Required + req.Required
			elseif existing.Type == 'Faction' then
				existing.Standing = max(existing.Standing, req.Standing)
			end
		else
			-- Copy requirement
			local new = { Type = req.Type }
			if req.Type == 'Honor' then
				new.Honor = req.Honor
			elseif req.Type == 'Money' then
				new.Money = req.Money
			elseif req.Type == 'Arena' then
				new.ArenaPoints = req.ArenaPoints
			elseif req.Type == 'Item' then
				new.Name = req.Name
				new.Required = req.Required
			elseif req.Type == 'Faction' then
				new.Faction = req.Faction
				new.Standing = req.Standing
			end
			table.insert(original, new)
		end
	end
end

function Wishlist:FindMatchingRequirement(requirements, match)
	local _, req
	for _, req in ipairs(requirements) do
		if (req.Type == match.Type) and ((req.Type ~= 'Item' and req.Type ~= 'Faction') or (req.Type == 'Item' and req.Name == match.Name) or (req.Type == 'Faction' and req.Faction == match.Faction)) then
			return req
		end
	end
	return nil -- Not found
end

function Wishlist:FormatMoneySimple(value, colorize)
  -- Helper method if Abacus isn't found
	return ('%d.%d.%d'):format(value / 10000, (value / 100) % 100, value % 100)
end

function Wishlist:GetFaction(name)
	return self.Data.Factions[name] or { Standing = 4, EarnedValue = 0 } -- Default Neutral
end

-- 0 - Unknown
-- 1 - Hated
-- 2 - Hostile
-- 3 - Unfriendly
-- 4 - Neutral
-- 5 - Friendly
-- 6 - Honored
-- 7 - Revered
-- 8 - Exalted
function Wishlist:GetFactionStanding(name)
	local faction = self:GetFaction(name)
	return not faction and 0 or faction.Standing
end

function Wishlist:GetItemCount(name)
	return (self.Data.Items[name] or 0)
end

function Wishlist:GetMoneyText()
	return self:FormatMoney(self.Data.Money, true)
end

function Wishlist:GetShortName(txt)
	return self.ShortNames[txt] or txt
end

-- Converts a percentage (0.00-1.00) to (red-green)
function Wishlist:PercentageToColoredText(value)
  local h, s, v = (value / 2.55) * 360, 1, 1

  local sectorPos = h / 60
  local sectorNumber = floor(sectorPos)
  local fractionalSector = sectorPos - sectorNumber

  local p = 0
  local q = 1 - fractionalSector
  local t = 1 - (1 - fractionalSector)

  local r, g, b
  if sectorNumber == 0 then
    r, g, b = v, t, p
  elseif sectorNumber == 1 then
    r, g, b = q, v, p
  elseif sectorNumber == 2 then
    r, g, b = p, v, t
  elseif sectorNumber == 3 then
    r, g, b = p, q, v
  elseif sectorNumber == 4 then
    r, g, b = t, p, v
  else
    r, g, b = v, p, q
  end

  return self:ColorToHex({ r = r, g = g, b = b }) .. ('%d'):format(value * 100) .. '%'
end

function Wishlist:PrintFaction(name)
	local standing = self:GetFactionStanding(name)

	if not standing then
		self:Print((L["Faction %s not found"]):format(name))
	else
		self:Print(name .. ' rep: ' .. self:ColorToHex(FACTION_BAR_COLORS[standing]) .. getglobal('FACTION_STANDING_LABEL' .. standing))
	end
end

function Wishlist:PrintHonor()
	if self.db.profile.Settings.ShowHonorChanges then
		self:Print((L["Honor: %d"]):format(self.Data.Honor))
	end
end

function Wishlist:PrintItems()
	if self.db.profile.Settings.ShowItemsChanges then
		local index, value
		for index, value in pairs(self.Data.Items) do
			self:Print((L["Items"]):format(index, value))
		end
	end
end

function Wishlist:PrintMoney()
	if self.db.profile.Settings.ShowMoneyChanges then
		self:Print((L["Money: %s"]):format(self:GetMoneyText()))
	end
end

function Wishlist:RemoveItem(index, hideMessage)
	local _, link = GetItemInfo(self.db.char.Items[index].ItemId)
	table.remove(self.db.char.Items, index)

	self:UpdateTotals()
	self:Update()
	self:UpdateItemsMenu()

	if not hideMessage then
	  self:Print((L['Removed %s from watch list']):format(link))
	end
end

function Wishlist:RemoveRequirement(item, reqIndex)
	table.remove(item.Requirements, reqIndex)

	self:UpdateTotals()
	self:Update()
	self:UpdateItemsMenu()
end

function Wishlist:RequirementToString(req)
	if req then
		if req.Type == 'Honor' then
			return (L['%d honor']):format(req.Honor), self.Textures.Honor
		elseif req.Type == 'Money' then
			return self:FormatMoney(req.Money, true), self.Textures.Money
		elseif req.Type == 'Arena' then
			return (L['%d Arena Point(s)']):format(req.ArenaPoints), self.Textures.ArenaPoints
		elseif req.Type == 'Faction' then
			return (L['%s|r with %s']):format(self:StandingToText(req.Standing), req.Faction)
		elseif req.Type == 'Item' then
			return (L['%dx %s']):format(req.Required, self:GetShortName(req.Name)), self.Textures[req.Name]
		end
	end
	return ''
end

function Wishlist:RequirementToNeeded(req)
	if req then
		if req.Type == 'Honor' then
			return (L['%d honor']):format(req.Honor - self.Data.Honor)
		elseif req.Type == 'Money' then
			return self:FormatMoney(req.Money - self.Data.Money, true)
		elseif req.Type == 'Arena' then
			return (L['%d Arena Point(s)']):format(req.ArenaPoints - self.Data.ArenaPoints)
		elseif req.Type == 'Faction' then
			return (L['%s|r with %s']):format(self:StandingToText(req.Standing), req.Faction)
		elseif req.Type == 'Item' then
			local count = self:GetItemCount(req.Name)
			return (L['%dx %s']):format(req.Required - count, self:GetShortName(req.Name))
		end
	end
	return ''
end

function Wishlist:RequirementToTooltip(req)
	if req then
		if req.Type == 'Honor' then
			return L['Honor'], ('|cffffffff%d/%d'):format(req.Honor - self.Data.Honor, req.Honor), self.Data.Honor / req.Honor
		elseif req.Type == 'Money' then
			return L['Money'], self:FormatMoney(req.Money - self.Data.Money, true), self.Data.Money / req.Money
		elseif req.Type == 'Arena' then
			return L['Arena Points'], ('|cffffffff%d/%d'):format(req.ArenaPoints - self.Data.ArenaPoints, req.ArenaPoints), self.Data.ArenaPoints / req.ArenaPoints
		elseif req.Type == 'Faction' then
			local earnedValue = self:GetFaction(req.Faction).EarnedValue
			local needed = self:StandingToValue(req.Standing)
			return req.Faction, (L['%s|r |cffffffff(%d)']):format(self:StandingToText(req.Standing), needed - earnedValue), earnedValue / needed
		elseif req.Type == 'Item' then
			local count = self:GetItemCount(req.Name)
			return self:GetShortName(req.Name), ('|cffffffff%d/%d'):format(req.Required - count, req.Required), count / req.Required
		end
	end
	return '', '', 0
end

function Wishlist:StandingToText(standing)
  return self:ColorToHex(FACTION_BAR_COLORS[standing]) .. getglobal('FACTION_STANDING_LABEL' .. standing)
end

function Wishlist:StandingTextToValue(text)
  local s
  for s = 1, 8 do
    if text == self:StandingToText(s) then
      return s
    end
  end
  return 0
end

function Wishlist:StandingToValue(standing)
  if (standing <= 5) then
    return 3000
  elseif (standing == 6) then
    return 9000
  elseif (standing == 7) then
    return 21000
  else
    return 42000
  end
end

function Wishlist:UpdateItems()
	-- Check each item
	local isChanged = false
	local item, value
	for item, value in pairs(self.Data.Items) do
		local count = GetItemCount(item, true)

		if value ~= count then
			self.Data.Items[item] = count

			if self.db.profile.Settings.ShowItemsChanges then
				self:Print((L["Items"]):format(item, count))
			end

			isChanged = true
		end
	end

	-- Check wished items
	local hasItems = {}
	local index
	for index, item in ipairs(self.db.char.Items) do
		if GetItemCount(item.ItemId) > 0 then
			local _, itemLink = GetItemInfo(item.ItemId)
			self:Print((L["You got item %s, congratulations!"]):format(itemLink))

			table.insert(hasItems, 1, index)
			isChanged = true
		end
	end
	while #hasItems > 0 do self:RemoveItem(table.remove(hasItems), true) end

	return isChanged
end

function Wishlist:UpdateTotals()
	-- Reset totals
	self.Totals = { Requirements = {} }

	-- Combine them
	local index, item
	for index, item in ipairs(self.db.char.Items) do
		self:CombineRequirements(self.Totals.Requirements, item.Requirements)
	end
end

----------------------
--- Event Handlers ---
----------------------

function Wishlist:BAG_UPDATE()
	if self:UpdateItems() then
		self:Update()
	end
end

function Wishlist:HONOR_CURRENCY_UPDATE()
	-- Keep honor
	self.Data.Honor = GetHonorCurrency()
	self:PrintHonor()
	self:Update()
end

function Wishlist:PLAYER_MONEY()
	-- Keep money
	self.Data.Money = GetMoney()
	self:PrintMoney()
	self:Update()
end

function Wishlist:UPDATE_FACTION()
	-- Keep faction reputation
	for factionIndex = 1, GetNumFactions() do
		local name, _, standingId, _, _, earnedValue, _, _, isHeader = GetFactionInfo(factionIndex)
		if not isHeader then
			self.Data.Factions[name] = { Standing = standingId, EarnedValue = earnedValue }
		end
	end
	self:Update()
end

-------------------------
--- AtlasLoot Methods ---
-------------------------

function Wishlist:BuildFromAtlasLoot(atlasData)
  local unitClass = UnitClass('player')

  local pvp = {}

  self:ExtractAtlasLootItems(pvp, 'Outland Reputation', atlasData['PVP70Rep' .. unitClass])
  self:ExtractAtlasLootItems(pvp, 'Arena S1 - ' .. unitClass, atlasData['Arena' .. unitClass])
  self:ExtractAtlasLootItems(pvp, 'Arena S1 - Weapons 1', atlasData.Arena1Weapons1)
  self:ExtractAtlasLootItems(pvp, 'Arena S1 - Weapons 2', atlasData.Arena1Weapons2)
  self:ExtractAtlasLootItems(pvp, 'Arena S2 - ' .. unitClass, atlasData['Arena2' .. unitClass])
  self:ExtractAtlasLootItems(pvp, 'Arena S2 - Weapons 1', atlasData.Arena2Weapons1)
  self:ExtractAtlasLootItems(pvp, 'Arena S2 - Weapons 2', atlasData.Arena2Weapons2)
  self:ExtractAtlasLootItems(pvp, 'Arena S3 - ' .. unitClass, atlasData['Arena3' .. unitClass])
  self:ExtractAtlasLootItems(pvp, 'Arena S3 - Weapons 1', atlasData.Arena3Weapons1)
  self:ExtractAtlasLootItems(pvp, 'Arena S3 - Weapons 2', atlasData.Arena3Weapons2)
  self:ExtractAtlasLootItems(pvp, 'Lvl70 Non-Set Epics 1', atlasData.PvP70NonSet1)
  self:ExtractAtlasLootItems(pvp, 'Lvl70 Non-Set Epics 2', atlasData.PvP70NonSet2)
  self:ExtractAtlasLootItems(pvp, 'Lvl70 Non-Set Epics 3', atlasData.PvP70NonSet3)

  local atlasMenu = {
    { type = 'group', name = L['PvP'], desc = L['PvP'], order = 1, args = pvp },
  }

  self.AtlasMenu = { type = 'group', name = L['Import from AtlasLoot'], desc = L['Import from AtlasLoot'], order = 1000, args = atlasMenu }
end

function Wishlist:ExtractAtlasLootItems(menuArgs, name, atlasPart)
  local args = {}

  local index = 1
  local _, part
  for _, part in ipairs(atlasPart) do
    local id, texture = part[1], part[2]
    if id ~= 0 then
	    local name, link = GetItemInfo(id)
	    if name then
	      local requirements = {}
	      self:ExtractAtlasLootRequirements(requirements, part[6], part[7])
	      self:ExtractAtlasLootRequirements(requirements, part[8], part[9])
        -- TODO: Requirements in description
    	  table.insert(args, { type = 'execute', name = link, desc = name, order = index, icon = 'Interface/Icons/' .. texture, func = function() self:AddNewItem({ ItemId = id, Requirements = requirements }, link) end })
    	  index = index + 1
      end
    end
  end

  if #args > 0 then
    table.insert(menuArgs, { type = 'group', name = name, desc = name, order = #menuArgs + 1, args = args })
  end
end

function Wishlist:ExtractAtlasLootRequirements(requirements, req1, req2)
  if req1 and req2 then
    if req2 == '#faction#' then
      table.insert(requirements, { Type = 'Honor', Honor = tonumber(req1) })
    elseif req2 == '#arena#' then
      table.insert(requirements, { Type = 'Arena', ArenaPoints = tonumber(req1) })
    elseif req2 == '#av#' then
      table.insert(requirements, { Type = 'Item', Name = 'Alterac Valley Mark of Honor', Required = tonumber(req1) })
    elseif req2 == '#ab#' then
      table.insert(requirements, { Type = 'Item', Name = 'Arathi Basin Mark of Honor', Required = tonumber(req1) })
    elseif req2 == '#wsg#' then
      table.insert(requirements, { Type = 'Item', Name = 'Warsong Gulch Mark of Honor', Required = tonumber(req1) })
    elseif req2 == '#eos#' then
      table.insert(requirements, { Type = 'Item', Name = 'Eye of the Storm Mark of Honor', Required = tonumber(req1) })
    end
    -- TODO: Reputation...
  end
end

-----------------------
--- Options Methods ---
-----------------------

self.ItemsMenu = {}
self.Options = {
	type = 'group',
	args = {
		Items = { type = 'group', name = L['Options Items'], desc = L['Options Items'], order = 1, args = self.ItemsMenu },
		Settings = {
			type = 'group', name = L['Options Settings'], desc = L['Options Settings'], order = 2, args = {
				Text = {
					type = 'group', name = L['Options Text'], desc = L['Options Text'], order = 1, args = {
						ShowMoney = { type = 'toggle', name = L['Show Money'], desc = L['Show Money'], get = 'IsShowingMoney', set = 'ToggleShowMoney' },
					},
				},
				ShowMoneyChanges = { type = 'toggle', name = L['Show Money Changes'], desc = L['Show Money Changes'], get = 'IsShowingMoneyChanges', set = 'ToggleShowMoneyChanges' },
				ShowHonorChanges = { type = 'toggle', name = L['Show Honor Changes'], desc = L['Show Honor Changes'], get = 'IsShowingHonorChanges', set = 'ToggleShowHonorChanges' },
				ShowItemsChanges = { type = 'toggle', name = L['Show Items Changes'], desc = L['Show Items Changes'], get = 'IsShowingItemsChanges', set = 'ToggleShowItemsChanges' },
			},
		},
	}
}

function Wishlist:RegisterMenu()
	self:UpdateItemsMenu()

	self:RegisterChatCommand({'/wishlist', '/wl'}, self.Options)
	self.OnMenuRequest = self.Options
end

function Wishlist:BuildCustom(item)
	local items = {}
	local name, _
	local index = 1
	for name, _ in pairs(self.Data.Items) do
		local shortName = self:GetShortName(name)
		table.insert(items, {
			type = 'text',
			name = shortName,
			desc = name,
			usage = name,
			icon = self.Textures[name],
			order = index,
			get = false,
			set = function(v) self:AddRequirements(item, { { Type = 'Item', Name = name, Required = tonumber(v) } }) end,
		})

		index = index + 1
	end

	local factions = {}
	local factionValidate = { [0] = self:StandingToText(5), [1] = self:StandingToText(6), [2] = self:StandingToText(7), [3] = self:StandingToText(8) }
	for index, name in ipairs(self.Factions) do
		table.insert(factions, {
			type = 'text',
			name = name,
			desc = name,
			usage = name,
			validate = factionValidate,
			order = index,
			get = false,
			set = function(v) self:AddRequirements(item, { { Type = 'Faction', Faction = name, Standing = self:StandingTextToValue(v) } }) end,
		})
	end

	return {
		{
			type = 'text',
			name = L['Honor'],
			desc = L['Honor'],
			usage = L['Honor'],
			order = 1,
			icon = self.Textures.Honor,
			get = false,
			set = function(v) self:AddRequirements(item, { { Type = 'Honor', Honor = tonumber(v) } }) end,
		},
		{
			type = 'text',
			name = L['Arena Points'],
			desc = L['Arena Points'],
			usage = L['Arena Points'],
			order = 2,
			icon = self.Textures.ArenaPoints,
			get = false,
			set = function(v) self:AddRequirements(item, { { Type = 'Arena', ArenaPoints = tonumber(v) } }) end,
		},
		{
			type = 'text',
			name = L['Money (copper)'],
			desc = L['Money (copper)'],
			usage = L['Money (copper)'],
			order = 3,
			icon = self.Textures.Money,
			get = false,
			set = function(v) self:AddRequirements(item, { { Type = 'Money', Money = tonumber(v) } }) end,
		},
		{
			type = 'group',
			name = L['Options Items'],
			desc = L['Options Items'],
			order = 4,
			args = items,
		},
		{
			type = 'group',
			name = L['Factions'],
			desc = L['Factions'],
			order = 5,
			args = factions,
		},
	}
end

function Wishlist:BuildPredefined(item, list)
	list = list or data.Predefined

	local result = {}
	local index, predefined
	for index, predefined in ipairs(list) do
		local args = predefined.Args
		if args then
			table.insert(result, { type = 'group', name = predefined.Name, desc = predefined.Name, order = index, icon = predefined.Icon, args = self:BuildPredefined(item, args) })
		else
			table.insert(result, { type = 'execute', name = predefined.Name, desc = predefined.Name, order = index, icon = predefined.Icon, func = function() self:AddRequirements(item, predefined.Requirements) end })
		end
	end

	return result
end

function Wishlist:UpdateItemsMenu()
  -- Remove everything
	while #self.ItemsMenu > 0 do table.remove(self.ItemsMenu) end

	local index, item
	for index, item in ipairs(self.db.char.Items) do
		local itemName, _, _, _, _, _, _, _, _, itemTexture = GetItemInfo(item.ItemId)

		local itemMenu = {}
		local reqIndex, req
		for reqIndex, req in ipairs(item.Requirements) do
			local reqString, icon = self:RequirementToString(req)
			table.insert(itemMenu, {
				type = 'group',
				name = reqString,
				desc = reqString,
				icon = icon,
				order = reqIndex,
				args = {
					RemoveRequirement = {
						type = 'execute',
						name = L['Remove'],
						desc = L['Remove'],
						order = 2,
						func = function() self:RemoveRequirement(item, reqIndex) end,
					},
				},
			})
		end

		table.insert(itemMenu, {
			type = 'group',
			name = L['Add Custom'],
			desc = L['Add Custom'],
			order = 997,
			args = self:BuildCustom(item),
		})
		table.insert(itemMenu, {
			type = 'group',
			name = L['Add Predefined'],
			desc = L['Add Predefined'],
			order = 998,
			args = self:BuildPredefined(item),
		})
		table.insert(itemMenu, {
			type = 'execute',
			name = L['Remove Item'],
			desc = L['Remove Item'],
			order = 999,
			func = function() self:RemoveItem(index) end,
		})
		table.insert(self.ItemsMenu, { type = 'group', name = itemName, desc = itemName, order = index, icon = itemTexture, args = itemMenu })
	end

	-- Add New Item Menu
	self.ItemsMenu.Add = { type = 'text', name = L['Add Item'], desc = L['Add Item'], usage = L['Add Item'], get = false, set = 'AddItem', order = 999 }

	-- AtlasLoot hook
	if self.AtlasMenu then
	  table.insert(self.ItemsMenu, self.AtlasMenu)
	end
end

function Wishlist:IsShowingHonorChanges()
	return self.db.profile.Settings.ShowHonorChanges
end

function Wishlist:IsShowingItemsChanges()
	return self.db.profile.Settings.ShowItemsChanges
end

function Wishlist:IsShowingMoney()
	return self.db.profile.Text.ShowMoney
end

function Wishlist:IsShowingMoneyChanges()
	return self.db.profile.Settings.ShowMoneyChanges
end

function Wishlist:ToggleShowHonorChanges()
	self.db.profile.Settings.ShowHonorChanges = not self.db.profile.Settings.ShowHonorChanges
	self:PrintHonor()
	return self.db.profile.Settings.ShowHonorChanges
end

function Wishlist:ToggleShowItemsChanges()
	self.db.profile.Settings.ShowItemsChanges = not self.db.profile.Settings.ShowItemsChanges
	self:PrintItems()
	return self.db.profile.Settings.ShowItemsChanges
end
function Wishlist:ToggleShowMoney()
	self.db.profile.Text.ShowMoney = not self.db.profile.Text.ShowMoney
	self:OnTextUpdate()
	return self.db.profile.Text.ShowMoney
end

function Wishlist:ToggleShowMoneyChanges()
	self.db.profile.Settings.ShowMoneyChanges = not self.db.profile.Settings.ShowMoneyChanges
	self:PrintMoney()
	return self.db.profile.Settings.ShowMoneyChanges
end