local ADDONNAME, WIC = ...

LibStub("AceAddon-3.0"):NewAddon(WIC, ADDONNAME, "AceConsole-3.0", "AceEvent-3.0", "AceHook-3.0")

local L = LibStub("AceLocale-3.0"):GetLocale(ADDONNAME)

-- GLOBALS: LibStub, GAME_LOCALE
local _G = _G
local wipe = wipe
local type, select, pairs = type, select, pairs


WIC.RegisteredModules = {}
WIC.RegisteredModulesWithLoadOnDemand = {}
WIC.RegisteredModulesDescription = {}
WIC.RegisteredModulesTranslatedName = {}
WIC.RegisteredModulesList = L["No Modules Registered"]

function WIC:RegisterModule(name, handle, addonName)
    self.RegisteredModules[name] = handle
    if type(addonName) == "string" then
        self.RegisteredModulesWithLoadOnDemand[name] = addonName
    end
end
function WIC:AddTranslatedNameToModule(name, localeName)
    self.RegisteredModulesTranslatedName[name] = localeName
end
function WIC:AddDescriptionToModule(name, description)
    self.RegisteredModulesDescription[name] = description
end


do-- Adding modules
    WIC:RegisterModule("Basic", "WhisperInviteBasic", "WhisperInviteBasic")
    WIC:AddTranslatedNameToModule("Basic", L["Basic"])
    WIC:AddDescriptionToModule("Basic", L["Invite player when they whisper you with a defined keyword."])
    
    WIC:RegisterModule("Advanced", "WhisperInviteAdvanced", "WhisperInviteAdvanced")
    WIC:AddTranslatedNameToModule("Advanced" ,L["Advanced"])
    WIC:AddDescriptionToModule("Advanced" ,L["Invite player when they whisper you with a defined keyword where they are allowed to use."])
    

    -- Search for third-party Modules
    -- Add this to your TOC-File:
    --[[
    RequiredDeps: WhisperInvite
    X-WisperInvite-Name: ModuleName
    X-WisperInvite-Handle: AddonName -> used as in LibStub("AceAddon-3.0"):GetAddon(AddonName, true) or Global Variable Name
    X-WisperInvite-Version: Version -> See MODULEVERSION
    X-WisperInvite-Description: Small Description

    LoadOnDemand: 1 -> for when you only want be loaded when needed
    --]]
    --Name and Description supports localization e.g: X-WisperInvite-Description-deDE
    
    -- Needed functions when not a AceAddon:
    --[[
        Module:Enable()
        Module:Disable()
    --]]
    
    local MODULEVERSION = 0
    local locale = GAME_LOCALE or _G.GetLocale()

    for index=1, GetNumAddOns() do
        local addonName, title, notes, enabled, loadable, reason, security = _G.GetAddOnInfo(index)
        
        local version = GetAddOnMetadata(addonName, "X-WisperInvite-Version") 
        if version and tonumber(version) == MODULEVERSION and loadable then
            local name = GetAddOnMetadata(addonName, "X-WisperInvite-Name")
            local localeName = GetAddOnMetadata(addonName, "X-WisperInvite-Name-"..locale) or name
            local description = GetAddOnMetadata(addonName, "X-WisperInvite-Description-"..locale) or GetAddOnMetadata(addonName, "X-WisperInvite-Description") or L["No description for this module."]
            local handle = GetAddOnMetadata(addonName, "X-WisperInvite-Handle")

            if name and handle then
                WIC:RegisterModule(name, handle, addonName)
                WIC:AddTranslatedNameToModule(name, name)
            end

            if name and localeName then
                WIC:AddTranslatedNameToModule(name, localeName)
            end            
            if name and description then
                WIC:AddDescriptionToModule(name, description)
            else
                WIC:AddDescriptionToModule(name, L["No description for this module."])
            end
        end
    end

    local list = {}
    for moduleName in pairs(WIC.RegisteredModules) do
        local name = moduleName--(WIC.RegisteredModulesTranslatedName[name] or name).."<"..name..">"
        
        table.insert(list, name)
    end
    WIC.RegisteredModulesList = (", "):join(unpack(list) )
end

local defaults = {
    global = {},
    char = {
        info = {
            ['*'] = 0,
        },
        infoMax = {
            ['*'] = 3,
            setup = 5,
        },
    },
    profile = {
        active = true,
        selectedModuleName = false,
        convertGroupToRaid = true,
        inviteThrottleTime = 10, -- seconds
    },
}

WIC.TYPES = {
    LINK = "whisperinvite",
}

WIC.LOOKUP_LOCALIZED_CLASS_NAMES = {}

function WIC:OnInitialize()
    self.db = LibStub("AceDB-3.0"):New("WhisperInviteCoreDB", defaults, true)

    self:RegisterConfig()

    -- To enable wisperinvite when disabled
    self:RegisterChatCommand("wisperinviteenable", "Enable", true)
    self:RegisterChatCommand("wienable", "Enable", true)

    self:RegisterChatCommand("wi", "CMD", false)
    self:RegisterChatCommand("wisperinvite", "CMD", false)
    
    -- build localized lookup table for class names
    for class, classLocalized in pairs(_G.LOCALIZED_CLASS_NAMES_MALE) do
        self.LOOKUP_LOCALIZED_CLASS_NAMES[classLocalized] = class
    end
    for class, classLocalized in pairs(_G.LOCALIZED_CLASS_NAMES_FEMALE) do
        self.LOOKUP_LOCALIZED_CLASS_NAMES[classLocalized] = class
    end

    self:SetEnabledState(self.db.profile.active)
    if not self:IsEnabled() and self.db.char.info.enable < self.db.char.infoMax.enable then
        self.db.char.info.enable = self.db.char.info.enable + 1
        self:Print(L["You can run /wienable to enable WisperInvite."])
    end    
end

function WIC:OnEnable()
    if type(self.db.profile.selectedModuleName) == "string" and self:EnableSelectedModule() then
        -- Module is registered and enabled
    else
        if self.db.char.info.setup < self.db.char.infoMax.setup then
            self.db.char.info.setup = self.db.char.info.setup + 1
            self:Print(L["Run /wi modules or /wi options to setup WisperInvite."])
        end
    end

    -- Hooking 
    self:RawHook("ChatFrame_OnHyperlinkShow")
    --ChatFrame_OnHyperlinkShow(self, link, text, button)
end

function WIC:ChatFrame_OnHyperlinkShow(frame, linkData, link, button, ...)
    if (":"):split(linkData) == self.TYPES.LINK then
        local toonID, toonName, presenceName = select(2, (":"):split(linkData) )
        --[===[@debug@
        if select(5, (":"):split(linkData) ) == "DEBUG" then
            self:Printf("Invite: %s(%s), id:%s", toonName, presenceName, toonID)
            return
        end
        --@end-debug@]===]

        local result, code = self:InviteToon(toonID, toonName, presenceName)
        
        --[===[@debug@
        self:Printf("Invite Sent: %s", result and "OK" or "Error: "..code )
        --@end-debug@]===]
    else
        self.hooks.ChatFrame_OnHyperlinkShow(frame ,linkData, link, button, ...)
    end
end

function WIC:OnDisable()
    self:DisableSelectedModule()
end


function WIC:CMD(input, editBox)
    --GetArgs(str, numargs, startpos)
    local arg1, lastPos = self:GetArgs(input, 1)
    arg1 = arg1 and arg1:lower() or ""
    if arg1 == "modules" then
        local moduleName = self:GetArgs(input, 1, lastPos)
        if self.RegisteredModules[moduleName] then
            self:SelectModule(moduleName)
        else
            self:Printf(L["Usage: /wi modules moduleName (case sensitive)"])
            self:Printf(L["Modules: %s"], self.RegisteredModulesList )
        end
    elseif not arg1 or arg1 == "" or arg1 == "help" or arg1 == "usage" or arg1 == L["help"] or arg1 == L["usage"] then
        self:Print(L["Usage: /wi <command>\nCommands: modules, options"])
    elseif arg1 == "options" or arg1 == "op" then
        -- Load InterfaceOptionsFrameAddOns frame...
        _G.InterfaceOptionsFrame_OpenToCategory(ADDONNAME)

        -- Let's scroll down, so your page can be open
        _G.InterfaceOptionsFrameAddOnsListScrollBar:SetValue(select(2, _G.InterfaceOptionsFrameAddOnsListScrollBar:GetMinMaxValues() ) )
        _G.InterfaceOptionsFrame_OpenToCategory(ADDONNAME)
    else
        LibStub("AceConfigCmd-3.0").HandleCommand(self, "wi", ADDONNAME.."Options", input)
    end
end

function WIC:SelectModule(moduleName)
    if self.RegisteredModules[moduleName] then
        self.db.profile.selectedModuleName = moduleName
                
        return self:EnableSelectedModule()
    end
    return false
end

do
    local oldModule
    function WIC:EnableSelectedModule()
        local name = self.db.profile.selectedModuleName
        local addonName = self.RegisteredModulesWithLoadOnDemand[name]

        if name and addonName then        
            if not _G.IsAddOnLoaded(addonName) then
                local loaded, resason = _G.LoadAddOn(self.RegisteredModulesWithLoadOnDemand[name])
                if not loaded then
                    self:Printf(L["Can't load module %s because %s"], name, _G["ADDON_"..resason] or resason)

                    return false, resason
                end
            end


            local handle = self.RegisteredModules[name]
            local selectModule = LibStub("AceAddon-3.0"):GetAddon(handle, true) or _G[handle]
            if selectModule and type(selectModule.Enable) == "function" then
                if oldModule and type(oldModule.Disable) == "function" then
                    oldModule:Disable()
                end

                oldModule = selectModule

                return selectModule:Enable()
            else
                self:Print(not selectModule and L["Module not found."] or L["Module hasn't needed functions."] )
                return false
            end
        else
            self:Printf(L["Can't load module. %s"], (not name and L["No module selected!"] or not addonName and L["Addon name not found!"] or "") )
            return false
        end
    end
end

function WIC:DisableSelectedModule()
    local name = self.db.profile.selectedModuleName
    if name then
        local selectModule = LibStub("AceAddon-3.0"):GetAddon(name, true) or _G[name]
        if selectModule and type(selectModule.Disable) == "function" then
            selectModule:Disable()
        end
    end
end


function WIC:CanInvite()
    if _G.IsInGroup(_G.LE_PARTY_CATEGORY_INSTANCE) and select(3, _G.GetInstanceInfo() ) ~= 14 then
        -- 14 is flex and you can invite in flex.
        -- we can not invite when in a instance group or can we?
        return false
    else
        return _G.UnitIsGroupLeader("player") or _G.UnitIsGroupAssistant("player") or not _G.IsInGroup()
    end
end

function WIC:CheckGroupSize()
    if _G.GetNumGroupMembers(_G.LE_PARTY_CATEGORY_HOME) == 4 then
        if self.db.profile.convertGroupToRaid then
            _G.ConvertToRaid()
            return true
        else
            return false
        end
    end

    return true
end


--- InviteErrorCodeList
-- 1: Player is not allowed to invite other players
-- 2: No BNet/RealID friend is online with this presenceID
-- 3: Invalid parameters
-- 4: Maximal group size reached
-- 5: No WoW toon is online 
-- 6: Invite for this key was throttled

local playerName_Throttle = {}
local battleNet_Throttle = {}

--- Invite player with name
-- @param (String) name
-- @return1 (Boolean) true == Invite sent, false == Invite not possible
-- @return2 (Number) errorCode @see InviteErrorCodeList
function WIC:InvitePlayer(name)
    if not name then
        return false, 3
    end

    if self:CanInvite() then
        if self:CheckGroupSize() then
            local time = _G.GetTime()
            if (playerName_Throttle[name] or 0) < time then
                playerName_Throttle[name] = time + self.db.profile.inviteThrottleTime
                _G.InviteUnit( self:UniformPlayerName(name) )
                return true
            else
                return false, 6
            end            
        else
            self:Printf(L["Maximal group size reached. Can't invite %s"], name)
            return false, 4
        end
    else
        return false, 1
    end
end



do
    local toonCache = {}
    --- Invite player over Battle.net
    -- @paramsig presenceID [, friendIndex]
    -- @paramsig nil, friendIndex
    -- @param (presenceID) presenceID
    -- @param (Number) friendIndex
    -- @return1 (Boolean) true: Invite sent, false: Invite not possible
    -- @return2 (Number) errorCode @see InviteErrorCodeList
    --[===[@debug@
    function WIC:InviteBNet(presenceID, friendIndex, debug)
    --@end-debug@]===]
    --@non-debug@
    function WIC:InviteBNet(presenceID, friendIndex)
    --@end-non-debug@
        if not self:CanInvite() then
            return false, 1
        end        
        
        if type(friendIndex) ~= "number" and presenceID then
            friendIndex = _G.BNGetFriendIndex(presenceID)
            if not friendIndex then
                return false, 2
            end
        elseif type(friendIndex) ~= "number" then
            return false, 3
        end
        
        local numToons = _G.BNGetNumFriendToons(friendIndex)
        local toonCount = 0
        for toonIndex=1, numToons do 
            local hasFocus, toonName, client, realmName, realmID, faction, race, classLocalized, guild, zoneName, level, gameText, broadcastText, broadcastTime, canSoR, toonID = _G.BNGetFriendToonInfo(friendIndex, toonIndex)
            if client == _G.BNET_CLIENT_WOW then
                toonCount = toonCount + 1
                toonCache[toonCount] = toonCache[toonCount] or {}
                
                toonCache[toonCount].class = self.LOOKUP_LOCALIZED_CLASS_NAMES[classLocalized] or "WARRIOR"
                toonCache[toonCount].toonID = toonID
                toonCache[toonCount].faction = faction
                toonCache[toonCount].toonName = toonName
                toonCache[toonCount].realmName = realmName
                toonCache[toonCount].presenceName = select(2, _G.BNGetFriendInfo(friendIndex) )

                break
            end
        end

        --[===[@debug@
        if debug then
            toonCount = 3

            toonCache[1] = toonCache[1] or {}
            toonCache[1].class = "DRUID"
            toonCache[1].toonID = 0
            toonCache[1].faction = "Alliance"
            toonCache[1].toonName = "Areko"
            toonCache[1].realmName = "Alleria"
            toonCache[1].presenceName = "Areko#0000"

            toonCache[2] = toonCache[2] or {}
            toonCache[2].class = "HUNTER"
            toonCache[2].toonID = 0
            toonCache[2].faction = "Alliance"
            toonCache[2].toonName = "Tânuri"
            toonCache[2].realmName = "Alleria"
            toonCache[2].presenceName = "Tanuri#0000"

            toonCache[3] = toonCache[3] or {}
            toonCache[3].class = "MAGE"
            toonCache[3].toonID = 0
            toonCache[3].faction = "Horde"
            toonCache[3].toonName = "Keks"
            toonCache[3].realmName = "Alleria"
            toonCache[3].presenceName = "Keks#0000"
        end
        --@end-debug@]===]

        if toonCount > 0 then
            if toonCount == 1 then
                return self:InviteToon(toonCache[1].toonID, toonCache[1].toonName, toonCache[1].presenceName)
            else
                -- \124 > | 
                local linkTemplate = "[|H%s:%s|h%s|h]"
                local colourTemplate = "|c%s%s|r"
                local alliance
                local horde

                for i=1,toonCount do
                    local linkData = toonCache[i].toonID..":"..toonCache[i].toonName..":"..toonCache[i].presenceName                    
                    local name = self:UniformPlayerName(toonCache[i].toonName, toonCache[i].realmName)
                    local colour = (_G.CUSTOM_CLASS_COLORS or _G.RAID_CLASS_COLORS)[toonCache[i].class].colorStr or "FFFFFFFF"
                    --[===[@debug@
                    if debug then
                        linkData = linkData..":DEBUG"
                    end
                    --@end-debug@]===]

                    local msg = colourTemplate:format(colour, linkTemplate:format(self.TYPES.LINK, linkData, name) )

                    if toonCache[i].faction == "Horde" then
                        horde = horde and horde..", "..msg or msg
                    elseif toonCache[i].faction == "Alliance" then
                        alliance = alliance and alliance..", "..msg or msg
                    else
                        horde = horde and horde..", "..msg or msg
                        alliance = alliance and alliance..", "..msg or msg
                    end
                end

                local factionGroup = _G.UnitFactionGroup("player")
                self:Printf(L["%s is with more then one toon online. Choose which toons should be invited. Click on the name to invite."], toonCache[1].presenceName )
                
                if factionGroup == "Alliance" then
                    if alliance then
                        self:Printf(L["Alliance toons: %s"], alliance )
                    end
                    if horde then
                        self:Printf(L["Horde toons: %s"], horde )
                    end
                else
                    if horde then
                        self:Printf(L["Horde toons: %s"], horde )
                    end
                    if alliance then
                        self:Printf(L["Alliance toons: %s"], alliance )
                    end
                end
                
                return true, toonCount
            end
        else
            local presenceName = select(2, _G.BNGetFriendInfo(friendIndex) )
            self:Printf(L["%s is not online in World of Warcraft."], presenceName)
            return false, 5
        end
    end
end

function WIC:InviteToon(toonID, toonName, presenceName)
    if not toonID then
        return false, 3
    end

    if self:CheckGroupSize() then
        local time = _G.GetTime()
        if (battleNet_Throttle[toonID] or 0) < time then
            battleNet_Throttle[toonID] = time + self.db.profile.inviteThrottleTime
            _G.BNInviteFriend( toonID )
            return true
        else
            return false, 6
        end
    else
        self:Printf(L["Maximal group size reached. Can't invite %s"], (L["%s (%s)"]):format(presenceName or L["<No name given>"], toonName or L["<No toon name given>"]) )
        return false, 4
    end
end

--- Returns uniformed player or guild name
-- @paramsig playerName [, realmOverride]
-- @param (String) playerName/guildName
-- @param (String, optional) realmOverride
--
-- @return "Name-Realm", "Name", "Realm"
do
    local homeRealm = _G.GetRealmName()
    local dash = "-"
    function WIC:UniformPlayerName(playerName, realmOverride)
        local name = playerName
        local realm = homeRealm
        
        if playerName:find(dash) then
            name, realm = dash:split(playerName, 2)
        end
        
        realm = realmOverride or realm
        
        return name..dash..realm, name, realm
    end

    function WIC:UniformGuildName(guildName, realmOverride)
        -- same as player
        return self:UniformPlayerName(guildName, realmOverride)
    end
end

local optionHandler = {}
function optionHandler:Set(info, value, ...)
    local name = info[#info]
    WIC.db.profile[name] = value
end

function optionHandler:Get(info, value, ...)
    local name = info[#info]
    return WIC.db.profile[name]
end

function optionHandler:SetModule(info, value, ...)
    self:Set(info, value, ...)
    WIC:EnableSelectedModule()
end

local options = {
    type = "group", 
    name = L["WisperInvite Core Settings"],
    handler = optionHandler,
    set = "Set",
    get = "Get",
    args = {
        active = {
            type = "toggle",
            name = L["Active"],
            desc = L["Is this profile enabled."],
            width = "full",
            order = 1,
        },
        selectedModuleName = {
            type = "select",
            name = L["Active module"],
            desc = L["Choose active module."],
            values = WIC.RegisteredModulesTranslatedName,
            set = "SetModule",
            order = 2,
        },
        convertGroupToRaid = {
            type = "toggle",
            name = L["Auto convert"],
            desc = L["Convert group to raid when group has reached maximal size."],
            width = "full",
            order = 3,
        },
        inviteThrottleTime = {
            type = "range",
            name = L["Invite Throttle"],
            desc = L["Delay(seconds) needed between to invites of the same player before, WhisperInvite can send a new invite."],            
            min = 0.5,
            max = 120,
            softMin = 6,
            softMax = 30,
            bigStep = 1,
            step = 0.1,
            --width = "full",
            order = 4,
        },
    },
}

local configIsRegistered = false
local unregisteredConfigs = {}
function WIC:RegisterConfig()
    LibStub("AceConfigRegistry-3.0"):RegisterOptionsTable(ADDONNAME.."Options", options)
    LibStub("AceConfigDialog-3.0"):AddToBlizOptions(ADDONNAME.."Options", ADDONNAME)

    LibStub("AceConfigRegistry-3.0"):RegisterOptionsTable(ADDONNAME.."Profile", LibStub("AceDBOptions-3.0"):GetOptionsTable(self.db) )
    LibStub("AceConfigDialog-3.0"):AddToBlizOptions(ADDONNAME.."Profile", L["Profile"], ADDONNAME)
    
    configIsRegistered = true
    for appName, name in pairs(unregisteredConfigs) do
        self:AddModuleConfig(appName, name)
    end
    wipe(unregisteredConfigs)
end

--- Use to register your DB as namespace of the core DB
-- @param (String) name
-- @param (table) defaults
function WIC:RegisterNamespace(name, defaults)
    if type(defaults) ~= "table" then
        defaults = nil
    end
    return self.db:RegisterNamespace(name, defaults)
end

--- Add your config to BlizzOptions as child of WhisperInvite
-- @param (String) appName - same as used in AceConfigRegistry:RegisterOptionsTable()
-- @param (String) name - Display name
function WIC:AddModuleConfig(appName, name)
    if configIsRegistered then
        if appName and LibStub("AceConfigRegistry-3.0"):GetOptionsTable(appName) then
            LibStub("AceConfigDialog-3.0"):AddToBlizOptions(appName, name or appName, ADDONNAME)
            return true
        else
            return false
        end
    else
        unregisteredConfigs[appName] = name or appName
    end
end
