local UnderHood = UnderHood
local Module = UnderHood:GetModule("Texts")
local L = LibStub("AceLocale-3.0"):GetLocale("UnderHood_Texts")
local DogTag = LibStub("LibDogTag-3.0")
local LSM = LibStub("LibSharedMedia-3.0")

local playerClass = UnitClass("player")
local pairs = pairs

local TextFrame = UnderHood:RegisterFrameClass("Text", L["Text"], "SecureUnitFrame")

function TextFrame:init()
	super(self)

	self.background = self.frame:CreateTexture(nil, "BACKGROUND")
	self.background:SetTexture(0, 0, 1, 0.15)
	self.background:SetAllPoints(self.frame)
	self.background:Hide()

	self.text = self.frame:CreateFontString()
	self.text:SetAllPoints(self.frame)
	self.text:SetShadowColor(0, 0, 0, 1)
	self.text:SetShadowOffset(1, -1)
end

function TextFrame:OnAcquire(settings)
	super(self, settings)

	self.frame:SetScript("OnShow", function() self:OnShow() end)

	self.text:SetJustifyH(settings.hAlign or "LEFT")
	self.text:SetJustifyV(settings.vAlign or "MIDDLE")

	self:UpdateText()

	if not UnderHood.UnitSendEvents[self:GetUnit()] then
		UnderHood:RegisterUpdateTarget(self)
	end

	if UnderHood.configMode then
		self.background:Show()
	end
end

function TextFrame:OnRelease()
	UnderHood:UnregisterUpdateTarget(self)
	DogTag:RemoveFontString(self.text)

	self.background:Hide()
	self.frame:SetScript("OnShow", nil)

	super(self)
end

function TextFrame:UpdateSettings(settings)
	settings = super(self, settings)

	if settings.category == "custom" then
		settings.category = "none"
	end

	return settings
end

function TextFrame:GetDefaultSettings()
	local settings = super(self)

	settings.category = "name"
	settings.style = L["Long"]
	settings.text = "[Level] [Name:ClassColor] [(AFK or DND):Angle]"
	settings.fontName = "Arrial Narrow"
	settings.fontSize = 14

	return settings
end

function TextFrame:UnitChanged(unit)
	super(self, unit)

	UnderHood:UnregisterUpdateTarget(self)

	self:UpdateText()

	if not UnderHood.UnitSendEvents[unit] then
		UnderHood:RegisterUpdateTarget(self)
	end
end

function TextFrame:ConfigurationModeChanged(mode)
	super(self, mode)

	if mode then
		self.background:Show()
	else
		self.background:Hide()
	end

	self:UpdateText()
end

function TextFrame:UpdateText()
	self.text:SetFont(LSM:Fetch("font", self.settings.fontName), self.settings.fontSize)

	local kwargs = { unit = self:GetUnit() }

	DogTag:AddFontString(self.text, self.frame, self.settings.text, "Items;Unit", kwargs)
end

function TextFrame:ProcessUpdate()
	if self.frame:IsVisible() and self.frame:GetAlpha() > 0 then
		DogTag:UpdateFontString(self.text)
	end
end

function TextFrame:OnShow()
	if UnderHood.UnitSendEvents[self:GetUnit()] then
		DogTag:UpdateFontString(self.text)
	end
end

local textStyles = {
	class = {
		[L["Standard"]] = "[Classification] [Level:DifficultyColor] [(if (IsPlayer or (IsEnemy and not IsPet)) then Class):ClassColor] [DruidForm:Paren] [SmartRace]",
		[L["Player Classes Only"]] = "[Classification] [Level:DifficultyColor] [(if IsPlayer then Class):ClassColor] [DruidForm:Paren] [SmartRace]",
		[L["Short"]] = "[(Level (if Classification then '+')):DifficultyColor] [SmartRace]",
	},
	health = {
		[L["Absolute"]] = "[Status or FractionalHP(known=true) or PercentHP:Percent]",
		[L["Absolute Short"]] = "[Status or FractionalHP(known=true):Short or PercentHP:Percent]",
		[L["Difference"]] = "[Status or -MissingHP:Hide(0)]",
		[L["Percent"]] = "[Status or PercentHP:Percent]",
		[L["Mini"]] = "[HP:VeryShort]",
		[L["Smart"]] = "[Status or (if IsFriend then MissingHP:Hide(0):Short:Color('ff7f7f') else FractionalHP(known=true):Short or PercentHP:Percent)]",
		[L["Absolute and Percent"]] = "[Status or (FractionalHP:Short ' || ' PercentHP:Percent)]",
		[L["Informational"]] = "[Status or (Concatenate((if IsFriend then MissingHP:Hide(0):Short:Color('ff7f7f')), ' || ') FractionalHP:Short ' || ' PercentHP:Percent)]",
	},
	name = {
		[L["Standard"]] = "[Name] [(AFK or DND):Angle]",
		[L["Hostility Colored"]] = "[Name:HostileColor] [(AFK or DND):Angle]",
		[L["Class Colored"]] = "[Name:ClassColor] [(AFK or DND):Angle]",
		[L["Long"]] = "[Level] [Name:ClassColor] [(AFK or DND):Angle]",
		[L["Long w/ Druid form"]] = "[Level] [Name:ClassColor] [DruidForm:Paren] [(AFK or DND):Angle]",
	},
	power = {
		[L["Absolute"]] = "[if HasMP then FractionalMP]",
		[L["Absolute Short"]] = "[if HasMP then FractionalMP:Short]",
		[L["Difference"]] = "[-MissingMP]",
		[L["Percent"]] = "[PercentMP:Percent]",
		[L["Mini"]] = "[if HasMP then CurMP:VeryShort]",
		[L["Smart"]] = "[MissingMP:Hide(0):Short:Color('7f7fff')]",
	},
	druidMana = playerClass == "DRUID" and {
		[L["Absolute"]] = "[if not IsMana then FractionalDruidMP]",
		[L["Absolute Short"]] = "[if not IsMana then FractionalDruidMP:Short]",
		[L["Difference"]] = "[if not IsMana then -MissingDruidMP]",
		[L["Percent"]] = "[if not IsMana then PercentDruidMP:Percent]",
		[L["Mini"]] = "[if not IsMana then DruidMP:VeryShort]",
		[L["Smart"]] = "[if not IsMana then MissingDruidMP:Hide(0):Short:Color('7f7fff')]",
	} or nil,
--	holyPower = playerClass == "PALADIN" and {
--		[L["Absolute"]] = "[HolyPower]",
--		[L["Absolute Short"]] = "[HolyPower:Short]",
--		[L["Difference"]] = "[MissingHolyPower]",
--		[L["Percent"]] = "[HolyPower:Percent]",
--		[L["Mini"]] = "[HolyPower:VeryShort]",
--		[L["Smart"]] = "[MissingHolyPower:Hide(0):Short:Color('7f7fff')]",
--	} or nil,
--	soulShards = playerClass == "WARLOCK" and {
--		[L["Absolute"]] = "[SoulShards]",
--		[L["Absolute Short"]] = "[SoulShards:Short]",
--		[L["Difference"]] = "[MissingSoulShards]",
--		[L["Percent"]] = "[SoulShards:Percent]",
--		[L["Mini"]] = "[SoulShards:VeryShort]",
--		[L["Smart"]] = "[SoulShards:Hide(0):Short:Color('7f7fff')]",
--	} or nil,]]
	threat = {
		[L["Percent"]] = "[PercentThreat:Short:Hide(0):Percent]",
		[L["Raw Percent"]] = "[RawPercentThreat:Short:Hide(0):Percent]",
		[L["Colored Percent"]] = "[PercentThreat:Short:Hide(0):Percent:ThreatStatusColor(ThreatStatus)]",
		[L["Colored Raw Percent"]] = "[RawPercentThreat:Short:Hide(0):Percent:ThreatStatusColor(ThreatStatus)]",
	},
	cast = {
		[L["Standard Name"]] = "[Alpha((-CastStopDuration or 0) + 1) CastStopMessage or (CastName ' ' CastTarget:Paren)]",
		[L["Standard Time"]] = "[if not CastStopDuration then Concatenate('+', CastDelay:Round(1):Hide(0)):Red ' ' [CastEndDuration >= 0 ? '%.1f':Format(CastEndDuration)]]",
	},
	combo = (playerClass == "DRUID" or playerClass == "ROGUE") and {
		[L["Standard"]] = playerClass == "DRUID" and "[if IsEnergy(unit='player') then Combos:Hide(0)]" or "[Combos:Hide(0)]",
	} or nil,
	experience = {
		[L["Standard"]] = "[FractionalXP] [PercentXP:Percent:Paren] [Concatenate('R: ', PercentRestXP:Hide(0):Percent)]",
		[L["On Mouse-over"]] = "[if IsMouseOver then FractionalXP ' ' PercentXP:Percent:Paren ' ' Concatenate('R: ', PercentRestXP:Hide(0):Percent)]",
	},
	reputation = {
		[L["Standard"]] = "[if IsMouseOver then ReputationName else if ReputationName then FractionalReputation ' ' PercentReputation:Percent:Paren]"
	},
	none = {},
}

local textCategories = {
	class = L["Class"],
	health = L["Health"],
	name = L["Name"],
	power = L["Power"],
	druidMana = playerClass == "DRUID" and L["Druid Mana"] or nil,
--	holyPower = playerClass == "PALADIN" and L["Paladin Holy Power"] or nil,
--	soulShards = playerClass == "WARLOCK" and L["Warlock Soul Shards"] or nil,
	threat = L["Threat"],
	cast = L["Cast"],
	combo = (playerClass == "DRUID" or playerClass == "ROGUE") and L["Combo Points"] or nil,
	experience = L["Experience"],
	reputation = L["Reputation"],
	none = L["None"],
}

local defaultStyle = {
	class = L["Standard"],
	health = L["Smart"],
	name = L["Standard"],
	power = L["Smart"],
	druidMana = L["Smart"],
--	holyPower = L["Smart"],
--	soulShards = L["Smart"],
	threat = L["Colored Percent"],
	cast = L["Standard Name"],
	combo = L["Standard"],
	experience = L["Standard"],
	reputation = L["Standard"],
}

local textStyles_helper = {}

for cName, cStyles in pairs(textStyles) do
	local t = {}

	for sName, sValue in pairs(cStyles) do
		t[sValue] = sName
	end

	textStyles_helper[cName] = t
end

function TextFrame:CreateOptions()
	local options = super(self)

	options.text = {
		type = "group",
		name = L["Text"],
		order = 20,
		args = {
			descr1 = {
				type = "description",
				name = "\n"..L["You can select one of the predefined styles or enter your own tag sequence below:"].."\n",
				order = 3,
			},
			category = {
				name = L["Category"],
				type = "select",
				order = 4,
				values = textCategories,
				get = function() return self.settings.category or "none" end,
				set = function(info, value)
					self.settings.category = value

					if value ~= "none" then
						local style = defaultStyle[value]

						self.settings.text = textStyles[value][style]

						self:UpdateText()
					else
						UnderHood:NotifyOptionsChanged()
					end
				end,
			},
			style = {
				type = "select",
				name = L["Style"],
				order = 5,
				values = function() return textStyles_helper[self.settings.category] end,
				disabled = function() return not (self.settings.category) or self.settings.category == "none" end,
				get = function() return self.settings.text end,
				set = function(info, value)
					self.settings.text = value

					self:UpdateText()
				end,
			},
			text = {
				type = "input",
				name = L["Text"],
				width = "full",
				order = 6,
				multiline = true,
				disabled = function() return self.settings.category and self.settings.category ~= "none" end,
				get = function(info) return self.settings.text end,
				set = function(info, value)
					self.settings.text = value
					self:UpdateText()
				end,
			},
			help = {
				type = "execute",
				name = L["Open DogTag Help"],
				order = 7,
				width = "full",
				func = function() DogTag:OpenHelp() end,
			},
		},
	}

	options.font = {
		type = "group",
		name = L["Font"],
		order = 30,
		args = {
			name = {
				name = L["Name"],
				type = "select",
				order = 1,
				dialogControl = "LSM30_Font",
				values = function() return AceGUIWidgetLSMlists.font end,
				get = function() return self.settings.fontName end,
				set = function(info, value) self.settings.fontName = value; self:UpdateText() end,
			},
			size = {
				name = L["Size"],
				type = "range",
				order = 2,
				min = 4,
				max = 38,
				step = 1,
				bigStep = 1,
				get = function() return self.settings.fontSize end,
				set = function(info, value) self.settings.fontSize = value; self:UpdateText() end,
			},
			hAlign = {
				name = L["Horizontal align"],
				type = "select",
				order = 3,
				values = { LEFT = L["Left"], CENTER = L["Center"], RIGHT = L["Right"] },
				get = function() return self.settings.hAlign or "LEFT" end,
				set = function(info, value) self.settings.hAlign = value; self.text:SetJustifyH(value) end,
			},
			vAlign = {
				name = L["Vertical align"],
				type = "select",
				order = 4,
				values = { TOP = L["Top"], MIDDLE = L["Middle"], BOTTOM = L["Bottom"] },
				get = function() return self.settings.vAlign or "MIDDLE" end,
				set = function(info, value) self.settings.vAlign = value; self.text:SetJustifyV(value) end,
			},
		},
	}

	return options
end
