if select(2, UnitClass("player")) ~= "DEATHKNIGHT" then return end

local Module = UnderHood:GetModule("Runes")
local L = LibStub("AceLocale-3.0"):GetLocale("UnderHood_Runes")

local GetTime = GetTime
local GetRuneCooldown = GetRuneCooldown
local GetRuneType = GetRuneType
local ipairs = ipairs

local RunesFrame = UnderHood:RegisterFrameClass("Runes", L["Runes"])

local runeDescriptors = {
	{ L["Blood"], "Interface\\PlayerFrame\\UI-PlayerFrame-Deathknight-Blood" },
	{ L["Unholy"], "Interface\\PlayerFrame\\UI-PlayerFrame-Deathknight-Unholy" },
	{ L["Frost"], "Interface\\PlayerFrame\\UI-PlayerFrame-Deathknight-Frost" },
	{ L["Death"], "Interface\\PlayerFrame\\UI-PlayerFrame-Deathknight-Death" },
}

local layouts = {
	D = {
		size = function(size, space) return size, size * 5 + space * 4 end,
		point = "TOP", relative = "BOTTOM", dx = 0, dy = -1,
	},
	U = {
		size = function(size, space) return size, size * 5 + space * 4 end,
		point = "BOTTOM", relative = "TOP", dx = 0, dy = 1,
	},
	L = {
		size = function(size, space) return size * 5 + space * 4, size end,
		point = "RIGHT", relative = "LEFT", dx = -1, dy = 0,
	},
	R = {
		size = function(size, space) return size * 5 + space * 4, size end,
		point = "LEFT", relative = "RIGHT", dx = 1, dy = 0,
	},
}

function RunesFrame:init()
	super(self)

	self.runes = {}

	for i = 1, 6 do
		local rune = {}

		rune.texture = self.frame:CreateTexture()
		rune.texture:Show()

		rune.cooldown = CreateFrame("Cooldown", nil, self.frame)
		rune.cooldown:SetAllPoints(rune.texture)

		self.runes[i] = rune
	end

	LibStub("AceEvent-3.0"):Embed(self)
end

function RunesFrame:OnAcquire(settings)
	super(self, settings)

	self.frame:Show()

	self:RegisterEvent("RUNE_POWER_UPDATE")
	self:RegisterEvent("RUNE_TYPE_UPDATE")

	self:PerformLayout()
	self:UpdateRunes()
end

function RunesFrame:OnRelease()
	self:UnregisterAllEvents()

	super(self)
end

function RunesFrame:GetDefaultSettings()
	local settings = super(self)

	settings.positionAndSize.width = 1
	settings.positionAndSize.height = 1

	settings.runes = {
		layout = "D",
		size = 16,
		spacing = 0,
		style = "default",
	}

	return settings
end

--[[
function RunesFrame:ConfigurationModeChanged(mode)
	super(self, mode)

	self:UpdateRunes()
end
--]]

function RunesFrame:PerformLayout()
	local settings = self.settings.runes
	local layout = layouts[settings.layout]
	local size = settings.size
	local spacing = settings.spacing
	local width, height = layout.size(size, spacing)

	self.frame:SetWidth(width)
	self.frame:SetHeight(height)

	for i, rune in ipairs(self.runes) do
		local tx = rune.texture

		tx:SetWidth(size)
		tx:SetHeight(size)
		tx:ClearAllPoints()

		if i == 1 then
			tx:SetPoint(layout.point, self.frame, layout.point, 0, 0)
		else
			tx:SetPoint(layout.point, self.runes[i-1].texture, layout.relative, spacing * layout.dx, spacing * layout.dy)
		end
	end
end

function RunesFrame:RUNE_POWER_UPDATE(event, rune, usable)
	if rune then
		self:UpdateRune(rune)
	end
end

function RunesFrame:RUNE_TYPE_UPDATE(event, rune)
	if rune then
		self:UpdateRune(rune)
	end
end

function RunesFrame:UpdateRunes()
	for i = 1, 6 do
		self:UpdateRune(i)
	end
end

function RunesFrame:UpdateRune(id)
	if id > 6 then return end

	local rune = self.runes[id]
	local start, duration, ready = GetRuneCooldown(id)
	local runeType = GetRuneType(id)
	local descr = runeDescriptors[runeType]

	if rune.type ~= runeType then
		rune.texture:SetTexture(descr[2])
		rune.type = runeType
	end

	if rune.ready ~= ready or rune.start ~= start or rune.duration ~= duration then
		rune.ready = ready
		rune.start = start
		rune.duration = duration

		if ready then
			rune.cooldown:Hide()
		else
			rune.cooldown:SetCooldown(start, duration)
			rune.cooldown:Show()
		end
	end
end

local validLayouts = {
	D = L["Down"],
	U = L["Up"],
	R = L["Right"],
	L = L["Left"],
}

function RunesFrame:CreateOptions()
	local options = super(self)

	options.positionAndSize.args.width = nil
	options.positionAndSize.args.height = nil

	options.positionAndSize.args.h2 = {
		type = "header",
		name = L["Runes"],
		order = 100,
	}

	options.positionAndSize.args.layout = {
		name = L["Layout"],
		type = "select",
		order = 110,
		values = validLayouts,
		get = function() return self.settings.runes.layout end,
		set = function(info, value)
			self.settings.runes.layout = value
			self:PerformLayout()
		end,
	}

	options.positionAndSize.args.size = {
		name = L["Rune size"],
		type = "range",
		order = 120,
		min = 10,
		max = 48,
		step = 1,
		get = function() return self.settings.runes.size end,
		set = function(info, value)
			self.settings.runes.size = value
			self:PerformLayout()
		end,
	}

	options.positionAndSize.args.spacing = {
		name = L["Space between runes"],
		type = "range",
		order = 150,
		min = 0,
		max = 40,
		step = 1,
		get = function() return self.settings.runes.spacing end,
		set = function(info, value)
			self.settings.runes.spacing = value
			self:PerformLayout()
		end,
	}

	return options
end
