local UnderHood = UnderHood
local Module = UnderHood:GetModule("Portraits")
local L = LibStub("AceLocale-3.0"):GetLocale("UnderHood_Portraits")

local ipairs = ipairs
local select = select
local SetPortraitTexture = SetPortraitTexture
local type = type
local UnitExists = UnitExists
local UnitIsConnected = UnitIsConnected
local UnitIsVisible = UnitIsVisible

local validStyles = {
	flat = L["Flat"],
	animated = L["Animated"],
	--class = "Class",
}

local borderDescr = {
	{ "UHE-TL", 8, { "TOPLEFT", "TOPLEFT", 0 } },
	{ "UHE-TR", 8, { "TOPRIGHT", "TOPRIGHT", 0 } },
	{ "UHE-BL", 8, { "BOTTOMLEFT", "BOTTOMLEFT", 0 } },
	{ "UHE-BR", 8, { "BOTTOMRIGHT", "BOTTOMRIGHT", 0 } },
	{ "UHE-T", 8, { { "TOPLEFT", "TOPRIGHT", 1 }, { "TOPRIGHT", "TOPLEFT", 2 } } },
	{ "UHE-B", 8, { { "BOTTOMLEFT", "BOTTOMRIGHT", 3 }, { "BOTTOMRIGHT", "BOTTOMLEFT", 4 } } },
	{ "UHE-L", 8, { { "TOPLEFT", "BOTTOMLEFT", 1 }, { "BOTTOMLEFT", "TOPLEFT", 3 } } },
	{ "UHE-R", 8, { { "TOPRIGHT", "BOTTOMRIGHT", 2 }, { "BOTTOMRIGHT", "TOPRIGHT", 4 } } },
}

local Portrait = UnderHood:RegisterFrameClass("Portrait", L["Portrait"], "SecureUnitFrame")

function Portrait:init()
	super(self)

	LibStub("AceEvent-3.0"):Embed(self)

	self.borderParts = {}

	for i, bd in ipairs(borderDescr) do
		local tx = self.frame:CreateTexture(nil, "BACKGROUND")

		tx:SetTexture(UnderHood.addonPath.."\\Media\\"..bd[1])
		tx:SetWidth(bd[2]); tx:SetHeight(bd[2])
		self.borderParts[i] = tx

		local points = bd[3]

		if type(points[1]) == "table" then
			for _, point in ipairs(points) do
				tx:SetPoint(point[1], point[3] == 0 and self.frame or self.borderParts[point[3]], point[2])
			end
		else
			tx:SetPoint(points[1], points[3] == 0 and self.frame or self.borderParts[points[3]], points[2])
		end
	end

	self.background = self.frame:CreateTexture(nil, "BACKGROUND")
	self.background:SetTexture(0, 0, 0, 0.25)
	self.background:Show()

	self.texture2d = nil
	self.texture3d = nil
end

function Portrait:OnAcquire(settings)
	super(self, settings)

	if settings.portrait.showBorder then
		self:ShowBorder()
	else
		self:HideBorder()
	end

	self:SetupEvents()
	self:DoUpdatePortrait(true)
end

function Portrait:OnRelease()
	self.frame:SetScript("OnShow", nil)

	UnderHood:UnregisterUpdateTarget(self)
	self:UnregisterAllEvents()

	super(self)
end

function Portrait:UnitChanged(unit)
	super(self, unit)

	self:SetupEvents()
end

function Portrait:SetupEvents()
	local unit = self:GetUnit()
	local event, withUnit, owner = UnderHood:GetUnitChangedEvent(unit)

	UnderHood:UnregisterUpdateTarget(self)
	self:UnregisterAllEvents()

	self:RegisterEvent("UNIT_PORTRAIT_UPDATE", "UpdatePortraitIfUnit")

	self.unitOwner = owner

	if event then
		if withOwner then
			self:RegisterEvent(event, "UpdatePortraitIfUnit")
		else
			self:RegisterEvent(event, "UpdatePortrait")
		end
	else
		UnderHood:RegisterUpdateTarget(self)
	end
end

function Portrait:GetDefaultSettings()
	local settings = super(self)

	settings.positionAndSize.width = 40
	settings.positionAndSize.height = 40

	settings.portrait = {
		style = "flat",
		showBorder = true,
	}

	return settings
end

function Portrait:ShowBorder()
	for _, tx in ipairs(self.borderParts) do
		tx:Show()
	end

	if self.texture then
		self.texture:ClearAllPoints()

		self.texture:SetPoint("TOPLEFT", self.borderParts[1], "BOTTOMRIGHT", -4, 4)
		self.texture:SetPoint("BOTTOMRIGHT", self.borderParts[4], "TOPLEFT", 4, -4)
		self.background:SetPoint("TOPLEFT", self.borderParts[1], "BOTTOMRIGHT", -4, 4)
		self.background:SetPoint("BOTTOMRIGHT", self.borderParts[4], "TOPLEFT", 4, -4)
	end
end

function Portrait:HideBorder()
	for _, tx in ipairs(self.borderParts) do
		tx:Hide()
	end

	if self.texture then
		self.texture:ClearAllPoints(); self.texture:SetAllPoints(self.frame)
		self.background:ClearAllPoints(); self.background:SetAllPoints(self.frame)
	end
end

function Portrait:UpdatePortraitIfUnit(event, unit)
	if (self.unitOwner and unit == self.unitOwner) or (unit == self:GetUnit()) then
		self:DoUpdatePortrait(event == "UNIT_PORTRAIT_UPDATE")
	end
end

function Portrait:UpdatePortrait()
	local guid = UnitGUID(self:GetUnit())

	if guid ~= self.unitGUID then
		self:DoUpdatePortrait(true)
		self.unitGUID = guid
	end
end

function Portrait:ProcessUpdate()
	self:UpdatePortrait()
end

local classIcons = {
	["WARRIOR"] = { 0, 0.25, 0, 0.25 },
	["MAGE"] = { 0.25, 0.49609375, 0, 0.25 },
	["ROGUE"] = { 0.49609375, 0.7421875, 0, 0.25 },
	["DRUID"] = { 0.7421875, 0.98828125, 0, 0.25 },
	["HUNTER"] = { 0, 0.25, 0.25, 0.5 },
	["SHAMAN"] = { 0.25, 0.49609375, 0.25, 0.5 },
	["PRIEST"] = { 0.49609375, 0.7421875, 0.25, 0.5 },
	["WARLOCK"] = { 0.7421875, 0.98828125, 0.25, 0.5 },
	["PALADIN"] = { 0, 0.25, 0.5, 0.75 },
}

function Portrait:DoUpdatePortrait(force)
	local texture = self.texture

	if self.activeStyle ~= self.settings.portrait.style or not (texture) then
		if texture then
			if self.activeStyle == "animated" then
				texture:SetScript("OnShow", nil)
			end

			texture:Hide()
		end

		self.activeStyle = self.settings.portrait.style

		if self.activeStyle == "animated" then
			if not self.texture3d then
				self.texture3d = CreateFrame("PlayerModel", nil, self.frame)
			end

			texture = self.texture3d
			texture:SetScript("OnShow", function() self:OnShowModel() end)
		else
			if not self.texture2d then
				self.texture2d = self.frame:CreateTexture(nil, "ARTWORK")
			end

			texture = self.texture2d
		end

		self.texture = texture

		if texture then
			if self.settings.portrait.showBorder then
				texture:SetPoint("TOPLEFT", self.borderParts[1], "BOTTOMRIGHT", -4, 4)
				texture:SetPoint("BOTTOMRIGHT", self.borderParts[4], "TOPLEFT", 4, -4)
			else
				texture:SetAllPoints(self.frame)
			end

			force = true
		else
			return
		end
	end

	if force then
		local unit = self:GetUnit()

		if self.activeStyle == "flat" then
			SetPortraitTexture(texture, unit)
			texture:SetTexCoord(0.14644660941, 0.85355339059, 0.14644660941, 0.85355339059)
		elseif self.activeStyle == "animated" then
			if not UnitExists(unit) or not UnitIsConnected(unit) or not UnitIsVisible(unit) then
				texture:SetModelScale(4.25)
				texture:SetPosition(0, 0, -1.5)
				texture:SetModel("Interface\\Buttons\\talktomequestionmark.mdx")
			else
				texture:SetUnit(unit)
				texture:SetModelScale(1)
				texture:Hide()
			end
		elseif style == "class" then
			local className = select(2, UnitClass(unit))

			if className then
				local class = classIcons[className]

				texture:SetTexture("Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes")
				texture:SetTexCoord(class[1], class[2], class[3], class[4])
			else
				texture:SetTexture("Interface\\Icons\\Ability_Hunter_BeastCall")
				texture:SetTexCoord(0, 1, 0, 1)
			end
		end

		texture:Show()
	end
end

function Portrait:OnShowModel()
	if self.texture then
		self.texture:SetCamera(0)
	end
end

function Portrait:CreateOptions()
	local options = super(self)

	options.general.args.style = {
		type = "select",
		name = L["Style"],
		order = 100,
		values = validStyles,
		get = function() return self.settings.portrait.style end,
		set = function(info, value) self.settings.portrait.style = value; self:DoUpdatePortrait(true) end,
	}

	options.general.args.showBorder = {
		type = "toggle",
		name = L["Show border"],
		order = 110,
		get = function() return self.settings.portrait.showBorder end,
		set = function(info, value)
			self.settings.portrait.showBorder = value

			if value then
				self:ShowBorder()
			else
				self:HideBorder()
			end
		end,
	}

	return options
end
