local UnderHood = UnderHood
local Module = UnderHood:GetModule("Icons")
local L = LibStub("AceLocale-3.0"):GetLocale("UnderHood_Icons")

local GetLootMethod = GetLootMethod
local GetPartyLeaderIndex = GetPartyLeaderIndex
local GetRaidTargetIndex = GetRaidTargetIndex
local UnitIsGroupLeader = UnitIsGroupLeader
local UnitIsGroupAssistant = UnitIsGroupAssistant
local IsResting = IsResting
local SetRaidTargetIconTexture = SetRaidTargetIconTexture
local UnitAffectingCombat = UnitAffectingCombat
local UnitFactionGroup = UnitFactionGroup
local UnitIsPVP = UnitIsPVP
local UnitIsPVPFreeForAll = UnitIsPVPFreeForAll

local pairs = pairs
local random = math.random
local strmatch = string.match
local tonumber = tonumber
local unpack = unpack

local validIcons = {
	resting = L["Resting"],
	combat = L["Combat"],
	pvp = L["PvP"],
	leader = L["Leader"],
	ml = L["Master Looter"],
	rt = L["Raid Target"],
}

local unitIcons = {
	player = {
		resting = true,
		combat = true,
		pvp = true,
		leader = true,
		ml = true,
		rt = true,
	},
	target = {
		combat = true,
		pvp = true,
		rt = true,
	},
	focus = {
		combat = true,
		pvp = true,
		rt = true,
	},
	targettarget = {
		combat = true,
		pvp = true,
		rt = true,
	},
	pet = {
		combat = true,
		pvp = true,
		rt = true,
	},
}

for i = 1, 4 do
	unitIcons["party"..i] = {
		combat = true,
		pvp = true,
		rt = true,
		leader = true,
		ml = true
	}
	unitIcons["partypet"..i] = {
		combat = true,
		pvp = true,
		rt = true,
	}
end

local iconTextures = {
	resting = { "Interface\\CharacterFrame\\UI-StateIcon", 0.09, 0.43, 0.08, 0.42 },
	combat = { "Interface\\CharacterFrame\\UI-StateIcon", 0.57, 0.9, 0.08, 0.41 },
	pvp = {
		{ "Interface\\TargetingFrame\\UI-PVP-FFA", 0.05, 0.605, 0.015, 0.57 },
		{ "Interface\\TargetingFrame\\UI-PVP-Horde", 0.08, 0.58, 0.045, 0.545 },
		{ "Interface\\TargetingFrame\\UI-PVP-Alliance", 0.07, 0.58, 0.06, 0.57 },
	},
	leader = { "Interface\\GroupFrame\\UI-Group-LeaderIcon", 0.1, 0.84, 0.14, 0.88 },
	ml = { "Interface\\GroupFrame\\UI-Group-MasterLooter", 0.15, 0.9, 0.15, 0.9 },
	rt = { "Interface\\TargetingFrame\\UI-RaidTargetingIcons", 0, 1, 0, 1 },
}

local IconFrame = UnderHood:RegisterFrameClass("Icon", L["Icon"], "SecureUnitFrame")

function IconFrame:init()
	super(self)

	LibStub("AceEvent-3.0"):Embed(self)

	self.rtiForConfig = random(8)

	self.texture = self.frame:CreateTexture()
	self.texture:SetAllPoints(self.frame)
	self.texture:Show()
end

local function registerEvents(self, settings)
	if settings.icon == "pvp" then
		self:RegisterEvent("UPDATE_FACTION", "UpdateByEvent")
		self:RegisterEvent("PLAYER_FLAGS_CHANGED", "UpdateByEvent")
		self:RegisterEvent("UNIT_FACTION", "UpdateByEvent")
	elseif settings.icon == "resting" then
		self:RegisterEvent("PLAYER_UPDATE_RESTING", "UpdateByEvent")
	elseif settings.icon == "combat" then
		self:RegisterEvent("PLAYER_REGEN_ENABLED", "UpdateByEvent")
		self:RegisterEvent("PLAYER_REGEN_DISABLED", "UpdateByEvent")
	elseif settings.icon == "leader" then
		self:RegisterEvent("PARTY_LEADER_CHANGED", "UpdateByEvent")
		self:RegisterEvent("GROUP_ROSTER_UPDATE", "UpdateByEvent")
	elseif settings.icon == "ml" then
		self:RegisterEvent("PARTY_LOOT_METHOD_CHANGED", "UpdateByEvent")
		self:RegisterEvent("GROUP_ROSTER_UPDATE", "UpdateByEvent")
	elseif settings.icon == "rt" then
		self:RegisterEvent("RAID_TARGET_UPDATE", "UpdateByEvent")
	end
end

function IconFrame:OnAcquire(settings)
	super(self, settings)

	registerEvents(self, settings)
end

function IconFrame:UpdateByEvent(event, unit)
	if not unit or unit == "player" then
		self:ProcessUpdate()
	end
end

function IconFrame:OnRelease()
	self:UnregisterAllEvents()

	super(self)
end

function IconFrame:GetDefaultSettings()
	local settings = super(self)

	settings.positionAndSize.width = 20
	settings.positionAndSize.height = 20

	settings.icon = "resting"

	return settings
end

function IconFrame:ProcessUpdate()
	local icon = self.settings.icon
	local unit = self:GetUnit()
	local configMode = self.configMode
	local visible, set = true, nil
	local iconIndex

	if icon == "resting" then
		if not IsResting() then visible = false end
	elseif icon == "combat" then
		if not (UnitAffectingCombat(unit)) then visible = false end
	elseif icon == "pvp" then
		local pvpt

		if UnitIsPVPFreeForAll(unit) then
			pvpt = 'FFA'
		else
			local faction = UnitFactionGroup(unit)

			if faction and UnitIsPVP(unit) then
				pvpt = faction
			end
		end

		if pvpt or configMode then
			if pvpt == "Horde" then
				iconIndex = 2
			elseif pvpt == "Alliance" then
				iconIndex = 3
			else
				iconIndex = 1
			end
		else
			visible = false
		end
	elseif icon == "leader" then
		if unit == "player" then
			if not (UnitIsGroupLeader("player") or UnitIsGroupAssistant("player")) then
				visible = false
			end
		else
			local pi = strmatch(unit, "^party(%d)$")

			if not (pi) or tonumber(pi) ~= GetPartyLeaderIndex() then
				visible = false
			end
		end
	elseif icon == "ml" then
		local _, lootmaster = GetLootMethod()

		if lootmaster then
			if lootmaster == 0 then
				if unit ~= "player" then
					visible = false
				end
			else
				local pi = strmatch(unit, "^party(%d)$")

				if pi ~= lootmaster then
					visible = false
				end
			end
		else
			visible = false
		end
	elseif icon == "rt" then
		local rti = GetRaidTargetIndex(unit)

		if not configMode then
			if rti then
				iconIndex = rti
				set = true
			else
				visible = false
			end
		else
			iconIndex = self.rtiForConfig
			set = true
		end
	end

	if visible or configMode then
		local tex, l, r, t, b

		if iconIndex and not (set) then
			tex, l, r, t, b = unpack(iconTextures[icon][iconIndex])
		else
			tex, l, r, t, b = unpack(iconTextures[icon])
		end

		self.texture:SetTexture(tex)
		self.texture:SetTexCoord(l, r, t, b)

		if set then
			SetRaidTargetIconTexture(self.texture, iconIndex)
		end

		if not self.texture:IsVisible() then
			self.texture:Show()
		end
	else
		self.texture:Hide()
	end
end

function IconFrame:UnitChanged(unit)
	super(self, unit)

	self:ChangeIcon()
end

function IconFrame:ChangeIcon(icon)
	local unit = self:GetUnit()
	local needRefresh = false
	icon = icon or self.settings.icon

	if not unitIcons[unit][icon] then
		for k in pairs(unitIcons[unit]) do
			icon = k
			needRefresh = true
			break
		end
	end

	self.settings.icon = icon

	if needRefresh then
		UnderHood:NotifyOptionsChanged()
		self:UnregisterAllEvents()
		registerEvents(self, settings)
	end
end

function IconFrame:CreateOptions()
	local options = super(self)

	options.general.args.icon = {
		type = "select",
		name = L["Icon"],
		values = function()
			local v = {}

			for icon in pairs(unitIcons[self:GetUnit()]) do
				v[icon] = validIcons[icon]
			end

			return v
		end,
		get = function() return self.settings.icon end,
		set = function(info, value) self:ChangeIcon(value) end,
	}

	return options
end
