local UnderHood = UnderHood
local Module = UnderHood:GetModule("ComboPoints")
local L = LibStub("AceLocale-3.0"):GetLocale("UnderHood_ComboPoints")

local GetComboPoints = GetComboPoints
local ipairs = ipairs
local UnitHasVehicleUI = UnitHasVehicleUI

local ComboPointsFrame = UnderHood:RegisterFrameClass("ComboPoints", L["Combo Points"])

local layouts = {
	D = {
		size = function(size, space) return size, size * 5 + space * 4 end,
		point = "TOP", relative = "BOTTOM", dx = 0, dy = -1,
	},
	U = {
		size = function(size, space) return size, size * 5 + space * 4 end,
		point = "BOTTOM", relative = "TOP", dx = 0, dy = 1,
	},
	L = {
		size = function(size, space) return size * 5 + space * 4, size end,
		point = "RIGHT", relative = "LEFT", dx = -1, dy = 0,
	},
	R = {
		size = function(size, space) return size * 5 + space * 4, size end,
		point = "LEFT", relative = "RIGHT", dx = 1, dy = 0,
	},
}

function ComboPointsFrame:init()
	super(self)

	self.points = {}

	for i = 1, 5 do
		local point = self.frame:CreateTexture()

		self.points[i] = point
	end

	LibStub("AceEvent-3.0"):Embed(self)
end

function ComboPointsFrame:OnAcquire(settings)
	super(self, settings)

	self.frame:Show()

	self:RegisterEvent("UNIT_COMBO_POINTS", "UpdateComboPoints")
	self:RegisterEvent("UNIT_ENTERED_VEHICLE", "UpdateComboPoints")
	self:RegisterEvent("UNIT_EXITED_VEHICLE", "UpdateComboPoints")
	self:RegisterEvent("PLAYER_TARGET_CHANGED", "UpdateComboPoints")

	self:UpdateStyle()
	self:PerformLayout()
	self:UpdateComboPoints()
end

function ComboPointsFrame:OnRelease()
	self:UnregisterAllEvents()

	super(self)
end

function ComboPointsFrame:GetDefaultSettings()
	local settings = super(self)

	settings.positionAndSize.width = 1
	settings.positionAndSize.height = 1

	settings.combo = {
		layout = "D",
		size = 16,
		spacing = 0,
		style = "default",
	}

	return settings
end

function ComboPointsFrame:ConfigurationModeChanged(mode)
	super(self, mode)

	self:UpdateComboPoints()
end

function ComboPointsFrame:UpdateStyle()
	for i, point in ipairs(self.points) do
		point:SetTexture(UnderHood.addonPath.."\\Media\\ColoredComboPoint-"..i)
	end
end

function ComboPointsFrame:PerformLayout()
	local settings = self.settings.combo
	local layout = layouts[settings.layout]
	local size = settings.size
	local spacing = settings.spacing
	local width, height = layout.size(size, spacing)

	--self.settings.positionAndSize.width = width
	--self.settings.positionAndSize.height = height
	self.frame:SetWidth(width)
	self.frame:SetHeight(height)

	for i, point in ipairs(self.points) do
		point:SetWidth(size)
		point:SetHeight(size)
		point:ClearAllPoints()

		if i == 1 then
			point:SetPoint(layout.point, self.frame, layout.point, 0, 0)
		else
			point:SetPoint(layout.point, self.points[i-1], layout.relative, spacing * layout.dx, spacing * layout.dy)
		end
	end
end

function ComboPointsFrame:UNIT_COMBO_POINTS(event, unit)
	if unit == "player" or unit == "vehicle" then
		self:UpdateComboPoints()
	end
end

function ComboPointsFrame:UpdateComboPoints()
	local points = UnderHood.configMode and 5 or GetComboPoints(UnitHasVehicleUI("player") and "vehicle" or "player", "target")

	for i, point in ipairs(self.points) do
		if points >= i then
			point:Show()
		else
			point:Hide()
		end
	end
end

local validLayouts = {
	D = L["Down"],
	U = L["Up"],
	R = L["Right"],
	L = L["Left"],
}

function ComboPointsFrame:CreateOptions()
	local options = super(self)

	options.positionAndSize.args.width = nil
	options.positionAndSize.args.height = nil

	options.positionAndSize.args.h2 = {
		type = "header",
		name = L["Combo Points"],
		order = 100,
	}

	options.positionAndSize.args.layout = {
		name = L["Layout"],
		type = "select",
		order = 110,
		values = validLayouts,
		get = function() return self.settings.combo.layout end,
		set = function(info, value)
			self.settings.combo.layout = value
			self:PerformLayout()
		end,
	}

	options.positionAndSize.args.size = {
		name = L["Combo Point size"],
		type = "range",
		order = 120,
		min = 10,
		max = 48,
		step = 1,
		get = function() return self.settings.combo.size end,
		set = function(info, value)
			self.settings.combo.size = value
			self:PerformLayout()
		end,
	}

	options.positionAndSize.args.spacing = {
		name = L["Space between points"],
		type = "range",
		order = 150,
		min = 0,
		max = 40,
		step = 1,
		get = function() return self.settings.combo.spacing end,
		set = function(info, value)
			self.settings.combo.spacing = value
			self:PerformLayout()
		end,
	}

	return options
end
