local UnderHood = UnderHood
local Module = UnderHood:GetModule("Bars")
local L = Module.L
local OO = UnderHood.OO
local LSM = LibStub("LibSharedMedia-3.0")

local ipairs = ipairs
local pairs = pairs
local twipe = table.wipe
local type = type
local unpack = unpack

local Style = OO:NewClass("UniversalBarStyle", "BarStyle")

function Style:init(factory, meta)
	super(self, factory)

	self.meta = meta
	self.textures = {}
	self.zones = {}
end

function Style:GetZones()
	return self.meta.zones
end

function Style:OnAcquire(bar, settings)
	super(self, bar, settings)

	local sbt = OO:GetClass("StatusBarTexture")
	local meta = self.meta
	local frame = bar:GetFrame()

	for tName, tData in pairs(meta.textures) do
		self.textures[tName] = sbt:Acquire(frame, tData, settings)
	end

	for z, zm in ipairs(meta.zones) do
		local zd = {}

		zd.config = zm
		zd.texture = self.textures[zm.texture]

		self.values[z] = zm.defaultValue or 0
		self.colors[z] = { r = 0, g = 0, b = 0, a = 0 }

		self.zones[z] = zd

		if type(zm.vertical) == "boolean" then
			zd.vertical = zm.vertical
		elseif type(zm.vertical) == "string" then
			zd.vertical = settings[zm.vertical]
		else
			zd.vertical = false
		end

		if type(zm.reversed) == "boolean" then
			zd.reversed = zm.reversed
		elseif type(zm.reversed) == "string" then
			zd.reversed = settings[zm.reversed]
		else
			zd.reversed = false
		end

		zd.texture:SetOrientation(zd.vertical, zd.reversed)
		zd.texture:SetStatusBarValue(self.values[z])
	end
end

function Style:OnRelease()
	local sbt = OO:GetClass("StatusBarTexture")

	if self.textures then
		for _, v in pairs(self.textures) do
			sbt:Release(v)
		end
	end

	self.textures = twipe(self.textures)
	self.zones = twipe(self.zones)

	super(self)
end

function Style:SetColorCore(zone, r, g, b, a)
	local zd = self.zones[zone]

	if not zd then return end

	zd.texture:SetColor(r, g, b, a)
end

function Style:SetValueCore(zone, value)
	local zd = self.zones[zone]

	if not zd then return end

	zd.texture:SetStatusBarValue(value)
end

local function HaveOptions(data)
	return	type(data.color) == "string" or
			type(data.visibility) == "string" or
			(type(data.lsmName) == "string" and LSM)
end

function Style:CreateOptions()
	local options = super(self)
	local order = 100
	local optCache = {}

	for i, zm in ipairs(self.meta.zones) do
		if type(zm.vertical) == "string" or type(zm.reversed) == "string" then
			local group = {
				name = zm.title,
				type = "group",
				inline = true,
				order = order,
				args = {},
			}

			optCache[group.name] = group

			order = order + 1

			if type(zm.vertical) == "string" then
				local vopt = {
					name = L["Vertical"],
					type = "toggle",
					order = 1,
					get = function() return self.settings[zm.vertical] end,
					set = function(info, value)
						self.settings[zm.vertical] = value

						local zd = self.zones[i]

						zd.vertical = value

						zd.texture:SetOrientation(zd.vertical, zd.reversed)
						zd.texture:SetStatusBarValue(self.values[i])
					end,
				}

				group.args.vertical = vopt
			end

			if type(zm.reversed) == "string" then
				local ropt = {
					name = L["Reversed"],
					type = "toggle",
					order = 2,
					get = function() return self.settings[zm.reversed] end,
					set = function(info, value)
						self.settings[zm.reversed] = value

						local zd = self.zones[i]

						zd.reversed = value

						zd.texture:SetOrientation(zd.vertical, zd.reversed)
						zd.texture:SetStatusBarValue(self.values[i])
					end,
				}

				group.args.reversed = ropt
			end

			options["zone"..i] = group
		end
	end

	for eName, eData in pairs(self.meta.textures) do
		if HaveOptions(eData) then
			local group = optCache[eData.name or eName]

			if not group then
				group = {
					name = eData.name or eName,
					type = "group",
					inline = true,
					order = order,
					args = {},
				}

				order = order + 1

				options[eName] = group

				optCache[group.name] = group
			end

			if type(eData.visibility) == "string" then
				local vopt = {
					name = L["Visible"],
					type = "toggle",
					order = 1,
					get = function() return self.settings[eData.visibility] end,
					set = function(info, value)
						self.settings[eData.visibility] = value

						if value then
							self.textures[eName]:Show()
						else
							self.textures[eName]:Hide()
						end
					end,
				}

				group.args.visible = vopt
			end

			if type(eData.color) == "string" then
				local copt = {
					name = L["Color"],
					type = "color",
					order = 2,
					hasAlpha = true,
					get = function() return unpack(self.settings[eData.color]) end,
					set = function(info, r, g, b, a)
						self.settings[eData.color] = { r, g, b, a }

						self.textures[eName]:SetColor(r, g, b, a)
					end,
				}

				group.args.color = copt
			end

			if type(eData.lsmName) == "string" and LSM then
				group.args.texture = {
					name = L["Texture"],
					type = "select",
					dialogControl = "LSM30_Statusbar",
					order = 3,
					values = function() return AceGUIWidgetLSMlists.statusbar end,
					get = function() return self.settings[eData.lsmName] end,
					set = function(info, value)
						self.settings[eData.lsmName] = value
						self.textures[eName]:SetTexture(LSM:Fetch(LSM.MediaType.STATUSBAR, value))
					end,
				}
			end
		end
	end

	return options
end

local Factory = OO:NewClass("UniversalBarStyleFactory", "BarStyleFactory")

function Factory:init(metaName, metaStyle)
	super(self, metaName, metaStyle.title or metaName)

	self.metaName = metaName
	self.metaStyle = metaStyle or metaStyles[metaName]
end

function Factory:GetDefaultSettings()
	local settings = super(self)

	if type(self.metaStyle.settings) == "table" then
		UnderHood:CopyTable(self.metaStyle.settings, settings)
	end

	return settings
end

function Factory:UpgradeSettings(settings)
	if type(self.metaStyle.UpgradeSettings) == "function" then
		settings = self.metaStyle.UpgradeSettings(settings)
	end

	return settings
end

function Factory:CreateStyle(bar)
	return Style:new(self, self.metaStyle)
end

function Factory:GetPreferredSize()
	if self.metaStyle.preferredSize then
		return self.metaStyle.preferredSize.width, self.metaStyle.preferredSize.height
	end
end

local txPath = UnderHood.addonPath.."\\Media"

if LSM then
	LSM:Register("statusbar", "UnderHood Vertical", txPath.."\\UHVB")
end

local metaStyles = {
	UnderHood = {
		title = L["UnderHood"],
		preferredSize = {
			width = 22,
			height = 256,
		},
		settings = {
			showBackground = true,
			showBorder = true,
			bgColor = { 0.3, 0.3, 0.3, 0.75 },
			sbVertical = true,
			sbReversed = true,
			cbVertical = true,
			cbReversed = true,
			rcbVertical = true,
			rcbReversed = false,
			bgTexture = "UnderHood Vertical",
			sbTexture = "UnderHood Vertical",
		},
		UpgradeSettings = function(settings)
			if settings.showBorder == nil then
				settings.showBorder = true
			end

			return settings
		end,
		zones = {
			{
				title = L["Main bar"],
				texture = "statusbar",
				defaultValue = 1,
				vertical = "sbVertical",
				reversed = "sbReversed",
			},
			{
				title = L["Outline bar"],
				texture = "castingbar",
				defaultValue = 0,
				vertical = "cbVertical",
				reversed = "cbReversed",
			},
		},
		textures = {
			background = {
				name = L["Background"],
				class = "SimpleStatusBarTexture",
				lsmName = "bgTexture",
				path = txPath.."\\UHVB",
				blendMode = "BLEND",
				points = {
					{ "TOPLEFT", "TOPLEFT", 5, -5 },
					{ "BOTTOMRIGHT", "BOTTOMRIGHT", -5, 5 },
				},
				strata = "MEDIUM",
				level = 2,
				color = "bgColor",
				visibility = "showBackground",
			},
			border = {
				name = L["Border"],
				class = "CompoundStatusBarTexture",
				path = txPath,
				blendMode = "BLEND",
				strata = "MEDIUM",
				level = 3,
				static = true,
				points = true,
				parts = {
					{ "UHE-TL", 8 },
					{ "UHE-TR", 8 },
					{ "UHE-BL", 8 },
					{ "UHE-BR", 8 },
					{ "UHE-T", 8 },
					{ "UHE-B", 8 },
					{ "UHE-L", 8 },
					{ "UHE-R", 8 },
				},
				visibility = "showBorder",
			},
			statusbar = {
				name = L["Main bar"],
				class = "SimpleStatusBarTexture",
				lsmName = "sbTexture",
				path = txPath.."\\UHVB",
				blendMode = "BLEND",
				points = {
					{ "TOPLEFT", "TOPLEFT", 5, -5 },
					{ "BOTTOMRIGHT", "BOTTOMRIGHT", -5, 5 },
				},
				strata = "MEDIUM",
				level = 4,
			},
			castingbar = {
				class = "CompoundStatusBarTexture",
				path = txPath,
				blendMode = "BLEND",
				points = true,
				strata = "MEDIUM",
				level = 5,
				parts = {
					{ "UHC-TL", 8, 2 },
					{ "UHC-TR", 8, 2 },
					{ "UHC-BL", 8, 2 },
					{ "UHC-BR", 8, 2 },
					{ "UHC-T", 8, 2 },
					{ "UHC-B", 8, 2 },
					{ "UHC-L", 8, 2 },
					{ "UHC-R", 8, 2 },
				},
			},
		},
	},
}

for k, v in pairs(metaStyles) do
	local factory = Factory:new(k, v)

	Module:RegisterStyleFactory(k, factory)
end
