local OO = UnderHood.OO

local error = error
local pairs = pairs
local strformat = string.format
local type = type
local unpack = unpack

local defaults = {
	profile = {
		colors = {
			unit = {
				unknown = { 0.8, 0.8, 0.8 },
				hostile = { 226/255, 45/255, 75/255 },
				neutral = { 1, 1, 34/255 },
				friendly = { 0.2, 0.8, 0.15 },
				civilian = { 48/255, 113/255, 191/255 },
				tapped = { 0.5, 0.5, 0.5 },
				dead = { 0.6, 0.6, 0.6 },
				disconnected = { 0.7, 0.7, 0.7 },
				inCombat = { 1, 0, 0 },
			},
			power = {
				rage = { 226/255, 45/255, 75/255 },
				energy = { 1, 220/255, 25/255 },
				focus = { 1, 210/255, 0 },
				mana = { 48/255, 113/255, 191/255 },
				runicPower = { 0, 209/255, 1 },
				holyPower = { 218/255, 231/255, 31/255 },
				soulShards = { 255/255, 0, 255/255 },
				eclipseLunar = { 31/255, 31/255, 231/255 },
				eclipseLunarActive = { 0, 0, 255/255 },
				eclipseSolar = { 190/255, 210/255, 31/255 },
				eclipseSolarActive = { 238/255, 251/255, 31/255 },
			},
			health = {
				min = { 1, 0, 0 },
				mid = { 1, 1, 0 },
				max = { 0, 1, 0 },
			},
			cast = {
				casting = { 0, 1, 0 },
				channeling = { 0, 0, 1 },
				failed = { 1, 0, 0 },
				interrupted = { 1, 0, 0 },
				gcd = { 1, 220/255, 25/255 },
			},
			threat = {
				normal = { 0, 1, 0 },
				warning = { 1, 1, 0 },
				pull = { 1, 0, 0 },
			},
		},
	},
}

local Module = UnderHood:NewModule("Bars")
local L = LibStub("AceLocale-3.0"):GetLocale("UnderHood_Bars")

Module.L = L

function Module:OnInitialize()
	self.db = UnderHood.db:RegisterNamespace("Bars", defaults)

	self:CreateOptions()
end

local styleFactories = {}
local providerFactories = {}
local styleSelectorValues = {} -- { [""] = L["None"] }
local providerSelectorValues = { [""] = L["None"] }

function Module:RegisterStyleFactory(name, factory)
	if type(name) ~= "string" or #name == 0 then error("Style name must be nonempty string.") end
	if not OO:IsInstanceOf(factory, "BarStyleFactory") then error(strformat("Factory for style '%s' must be an instance of class derived from BarStyleFactory.", name)) end
	if styleFactories[name] then error(strformat("Factory for style '%s' is already registered.", name)) end

	styleFactories[name] = factory
	styleSelectorValues[name] = factory:GetTitle()
end

function Module:GetStyleFactory(name)
	return styleFactories[name]
end

function Module:IterateStyleFactories()
	return pairs(styleFactories)
end

function Module:GetStyleSelectorValues()
	return styleSelectorValues
end

function Module:RegisterProviderFactory(name, factory)
	if type(name) ~= "string" or #name == 0 then error("Provider name must be nonempty string.") end
	if not OO:IsInstanceOf(factory, "BarProviderFactory") then error(strformat("Factory for provider '%s' must be an instance of class derived from BarProviderFactory.", name)) end
	if providerFactories[name] then error(strformat("Factory for provider '%s' is already registered.", name)) end

	providerFactories[name] = factory
	providerSelectorValues[name] = factory:GetTitle()
end

function Module:GetProviderFactory(name)
	return providerFactories[name]
end

function Module:IterateProviderFactories()
	return pairs(providerFactories)
end

function Module:GetProviderSelectorValues()
	return providerSelectorValues
end

local colorOptions = {
	type = "group",
	name = L["Colors"],
	childGroups = "tab",
	args = {},
}

local function colorGroup(name, title, order)
	local group = {
		type = "group",
		name = title,
		order = order,
		args = {},
	}

	colorOptions.args[name] = group

	return group
end

local function colorEntry(order, title, group, key)
	local tbl = {}

	tbl.name = title
	tbl.type = "color"
	tbl.order = order
	tbl.get = function() return unpack(Module.db.profile.colors[group][key]) end
	tbl.set = function(info, r, g, b) Module.db.profile.colors[group][key] = { r, g, b } end

	return tbl
end

function Module:CreateOptions()
	local group

	-- Colors
	group = colorGroup("unit", L["Units"], 1)

	group.args.unknown = colorEntry(1, L["Unknown"], "unit", "unknown")
	group.args.hostile = colorEntry(2, L["Hostile"], "unit", "hostile")
	group.args.neutral = colorEntry(3, L["Neutral"], "unit", "neutral")
	group.args.friendly = colorEntry(4, L["Friendly"], "unit", "friendly")
	group.args.civilian = colorEntry(5, L["Civilian"], "unit", "civilian")
	group.args.tapped = colorEntry(6, L["Tapped"], "unit", "tapped")
	group.args.dead = colorEntry(7, L["Dead"], "unit", "dead")
	group.args.disconnected = colorEntry(8, L["Disconnected"], "unit", "disconnected")
	group.args.inCombat = colorEntry(9, L["In Combat"], "unit", "inCombat")

	group = colorGroup("power", L["Power"], 2)

	group.args.rage = colorEntry(1, L["Rage"], "power", "rage")
	group.args.energy = colorEntry(2, L["Energy"], "power", "energy")
	group.args.focus = colorEntry(3, L["Focus(P)"], "power", "focus")
	group.args.mana = colorEntry(4, L["Mana"], "power", "mana")
	group.args.runicPower = colorEntry(5, L["Runic power"], "power", "runicPower")
	group.args.holyPower = colorEntry(6, L["Holy Power"], "power", "holyPower")
	group.args.soulShards = colorEntry(7, L["Soul Shards"], "power", "soulShards")
	group.args.eclipseSolar = colorEntry(8, L["Eclipse Lunar"], "power", "eclipseSolar")
	group.args.eclipseLunar = colorEntry(9, L["Eclipse Solar"], "power", "eclipseLunar")
	group.args.eclipseSolarActive = colorEntry(10, L["Eclipse Lunar Active"], "power", "eclipseSolarActive")
	group.args.eclipseLunarActive = colorEntry(11, L["Eclipse Solar Active"], "power", "eclipseLunarActive")

	group = colorGroup("health", L["Health"], 3)

	group.args.min = colorEntry(1, L["Minimum"], "health", "min")
	group.args.mid = colorEntry(2, L["Middle"], "health", "mid")
	group.args.max  = colorEntry(3, L["Maximum"], "health", "max")

	group = colorGroup("cast", L["Spellcast"], 4)

	group.args.casting = colorEntry(1, L["Casting"], "cast", "casting")
	group.args.channeling = colorEntry(2, L["Channeling"], "cast", "channeling")
	group.args.failed  = colorEntry(3, L["Cast failed"], "cast", "failed")
	group.args.interrupted = colorEntry(4, L["Cast interrupted"], "cast", "interrupted")
	group.args.gcd = colorEntry(5, L["Global cooldown"], "cast", "gcd")

	group = colorGroup("threat", L["Threat"], 5)

	group.args.normal = colorEntry(1, L["Normal"], "threat", "normal")
	group.args.warning = colorEntry(2, L["Warning"], "threat", "warning")
	group.args.pull  = colorEntry(3, L["Pull aggro"], "threat", "pull")

	UnderHood:AddGeneralOptionsSection("colors", colorOptions)
end

