local UnderHood = UnderHood
local Module = UnderHood:GetModule("Bars")
local L = Module.L
local OO = UnderHood.OO

local GetThreatStatusColor = GetThreatStatusColor
local GetTime = GetTime
local UnitClass = UnitClass
local UnitName = UnitName
local UnitExists = UnitExists
local UnitIsPlayer = UnitIsPlayer
local UnitPlayerControlled = UnitPlayerControlled
local UnitCanAttack = UnitCanAttack
local UnitCastingInfo = UnitCastingInfo
local UnitDetailedThreatSituation = UnitDetailedThreatSituation
local UnitGUID = UnitGUID
local unpack = unpack

local Provider = OO:NewClass("ThreatBarProvider", "BarProviderWithUnit")

function Provider:init(factory)
	super(self, factory, "AceTimer-3.0")
end

function Provider:UnitChanged(oldUnit)
	super(self, oldUnit)

	self:RegisterEvent("UNIT_THREAT_LIST_UPDATE")
	self:RegisterEvent("UNIT_TARGET")

	self.threat = 0
	self.status = 0
	self.lastTargetSwitchTime = 0
	self.aggroUnit = nil
	self.hostileUnit = nil
	self.aggroGUID = nil
	self.hostileGUID = nil

	self:SetValue(0)
end

function Provider:OnEnterCombat()
	super(self)

	if self.unitGUID and self.hostileUnit then
		self:UpdateValue()
	else
		self:SetValue(0)
	end
end

function Provider:OnLeaveCombat()
	super(self)

	self:SetValue(0)
end

function Provider:UNIT_THREAT_LIST_UPDATE(event, unit)
	if not unit then return end
	if not (self.hostileUnit and self.unitGUID) then return end

	local unitGUID = UnitGUID(unit)

	if self.hostileGUID ~= unitGUID then return end

	local _, status, threatPct = UnitDetailedThreatSituation(self.unit, self.hostileUnit)

	self.threat = threatPct
	self.status = status

	self:UpdateValue()
end

local function isValidThreatTarget(unit)
	if not UnitExists(unit) or UnitIsPlayer(unit) or UnitPlayerControlled(unit) then
		return false
	end

	return true
end

function Provider:TargetTargetCheck()
	if not self.timerTargetSwitch and UnitGUID(self.aggroUnit) ~= self.aggroGUID then
		self:UNIT_TARGET(nil, self.hostileUnit)
	end
end

function Provider:SwitchAggroTarget()
	local n = self.aggroUnit and UnitName(self.aggroUnit)

	if n then
		self.aggroGUID = UnitGUID(self.aggroUnit)
		self.lastTargetSwitchTime = GetTime()
	end

	self.timerTargetSwitch = nil
end

--[[
	[Unit (unit)] --> [Enemy unit (hostile unit)] --> [Friendly Unit (aggro unit)]
	[Unit (unit)] --> [Friendly unit] --> [Enemy Unit (hostile unit)] --> [Friendly Unit (aggro unit)]
--]]
function Provider:UNIT_TARGET(event, unit)
	local target = self.unit == "player" and "target" or self.unit.."target"
	-- We are only interested in unit's target

	--UnderHood:Debug("UNIT_TARGET: %q", unit)

	if unit == self.unit then
		-- Watching unit has chagned target

		self.hostileUnit = nil
		self.hostileGUID = nil
		self.aggroUnit = nil
		self.aggroGUID = nil

		if self.timerTargetTargetCheck then
			self:CancelTimer(self.timerTargetTargetCheck, true)
			self.timerTargetTargetCheck = nil
		end

		if isValidThreatTarget(target) then
			self.hostileUnit = target
			self.aggroUnit = target.."target"
		elseif isValidThreatTarget(target.."target") then
			self.hostileUnit = target.."target"
			self.aggroUnit = target.."targettarget"
		end

		if not self.hostileUnit then
			self:SetValue(0)
			return
		end

		if not UnderHood.UnitSendEvents[self.hostileUnit] then
			self.timerTargetTargetCheck = self:ScheduleRepeatingTimer("TargetTargetCheck", 0.5)
		end

		self.hostileGUID = UnitGUID(self.hostileUnit)
		self.aggroGUID = UnitGUID(self.aggroUnit)
		--UnderHood:Debug("UNIT_TARGET: H: %q / A: %q", self.hostileGUID, self.aggroGUID or "nil")

		local _, status, threatPct = UnitDetailedThreatSituation(self.unit, self.hostileUnit)

		self.threat = threatPct
		self.status = status

		self:UpdateValue()

		unit = target -- to pass next check
	end

	if unit == target then
		if  (not self.hostileUnit and isValidThreatTarget(target.."target")) or
			(self.hostileUnit == target.."target" and self.hostileGUID ~= UnitGUID(self.hostileUnit)) then
			self:UNIT_TARGET(event, self.unit) -- Recurse back?
			return
		end

		if not self.hostileUnit then return end

		if self.timerTargetSwitch then
			self:CancelTimer(self.timerTargetSwitch, true)
			self.timerTargetSwitch = nil
		end

		if not UnitCastingInfo(self.hostileUnit) or not self.aggroGUID then
			--UnderHood:Debug("Starting check aggro timer")
			self.timerTargetSwitch = self:ScheduleTimer("SwitchAggroTarget", 0.5)
		end
	end
end

function Provider:RealUnitChanged(oldUnitId)
	if self.unitGUID then
		self:UNIT_TARGET(nil, self.unit)
	else
		self:SetValue(0)
	end
end

function Provider:UpdateColor()
	self:SetColor(0, 0, 1)
end

function Provider:UpdateValue()
	local threat, status = (self.threat or 0) / 100, self.status or 0
	local colorMode, tankMode = self.settings.colorMode, self.settings.tankMode
	local r, g, b

	if colorMode == "static" then
		r, g, b = unpack(Module.db.profile.colors.threat.pull)
	elseif colorMode == "custom" then
		if tankMode then
			if status == 3 then
				r, g, b = unpack(Module.db.profile.colors.threat.normal)
			elseif status > 0 then
				r, g, b = unpack(Module.db.profile.colors.threat.warning)
			else
				r, g, b = unpack(Module.db.profile.colors.threat.pull)
			end
		else
			if status > 1 then
				r, g, b = unpack(Module.db.profile.colors.threat.pull)
			elseif status == 1 then
				r, g, b = unpack(Module.db.profile.colors.threat.warning)
			else
				r, g, b = unpack(Module.db.profile.colors.threat.normal)
			end
		end
	else
		r, g, b = GetThreatStatusColor(status)
	end

	self:SetColor(r, g, b, 1)
	self:SetValue(threat)
end

local validColorModes = {
	default = L["Default"],
	custom = L["Custom"],
	static = L["Static"],
}

function Provider:CreateOptions()
	local options = super(self)

	options.colorMode = {
		type = "select",
		name = L["Color mode"],
		values = validColorModes,
		get = function() return self.settings.colorMode end,
		set = function(info, value) self.settings.colorMode = value end,
	}

	options.tankMode = {
		type = "toggle",
		name = L["Tank mode"],
		get = function() return self.settings.tankMode end,
		set = function(info, value) self.settings.tankMode = value end,
	}

	return options
end

local Factory = OO:NewClass("ThreatBarProviderFactory", "BarProviderFactory")

function Factory:init()
	super(self, "Threat", L["Threat"])
end

function Factory:GetDefaultSettings()
	local settings = super(self)

	settings.unit = "default"
	settings.colorMode = "default"

	return settings
end

function Factory:CreateProvider()
	return Provider:new(self)
end

local factory = Factory:new()

Module:RegisterProviderFactory(factory:GetName(), factory)
