local Module = UnderHood:GetModule("Bars")
local OO = UnderHood.OO

local assert = assert
local next = next
local setmetatable = setmetatable
local strformat = string.format
local twipe = table.wipe

local Style = OO:NewClass("BarStyle")

function Style:init(factory)
	self.factory = factory
	self.values = {}
	self.colors = {}
end

function Style:Release()
	assert(self.factory)

	self.factory:ReleaseStyle(self)
end

function Style:OnAcquire(bar, settings)
	assert(bar, "Bar is empty")
	assert(settings, "Settings is empty")

	twipe(self.values)
	twipe(self.colors)

	self.bar = bar
	self.settings = settings
end

function Style:OnRelease()
	self.bar = nil
end

function Style:GetFactory()
	return self.factory
end

function Style:GetSettings()
	return self.settings
end

Style:mustImplement("GetZones")
Style:mustImplement("SetColorCore")
Style:mustImplement("SetValueCore")

function Style:SetColor(zone, r, g, b, a)
	local color = self.colors[zone] or {}

	color.r = r
	color.g = g
	color.b = b
	color.a = a

	self.colors[zone] = color

	self:SetColorCore(zone, r, g, b, a)
end

function Style:SetValue(zone, value)
	if not value then value = 0 end
	if value < 0 then value = 0 end
	if value > 1 then value = 1 end

	if value ~= self.values[zone] then
		self.values[zone] = value

		self:SetValueCore(zone, value)
	end
end

local gray = { r = 0.5, g = 0.5, b = 0.5, a = 1 }

function Style:Update()
	local zones = self:GetZones()
	local inConfigMode = self.bar.inConfigMode

	for i=1, #zones do
		local color = self.colors[i] or gray
		local value = inConfigMode and 1 or (self.values[i] or 0)

		self:SetColorCore(i, unpack(color))
		self:SetValueCore(i, value)
	end
end

function Style:GetOptions()
	if not self.options then
		self.options = self:CreateOptions()
	end

	return self.options
end

function Style:CreateOptions()
	return {}
end

local Factory = OO:NewClass("BarStyleFactory")

function Factory:init(name, title)
	self.name = name
	self.title = title
end

function Factory:GetName()
	return self.name
end

function Factory:GetTitle()
	return self.title
end

function Factory:GetDefaultSettings()
	return {}
end

function Factory:UpgradeSettings(settings)
	return settings
end

function Factory:GetPreferredSize()
end

Factory:mustImplement("CreateStyle")

function Factory:AcquireStyle(bar, settings)
	local style

	if self.cache then
		style = next(self.cache)

		if style then
			self.cache[style] = nil
		end
	end

	if not style then
		style = self:CreateStyle()
		assert(OO:IsInstanceOf(style, Style), strformat("%s:CreateStyle() must return an instance of BarStyle or one of it's deriviatives.", self:class().name))
	end

	if not settings then
		settings = self:GetDefaultSettings()
	else
		settings = self:UpgradeSettings(settings)
	end

	style:OnAcquire(bar, settings)

	return style, settings
end

function Factory:ReleaseStyle(style)
	assert(OO:IsInstanceOf(style, Style))

	style:OnRelease()

	if not self.cache then self.cache = setmetatable({}, { __mode = 'k' }) end

	self.cache[style] = true
end
