local Module = UnderHood:GetModule("Bars")
local OO = UnderHood.OO

local assert = assert
local next = next
local setmetatable = setmetatable
local strformat = string.format
local type = type

local Provider = OO:NewClass("BarProvider")

function Provider:init(factory, ...)
	self.factory = factory

	local nEmbeds = select("#", ...)

	for i = 1, nEmbeds do
		local library = LibStub(select(i, ...))

		library:Embed(self)
	end
end

function Provider:Release()
	assert(self.factory)

	self.factory:ReleaseProvider(self)
end

function Provider:OnAcquire(bar, zone, settings)
	self.bar = bar
	self.zone = zone
	self.settings = settings

	self.value = 0
	self.red = 0
	self.green = 0
	self.blue = 0
	self.alpha = 0
end

function Provider:OnRelease()
	if type(self.UnregisterAllEvents) == "function" then
		self:UnregisterAllEvents()
		self:UnregisterAllMessages()
	end

	if type(self.CancelAllTimers) == "function" then
		self:CancelAllTimers()
	end

	self.bar = nil
	self.settings = nil
end

function Provider:GetBar()
	return self.bar
end

function Provider:GetZone()
	return self.zone
end

function Provider:GetSettings()
	return self.settings
end

function Provider:BarUnitChanged(unit)
end

function Provider:IsRealtime()
	return false
end

function Provider:GetRealtimeValueAndColor(currentTime)
	return 0, 0, 0, 0, 0
end

function Provider:Update()
	self:UpdateColor()
	self:UpdateValue()
end

function Provider:OnEnterCombat()
	self.inCombat = true
end

function Provider:OnLeaveCombat()
	self.inCombat = false
end

function Provider:UpdateColor()
end

function Provider:UpdateValue()
end

function Provider:SetColor(r, g, b, a)
	self.red = r
	self.green = g
	self.blue = b
	self.alpha = a

	self.bar:SetColor(self.zone, r, g, b, a)
end

function Provider:SetValue(value)
	self.value = value

	self.bar:SetValue(self.zone, value)
end

function Provider:GetOptions()
	if not self.options then
		self.options = self:CreateOptions()
	end

	return self.options
end

function Provider:CreateOptions()
	return {}
end

local Factory = OO:NewClass("BarProviderFactory")

function Factory:init(name, title)
	self.name = name
	self.title = title
end

function Factory:GetName()
	return self.name
end

function Factory:GetTitle()
	return self.title
end

function Factory:UnitSupported(unit)
	return true
end

function Factory:GetDefaultSettings()
	return {}
end

function Factory:UpgradeSettings(settings)
	return settings
end

Factory:mustImplement("CreateProvider")

function Factory:AcquireProvider(bar, zone, settings)
	local provider

	if self.cache then
		provider = next(self.cache)

		if provider then
			self.cache[provider] = nil
		end
	end

	if not provider then
		provider = self:CreateProvider()
		assert(OO:IsInstanceOf(provider, Provider), strformat("%s:CreateProvider() must return an instance of BarProvider or one of it's deriviatives.", self:class().name))
	end

	if not settings then
		settings = self:GetDefaultSettings()
	else
		settings = self:UpgradeSettings(settings)
	end

	provider:OnAcquire(bar, zone, settings)

	return provider, settings
end

function Factory:ReleaseProvider(provider)
	assert(OO:IsInstanceOf(provider, Provider))

	provider:OnRelease()

	if not self.cache then self.cache = setmetatable({}, { __mode = 'k' }) end

	self.cache[provider] = true
end
