local UnderHood = UnderHood
local Module = UnderHood:GetModule("Bars")
local L = Module.L

local UnitCanAttack = UnitCanAttack
local UnitClass = UnitClass
local UnitExists = UnitExists
local UnitHealth = UnitHealth
local UnitHealthMax = UnitHealthMax
local UnitIsDead = UnitIsDead
local UnitIsPlayer = UnitIsPlayer
local UnitIsPVP = UnitIsPVP
local UnitIsTapped = UnitIsTapped
local UnitIsTappedByPlayer = UnitIsTappedByPlayer
local UnitPlayerControlled = UnitPlayerControlled
local UnitReaction = UnitReaction
local RAID_CLASS_COLORS = RAID_CLASS_COLORS
local unpack = unpack

local Provider = UnderHood.OO:NewClass("HealthBarProvider", "BarProviderWithUnit")

function Provider:init(factory)
	super(self, factory, "AceTimer-3.0")
end

function Provider:UnitChanged(oldUnit)
	super(self, oldUnit)

	if self.unit == "player" or self.unit == "pet" then
		self:ScheduleRepeatingTimer("UpdateValue", 0.2)
		self:RegisterEvent("UNIT_FACTION", "UpdateIfUnit")
	elseif UnderHood.UnitSendEvents[self.unit] then
		self:RegisterEvent("UNIT_HEALTH", "UpdateValueIfUnit")
		self:RegisterEvent("UNIT_MAXHEALTH", "UpdateValueIfUnit")
		self:RegisterEvent("UNIT_FACTION", "UpdateColorIfUnit")
	end
end

function Provider:UpdateColor()
	super(self)

	local colors = Module.db.profile.colors
	local r, g, b = unpack(colors.health.max)
	local unit = self.unit
	local _, class = UnitClass(unit)
	local rc = RAID_CLASS_COLORS[class]
	local ps = self.settings

	if not UnitExists(unit) then return end

	if ps and ps.colorByHostility then
		if UnitIsPlayer(unit) or UnitPlayerControlled(unit) then
			if UnitCanAttack(unit, "player") then
				-- they can attack me
				if UnitCanAttack("player", unit) then
					-- and I can attack them
					r, g, b = unpack(colors.unit.hostile)
				else
					-- but I can't attack them
					r, g, b = unpack(colors.unit.civilian)
				end
			elseif UnitCanAttack("player", unit) then
				-- they can't attack me, but I can attack them
				r, g, b = unpack(colors.unit.neutral)
			elseif UnitIsPVP(unit) then
				-- on my team
				if ps.colorByClass and rc then
					r, g, b = rc.r, rc.g, rc.b
				else
					r, g, b = unpack(colors.unit.friendly)
				end
			else
				-- either enemy or friend, no violence
				r, g, b = unpack(colors.unit.civilian)
			end
		elseif (UnitIsTapped(unit) and not UnitIsTappedByPlayer(unit)) or UnitIsDead(unit) then
			r, g, b = unpack(colors.unit.tapped)
		else
			local reaction = UnitReaction(unit, "player")
			if reaction then
				if reaction >= 5 then
					if ps.colorByClass and rc then
						r, g, b = rc.r, rc.g, rc.b
					else
						r, g, b = unpack(colors.unit.friendly)
					end
				elseif reaction == 4 then
					r, g, b = unpack(colors.unit.neutral)
				else
					r, g, b = unpack(colors.unit.hostile)
				end
			else
				r, g, b = unpack(colors.unit.unknown)
			end
		end
	elseif ps and ps.colorByClass then
		if rc then
			r, g, b = rc.r, rc.g, rc.b
		else
			r, g, b = unpack(colors.unit.unknown)
		end
	end

	self:SetColor(r, g, b, 1)
end

function Provider:UpdateValue()
	super(self)

	local unit = self.unit
	local current, max = UnitHealth(unit), UnitHealthMax(unit)

	current = (max > 0) and (current / max) or 0

	self:SetValue(current)
end

function Provider:CreateOptions()
	local options = super(self)

	options.colorByHostility = {
		name = L["Color by hostility"],
		type = "toggle",
		get = function() return self.settings.colorByHostility end,
		set = function(info, value)
			self.settings.colorByHostility = value
			self:UpdateColor()
		end,
	}

	options.colorByClass = {
		name = L["Color by class"],
		type = "toggle",
		get = function() return self.settings.colorByClass end,
		set = function(info, value)
			self.settings.colorByClass = value
			self:UpdateColor()
		end,
	}

	return options
end

local Factory = UnderHood.OO:NewClass("HealthBarProviderFactory", "BarProviderFactory")

function Factory:init()
	super(self, "Health", L["Health"])
end

function Factory:GetDefaultSettings()
	local settings = super(self)

	settings.unit = "default"

	return settings
end

function Factory:CreateProvider()
	return Provider:new(self)
end

local factory = Factory:new()

Module:RegisterProviderFactory(factory:GetName(), factory)
