local Module = UnderHood:GetModule("Bars")
local L = Module.L

local BOOKTYPE_SPELL = BOOKTYPE_SPELL
local GetSpellBookItemName = GetSpellBookItemName
local GetSpellCooldown = GetSpellCooldown
local GetSpellTabInfo = GetSpellTabInfo
local unpack = unpack

local Provider = UnderHood.OO:NewClass("GlobalCooldownBarProvider", "BarProvider")

function Provider:init(factory)
	super(self, factory, "AceEvent-3.0")
end

function Provider:IsRealtime()
	return true
end

function Provider:OnAcquire(bar, zone, settings)
	super(self, bar, zone, settings)

	self:UpdateSpells()
	self:UpdateColor()

	self:RegisterEvent("ACTIONBAR_UPDATE_COOLDOWN")
	self:RegisterEvent("SPELLS_CHANGED", "UpdateSpells")
end

--
-- Called from OnUpdate of bar
--
-- arguments: currentTime = time when OnUpdate started
-- return values: value, r, g, b, a
--
function Provider:GetRealtimeValueAndColor(currentTime)
	if not self.active then return 0 end

	local value = (currentTime - self.startTime) / self.duration

	if value > 1 then
		self.active = nil
		return 0
	end

	return value
end

function Provider:ACTIONBAR_UPDATE_COOLDOWN()
	if self.primarySpellId then
		local start, duration = GetSpellCooldown(self.primarySpellId, BOOKTYPE_SPELL)

		if duration > 0 and duration <= 2 then
			self.currentSpell = 1
			self.startTime = start
			self.duration = duration
			self.active = true
		elseif self.currentSpell == 1 and duration == 0 then
			self.active = nil
		end
	end

	if self.secondarySpellId then
		local start, duration = GetSpellCooldown(self.secondarySpellId, BOOKTYPE_SPELL)

		if duration > 0 and duration <= 2 then
			self.currentSpell = 2
			self.startTime = start
			self.duration = duration
			self.active = true
		elseif self.currentSpell == 2 and duration == 0 then
			self.active = nil
		end
	end
end

function Provider:UpdateColor()
	local r, g, b = unpack(Module.db.profile.colors.cast.gcd)

	self:SetColor(r, g, b, 1)
end

function Provider:UpdateSpells()
	local primarySpell = self.settings.primarySpell
	local secondarySpell = self.settings.secondarySpell

	if primarySpell then primarySpell = primarySpell:lower() end
	if secondarySpell then secondarySpell = secondarySpell:lower() end

	self.primarySpellId = nil
	self.secondarySpellId = nil

	for spellTab = 1, 4 do
		local _, _, offset, numSpells = GetSpellTabInfo(spellTab)

		for i = (1+offset), (offset+numSpells) do
			local spell = GetSpellBookItemName(i, BOOKTYPE_SPELL)

			spell = spell:lower()

			if primarySpell and spell == primarySpell then
				self.primarySpellId = i
			elseif secondarySpell and spell == secondarySpell then
				self.secondarySpellId = i
			end
		end
	end
end

local function validateSpell(info, value)
	if not value or #value == 0 then return true end

	for spellTab = 1, 4 do
		local _, _, offset, numSpells = GetSpellTabInfo(spellTab)

		for i = (1+offset), (offset+numSpells) do
			local spell = GetSpellBookItemName(i, BOOKTYPE_SPELL)

			if spell:lower() == value:lower() then
				return true
			end
		end
	end

	return L["Invalid spell"]
end

function Provider:CreateOptions()
	local options = super(self)

	options.primarySpell = {
		name = L["Primary spell"],
		type = "input",
		order = 100,
		validate = validateSpell,
		get = function() return self.settings.primarySpell end,
		set = function(info, value)
			self.settings.primarySpell = value
			self:UpdateSpells()
		end,
	}

	options.secondarySpell = {
		name = L["Secondary spell"],
		type = "input",
		order = 101,
		validate = validateSpell,
		get = function() return self.settings.secondarySpell end,
		set = function(info, value)
			self.settings.secondarySpell = value
			self:UpdateSpells()
		end,
	}

	return options
end

local Factory = UnderHood.OO:NewClass("GlobalCooldownBarProviderFactory", "BarProviderFactory")

function Factory:init()
	super(self, "GCD", L["Global cooldown"])
end

function Factory:CreateProvider()
	return Provider:new(self)
end

local factory = Factory:new()

Module:RegisterProviderFactory(factory:GetName(), factory)
