local Module = UnderHood:GetModule("Bars")

local GetTime = GetTime
local min = math.min
local UnitCastingInfo = UnitCastingInfo
local UnitChannelInfo = UnitChannelInfo
local UnitExists = UnitExists
local unpack = unpack

local Provider = UnderHood.OO:NewClass("CastBarProvider", "BarProviderWithUnit")

function Provider:init(factory)
	super(self, factory)
end

function Provider:IsRealtime()
	return true
end

function Provider:UnitChanged(oldUnit)
	super(self, oldUnit)

	self.casting = nil
	self.channeling = nil
	self.fadeOut = nil

	self:RegisterEvent("UNIT_SPELLCAST_START")
	self:RegisterEvent("UNIT_SPELLCAST_STOP")
	self:RegisterEvent("UNIT_SPELLCAST_FAILED")
	self:RegisterEvent("UNIT_SPELLCAST_DELAYED")
	self:RegisterEvent("UNIT_SPELLCAST_INTERRUPTED")
	self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_START")
	self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_UPDATE")
	self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_STOP")
	self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_INTERRUPTED", "UNIT_SPELLCAST_INTERRUPTED")
end

--
-- Called from OnUpdate of bar
--
-- arguments: currentTime = time when OnUpdate started
-- return values: value, r, g, b, a
--
function Provider:GetRealtimeValueAndColor(currentTime)
	local value, r, g, b, a

	if self.casting then
		if currentTime > self.castEndTime then
			self.casting = nil
			self.fadeOut = true
			self.stopTime = currentTime
		end

		local showTime = min(currentTime, self.castEndTime)

		value = (showTime - self.castStartTime) / (self.castEndTime - self.castStartTime)
	elseif self.channeling then
		if currentTime > self.castEndTime then
			self.channeling = nil
			self.fadeOut = true
			self.stopTime = currentTime
		end

		local remainingTime = self.castEndTime - currentTime

		value = remainingTime / (self.castEndTime - self.castStartTime)
	elseif self.fadeOut then
		if self.stopTime then
			a = self.stopTime - currentTime + 1
		else
			a = 0
		end

		if a >= 1 then
			a = 1
		end

		if a <= 0 then
			self.fadeOut = nil
			self.stopTime = nil
			value = 0
			r = 0
			g = 0
			b = 0
		else
			r, g, b = unpack(self.castingColor)
			value = 1
		end
	else
		value = 0
	end

	return value, r, g, b, a
end

function Provider:UNIT_SPELLCAST_START(event, unit)
	if unit ~= self.unit then return end

	local spell, _, _, _, startTime, endTime = UnitCastingInfo(unit)

	self:SetValue(0)

	if not startTime then
		return
	end

	self.castStartTime = startTime / 1000
	self.castEndTime = endTime / 1000
	self.castDelay = 0
	self.casting = true
	self.channeling = nil
	self.fadeOut = nil
	self.castingColor = Module.db.profile.colors.cast.casting

	local r, g, b = unpack(self.castingColor)

	self:SetColor(r, g, b, 1)
end

function Provider:UNIT_SPELLCAST_CHANNEL_START(event, unit)
	if unit ~= self.unit then return end

	local spell, _, _, _, startTime, endTime = UnitChannelInfo(unit)

	self:SetValue(0)

	if not startTime then
		return
	end

	self.castStartTime = startTime / 1000
	self.castEndTime = endTime / 1000
	self.castDelay = 0
	self.casting = nil
	self.channeling = true
	self.fadeOut = nil
	self.castingColor = Module.db.profile.colors.cast.channeling

	local r, g, b = unpack(self.castingColor)

	self:SetColor(r, g, b, 1)
end

function Provider:UNIT_SPELLCAST_STOP(event, unit)
	if unit ~= self.unit then return end

	if self.casting then
		self.casting = nil
		self.fadeOut = true
		self.stopTime = GetTime()

		self:SetValue(1)
	end
end

function Provider:UNIT_SPELLCAST_CHANNEL_STOP(event, unit)
	if unit ~= self.unit then return end

	if self.channeling then
		self.channeling = nil
		self.fadeOut = true
		self.stopTime = GetTime()

		self:SetValue(1)
	end
end

function Provider:UNIT_SPELLCAST_FAILED(event, unit)
	if unit ~= self.unit or self.channeling then return end

	self.casting = nil
	self.channeling = nil
	self.fadeOut = true

	if not (self.stopTime) then
		self.stopTime = GetTime()
	end

	self.castingColor = Module.db.profile.colors.cast.failed

	local r, g, b = unpack(self.castingColor)

	self:SetValue(1)
	self:SetColor(r, g, b, 1)
end

function Provider:UNIT_SPELLCAST_DELAYED(event, unit)
	if unit ~= self.unit then return end

	local spell, rank, displayName, icon, startTime, endTime = UnitCastingInfo(unit)

	if startTime then
		local oldStartTime = self.castStartTime

		self.castStartTime = startTime / 1000
		self.castEndTime = endTime / 1000

		self.castDelay = (self.castDelay or 0) + (startTime - (oldStartTime or startTime))
	end
end

function Provider:UNIT_SPELLCAST_INTERRUPTED(event, unit)
	if unit ~= self.unit then return end

	self.casting = nil
	self.channeling = nil
	self.fadeOut = true

	if not (self.stopTime) then
		self.stopTime = GetTime()
	end

	self.castingColor = Module.db.profile.colors.cast.interrupted

	local r, g, b = unpack(self.castingColor)

	self:SetValue(1)
	self:SetColor(r, g, b, 1)
end

function Provider:UNIT_SPELLCAST_CHANNEL_UPDATE(event, unit)
	if unit ~= self.unit then return end

	local spell, _, _, _, startTime, endTime = UnitChannelInfo(unit)

	if startTime then
		local oldStartTime = self.castStartTime

		self.castStartTime = startTime / 1000
		self.castEndTime = endTime / 1000

		self.castDelay = (self.castDelay or 0) + (startTime - (oldStartTime or startTime))
	end
end

function Provider:Update()
	self.casting = nil
	self.channeling = nil
	self.fadeOut = nil

	self:SetValue(0)

	if not UnitExists(self.unit) then return end

	local spell, _, _, _, startTime, endTime = UnitCastingInfo(self.unit)

	if spell then
		self.castStartTime = startTime / 1000
		self.castEndTime = endTime / 1000
		self.castDelay = 0
		self.casting = true
		self.channeling = nil
		self.fadeOut = nil
		self.castingColor = Module.db.profile.colors.cast.casting
	else
		spell, _, _, _, startTime, endTime = UnitChannelInfo(self.unit)

		if spell then
			self.castStartTime = startTime / 1000
			self.castEndTime = endTime / 1000
			self.castDelay = 0
			self.casting = nil
			self.channeling = true
			self.fadeOut = nil
			self.castingColor = Module.db.profile.colors.cast.channeling
		end
	end

	if self.casting or self.channeling then
		local r, g, b = unpack(self.castingColor)

		self:SetValue(0)
		self:SetColor(r, g, b, 1)
	end
end

local Factory = UnderHood.OO:NewClass("CastBarProviderFactory", "BarProviderFactory")

function Factory:init()
	local L = Module.L

	super(self, "Cast", L["Cast"])
end

function Factory:GetDefaultSettings()
	local settings = super(self)

	settings.unit = "default"

	return settings
end

function Factory:CreateProvider()
	return Provider:new(self)
end

local factory = Factory:new()

Module:RegisterProviderFactory(factory:GetName(), factory)
