local UnderHood = UnderHood
local Module = UnderHood:GetModule("Bars")
local L = Module.L

local DURATION = 0.5
local GetTime = GetTime
local ipairs = ipairs
local cos, pi, max = math.cos, math.pi, math.max
local twipe = table.wipe


local BarFrame = UnderHood:RegisterFrameClass("Bar", L["Bar"], "SecureUnitFrame")

function BarFrame:init()
	super(self)
end

function BarFrame:OnAcquire(settings)
	super(self, settings)

	local styleFactory = Module:GetStyleFactory(settings.style)

	if not styleFactory then
		UnderHood:Print("Unable to create bar: style '%s' is not defined.", settings.style)
		return -- What else can I do?
	end

	self.style, settings.styleSettings = styleFactory:AcquireStyle(self, settings.styleSettings)

	local styleZones, settingsZones = self.style:GetZones(), settings.zones
	local count = max(#styleZones, #settingsZones)

	self.zones = {}

	for z = 1, count do
		local sz, zm

		if z > #styleZones then
			settingsZones[z] = nil
		else
			sz = styleZones[z]

			if z > #settingsZones then
				zm = { provider = "" }
				settingsZones[z] = zm
			else
				zm = settingsZones[z]
			end

			local zd = {}

			zd.map = zm
			zd.currentValue = nil
			zd.targetValue = nil
			zd.color = { r = 0, g = 0, b = 0, a = 0 }

			self.zones[#self.zones+1] = zd

			self.style:SetValue(z, 0)

			if #zm.provider > 0 then
				zd.provider, zm.settings = Module:GetProviderFactory(zm.provider):AcquireProvider(self, z, zm.settings)
			end
		end
	end

	self:UpdateOptions()

	self.frame:SetScript("OnUpdate", function(this)
		if this:IsVisible() then
			self:OnUpdate()
		end
	end)

	self.frame:SetScript("OnEnter", function(this)
		if this.uhFrame.settings.tooltips == "always"
			or (this.uhFrame.settings.tooltips == "ooc" and not (InCombatLockdown())) then
			if not (this:IsVisible()) or this:GetAlpha() == 0 then return end

			GameTooltip_SetDefaultAnchor(GameTooltip, this)
			GameTooltip:SetUnit(this.uhFrame:GetUnit())

			GameTooltip:Show()
		end
	end)

	self.frame:SetScript("OnLeave", function(this)
		if GameTooltip:IsOwned(this) then
			GameTooltip:Hide()
		end
	end)

end

function BarFrame:OnRelease()
	self.frame:SetScript("OnUpdate", nil)
	self.frame:SetScript("OnEnter", nil)
	self.frame:SetScript("OnLeave", nil)

	if self.style then
		self.style:Release()

		self.style = nil
	end

	for _, zd in ipairs(self.zones) do
		if zd.provider then
			zd.provider:Release()
			zd.provider = nil
		end
	end

	self.zones = twipe(self.zones)

	super(self)
end

function BarFrame:GetDefaultSettings()
	local settings = super(self)
	local factory = Module:GetStyleFactory("UnderHood")

	settings.positionAndSize.width, settings.positionAndSize.height = factory:GetPreferredSize()
	settings.tooltips = "never"
	settings.style = "UnderHood"
	settings.zones = {}

	return settings
end

function BarFrame:UpdateSettings(settings)
	settings = super(self, settings)

	if not settings.tooltips then
		settings.tooltips = "never"
	end

	return settings
end

function BarFrame:OnEnterCombat()
	super(self)

	for _, zd in ipairs(self.zones) do
		if zd.provider then
			zd.provider:OnEnterCombat()
		end
	end
end

function BarFrame:OnLeaveCombat()
	super(self)

	for _, zd in ipairs(self.zones) do
		if zd.provider then
			zd.provider:OnLeaveCombat()
		end
	end
end
function BarFrame:SetColor(zone, r, g, b, a)
	local zd = self.zones[zone]

	zd.color.r = r
	zd.color.g = g
	zd.color.b = b
	zd.color.a = a

	self.style:SetColor(zone, r, g, b, a)
end

function BarFrame:SetValue(zone, value)
	local zd = self.zones[zone]

	if zd.targetValue == value then return end

	if not zd.map.animate or not self.frame:IsVisible() then
		zd.currentValue = value
		zd.targetValue = value

		self.style:SetValue(zone, value)
	else
		zd.targetValue = value

		if zd.currentValue ~= zd.targetValue then
			zd.animationEndTime = GetTime() + DURATION
			zd.animating = true
		end
	end
end

local function CosineInterpolate(y1, y2, mu)
	local mu2 = (1 - cos(mu * pi)) / 2
	return y1 * (1 - mu2) + y2 * mu2
end

function BarFrame:OnUpdate()
	local currentTime = GetTime()

	for z, zd in ipairs(self.zones) do
		if zd.provider then
			if zd.provider:IsRealtime() then
				local value, r, g, b, a = zd.provider:GetRealtimeValueAndColor(currentTime)

				if value then self.style:SetValue(z, value) end
				if r and g and b and a then self.style:SetColor(z, r, g, b, a) end
			elseif zd.animating and zd.targetValue then
				local value = CosineInterpolate(zd.currentValue or 0, zd.targetValue, 1 - ((zd.animationEndTime - currentTime) / DURATION))

				zd.currentValue = value
				self.style:SetValue(z, value)

				if zd.currentValue == zd.targetValue then
					zd.animating = nil
				end
			end
		end
	end
end

function BarFrame:UnitChanged(unit)
	super(self, unit)

	for _, zd in ipairs(self.zones) do
		if zd.provider then
			zd.provider:BarUnitChanged(unit)
		end
	end
end

function BarFrame:ChangeStyle(newStyleName)
	self.style = self.style:Release()

	self.settings.style = newStyleName

	local factory = Module:GetStyleFactory(newStyleName)

	-- Adjusting frame size in case style has changed it

	local preferredWidth, preferredHeight = factory:GetPreferredSize()

	if preferredWidth and self.frame:GetWidth() ~= preferredWidth then
		self.settings.positionAndSize.width = preferredWidth
		self.frame:SetWidth(preferredWidth)
	end

	if preferredHeight and self.frame:GetHeight() ~= preferredHeight then
		self.settings.positionAndSize.height = preferredHeight
		self.frame:SetHeight(preferredHeight)
	end

	self.style, self.settings.styleSettings = factory:AcquireStyle(self)

	self:UpdateStyleOptions()

	-- Validating mappings
	local styleZones, settingsZones = self.style:GetZones(), self.settings.zones
	local count = max(#styleZones, #settingsZones)

	for z = 1, count do
		local sz, zm, zd

		if z > #styleZones then
			-- That zone doesn't exists
			zd = self.zones[z]

			if zd.provider then
				zd.provider:Release()
				zd.provider = nil
			end

			self.zones[z] = nil
			settingsZones[z] = nil
		else
			sz = styleZones[z]

			if z > #settingsZones then
				zm = { provider = "" }
				settingsZones[z] = zm
			else
				zm = settingsZones[z]
				zd = self.zones[z]
			end

			if not zd then
				zd = {}
				zd.map = zm
				zd.currentValue = 0
				zd.targetValue = 0
				zd.color = { r = 0, g = 0, b = 0, a = 0 }

				self.zones[z] = zd
			else
				zd.currentValue = zd.targetValue
				zd.animationEndTime = nil
				zd.animating = nil

				self.style:SetColor(z, zd.color.r, zd.color.g, zd.color.b, zd.color.a)
				self.style:SetValue(z, zd.targetValue)
			end
		end

		self:UpdateZoneOptions(z)
	end

	UnderHood:NotifyOptionsChanged()
end

function BarFrame:ChangeProvider(z, provider)
	local zd = self.zones[z]

	if not zd then return end

	if zd.provider then
		zd.provider:Release()
		zd.provider = nil
		zd.animationEndTime = nil
		zd.animating = nil
		zd.currentValue = 0
		zd.targetValue = 0
		zd.color = { r = 0, g = 0, b = 0, a = 0 }
	end

	if #provider > 0 then
		zd.map.provider = provider

		local factory = Module:GetProviderFactory(provider)

		zd.provider, zd.map.settings = factory:AcquireProvider(self, z)

		if zd.provider:IsRealtime() then
			zd.map.animate = nil
		end
	else
		zd.map.provider = ""
		zd.map.settings = nil
		zd.map.animate = nil

		self.style:SetValue(z, 0)
	end

	self:UpdateZoneOptions(z)

	UnderHood:NotifyOptionsChanged()
end

function BarFrame:CreateOptions()
	local options = super(self)

	options.general.args.tooltips = {
		name = L["Show Tooltips"],
		type = "select",
		order = 102,
		values = { always = L["Always"], ooc = L["Out of Combat"], never = L["Never"] },
		get = function() return self.settings.tooltips end,
		set = function(info, value) self.settings.tooltips = value end,
		disabled = function() return not self.settings.general.interactive end,
	}

	options.style = {
		type = "group",
		order = 20,
		name = L["Style"],
		args = {
			style = {
				type = "select",
				order = 1,
				name = L["Select style"]..":",
				values = function() return Module:GetStyleSelectorValues() end,
				get = function() return self.settings.style end,
				set = function(info, value) self:ChangeStyle(value) end,
			},
		},
		plugins = {},
	}

	options.providers = {
		type = "group",
		order = 30,
		name = L["Providers"],
		args = {},
	}

	return options
end

function BarFrame:UpdateOptions(clear)
	self:UpdateStyleOptions(clear)
	self:UpdateZonesOptions(clear)
end

function BarFrame:UpdateStyleOptions(clear)
	local options = self:GetOptions().style

	options.plugins.style = nil

	if self.style and not clear then
		options.plugins.style = self.style:GetOptions() 
	end
end

function BarFrame:UpdateZonesOptions(clear)
	local options = self:GetOptions().providers

	options.args = {}

	if clear then return end

	for z in ipairs(self.zones) do
		self:UpdateZoneOptions(z)
	end
end

function BarFrame:UpdateZoneOptions(z)
	local options = self:GetOptions().providers
	local zd = self.zones[z]
	local zo

	if zd then
		local styleZones = self.style:GetZones()

		zo = {
			type = "group",
			order = 10 + z,
			inline = true,
			name = L["Zone"]..": "..styleZones[z].title,
			args = {
				provider = {
					name = L["Provider"],
					type = "select",
					order = 1,
					values = function() return Module:GetProviderSelectorValues() end,
					get = function(info) return zd.map.provider or false end,
					set = function(info, value) self:ChangeProvider(z, value) end,
					--[[
					validate = function(info, value)
						if #value == 0 then return true end

						local factory = Module:GetProviderFactory(value)

						if not factory:UnitSupported(self:GetUnit()) then
							return (L["Sorry, but provider '%s' doesn't support unit '%s'."]):format(
								self.module:GetProvidersNameTableWithNone()[value],
								UnderHood.OptionsHelper.Unit[self.config.unit])
						end

						return true
					end,
					--]]
				},
				animate = {
					name = L["Animate"],
					type = "toggle",
					order = 2,
					get = function(info) return zd.provider and zd.map.animate end,
					set = function(info, value) zd.map.animate = value end,
					disabled = function(info) return not zd.provider or zd.provider:IsRealtime() end,
				},
			},
			plugins = {},
		}

		if zd.provider then
			zo.plugins.provider = zd.provider:GetOptions()
		end
	end

	options.args["Zone"..z] = zo
end
