local UnderHood = UnderHood
local Module = UnderHood:GetModule("Auras")
local L = Module.L
local Spells = Module.Spells

local CancelUnitBuff = CancelUnitBuff
local CreateFrame = CreateFrame
local floor, min, max = math.floor, math.min, math.max
local GetNumGroupMembers = GetNumGroupMembers
local GetTime = GetTime
local ipairs = ipairs
local UnitAura = UnitAura
local UnitBuff = UnitBuff
local UnitDebuff = UnitDebuff
local UnitIsFriend = UnitIsFriend
local UnitGUID = UnitGUID
local UnitIsUnit = UnitIsUnit
local tinsert = table.insert
local tsort = table.sort

local MAX_AURAS = 32
local auras = {}

-- Global auras storage

for i = 1, MAX_AURAS do
	auras[i] = {}
end

local validTypes = {
	buffs = L["Buffs"],
	debuffs = L["Debuffs"],
}

local validLayouts = {
	LTD = L["Left, then Down"],
	RTD = L["Right, then Down"],
	LTU = L["Left, then Up"],
	RTU = L["Right, then Up"],
	DTL = L["Down, then Left"],
	DTR = L["Down, then Right"],
	UTL = L["Up, then Left"],
	UTR = L["Up, then Right"],
}

local AurasFrame = UnderHood:RegisterFrameClass("AurasFrame", L["Auras"], "SecureUnitFrame")

function AurasFrame:init()
	super(self)

	LibStub("AceEvent-3.0"):Embed(self)

	self.buttons = {}

	for i = 1, MAX_AURAS do
		local button = CreateFrame("Button", nil, self.frame)

		button:RegisterForClicks("RightButtonUp")

		button:SetID(i)

		button:SetBackdrop({
			bgFile = "Interface\\Buttons\\WHITE8X8", tile = true, tileSize = 8,
			edgeFile = "Interface\\Buttons\\WHITE8X8", edgeSize = 1,
			insets = { left = 1, right = 1, top = 1, bottom = 1 },
		})

		button:SetBackdropColor(0, 0, 0, 0)
		button:SetBackdropBorderColor(0, 0, 0, 1)

		button.Icon = button:CreateTexture(nil, "BACKGROUND")
		button.Icon:SetTexture("Interface\\Icons\\INV_Misc_Ear_Human_02")
		button.Icon:SetTexCoord(0.07, 0.93, 0.07, 0.93)
		button.Icon:SetPoint("TOPLEFT", button, "TOPLEFT", 1, -1)
		button.Icon:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", -1, 1)
		button.Icon:Show()

		button.Count = button:CreateFontString(nil, "OVERLAY", "NumberFontNormalSmall")
		button.Count:SetJustifyH("RIGHT")
		button.Count:SetJustifyV("BOTTOM")
		button.Count:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", -1, 1)
		button.Count:Show()

		button.cooldown = CreateFrame("Cooldown", nil, button)
		button.cooldown:SetReverse(true)
		button.cooldown:SetPoint("TOPLEFT", button, "TOPLEFT", 1, -1)
		button.cooldown:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", -1, 1)
		button.cooldown:Hide()

		button.cooldownText = button.cooldown:CreateFontString(nil, "OVERLAY", "NumberFontNormalSmall")
		button.cooldownText:SetTextColor(1, 1, 1, 1)
		button.cooldownText:SetShadowColor(0, 0, 0, 1)
		button.cooldownText:SetShadowOffset(0.8, -0.8)
		button.cooldownText:SetPoint("TOPLEFT", button, "TOPLEFT", 1, -1)
		button.cooldownText:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", -1, 1)
		button.cooldownText:Hide()

		button.owner = self

		button:SetScript("OnEnter", function(this)
			if this.owner.settings.auras.tooltips == "always"
				or (this.owner.settings.auras.tooltips == "ooc" and not (InCombatLockdown())) then
				if not (this:IsVisible() and this.id) or this.owner.frame:GetAlpha() == 0 then return end

				GameTooltip:SetOwner(this, "ANCHOR_BOTTOMRIGHT")

				if this.id == 0 then
					GameTooltip:SetText(this.isBuff and "Test buff" or "Test debuff")
				elseif this.isBuff then
					GameTooltip:SetUnitBuff(this.unit, this.id)
				else
					GameTooltip:SetUnitDebuff(this.unit, this.id)
				end

				GameTooltip:Show()
			end
		end)

		button:SetScript("OnLeave", function(this)
			if GameTooltip:IsOwned(this) then
				GameTooltip:Hide()
			end
		end)

		button:SetScript("OnClick", function(this)
			if not this.isPlayer then return end

			CancelUnitBuff("player", this.id)
		end)

		button:Hide()

		tinsert(self.buttons, button)
	end
end

function AurasFrame:OnAcquire(settings)
	super(self, settings)
	if self.enabled then return end

	self:PerformLayout()

	self:RegisterEvent("UNIT_AURA")
	self:RegisterEvent("PLAYER_TARGET_CHANGED")
	self:RegisterEvent("PLAYER_FOCUS_CHANGED")
	self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
	self:RegisterEvent("GROUP_ROSTER_UPDATE")

	self:ForceUpdate()

	UnderHood:RegisterUpdateTarget(self)
end

function AurasFrame:OnRelease()
	UnderHood:UnregisterUpdateTarget(self)
	self:UnregisterAllEvents()

	super(self)
end

function AurasFrame:UpdateSettings(settings)
	settings = super(self, settings)

	settings.positionAndSize.width = 1
	settings.positionAndSize.height = 1

	return settings
end

function AurasFrame:GetDefaultSettings()
	local settings = super(self)

	settings.positionAndSize.width = 1
	settings.positionAndSize.height = 1

	settings.auras = {
		type = "buffs",
		tooltips = "ooc",
		maxAuras = 16,
		columns = 8,
		size = 20,
		spacing = 1,
		layout = "LTD",
	}

	return settings
end

local auraEvents = {
	SPELL_AURA_DISPELLED = true,
	SPELL_AURA_STOLEN = true,
	SPELL_AURA_APPLIED = true,
	SPELL_AURA_REMOVED = true,
	SPELL_AURA_APPLIED_DOSE = true,
	SPELL_AURA_REMOVED_DOSE = true,
}

function AurasFrame:COMBAT_LOG_EVENT_UNFILTERED(timestamp, event, hideCaster, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, spellId, spellName, spellSchool, ...)
	if self:GetUnit() ~= "targettarget" then return end
	if not auraEvents[event] then return end

	if not (self.totGUID) or (destGUID ~= self.totGUID and sourceGUID ~= self.totGUID) then return end

	self.needUpdate = true
end

local auraSort__isFriend
local auraSort__isBuff
local function auraSort(alpha, bravo)
	if not alpha or alpha[1] == -1 then
		return false
	elseif not bravo or bravo[1] == -1 then
		return true
	end

	-- show your own buffs/debuffs first
	local alpha_isMine, bravo_isMine = alpha[9], bravo[9]
	if not alpha_isMine ~= not bravo_isMine then
		if alpha_isMine then
			return true
		else
			return false
		end
	end

	if not auraSort__isBuff then
		if auraSort__isFriend then
			-- sort by dispel type
			local alpha_dispelType, bravo_dispelType = alpha[5], bravo[5]
			if alpha_dispelType ~= bravo_dispelType then
				if not alpha_dispelType then
					return false
				elseif not bravo_dispelType then
					return true
				end
				local canDispel_alpha_dispelType = Spells.canDispel[alpha_dispelType]
				if not canDispel_alpha_dispelType ~= not Spells.canDispel[bravo_dispelType] then
					-- show debuffs you can dispel first
					if canDispel_alpha_dispelType then
						return true
					else
						return false
					end
				end
				return alpha_dispelType < bravo_dispelType
			end
		end
	end

	local alpha_TimeLeft, bravo_TimeLeft = alpha[7], bravo[7]
	if alpha_TimeLeft == 0 then
		return false
	elseif bravo_TimeLeft == 0 then
		return true
	else
		return alpha_TimeLeft > bravo_TimeLeft
	end
end

function AurasFrame:CollectAuras()
	local settings = self.settings
	local uid = self:GetUnit()
	local isBuff = settings.auras.type == "buffs"
	local maxAuras = settings.auras.maxAuras
	local isPlayer = UnitIsUnit(uid, "player")
	local isPet = not isPlayer and UnitIsUnit(uid, "pet")
	local isFriend = isPlayer or isPet or UnitIsFriend("player", uid)
	local filtering = settings.auras.filter
	local extraFilteredSpells

	if filtering then
		if isBuff then
			if isFriend then
				extraFilteredSpells = Module.db.profile.filter.extraFriendBuffs
			else
				filtering = false
			end
		else
			if isFriend then
				extraFilteredSpells = Module.db.profile.filter.extraFriendDebuffs
			else
				extraFilteredSpells = Module.db.profile.filter.extraEnemyDebuffs
			end
		end
	end

	local nAuras = 0
	local i = 1

	while true do
		-- if #auras > maxAuras then break end

		local name, rank, icon, count, debuffType, duration, expirationTime, unitCaster = UnitAura(uid, i, isBuff and "HELPFUL" or "HARMFUL")

		if not name then
			break
		end

		local filtered = false

		if filtering then
			if isPlayer then
				if extraFilteredSpells and extraFilteredSpells[name] ~= nil then
					filtered = not extraFilteredSpells[name]
				else
					filtered = not Spells.canDispel[debuffType]
				end
			else
				if extraFilteredSpells and extraFilteredSpells[name] ~= nil then
					filtered = not extraFilteredSpells[name]
				elseif (extraFilteredSpells and extraFilteredSpells[name] ~= nil and (not extraFilteredSpells[name])) then
					filtered = true
				else
					filtered = not Spells.canDispel[debuffType]
				end
			end
		end

		if not filtered then
			local startTime

			if expirationTime and duration and duration > 0 then
				startTime = floor(expirationTime - duration)
			end

			if expirationTime then
				expirationTime = GetTime() - expirationTime -- now timeLeft
			end

			nAuras = nAuras + 1
			local a = auras[nAuras]

			a[1] = i
			a[2] = name
			a[3] = icon
			a[4] = count
			a[5] = debuffType
			a[6] = duration
			a[7] = expirationTime
			a[8] = startTime
			a[9] = unitCaster == "player"
			--auras[#auras+1] = { i, name, iconTexture, count, debuffType, duration, timeLeft, startTime, isMine }
		end

		i = i + 1
	end

	auraSort__isFriend = isFriend
	auraSort__isBuff = isBuff

	for i = nAuras+1, #auras do
		auras[i][1] = -1 --nil --releaseTable(auras[i])
	end

	tsort(auras, auraSort)

	return nAuras
end

function AurasFrame:UpdateButtons()
	local settings = self.settings
	local nAuras = self:CollectAuras()
	local maxButtons = settings.auras.maxAuras
	local unit = self:GetUnit()
	local isBuff = settings.auras.type == "buffs"
	local isPlayer = UnitIsUnit(unit, "player")
	local inConfigMode = UnderHood.configMode
	local groupCount = GetNumGroupMembers()

	for i, button in ipairs(self.buttons) do
		if i > maxButtons then
			if not button:IsVisible() then break end

			button.id = nil
			button.isBuff = nil
			button.isPlayer = nil
			button.unit = nil
			button.count = nil
			button.duration = nil
			button.isLarge = nil
			button.startTime = nil
			button.isMine = nil

			button:Hide()
			button = nil
		else
			if i > nAuras then
				button.isBuff = nil
				button.isPlayer = nil
				button.unit = nil
				button.count = nil
				button.duration = nil
				button.isLarge = nil
				button.startTime = nil
				button.isMine = nil

				if inConfigMode then
					button.id = 0
					button.Icon:SetTexture(isBuff and "Interface\\Icons\\INV_Misc_Ear_Human_02" or "Interface\\Icons\\INV_Misc_Bone_HumanSkull_01") --"Interface\\Icons\\INV_Misc_Ear_Human_02")
					button.Count:SetText(i)
					button.Count:Show()

					button:Show()
				else
					button.id = nil
					if not button:IsVisible() then break end

					button:Hide()
				end
			else
				-- We have to use assignments vs unpack() due to possible nil values inside a row
				local a = auras[i]
				local id, name, icon, count, debuffType, duration, timeLeft, startTime, isMine = a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9]

				if button.id ~= id or button.unit ~= unit or
					button.isBuff ~= isBuff or button.isPlayer ~= isPlayer or
					button.count ~= count or button.duration ~= duration or
					button.startTime ~= startTime or
					button.name ~= name or
					button.isMine ~= isMine or
					not (button:IsVisible()) or self.forceUpdate then

					--[[
					if unit == "target" then
					  UnderHood:Debug(("%sOld:: %s/%d: [%d] %s(%d) %s %f/%f @ %f"):format(self.forceUpdate and "<F>" or "<R>", button.unit or "nil", i, button.id or -1, button.name or "nil", count or -1, isMine and "M" or "O", timeLeft or -1, duration or -1, button.startTime or -1))
					  UnderHood:Debug(("%sNew:: %s/%d: [%d] %s(%d) %s %f/%f @ %f"):format(self.forceUpdate and "<F>" or "<R>", unit or "nil", i, id or -1, name or "nil", count or -1, isMine and "M" or "O", timeLeft or -1, duration or -1, startTime or -1))
					end
					--]]

					button.id = id
					button.unit = unit
					button.count = count
					button.duration = duration
					button.startTime = startTime
					button.isBuff = isBuff
					button.isPlayer = isPlayer
					button.isMine = isMine
					button.name = name

					button.Icon:SetTexture(icon)

					if count and count > 1 then
						button.Count:SetText(count)
						button.Count:Show()
					else
						button.Count:Hide()
					end

					if isMine and (groupCount > 0 or self.doubleWhenSolo) then
						button.isLarge = ((isBuff and self.doubleBuffs) or (not (isBuff) and self.doubleDebuffs)) and true or nil
					else
						button.isLarge = nil
					end

					if startTime then
						button.cooldown:Show()
						button.cooldown:SetCooldown(startTime, duration)
					else
						button.cooldown:Hide()
					end

					if isBuff or not self.colorDebuffBorders then
						button:SetBackdropBorderColor(0, 0, 0, 1)
					else
						local color = DebuffTypeColor[debuffType or "none"]

						button:SetBackdropBorderColor(color.r, color.g, color.b, 1)
					end

					button:Show()
				end
			end
		end
	end

	self:PerformLayout()

	--auras = table.wipe(auras)
	--[[
	for i in ipairs(auras) do
		auras[i] = releaseTable(auras[i])
	end

	auras = releaseTable(auras)
	--]]
end

function AurasFrame:ProcessUpdate()
	if self:GetUnit() == "targettarget" then
		local totGUID = UnitGUID("targettarget")

		if totGUID ~= self.totGUID then
			self.totGUID = totGUID

			self.needUpdate = true
		end
	end

	if self.needUpdate then
		self.needUpdate = nil
		self:UpdateButtons()
		self.forceUpdate = nil
	end
end

function AurasFrame:ForceUpdate()
	self.colorDebuffBorders = Module.db.profile.colorDebuffBorders
	self.doubleBuffs = Module.db.profile.doubleSelfBuffs
	self.doubleDebuffs = Module.db.profile.doubleSelfDebuffs
	self.doubleWhenSolo = Module.db.profile.doubleWhenSolo

	self.forceUpdate = true
	self.needUpdate = true
end

function AurasFrame:UNIT_AURA(eventName, unitId)
	if unitId == self:GetUnit() then
		self.needUpdate = true
	end
end

function AurasFrame:PLAYER_TARGET_CHANGED()
	local unit = self:GetUnit()

	if unit == "target" or unit == "targettarget" then
		self.forceUpdate = true
		self.needUpdate = true
	end
end

function AurasFrame:PLAYER_FOCUS_CHANGED()
	if self:GetUnit() == "focus" then
		self.needUpdate = true
	end
end

function AurasFrame:GROUP_ROSTER_UPDATE()
	self.needUpdate = true
	self.forceUpdate = true
end

function AurasFrame:ConfigurationModeChanged(mode)
	super(self, mode)

	self.needUpdate = true
	self.forceUpdate = true
end

function AurasFrame:UnitChanged(unit)
	super(self, unit)

	self.needUpdate = true
	self.forceUpdate = true
end

function AurasFrame:CreateOptions()
	local options = super(self)

	options.general.args.h1 = {
		type = "header",
		name = L["Aura settings"],
		order = 100,
	}

	options.general.args.type = {
		type = "select",
		name = L["Type"],
		order = 101,
		values = validTypes,
		get = function() return self.settings.auras.type end,
		set = function(info, value) self.settings.auras.type = value; self.needUpdate = true end,
	}

	options.general.args.tooltips = {
		name = L["Show Tooltips"],
		type = "select",
		order = 102,
		values = { always = L["Always"], ooc = L["Out of Combat"], never = L["Never"] },
		get = function() return self.settings.auras.tooltips end,
		set = function(info, value) self.settings.auras.tooltips = value end,
	}

	options.general.args.filter = {
		name = L["Filter auras"],
		desc = L["Filter certain auras based on your class"],
		type = "toggle",
		order = 103,
		get = function() return self.settings.auras.filter end,
		set = function(info, value) self.settings.auras.filter = value; self.needUpdate = true end,
	}

	options.positionAndSize.args.width = nil
	options.positionAndSize.args.height = nil

	options.positionAndSize.args.h2 = {
		type = "header",
		name = L["Aura buttons"],
		order = 100,
	}

	options.positionAndSize.args.layout = {
		name = L["Layout"],
		type = "select",
		order = 110,
		values = validLayouts,
		get = function() return self.settings.auras.layout end,
		set = function(info, value)
			self.settings.auras.layout = value
			self:PerformLayout()
		end,
	}

	options.positionAndSize.args.size = {
		name = L["Button size"],
		type = "range",
		order = 120,
		min = 10,
		max = 48,
		step = 1,
		get = function() return self.settings.auras.size end,
		set = function(info, value)
			self.settings.auras.size = value
			self:PerformLayout()
		end,
	}

	options.positionAndSize.args.maxAuras = {
		name = L["Auras to show"],
		desc = L["Maximum number of auras to show"],
		order = 130,
		type = "range",
		min = 1,
		max = MAX_AURAS,
		step = 1,
		get = function() return self.settings.auras.maxAuras end,
		set = function(info, value)
			if self.settings.auras.columns > value then
				self.settings.auras.columns = value
			end

			self.settings.auras.maxAuras = value
			self.needUpdate = true
		end,
	}

	options.positionAndSize.args.columns = {
		name = L["Columns"],
		type = "range",
		order = 140,
		min = 1,
		max = MAX_AURAS,
		step = 1,
		get = function() return self.settings.auras.columns end,
		set = function(info, value)
			self.settings.auras.columns = min(value, self.settings.auras.maxAuras)
			self:PerformLayout()
		end,
	}

	options.positionAndSize.args.spacing = {
		name = L["Space between buttons"],
		type = "range",
		order = 150,
		min = 0,
		max = 40,
		step = 1,
		get = function() return self.settings.auras.spacing end,
		set = function(info, value)
			self.settings.auras.spacing = value
			self:PerformLayout()
		end,
	}

	return options
end
