local UnderHood = UnderHood

local error = error
local next = next
local pairs = pairs
local setmetatable = setmetatable
local tonumber = tonumber
local type = type

local strformat = string.format
local strmatch = string.match

--[[
	Regular Update handling.

	Each object that wants to participate in unified update should register in
	UnderHood and implement ProcessUpdate method.
--]]

local objectsToUpdate = {}

function UnderHood:RegisterUpdateTarget(target, method)
	if type(target) ~= "table" then
		error(strformat("Bad argument #2 to 'RegisterUpdateTarget'. Expected 'table', got %q.", type(target)), 2)
	end

	if not method then method = "ProcessUpdate" end

	if type(target[method]) ~= "function" then
		error(strformat("Bad argument #2 to 'RegisterUpdateTarget'. No %q method declared.", method), 2)
	end

	if objectsToUpdate[target] then
		error("Double registration detected in 'RegisterUpdateTarget'.", 2)
	end

	objectsToUpdate[target] = target[method]
end

function UnderHood:UnregisterUpdateTarget(target)
	if type(target) ~= "table" then
		error(strformat("Bad argument #2 to 'UnregisterUpdateTarget'. Expected 'table', got %q.", type(target)), 2)
	end

	if objectsToUpdate[target] then
		objectsToUpdate[target] = nil
	end
end

function UnderHood:ProcessUpdate()
	if not next(objectsToUpdate) then return end

	for target, method in pairs(objectsToUpdate) do
		method(target)
	end
end

local UnitSendEvents = setmetatable({ player=true, target=true, pet=true, mouseover=true, focus=true, vehicle=true }, { __index = function(self, unit)
	if not unit then
		error("Argument #1 to `UnitSendEvents' should be a string, got nil.", 2)
	end

	local party = strmatch(unit, "^party(%d)$") or strmatch(unit, "^partypet(%d)$")
	if party then
		party = tonumber(party)
		if party and party >= 1 and party <= 4 then
			self[unit] = true
			return true
		end
		self[unit] = false
		return false
	end

	local raid = strmatch(unit, "^raid(%d%d?)$")
	if raid then
		raid = tonumber(raid)
		if raid and raid >= 1 and raid <= 40 then
			self[unit] = true
			return true
		end
		self[unit] = false
		return false
	end
	self[unit] = false
	return false
end})

UnderHood.UnitSendEvents = UnitSendEvents

function UnderHood:GetUnitChangedEvent(unit)
	if not unit then
		error("Argument #1 to `GetUnitChangedEvent' should be a string, got nil.", 2)
	end

	if unit == "player" then return "PLAYER_ENTERING_WORLD"
	elseif unit == "target" then return "PLAYER_TARGET_CHANGED"
	elseif unit == "focus" then return "PLAYER_FOCUS_CHANGED"
	elseif unit == "pet" then return "UNIT_PET", true, "player"
	end

	local idx = strmatch(unit, "^partypet(%d)$")
	if idx then return "UNIT_PET", true, "party"..idx end

	if strmatch(unit, "^party(%d)$") then return "GROUP_ROSTER_UPDATE" end
end
