local UnderHood = UnderHood
local L = UnderHood.L
local OO = UnderHood.OO

local RegisterUnitWatch = RegisterUnitWatch
local UnregisterUnitWatch = UnregisterUnitWatch

local pairs = pairs
local UnitInParty = UnitInParty
local UnitInRaid = UnitInRaid
local UnitIsPlayer = UnitIsPlayer
local UnitIsUnit = UnitIsUnit
local tContains = tContains

local _, unitClass = UnitClass("player")

local SecureUnitFrame = OO:NewClass("SecureUnitFrame", "Frame")

function SecureUnitFrame:init()
	super(self, "Button", "SecureUnitButtonTemplate")
end

function SecureUnitFrame:OnAcquire(settings)
	super(self, settings)

	self.frame:SetAttribute("unit", settings.general.unit)

	self:UpdateVisibility()
	self:UpdateInteractivity()
end

function SecureUnitFrame:OnRelease()
	self:UpdateInteractivity(true)
	self:UpdateVisibility(true)

	self.frame:SetAttribute("unit", nil)

	super(self)
end

function SecureUnitFrame:GetDefaultSettings()
	local settings = super(self)

	settings.general = {
		unit = "player",
		visibility = "auto", -- "always", "combat", "ooc", "auto", "autocombat"
		interactive = true,
		class = "none"
	}

	return settings
end

function SecureUnitFrame:GetUnit()
	return self.settings.general.unit
end

function SecureUnitFrame:ConfigurationModeChanged(mode)
	super(self, mode)

	self:UpdateVisibility()
	self:UpdateInteractivity()
end

function SecureUnitFrame:UpdateVisibility(clear)
	local frame, settings = self.frame, self.settings.general

	UnregisterUnitWatch(frame)

	if clear then
		frame:Hide()
		return
	end

	if settings.class and settings.unit == "player" then
		if settings.class ~= "none" and settings.class ~= unitClass then
			frame:Hide()
			return
		end
	end

	local visibility, inCombat = settings.visibility, InCombatLockdown()

	if UnderHood.configMode or visibility == "always" or (visibility == "combat" and inCombat) or (visibility == "ooc" and not inCombat) then
		frame:Show()
	elseif visibility == "auto" then
		RegisterUnitWatch(frame)
	else
		frame:Hide()
	end
end

function SecureUnitFrame:OnEnterCombat()
	super(self)

	local frame, settings = self.frame, self.settings.general

	if settings.class and settings.unit == "player"  and settings.class ~= "none" and settings.class ~= unitClass then
		return
	elseif settings.visibility == "ooc" then
		frame:Hide()
	elseif settings.visibility == "combat" then
		frame:Show()
	elseif settings.visibility == "autocombat" then
		RegisterUnitWatch(frame)
	end
end

function SecureUnitFrame:OnLeaveCombat()
	super(self)

	local frame, settings = self.frame, self.settings.general

	if settings.class and settings.unit == "player" and settings.class ~= "none" and settings.class ~= unitClass then
		return
	elseif settings.visibility == "ooc" then
		frame:Show()
	elseif settings.visibility == "combat" then
		frame:Hide()
	elseif settings.visibility == "autocombat" then
		UnregisterUnitWatch(frame)
		frame:Hide()
	end
end

local UnderHood_SecureUnitFrame_DropDown = CreateFrame("Frame", "UnderHood_SecureUnitFrame_DropDown", UIParent, "UIDropDownMenuTemplate")

local function GetMenuForUnit(unit)
	if UnitIsUnit(unit, "player") then return "SELF" end
	if UnitIsUnit(unit, "vehicle") then return "VEHICLE" end
	if UnitIsUnit(unit, "pet") then return "PET" end
	if not UnitIsPlayer(unit) then return "RAID_TARGET_ICON" end

	local id = UnitInRaid(unit)

	if id then return "RAID_PLAYER", id end
	if UnitInParty(unit) then return "PARTY" end

	return "PLAYER"
end

local function MenuInitializer()
	local unit = UnderHood_SecureUnitFrame_DropDown.unit

	if not unit then return end

	local menu, id = GetMenuForUnit(unit)

	if menu then
		UnitPopup_ShowMenu(UnderHood_SecureUnitFrame_DropDown, menu, unit, nil, id)
	end
end

UIDropDownMenu_Initialize(UnderHood_SecureUnitFrame_DropDown, MenuInitializer, "MENU", nil)

local function MenuFunction(frame, unit)
	UnderHood_SecureUnitFrame_DropDown.unit = unit
	ToggleDropDownMenu(1, nil, UnderHood_SecureUnitFrame_DropDown, "cursor")
end

function SecureUnitFrame:UpdateInteractivity(turnOff)
	local frame, settings = self.frame, self.settings.general
	local interactive = settings.interactive and not turnOff

	if interactive then
		self.frame:SetAttribute("*type1", "target")
		self.frame:SetAttribute("*type2", "menu")
		self.frame:EnableMouse(true)
		self.frame:RegisterForClicks("AnyUp")

		self.frame.menu = MenuFunction

		ClickCastFrames[self.frame] = true
	else
		self.frame:SetAttribute("*type1", nil)
		self.frame:SetAttribute("*type2", nil)
		self.frame:RegisterForClicks()
		self.frame:EnableMouse(false)

		self.frame.menu = nil

		ClickCastFrames[self.frame] = nil
	end
end

local unitValues, unitValuesWithDefault

local classValues = { -- TODO: Locale
	none = "none",
	WARRIOR = "Warrior",
	PALADIN = "Paladin",
	HUNTER = "Hunter",
	ROGUE = "Rogue",
	PRIEST = "Priest",
	DEATHKNIGHT = "Death Knight",
	SHAMAN = "Shaman",
	MAGE = "Mage",
	WARLOCK = "Warlock",
	DRUID = "Druid",
}

local visibilityValues = {
	always = L["Always show"],
	combat = L["Show in combat"],
	ooc = L["Show out of combat"],
	auto = L["Show when unit exists"],
	autocombat = L["Show in combat when unit exists"],
}

function SecureUnitFrame.static:GetUnitOptionValues(includeDefault)
	if not unitValues then
		unitValues = {
			player = L["Player"],
			target = L["Target"],
			focus = L["Focus"],
			pet = L["Pet"],
			vehicle = L["Vehicle"],
			targettarget = L["Target's target"],
			focustarget = L["Focus target"],
			pettarget = L["Pet target"],
			targettargettarget = L["Target of target's target"],
		}

		for i = 1, 4 do
			unitValues["party"..i] = L["Party"].." "..i
			unitValues["partypet"..i] = L["Party pet"].." "..i
			unitValues["party"..i.."target"] = L["Party"].." "..i.." "..L["Target"]
			unitValues["partypet"..i.."target"] = L["Party pet"].." "..i.." "..L["Target"]
		end

		unitValuesWithDefault = UnderHood:CopyTable(unitValues)

		unitValuesWithDefault.default = L["Default"]
	end

	return includeDefault and unitValuesWithDefault or unitValues
end

function SecureUnitFrame:UnitChanged(unit)
	self.settings.general.unit = unit

	self.frame:SetAttribute("unit", unit)

	self:UpdateInteractivity()
end

function SecureUnitFrame:CreateOptions()
	local options = super(self)

	options.general = {
		type = "group",
		order = 10,
		name = L["General"],
		args = {
			unit = {
				type = "select",
				order = 10,
				name = L["Unit"],
				values = function() return SecureUnitFrame:GetUnitOptionValues() end,
				get = function() return self.settings.general.unit end,
				set = function(info, value)
					if value ~= "player" then
						self.settings.general.class = "none"
					end
					self:UnitChanged(value)
				end,
			},
			visibility = {
				type = "select",
				order = 20,
				name = L["Visibility"],
				values = visibilityValues,
				get = function() return self.settings.general.visibility end,
				set = function(info, value)
					self.settings.general.visibility = value
					self:UpdateVisibility()
				end,
			},
			interactive = {
				type = "toggle",
				order = 30,
				name = L["Interactive"],
				get = function() return self.settings.general.interactive end,
				set = function(info, value)
					self.settings.general.interactive = value
					self:UpdateInteractivity()
				end,
			},
			class = {
				type = "select",
				order = 30,
				name = L["Texts/Class"],
				values = classValues,
				get = function() return self.settings.general.class end,
				set = function(info, value)
					self.settings.general.class = value
					self:UpdateVisibility()
				end,
				hidden = function()
					return self.class ~= "Bar"
				end,
				disabled = function()
					return self.settings.general.unit ~= "player"
				end,
			},
		},
	}

	return options
end
