local anonymousCount = 1

local classes = {}

local OO = {}
OO.__index = OO

UnderHood.OO = OO

local assert = assert
local error = error
local rawget = rawget
local rawset = rawset
local setfenv = setfenv
local setmetatable = setmetatable
local strformat = string.format
local type = type

local function instanceOf(instance, class)
	if type(instance) ~= "table" or type(instance.class) ~= "function" then return false end
	if type(class) == "string" then class = classes[class] end
	if type(class) ~= "table" or type(class.super) ~= "function" then return false end

	local iClass = instance:class()

	while iClass do
		if iClass == class then
			return true
		end

		iClass = iClass:super()
	end
end

local function inheritsFrom(class, base)
	if type(class) == "string" then class = classes[class] end
	if type(base) == "string" then base = classes[base] end
	if type(class) ~= "table" or type(class.super) ~= "function" then return false end
	if type(base) ~= "table" or type(base.super) ~= "function" then return false end

	while class do
		if class == base then
			return true
		end

		class = class:super()
	end
end

local function mustImplement(class, method)
	assert(type(method) == "string" and #method > 0, strformat("Method name for %s:mustImplement() must be nonempty string.", class.name))

	if rawget(class.methods, method) then
		error(strformat("Method %s:%s is already implemented and can't be marked as abstract.", class.name, method))
	end

	local v = function(self)
		error(strformat("Method %s must be implemented in classes derived from %s.", method, class.name))
	end

	rawset(class.methods, method, v)
end

function OO:IsClass(class)
	if type(class) == "string" then class = classes[class] end

	return type(class) == "table" and type(class.name) == "string" and type(class.super) == "function"
end

function OO:ClassOf(obj)
	if type(obj) ~= "table" or type(obj.class)  ~= "function" then return end

	local class = obj:class()

	if self:IsClass(class) then return class end
end

function OO:IsInstance(obj)
	return self:ClassOf(obj) and true or false
end

function OO:IsInstanceOf(obj, class)
	if type(class) == "string" then class = classes[class] end
	if type(class) ~= "table" or type(class.super) ~= "function" then return false end
	if type(obj) ~= "table" or type(obj.class)  ~= "function" then return nil end

	return instanceOf(obj, class)
end

function OO:InheritsFrom(class, base)
	return inheritsFrom(class, base)
end

function OO:NewClass(name, base)
	if not base and type(name) == "table" and name.methods then
		base = name
		name = nil
	end

	if type(base) == "string" then
		if not classes[base] then
			error(strformat("Class '%s' is not registered.", base))
		end

		base = classes[base]
	end

	if not name then
		name = "Anonymous"..anonymousCount
		anonymousCount = anonymousCount + 1
	end

	if classes[name] then
		error(strformat("Class '%s' is already registered.", name))
	end

	--print(strformat("Defining class %s based on %s.", name, (base and base.name) or "none"))

	local class = { static = {}, methods = setmetatable({}, { __index = (base and base.methods) or nil }) }
	local class_mt = { __index = class.methods }

	class.name = name
	class.static.super = function() return base end
	class.static.inheritsFrom = inheritsFrom
	class.static.mustImplement = mustImplement
	class.methods.class = function() return class end
	class.methods.instanceOf = instanceOf

	function class.static:new(...)
		local inst = setmetatable({}, class_mt)

		if type(inst.init) == "function" then
			inst:init(...)
		end

		return inst
	end

	local function extendMethod(inst, k, v)
		if type(v) == "function" then
			--print(strformat("Defining method %s of class %s", k, name))
			local superMethod = base and base.methods[k] --rawget(base.methods, k)

			if superMethod then
				--print(strformat("Wrapping method %s of class %s", k, name))
				setfenv(v, setmetatable({ super = superMethod }, { __index = getfenv(v) }))
			end

			inst = class.methods
		end

		rawset(inst, k, v)
	end

	setmetatable(class, { __index = class.static, __newindex = extendMethod, __call = function(self, ...) return self.static:new(...) end })

	classes[name] = class

	return class
end

function OO:GetClass(name)
	return classes[name]
end
