local UnderHood = UnderHood
local L = UnderHood.L

local error = error
local next = next
local pairs = pairs
local setmetatable = setmetatable
local strfind = string.find
local strformat = string.format
local tonumber = tonumber
local type = type
local twipe = table.wipe

local idToClass = {}
local idToTitle = {}
local classToId = {}
local frameCache = {}
local activeFrames = {}
local frameNames = {}

local frameClass
local nextFrameNumber = 1
local frameName = "frame1"

local function ValidateFrameName(frame, newName)
	if strfind(newName, "[ \(\[\]\)\.\:]") then
		return L["Invalid character in name."]
	end

	for k, v in pairs(UnderHood.db.profile.frames) do
		if k == newName and v ~= frame then
			return L["Frame with the same name already exists."]
		end
	end

	return true
end

local options = {
	type = "group",
	name = L["Frames"],
	order = 3,
	args = {
		header1 = {
			type = "description",
			name = L["You can create new frame by typing frame name in the box below and clicking appropriate button:"].."\n",
			order = 1,
		},
		name = {
			type = "input",
			name = L["New frame name:"],
			order = 2,
			width = "full",
			validate = function(info, value) return ValidateFrameName(nil, value) end,
			get = function() return frameName end,
			set = function(info, value) frameName = value end,
		},
		--[[
		header2 = {
			type = "description",
			name = "\n"..L["You can enable or disable individual frames by toggling checks below:"].."\n",
			order = 4,
		},
		--]]
		byName = {
			type = "group",
			name = "By Name",
			order = 5,
			args = {},
		},
		byType = {
			type = "group",
			name = "By Type",
			order = 5,
			args = {},
		},
	},
	plugins = { classes = {}, frames = {} },
}

UnderHood:AddToplevelOptionsSection("frames", options)

function UnderHood:RegisterFrameClass(id, title, class)
	if type(id) ~= "string" then error("FrameClass id must be a string.") end
	if type(title) ~= "string" or #title == 0 then error("FrameClass title must be a nonempty string.") end
	if idToClass[id] then error(strformat("FrameClass <%s> is already registered.", id)) end

	if not class then
		class = self.OO:NewClass(id, "Frame")
	else
		if type(class) == "string" then
			class = self.OO:GetClass(class)
		end

		if not class or not self.OO:IsClass(class) or not class:inheritsFrom("Frame") then
			error(strformat("Parent frame class for %s must be Frame or one of it's descendants.", id))
		end

		class = self.OO:NewClass(id, class)
	end

	idToClass[id] = class
	classToId[class] = id
	idToTitle[id] = title

	options.plugins.classes["create_"..id] = {
		type = "execute",
		name = strformat(L["Create %s"], title),
		order = 3,
		func = function() UnderHood:CreateFrame(id) end,
	}

	return class
end

function UnderHood:IterateFrameClassTitles()
	return pairs(idToTitle)
end

local function AcquireFrame(id, settings)
	if type(id) ~= "string" then error("FrameClass id must be a string.") end

	local frameClass = idToClass[id]

	if not frameClass then
		UnderHood:Print(L["Unable to create frame of class '%s' because this class is not registered. Please ensure that corresponding module is enabled and loaded."], frameClass)
		return
	end

	local frameClassCache = frameCache[frameClass]
	local frame

	if frameClassCache then
		frame = next(frameClassCache)

		if frame then
			frameClassCache[frame] = nil
		end
	end

	if not frame then
		frame = frameClass:new()
	end

	if not settings then
		settings = frame:GetDefaultSettings()
	else
		settings = frame:UpdateSettings(settings)
	end

	frame:OnAcquire(settings)

	return frame
end

local function ReleaseFrame(frame)
	local frameClass = UnderHood.OO:ClassOf(frame)

	if not frameClass or not classToId[frameClass] then
		error("Invalid object is passed to ReleaseFrame().")
	end

	local frameClassCache = frameCache[frameClass]

	if not frameClassCache then
		frameClassCache = setmetatable({}, { __mode = 'k' })
		frameCache[frameClass] = frameClassCache
	end

	if frameClassCache[frame] then
		error(strformat("Attempt to release frame of type <%s> that was already released.", classToId[frameClass]))
	end

	frame:OnRelease()

	frameClassCache[frame] = true
end

local function fixNextFrameNumber()
	local pattern = "frame(%d+)"
	local frames = UnderHood.db.profile.frames
	local seed = 1
	local num

	for k in pairs(frames) do
		_, _, num = strfind(k, pattern)

		if num and tonumber(num) >= seed then
			seed = tonumber(num) + 1
		end
	end

	nextFrameNumber = seed
	frameName = "frame"..seed
end

local function RenameFrame(frame, newName)
	local name = frameNames[frame]
	local descr = UnderHood.db.profile.frames[name]
	--local frameOptions = options.plugins.frames[name]
	local frameOptions = options.args.byName.args[name]

	UnderHood.db.profile.frames[newName] = descr
	UnderHood.db.profile.frames[name] = nil

	--options.plugins.frames[newName] = frameOptions
	--options.plugins.frames[name] = nil
	options.args.byName.args[newName] = frameOptions
	options.args.byName.args[name] = nil

	local frameClass = UnderHood.OO:ClassOf(frame)
	local id = classToId[frameClass]

	options.args.byType.args[id].args[newName] = frameOptions
	options.args.byType.args[id].args[name] = nil

	activeFrames[name] = nil
	activeFrames[newName] = frame
	frameNames[frame] = newName

	fixNextFrameNumber()

	for _, v in pairs(activeFrames) do
		v:FrameNameChanged(name, newName)
	end

	UnderHood:NotifyOptionsChanged()
end

local function AdoptFrame(name, frame)
	activeFrames[name] = frame
	frameNames[frame] = name

	local frameOptionsContent = frame:GetOptions()
	local frameOptions = {
		type = "group",
		name = function() return frameNames[frame] end,
		args = {
			name = {
				type = "input",
				name = L["Frame name"],
				order = 1,
				width = "full",
				validate = function(info, value) return ValidateFrameName(frame, value) end,
				get = function() return frameNames[frame] end,
				set = function(info, value) RenameFrame(frame, value) end,
			},
			delete = {
				type = "execute",
				name = L["Delete"],
				order = 2,
				confirm = true,
				confirmText = L["Do you really want to delete this frame?"],
				func = function() UnderHood:DeleteFrame(frame) end,
			},
			duplicate = {
				type = "execute",
				name = L["Duplicate"],
				order = 3,
				func = function() UnderHood:DuplicateFrame(frame) end,
			},
		},
	}

	if frameOptionsContent then
		for k, v in pairs(frameOptionsContent) do
			frameOptions.args[k] = v
		end

		local frameClass = UnderHood.OO:ClassOf(frame)
		local id = classToId[frameClass]
		local title = idToTitle[id]

		--options.plugins.frames[name] = frameOptions
		options.args.byName.args[name] = frameOptions

		local typeGroup = options.args.byType.args[id]

		if not typeGroup then
			typeGroup = {
				type = "group",
				name = title,
				args = {},
			}

			options.args.byType.args[id] = typeGroup
		end

		typeGroup.args[name] = frameOptions
	end
end

function UnderHood:CreateFrame(class, name)
	local frame = AcquireFrame(class)
	local name = name or frameName

	if not self.db.profile.frames then
		self.db.profile.frames = {}
	end

	self.db.profile.frames[name] = {
		class = class,
		enable = true,
		settings = frame:GetSettings(),
	}

	AdoptFrame(name, frame)
	fixNextFrameNumber()

	frame:PositionFrame()

	self:NotifyOptionsChanged()
end

function UnderHood:DeleteFrame(frame)
	local name = frameNames[frame]

	if not name then error("Attempt to delete unregistered frame.") end

	self.db.profile.frames[name] = nil

	--options.plugins.frames[name] = nil
	local frameClass = UnderHood.OO:ClassOf(frame)
	local id = classToId[frameClass]

	options.args.byName.args[name] = nil
	options.args.byType.args[id].args[name] = nil

	if not next(options.args.byType.args[id].args) then
		options.args.byType.args[id] = nil
	end

	activeFrames[name] = nil
	frameNames[frame] = nil

	for _, v in pairs(activeFrames) do
		v:FrameNameChanged(name, nil)
	end

	ReleaseFrame(frame)
	fixNextFrameNumber()

	self:NotifyOptionsChanged()
end

function UnderHood:DuplicateFrame(originalFrame)
	local originalName = frameNames[originalFrame]

	if not originalName then error("Attempt to duplicate unregistered frame.") end

	local originalDescr = self.db.profile.frames[originalName]
	local newDescr = self:CopyTable(originalDescr)
	local newName = frameName

	self.db.profile.frames[newName] = newDescr

	local newFrame = AcquireFrame(newDescr.class, newDescr.settings)

	newDescr.settings = newFrame:GetSettings()

	AdoptFrame(newName, newFrame)

	fixNextFrameNumber()

	newFrame:PositionFrame()

	self:NotifyOptionsChanged()
end

function UnderHood:ReloadFrames()
	for _, frame in pairs(activeFrames) do
		ReleaseFrame(frame)
	end

	activeFrames = twipe(activeFrames)
	frameNames = twipe(frameNames)

	--options.plugins.frames = twipe(options.plugins.frames)
	options.args.byName.args = twipe(options.args.byName.args)
	options.args.byType.args = twipe(options.args.byType.args)

	for name, descr in pairs(self.db.profile.frames) do
		local frame = AcquireFrame(descr.class, descr.settings)

		-- frame MAY BE nil, if corresponding module is not installed/enabled.
		-- In that case we just skip it.

		if frame then
			descr.settings = frame:GetSettings()

			AdoptFrame(name, frame)
		end
	end

	for _, frame in pairs(activeFrames) do
		frame:PositionFrame()
	end

	fixNextFrameNumber()

	self:NotifyOptionsChanged()
end

function UnderHood:GetFrame(name)
	return activeFrames[name]
end

function UnderHood:IterateFrames()
	return pairs(activeFrames)
end
