local UnderHood = UnderHood
local L = UnderHood.L

local CreateFrame = CreateFrame
local floor = math.floor
local strformat = string.format

local Frame = UnderHood.OO:NewClass("Frame")

function Frame:init(frameType, inheritsFrom)
	frameType = frameType or "Frame"

	self.frame = CreateFrame(frameType, nil, UIParent, inheritsFrom)
	self.frame.uhFrame = self
	self.frame:Hide()
end

function Frame:OnAcquire(settings)
	self.settings = settings
end

function Frame:OnRelease()
	self.frame:Hide()
	self.frame:ClearAllPoints()

	self.settings = nil
	self.options = nil
end

function Frame:GetFrame()
	return self.frame
end

function Frame:GetSettings()
	return self.settings
end

function Frame:GetDefaultSettings()
	return {
		alpha = {
			inherit = true,
			enablePet = true,
			enableFocus = true,
			enableTarget = true,
			enableCasting = true,
			enableRegen = true,
			combat = 1,
			ooc = 0,
			regen = 0.75,
			target = 0.75,
			focus = 0.75,
			pet = 0.75,
			casting = 0.75,
		},
		positionAndSize = {
			x = 0,
			y = 0,
			point = "CENTER",
			targetPoint = "CENTER",
			targetFrame = ":uiparent:",
			width = 20,
			height = 100,
			scale = 1,
		},
	}
end

function Frame:UpdateSettings(settings)
	return settings
end

function Frame:FrameNameChanged(oldName, newName)
	local pAs = self.settings.positionAndSize

	if pAs.targetFrame == oldName then
		if not newName then
			-- frame that we were anchored to was removed. reanchoring to UIParent

			pAs.x = floor(self.frame:GetLeft() + 0.5)
			pAs.y = floor(UIParent:GetHeight() - self.frame:GetTop() + 0.5)
			pAs.point = "TOPLEFT"
			pAs.targetPoint = "TOPLEFT"
			pAs.targetFrame = ":uiparent:"

			self:PositionFrame()
		else
			pAs.targetFrame = newName
		end
	end
end

function Frame:ConfigurationModeChanged(mode)
end

function Frame:PositionFrame(changedElementName)
	local frame = self:GetFrame()
	local settings = self.settings.positionAndSize
	local strata = settings.strata or "MEDIUM"
	local targetFrame = UIParent

	if settings.targetFrame ~= ":uiparent:" then
		targetFrame = UnderHood:GetFrame(settings.targetFrame)

		targetFrame = (targetFrame and targetFrame:GetFrame()) or UIParent
	end

	frame:ClearAllPoints()

	frame:SetPoint(settings.point, targetFrame, settings.targetPoint, settings.x, settings.y)
	frame:SetWidth(settings.width)
	frame:SetHeight(settings.height)
	frame:SetScale(settings.scale)
	frame:SetFrameStrata(strata)
end

function Frame:UpdateAlpha()
	local alpha = UnderHood.globalAlpha
	local as = self.settings.alpha

	if not as.inherit then
		alpha = UnderHood:CalculateAlpha(as)
	end

	self.frame:SetAlpha(alpha)
end

local validPoints = {
	CENTER = L["Center"],
	TOP = L["Top"],
	LEFT = L["Left"],
	RIGHT = L["Right"],
	BOTTOM = L["Bottom"],
	TOPLEFT = L["Top left"],
	TOPRIGHT = L["Top right"],
	BOTTOMLEFT = L["Bottom left"],
	BOTTOMRIGHT = L["Bottom right"],
}

local validStratas = {
	BACKGROUND = L["Background"],
	LOW = L["Low"],
	MEDIUM = L["Medium"],
	HIGH = L["High"],
	DIALOG = L["Dialog"],
	TOOLTIP = L["Tooltip"],
}

function Frame:CreateOptions()
	return {
		alpha = {
			name = L["Alpha settings"],
			type = "group",
			order = -20,
			get = function(info) return self.settings.alpha[info[#info]] end,
			set = function(info, value) self.settings.alpha[info[#info]] = value; self:UpdateAlpha() end,
			disabled = function(info) if info[#info] ~= "alpha" then return self.settings.alpha.inherit end end,
			args = {
				inherit = {
					type = "toggle",
					name = L["Use global Alpha settings"],
					width = "full",
					order = 0,
					disabled = false,
				},
				header1 = {
					type = "header",
					name = L["Alpha settings for this frame"],
					order = 1,
				},
				combat = {
					name = L["Combat"],
					type = "range",
					min = 0,
					max = 1,
					step = 0.01,
					bigStep = 0.01,
					order = 2,
				},
				ooc = {
					name = L["Out of Combat"],
					type = "range",
					min = 0,
					max = 1,
					step = 0.01,
					bigStep = 0.01,
					order = 3,
				},
				regen = {
					name = L["Regenerating"],
					type = "range",
					min = 0,
					max = 1,
					step = 0.01,
					bigStep = 0.01,
					order = 4,
				},
				casting = {
					name = L["Casting"],
					type = "range",
					min = 0,
					max = 1,
					step = 0.01,
					bigStep = 0.01,
					order = 5,
				},
				target = {
					name = L["Have target"],
					type = "range",
					min = 0,
					max = 1,
					step = 0.01,
					bigStep = 0.01,
					order = 6,
				},
				focus = {
					name = L["Have focus"],
					type = "range",
					min = 0,
					max = 1,
					step = 0.01,
					bigStep = 0.01,
					order = 7,
				},
				pet = {
					name = L["Have pet"],
					type = "range",
					min = 0,
					max = 1,
					step = 0.01,
					bigStep = 0.01,
					order = 8,
				},
				header2 = {
					name = "\n"..L["You can enable or disable some conditions by checking or unchecking boxes below:"].."\n",
					type = "description",
					order = 9,
				},
				enableCasting = {
					name = L["Enable |cffffff00Casting|r"],
					type = "toggle",
					order = 10,
				},
				enableRegen = {
					name = L["Enable |cffffff00Regenerating|r"],
					type = "toggle",
					order = 11,
				},
				enableTarget = {
					name = L["Enable |cffffff00Have Target|r"],
					type = "toggle",
					order = 12,
				},
				enableFocus = {
					name = L["Enable |cffffff00Have Focus|r"],
					type = "toggle",
					order = 13,
				},
				enablePet = {
					name = L["Enable |cffffff00Have Pet|r"],
					type = "toggle",
					order = 14,
				},
			},
		},
		positionAndSize = {
			type = "group",
			name = L["Position and Size"],
			order = -10,
			get = function(info) return self.settings.positionAndSize[info[#info]] end,
			set = function(info, value) self.settings.positionAndSize[info[#info]] = value; self:PositionFrame(info[#info]) end,
			args = {
				headerPosition = {
					type = "header",
					name = L["Position"],
					order = 0,
				},
				point = {
					type = "select",
					name = L["Frame anchor point"],
					order = 10,
					values = validPoints,
				},
				targetPoint = {
					type = "select",
					name = L["Target frame anchor point"],
					order = 20,
					values = validPoints,
				},
				targetFrame = {
					type = "select",
					name = L["Target frame"],
					order = 30,
					values = function()
						local list = {}

						list[":uiparent:"] = L["-- Screen --"]

						for k in UnderHood:IterateFrames() do
							list[k] = k
						end

						return list
					end,
					validate = function(info, value)
						if value == ":uiparent:" then return true end

						local frame = UnderHood:GetFrame(value)
						local originalValue = value

						if frame == self then
							return L["You can't anchor frame to itself."]
						end

						while frame do
							if frame == self then
								return strformat(L["You can't anchor to frame '%s' because it will create circular anchoring ('%s' is anchored to this frame)."], originalValue, value)
							end

							value = frame:GetSettings().positionAndSize.targetFrame

							if value == ":uiparent:" then break end

							frame = UnderHood:GetFrame(value)
						end

						return true
					end,
				},
				strata = {
					name = L["Strata"],
					type = "select",
					order = 35,
					values = validStratas,
					get = function() return self.settings.positionAndSize.strata end,
					set = function(info, value) self.settings.positionAndSize.strata = value; self.frame:SetFrameStrata(value) end,
				},
				x = {
					type = "range",
					name = L["X"],
					order = 40,
					min = -800,
					max = 800,
					step = 1,
				},
				y = {
					type = "range",
					name = L["Y"],
					order = 50,
					min = -800,
					max = 800,
					step = 1,
				},
				headerSize = {
					type = "header",
					name = L["Size"],
					order = 60,
				},
				width = {
					type = "range",
					name = L["Width"],
					order = 70,
					min = 1,
					max = 500,
					step = 1,
				},
				height = {
					type = "range",
					name = L["Height"],
					order = 80,
					min = 1,
					max = 500,
					step = 1,
				},
				scale = {
					name = L["Scale"],
					type = "range",
					order = 90,
					min = 0.01,
					max = 5,
					step = 0.01,
				},
			},
		},
	}
end

function Frame:GetOptions()
	if not self.options then
		self.options = self:CreateOptions()
	end

	return self.options
end

function Frame:OnEnterCombat()
end

function Frame:OnLeaveCombat()
end

