-- Init
local frame = CreateFrame("FRAME", "TurnInEverything_Frame");
frame:RegisterEvent("GOSSIP_SHOW");
frame:RegisterEvent("QUEST_GREETING");
frame:RegisterEvent("QUEST_PROGRESS");	
frame:RegisterEvent("QUEST_COMPLETE");
frame:RegisterEvent("MODIFIER_STATE_CHANGED");
-- Saved globals
tie_PermDisable = false;
tie_TournReward = 0;
tie_DEBUG = false;
-- Locals
local tie_TempDisable = false;
local tie_WasGossip = true;
-- Forward function declarations (to limit scope)
local tie_CompletedQuests;
local tie_QuestGreeting;
local tie_GossipShow;
local tie_QuestProgress;
local tie_QuestComplete;
local tie_ModiferChange;
local tie_isDisabled;

-- Event handler
local function tie_EventHandler(self, event)
	if (tie_DEBUG) then
		print('tie_eventHandler:', event);
		print('tie_PermDisable', tie_PermDisable);
		print('tie_TempDisable', tie_TempDisable);
	end

	-- Track modifier changes
	if (event == "MODIFIER_STATE_CHANGED") then
		tie_ModiferChange();
		return;
	
	-- Track quest completion
	elseif (event == "QUEST_COMPLETE") then
		tie_QuestComplete();
		return;

	-- Track quest progress
	elseif (event == "QUEST_PROGRESS") then
		tie_QuestProgress();
		return;

	-- Track new gossip/greeting frames
	elseif (event == "GOSSIP_SHOW" or event == "QUEST_GREETING") then
		-- Do not process while disabled
		if (tie_isDisabled()) then
			return;
		end
		
		if (event == "GOSSIP_SHOW") then
			tie_GossipShow();
		else
			tie_QuestGreeting();
		end
		return;
	end
end
frame:SetScript('OnEvent', tie_EventHandler);

-- Slash commands
SLASH_TURNINEVERYTHING1, SLASH_TURNINEVERYTHING2 = '/tie', '/turnineverything';
SlashCmdList["TURNINEVERYTHING"] = function(msg, editbox)
	local outmsg = "Unrecognized command";
	msg = string.lower(msg);
	
	-- Master enable/disable
	if (msg == 'disable') then
		tie_PermDisable = true;
		outmsg = "Disabled";
	elseif (msg == 'enable') then
		tie_PermDisable = false;
		outmsg = "Enabled";
	
	-- Allow automatic acceptance of Argent Tournament rewards
	elseif (msg == 'tourn' or string.find(msg, '^tourn%s+%a+')) then
		if (string.find(msg, '%swrit') or string.find(msg, '2$')) then
			tie_TournReward = 1;
		elseif (string.find(msg, '%spurse') or string.find(msg, '1$')) then
			tie_TournReward = 2;
		elseif (msg ~= 'tourn') then
			tie_TournReward = 0;
		end
		
		local t = {'Argent Tournament Reward: '};
		if (tie_TournReward == 1) then
			table.insert(t, 'Writ');
		elseif (tie_TournReward == 2) then
			table.insert(t, 'Purse');
		else
			table.insert(t, 'None');
		end
		outmsg = table.concat(t);
	
	-- Debug
	elseif (msg == 'debug') then
		if (tie_DEBUG) then
			tie_DEBUG = false;
			outmsg = "Debug Disabled";
		else
			tie_DEBUG = true;
			outmsg = "Debug Enabled";
		end
	end
	print("TurnInEverything: " .. outmsg);
end

-- Return a list of completed quests from the quest log
function tie_CompletedQuests()
	local i = 1
	local completedQuests = {};
	while GetQuestLogTitle(i) do
		-- Grab the quest details
		local questTitle, level, questTag, suggestedGroup, isHeader, isCollapsed, isComplete, isDaily, questID = GetQuestLogTitle(i);

		-- Record the names of completed quests
		if (not isHeader and isComplete == 1) then
			questTitle = string.gsub(questTitle, '%[%d+%a?%+?%]%s+', '');
			table.insert(completedQuests, questTitle);
			if (tie_DEBUG) then
				print('tie_CompletedQuests: Found completed quest:', questTitle);
			end
		end
		
		-- Loop control
		i = i + 1;
	end
	return completedQuests;
end

-- Check each active quest
function tie_QuestGreeting()
	-- Remember that we're in a quest greeting frame
	tie_WasGossip = false;

	-- If there are any active quests
	numQuests = GetNumActiveQuests();
	if (numQuests ~= nil) then
	
		-- See what quests are complete
		local completedQuests = tie_CompletedQuests();

		-- Check each active quest
		for i = 1, numQuests do
			local title = GetActiveTitle(i);
			if (tie_DEBUG) then
				print('tie_QuestGreeting: Found active quest:', title);
			end
	
			-- Check each quest against our completed list
			for j,v in ipairs(completedQuests) do
				if (v == title) then
					if (tie_DEBUG) then
						print('tie_QuestGreeting: Matched completed quest:', v);
					end
					
					-- Select the quest and stop (i.e. process at most one completed quest per click)
					SelectActiveQuest(i);
					return;
				end
			end
		end
	end
end

-- Check each active quest
function tie_GossipShow()
	-- Remember that we're in a gossip frame
	tie_WasGossip = true;

	-- See what quests are complete
	local completedQuests = tie_CompletedQuests();

	-- Check each active quest
	local activeQuests = { GetGossipActiveQuests() };
	local i = 1;
	while(activeQuests[i]) do
		if (tie_DEBUG) then
			print('tie_GossipShow: Found active quest:', activeQuests[i]);
		end

		-- Check each quest against our completed list
		for j,v in ipairs(completedQuests) do
			if (v == activeQuests[i]) then
				if (tie_DEBUG) then
					print('tie_GossipShow: Matched completed quest:', v);
				end
				
				-- Select the quest and stop (i.e. process at most one completed quest per click)
				SelectGossipActiveQuest(((i - 1) / 4) + 1);
				return;
			end
		end

		-- Data from GetGossipActiveQuests() comes in triplets
		i = i + 4;
	end
end

-- If this quest is completable, click continue and spew out text until we get to the completion screen
function tie_QuestProgress()
	-- Display the progress text
	DEFAULT_CHAT_FRAME:AddMessage(GetTitleText() .. ' Progress: ' .. GetProgressText(), 1, 0.98, 0.8);
	
	-- Do not complete while disabled
	if (tie_isDisabled()) then
		return;
	end

	-- Bail if the quest is not completable
	-- This can happen with duplicate quest names or other state weirdness
	if(not IsQuestCompletable()) then
		DeclineQuest();
		return;
	end

	-- Complete the quest
	CompleteQuest();
end

-- Collect our reward when possible, then cleanup and stop listening
function tie_QuestComplete()
	-- Display the reward text
	DEFAULT_CHAT_FRAME:AddMessage(GetTitleText() .. ' Reward: ' .. GetRewardText(), 1, 0.98, 0.8);

	-- Do not accept the reward if disabled
	if (tie_isDisabled()) then
		return;
	end
	
	-- Close the parent frame to avoid race issues when there are multiple quests
	if (tie_WasGossip) then
		CloseGossip();
	else
		CloseQuest();
	end

	-- Choose a reward (if we can)
	local reward = 0;
	if (GetNumQuestChoices() < 2) then
		reward = 1;
	-- Always grab the selected Argent Tournament reward, if set
	elseif (GetNumQuestChoices() == 2 and "Champion's Writ" == GetQuestItemInfo("choice", 1) and "Champion's Purse" == GetQuestItemInfo("choice", 2)) then
		if (tie_TournReward >= 1 and tie_TournReward <= 2) then
			reward = tie_TournReward;
		end
	end
	
	-- Accept the selected reward (if possible and not debugging)
	if (tie_DEBUG) then
		print('tie_QuestComplete: Will not claim rewards in debug mode. Would have selected: ' .. reward);
	elseif (reward >= 1 and reward >= GetNumQuestChoices()) then
		GetQuestReward(reward);
	end
end

-- Temporarily disable processing if the hotkey is pressed
-- Currently hard-coded to shift, but abstracted to a function should I ever to to writing a UI to allow user configuration
function tie_ModiferChange()
	if (IsShiftKeyDown()) then
		tie_TempDisable = true;
	else
		tie_TempDisable = false;
	end
end

-- Return true if TIE is disabled for any reason
function tie_isDisabled()
	if (tie_TempDisable or tie_PermDisable) then
		return true;
	end
	return false;
end
