--[[

SpyGnome, World of Warcraft addon to gather addon version information from your party players.
Copyright (C) 2007 Rabbit

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

]]

local comm = AceLibrary("AceComm-2.0")
local dewdrop = AceLibrary("Dewdrop-2.0")
local tablet = AceLibrary("Tablet-2.0")
local L = AceLibrary("AceLocale-2.2"):new("SpyGnome")

local _G = getfenv(0)

local Rock = _G.Rock
local rockComm = type(Rock) == "table" and type(Rock.HasLibrary) == "function" and Rock:HasLibrary("LibRockComm-1.0") and Rock("LibRockComm-1.0") or nil

local addons
local responseTable
local versionColor = {}

local options = {
	type = "group",
	args = {},
}

SpyGnome = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "FuBarPlugin-2.0")
local SpyGnome = SpyGnome

SpyGnome.revision = tonumber(("$Revision: 48 $"):sub(12, -3))
SpyGnome.hasNoColor = true
SpyGnome.hasIcon = "Interface\\Icons\\INV_Misc_Spyglass_01"

local new, del
do
	local cache = setmetatable({},{__mode='k'})
	function new()
		local t = next(cache)
		if t then
			cache[t] = nil
			return t
		else
			return {}
		end
	end
	function del(t)
		if type(t) ~= "table" then return end
		for k in pairs(t) do
			t[k] = nil
		end
		cache[t] = true
		return nil
	end
end

local hexColors = {
	WTF = "|cffa0a0a0"
}
for k, v in pairs(RAID_CLASS_COLORS) do
	hexColors[k] = "|cff" .. string.format("%02x%02x%02x", v.r * 255, v.g * 255, v.b * 255)
end
local coloredNames = setmetatable({}, {__index =
	function(self, key)
		if type(key) == "nil" then return nil end
		local class = select(2, UnitClass(key)) or "WTF"
		self[key] = hexColors[class]  .. key .. "|r"
		return self[key]
	end
})

local function cleanVersion(input)
	if type(input) ~= "string" then return input end
	local version = input
	-- Remove any color codes
	version = version:gsub("(|c%x%x%x%x%x%x%x%x)", "")
	version = version:gsub("(|r)", "")
	-- Remove any SVN keywords
	if version:find("%$Revision: (%d+) %$") then
		version = version:gsub("%$Revision: (%d+) %$", "%1")
	elseif version:find("%$Rev: (%d+) %$") then
		version = version:gsub("%$Rev: (%d+) %$", "%1")
	elseif version:find("%$LastChangedRevision: (%d+) %$") then
		version = version:gsub("%$LastChangedRevision: (%d+) %$", "%1")
	end
	return version
end

-- Copied from AceComm who copied from BugGrabber.
local function GetAddonVersionString(addon)
	if not addon then return nil end

	local version
	local revision
	if type(Rock) == "table" and Rock:HasLibrary(addon, true) and Rock(addon).GetLibraryVersion then
		version, revision = Rock(addon):GetLibraryVersion()
		version = version .. "-" .. revision
	elseif type(Rock) == "table" and Rock:HasAddon(addon) then
		local addon = Rock:GetAddon(addon)
		revision = addon.revision
		version = addon.version
		if version then version = tostring(version) end
		if revision then revision = tostring(revision) end
		if revision and version and version:len() > 0 and not version:find(revision) then
			version = version .. "." .. revision
		end
		if not version and revision then version = revision end
	elseif type(LibStub) == "table" and type(LibStub.GetLibrary) == "function" and LibStub:GetLibrary(addon, true) then
		revision = select(2, LibStub:GetLibrary(addon, true))
		version = addon .. "-" .. revision
	elseif AceLibrary:HasInstance(addon, false) and type(AceLibrary(addon).GetLibraryVersion) == "function" then
		version, revision = AceLibrary(addon):GetLibraryVersion()
		version = version .. "-" .. revision
	else
		local _G_addon = _G[addon]
		if not _G_addon then
			_G_addon = _G[addon:match("^[^_]+_(.*)$")]
		end
		if type(_G_addon) == "table" then
			if rawget(_G_addon, "version") then version = _G_addon.version
			elseif rawget(_G_addon, "Version") then version = _G_addon.Version
			elseif rawget(_G_addon, "VERSION") then version = _G_addon.VERSION
			end
			if type(version) == "function" then version = tostring(select(2, pcall(version()))) end
			local revision = nil
			if rawget(_G_addon, "revision") then revision = _G_addon.revision
			elseif rawget(_G_addon, "Revision") then revision = _G_addon.Revision
			elseif rawget(_G_addon, "REVISION") then revision = _G_addon.REVISION
			elseif rawget(_G_addon, "rev") then revision = _G_addon.rev
			elseif rawget(_G_addon, "Rev") then revision = _G_addon.Rev
			elseif rawget(_G_addon, "REV") then revision = _G_addon.REV
			end
			if type(revision) == "function" then revision = tostring(select(2, pcall(revision()))) end

			if version then version = tostring(version) end
			if revision then revision = tostring(revision) end
			if type(revision) == "string" and type(version) == "string" and version:len() > 0 and not version:find(revision) then
				version = version .. "." .. revision
			end

			if not version and revision then version = revision end
		end

		if _G[addon:upper().."_VERSION"] then
			version = _G[addon:upper() .. "_VERSION"]
		end
		if _G[addon:upper().."_REVISION"] or _G[addon:upper().."_REV"] then
			local revision = _G[addon:upper() .. "_REVISION"] or _G[addon:upper().."_REV"]
			if type(revision) == "string" and type(version) == "string" and version:len() > 0 and not version:find(revision) then
				version = version .. "." .. revision
			end
			if (not version or version == "") and revision then version = revision end
		end
		if not version or version == "" then
			version = GetAddOnMetadata(addon, "Version")
		end
		if not version or version == "" then
			version = IsAddOnLoaded(addon) and L["Unknown"] or L["N/A"]
		end
	end
	return cleanVersion(version)
end

local function VersionCompare(a, b)
	if type(a) == "string" then a = a:trim() end
	if type(b) == "string" then b = b:trim() end
	if a == b then
		return 0
	elseif tonumber(a) and tonumber(b) then
		if tonumber(a) == tonumber(b) then
			return 0
		else
			return tonumber(b) > tonumber(a) and 1 or 2
		end
	else
		a = tostring(a):trim()
		b = tostring(b):trim()
		local numA = a:gsub("%D", "")
		local numB = b:gsub("%D", "")
		numA = tonumber(numA)
		numB = tonumber(numB)
		if type(numA) == "number" and type(numB) == "number" then
			if numA == numB then
				return 0
			else
				return numB > numA and 1 or 2
			end
		else
			if numB and not numA then
				return 1
			else
				return 2
			end
		end
	end
end

local myVersions = setmetatable({}, {__index =
	function(self, key)
		local value = GetAddonVersionString(key)
		self[key] = value
		return value
	end
})

function SpyGnome:GetVersionColor(addon, otherVersion)
	if type(versionColor[addon]) ~= "table" then
		versionColor[addon] = new()
	end

	if not versionColor[addon][otherVersion] then
		local color = nil
		if otherVersion == L["N/A"] or otherVersion == L["Unknown"] then
			color = "|cffff0000"
		else
			local res = VersionCompare(otherVersion, myVersions[addon])
			if res == 0 then -- Same version
				color = "|cff00ff00" -- Green
			elseif res == 1 then -- Mine is more recent
				color = "|cff68ccff" -- Blue
			elseif res == 2 then -- Other is more recent
				color = "|cffff0000" -- Red
			end
		end
		versionColor[addon][otherVersion] = color
	end
	return versionColor[addon][otherVersion]
end

function SpyGnome:OnInitialize()
	self:RegisterDB("SpyGnomeDB")
	self:RegisterDefaults("profile", {
		addons = {
			SpyGnome = true,
		},
	})

	self.version = (self.version or "1") .. "." .. (self.revision or "1")

	self:UpdateAddonList(true)
end

function SpyGnome:UpdateAddonList(dontQuery)
	addons = del(addons)
	addons = new()

	local db = self.db.profile.addons
	for k, v in pairs(db) do
		local exists = select(6, GetAddOnInfo(k))
		if db[k] and db[k] == true and exists ~= "MISSING" then
			table.insert(addons, k)
		end
	end

	-- Update in 10 seconds, so we get time to tick/untick a few more addons.
	if not dontQuery then
		self:ScheduleEvent("SpyGnomeDataUpdate", self.UpdateData, 10, self)
	end
end

function SpyGnome:OnEnable(first)
	if first then
		comm:RegisterAddonVersionReceptor(SpyGnome, "AddonPong")
		if rockComm then
			rockComm:AddAddonVersionReceptor(function(player, addon, version)
				self:AddonPong(player, addon, version)
			end)
		end
	end
	self:RegisterBucketEvent({"PARTY_MEMBERS_CHANGED", "RAID_ROSTER_UPDATE"}, 5, function()
		self:ScheduleEvent("SpyGnomeDataUpdate", self.UpdateData, 25, self)
	end)
	
	--self:RegisterEvent("CHAT_MSG_ADDON")
end

function SpyGnome:OnDataUpdate()
	if not addons then return end

	-- Query for new version revisions
	responseTable = del(responseTable)
	for i, addon in ipairs(addons) do
		comm:QueryAddonVersion(addon, "GROUP")
		if rockComm then
			rockComm:QueryAddonVersion(addon, "GROUP")
		end

		--SendAddonMessage("VQ1", addon, "RAID")
	end
end

do
	-- Version Query Protocol, revision 1
	--
	-- QUERY
	-- Queries are sent with a prefix of "VQ1", one addon message per addon to
	-- be queried, the message containing only the name of the addon.
	-- Example: SendAddonMessage("VQ1", "SpyGnome", "RAID")
	--
	-- RESPONSE
	-- Responses are sent with a prefix of "VR1". Responses contain the name of
	-- the addon and the version, separated by a sharp sign (#).
	-- Example: SendAddonMessage("VR1", "SpyGnome#1.43521", "WHISPER", sender)
	--

	local COMPATIBLE_PROTOCOL_VERSION = 1
	local replyFormatOne = "%s#%s" -- "Addon#Version"
	local p = UnitName("player")

	function SpyGnome:CHAT_MSG_ADDON(prefix, message, distribution, sender)
		if sender == p then return end

		if prefix == "VQ1" then -- Version Query, protocol revision 1
			local x = myVersions[message]
			if not x then return end
			SendAddonMessage("VR1", replyFormatOne:format(message, x), "WHISPER", sender)
		elseif prefix == "VR1" then -- Version Response, protocol revision 1
			local addon, version = select(3, message:find("^(%w+)#(.*)$"))
			if not addon or not version then return end
			if tonumber(protocol) > COMPATIBLE_PROTOCOL_VERSION then return end
			self:AddonPong(sender, addon, version)
		end
	end
end

function SpyGnome:AddonPong(player, addon, version)
	--self:Print("AddonPong from ["..tostring(player).."].")
	if type(responseTable) ~= "table" then
		responseTable = new()
	end
	if type(responseTable[player]) ~= "table" then
		responseTable[player] = new()
	end
	if type(version) == "boolean" or version == nil then
		version = version and L["Unknown"] or L["N/A"]
	end
	responseTable[player][addon] = cleanVersion(version)

	self:ScheduleEvent("SpyGnomeTooltipUpdate", self.UpdateTooltip, 3, self)
end

function SpyGnome:OnTooltipUpdate()
	if not addons then return end
	if responseTable then
		local tmp = new()
		table.insert(tmp, "text")
		table.insert(tmp, L["Player"])

		local columns = new()
		for i, v in ipairs(addons) do
			table.insert(tmp, string.format("%s%d", "text", i + 1))
			table.insert(tmp, v)
			columns[v] = i + 1
		end
		local cat = tablet:AddCategory("columns", #addons + 1, unpack(tmp))
		tmp = del(tmp)
		for player, playerAddons in pairs(responseTable) do
			if type(player) == "string" and type(playerAddons) == "table" then
				tmp = new()
				table.insert(tmp, "text")
				table.insert(tmp, coloredNames[player] or player)
				for addon, version in pairs(playerAddons) do
					if addon and columns[addon] ~= nil then
						table.insert(tmp, string.format("%s%d", "text", columns[addon]))
						table.insert(tmp, self:GetVersionColor(addon, tostring(version))..tostring(version).."|r")
					end
				end
				cat:AddLine(unpack(tmp))
				tmp = del(tmp)
			end
		end
		columns = del(columns)
		tablet:SetHint(L["|cffff0000Red|r is newer than yours, |cff0000ffBlue|r is older than yours, and |cff00ff00Green|r is the same version. \"N/A\" means the person does not have an addon, \"Unknown\" means that no version information could be found."])
	else
		tablet:SetHint(L["The SpyGnome is lurking just behind you, in the shadows. He has no new information for you at this time."])
	end
end

local function get(key)
	return SpyGnome.db.profile.addons[key]
end

local function set(key, val)
	SpyGnome.db.profile.addons[key] = val and true or nil
	SpyGnome:UpdateAddonList()
end

local addonsScanned = nil
function SpyGnome:OnMenuRequest(level)
	if not addonsScanned then
		local aa = AceLibrary("AceAddon-2.0")
		local numAddons = GetNumAddOns()
		for i = 1, numAddons do
			local name = GetAddOnInfo(i)
			local category = GetAddOnMetadata(name, "X-Category")
			if not category then
				category = L["Uncategorized"]
			else
				category = aa:GetLocalizedCategory(category)
			end
			if not options.args[category] then
				options.args[category] = {
					type = "group",
					name = category,
					desc = L["Addons in the %s category."]:format(category),
					pass = true,
					get = get,
					set = set,
					args = {},
				}
			end
			local myVer = myVersions[name]
			local n = myVer and (name .. " |cff797979(" .. myVer .. ")|r") or name
			options.args[category].args[name] = {
				type = "toggle",
				name = n,
				desc = L["Toggle whether to query for %s."]:format(n),
			}
		end
		addonsScanned = true
	end
	dewdrop:FeedAceOptionsTable(options)
	if level == 1 then
		dewdrop:AddLine()
	end
end

