-- Addon
local modName = "SpellSpy";
local core = CreateFrame("Frame",modName,UIParent);

-- Variables
local spellList = {};
local filter;

-- Constants
local POWERTYPE_NAMES = { [0] = MANA, [1] = RAGE, [2] = FOCUS, [3] = ENERGY, [4] = HAPPINESS, [5] = RUNES, [6] = RUNIC_POWER };
local MAX_SPELL_ID = 99999;
local MAX_ENTRIES = 10;
local ITEM_SIZE = 36;

--------------------------------------------------------------------------------------------------------
--                                          Initialize Frame                                          --
--------------------------------------------------------------------------------------------------------

core:SetWidth(444);
core:SetHeight(MAX_ENTRIES * ITEM_SIZE + 66);
core:SetPoint("CENTER");
core:SetBackdrop({ bgFile = "Interface\\ChatFrame\\ChatFrameBackground", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", edgeSize = 16, insets = { left = 3, right = 3, top = 3, bottom = 3 } });
core:SetBackdropColor(0.1,0.22,0.35,1);
core:SetBackdropBorderColor(0.1,0.1,0.1,1);
core:EnableMouse(1);
core:SetMovable(1);
core:SetToplevel(1);
core:SetClampedToScreen(1);
core:Hide();

core:SetScript("OnMouseDown",core.StartMoving);
core:SetScript("OnMouseUp",core.StopMovingOrSizing);

core.outline = CreateFrame("Frame",nil,core);
core.outline:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", edgeSize = 16, insets = { left = 4, right = 4, top = 4, bottom = 4 } });
core.outline:SetBackdropColor(0.1,0.1,0.2,1);
core.outline:SetBackdropBorderColor(0.8,0.8,0.9,0.4);
core.outline:SetPoint("TOPLEFT",12,-38);
core.outline:SetPoint("BOTTOMRIGHT",-12,12);

local function SpellSpyClose_OnClick(self,button,down)
	core:Hide();
	wipe(spellList);
	collectgarbage();
end

core.close = CreateFrame("Button",nil,core,"UIPanelCloseButton");
core.close:SetPoint("TOPRIGHT",-4,-4);
core.close:SetScript("OnClick",SpellSpyClose_OnClick);

core.header = core:CreateFontString(nil,"ARTWORK","GameFontHighlight");
core.header:SetPoint("TOPLEFT",12,-12);
core.header:SetFont(core.header:GetFont(),26,"THICKOUTLINE");

--------------------------------------------------------------------------------------------------------
--                                               Entries                                              --
--------------------------------------------------------------------------------------------------------

core.entries = {};

-- OnClick
local function Entry_OnClick(self,button)
	if (IsShiftKeyDown() and ChatEdit_GetActiveWindow():IsShown()) then
		local id = spellList[self.index].id;
		ChatEdit_GetActiveWindow():Insert(GetSpellLink(id) or format("|cff71d5ff|Hspell:%d|h[%s]|h|r",id,GetSpellInfo(id)));
	end
end

-- OnEnter
local function Entry_OnEnter(self)
	GameTooltip:SetOwner(self,"ANCHOR_LEFT");
	GameTooltip:SetHyperlink("spell:"..spellList[self.index].id);
end

-- Hide GTT
local function HideGTT()
	GameTooltip:Hide();
end

-- Make Entries
for i = 1, MAX_ENTRIES do
	local btn = CreateFrame("Button",nil,core);
	btn:SetHeight(ITEM_SIZE);

	btn:SetHighlightTexture("Interface\\QuestFrame\\UI-QuestLogTitleHighlight");
	btn:GetHighlightTexture():SetVertexColor(0.1,0.22,0.35);
	btn:SetScript("OnClick",Entry_OnClick);
	btn:SetScript("OnEnter",Entry_OnEnter);
	btn:SetScript("OnLeave",HideGTT);

	btn.icon = btn:CreateTexture(nil,"ARTWORK");
	btn.icon:SetPoint("TOPLEFT",2,-2);
	btn.icon:SetPoint("BOTTOMLEFT",2,2);
	btn.icon:SetWidth(ITEM_SIZE - 4);
	btn.icon:SetTexCoord(0.07,0.93,0.07,0.93);

	btn.name = btn:CreateFontString(nil,"ARTWORK","GameFontHighlight");
	btn.name:SetPoint("TOPLEFT",btn.icon,"TOPRIGHT",3,-3);
	btn.name:SetTextColor(1,1,1);

	btn.info = btn:CreateFontString(nil,"ARTWORK","GameFontNormalSmall");
	btn.info:SetPoint("BOTTOMLEFT",btn.icon,"BOTTOMRIGHT",8,3);
	btn.info:SetTextColor(1,1,1);
	btn.info:SetTextColor(0.6,0.6,0.6);

	if (i == 1) then
		btn:SetPoint("TOPLEFT",18,-46);
		btn:SetPoint("TOPRIGHT",-12,-46);
	else
		btn:SetPoint("TOPLEFT",core.entries[#core.entries],"BOTTOMLEFT");
		btn:SetPoint("TOPRIGHT",core.entries[#core.entries],"BOTTOMRIGHT");
	end

	core.entries[i] = btn;
end

--------------------------------------------------------------------------------------------------------
--                                                Code                                                --
--------------------------------------------------------------------------------------------------------

-- Update List
local function UpdateList()
	FauxScrollFrame_Update(core.scroll,#spellList,MAX_ENTRIES,ITEM_SIZE);
	local index = core.scroll.offset;
	local gttOwner = GameTooltip:GetOwner();
	-- Loop Entries
	for i = 1, MAX_ENTRIES do
		index = (index + 1);
		local btn = core.entries[i];
		if (spellList[index]) then
			local name, rank, icon, cost, isFunnel, powerType, castTime, minRange, maxRange = GetSpellInfo(spellList[index].id);
			local powerName = (POWERTYPE_NAMES[powerType] or " (type "..tostring(powerType)..")");
			local costText = (cost == 0 and powerType == 0 and "None") or (cost.." "..powerName);
			local castTimeText = castTime == 0 and "Instant" or format("%.3f sec",castTime / 1000);
			local rangeText = (maxRange == 0 and tostring(minRange) or minRange == 0 and tostring(maxRange) or tostring(minRange).." - "..tostring(maxRange)).." yards";

			btn.icon:SetTexture(icon);
			btn.name:SetFormattedText((rank and rank ~= "") and "%s |cff20ff20(%s)|r" or "%s",name,rank);
			btn.info:SetFormattedText("ID: |cffffff00%d|r, Cost: |cffffff00%s|r, Cast Time: |cffffff00%s|r, Range: |cffffff00%s%s|r.",spellList[index].id,costText,castTimeText,rangeText,(isFunnel and "|cffc0c0c0  Funneled" or ""));
			btn.index = index;

			if (btn == gttOwner) then
				Entry_OnEnter(btn);
			end

			btn:Show();
		else
			btn:Hide();
		end
	end
end

-- Scroll
core.scroll = CreateFrame("ScrollFrame","SpellSpyScroll",core,"FauxScrollFrameTemplate");
core.scroll:SetPoint("TOPLEFT",core.entries[1]);
core.scroll:SetPoint("BOTTOMRIGHT",core.entries[#core.entries],-28,-1);
core.scroll:SetScript("OnVerticalScroll",function(self,offset) FauxScrollFrame_OnVerticalScroll(self,offset,ITEM_SIZE,UpdateList); end);
FauxScrollFrame_Update(core.scroll,#core.entries,MAX_ENTRIES,ITEM_SIZE);

-- Sort Spells
local function SortSpellsFunc(a,b)
	if (a.name == b.name) then
		return (tonumber(a.rank:match("%d+")) or 0) < (tonumber(b.rank:match("%d+")) or 0);
	else
		return a.name < b.name;
	end
end

-- Build List, Sort and Update
local function BuildSpellTable()
	wipe(spellList);
	if (type(filter) == "number") then
		local name, rank = GetSpellInfo(filter);
		spellList[#spellList + 1] = { id = filter, name = name:lower(), rank = rank:lower() };
	else
		for i = 1, MAX_SPELL_ID do
			local name, rank = GetSpellInfo(i);
			if (name) and (not filter or name:lower():find(filter)) then
				spellList[#spellList + 1] = { id = i, name = name:lower(), rank = rank:lower() };
			end
		end
		sort(spellList,SortSpellsFunc);
	end
	core.header:SetFormattedText("SpellSpy (|cffffff20%d|r)",#spellList);
	UpdateList();
end

--------------------------------------------------------------------------------------------------------
--                                           Slash Handling                                           --
--------------------------------------------------------------------------------------------------------
_G["SLASH_"..modName.."1"] = "/ss";
SlashCmdList[modName] = function(cmd)
	local code = cmd:match("^%-%-(.+)");
	-- Show/Hide Dialog
	if (cmd == "") then
		if (core:IsVisible()) then
			core:Hide();
		else
			BuildSpellTable();
			core:Show();
		end
	-- Filter by SpellId
	elseif (code) and (tonumber(code)) then
		if (GetSpellInfo(tonumber(code))) then
			filter = tonumber(code);
			AzMsg("|2"..modName.."|r: Spell filter set to the spell |1"..GetSpellInfo(filter).."|r.");
			BuildSpellTable();
			core:Show();
		else
			AzMsg("|2"..modName.."|r: There is no spell with that id.");
		end
	-- Clears the Filter
	elseif (code == "clear") then
		filter = nil;
		AzMsg("|2"..modName.."|r: Spell filter cleared.");
		BuildSpellTable();
	-- Sets the Filter
	elseif (not code) then
		filter = cmd:lower();
		AzMsg("|2"..modName.."|r: Matching spells containing the string: |1"..filter.."|r.");
		BuildSpellTable();
		core:Show();
	-- Invalid or No Command
	else
		UpdateAddOnMemoryUsage();
		AzMsg(format("----- |2%s|r |1%s|r ----- |1%.2f |2kb|r -----",modName,GetAddOnMetadata(modName,"Version"),GetAddOnMemoryUsage(modName)));
		AzMsg("The following |2parameters|r are valid for this addon:");
		AzMsg(" |2--clear|r = Clears the current filter");
		AzMsg(" |2--<spellId>|r = Shows the spell with the given spellId, if one exists");
		AzMsg(" |2<filter>|r = Set the filter for which spells to display");
	end
end