--[[
Name: LibMobHealth-4.0
Revision: $Rev: 64458 $
Author: Cameron Kenneth Knight (ckknight@gmail.com)
Inspired By: MobHealth3 by Neronix
Website: http://www.wowace.com/
Description: Estimate a mob's health
License: LGPL v2.1
]]

local MAJOR_VERSION = "LibMobHealth-4.0"
local MINOR_VERSION = tonumber(("$Revision: 64458 $"):match("%d+")) or 0

-- #AUTODOC_NAMESPACE lib

local lib, oldMinor = LibStub:NewLibrary(MAJOR_VERSION, MINOR_VERSION)
if not lib then
	return
end
local oldLib
if oldMinor then
	oldLib = {}
	for k,v in pairs(lib) do
		oldLib[k] = v
		lib[k] = nil
	end
end

local _G = _G
local UnitLevel = _G.UnitLevel
local UnitIsPlayer = _G.UnitIsPlayer
local UnitPlayerControlled = _G.UnitPlayerControlled
local UnitName = _G.UnitName
local UnitHealth = _G.UnitHealth
local UnitHealthMax = _G.UnitHealthMax
local UnitIsFriend = _G.UnitIsFriend
local UnitIsDead = _G.UnitIsDead
local UnitCanAttack = _G.UnitCanAttack
local math_floor = _G.math.floor
local setmetatable = _G.setmetatable
local type = _G.type
local pairs = _G.pairs
local next = _G.next

local frame
if oldLib then
	frame = oldLib.frame
	frame:UnregisterAllEvents()
	frame:SetScript("OnEvent", nil)
	frame:SetScript("OnUpdate", nil)
	_G.LibMobHealth40DB = nil
else
	frame = _G.CreateFrame("Frame", MAJOR_VERSION .. "_Frame")
end

frame:RegisterEvent("UNIT_COMBAT")
frame:RegisterEvent("PLAYER_TARGET_CHANGED")
frame:RegisterEvent("PLAYER_FOCUS_CHANGED")
frame:RegisterEvent("UNIT_HEALTH")
frame:RegisterEvent("ADDON_LOADED")
frame:RegisterEvent("PLAYER_LOGIN")

frame:SetScript("OnEvent", function(this, event, ...)
	this[event](lib, ...)
end)

local mt = {__index = function(self, key)
	if key == nil then
		return nil
	end
	local t = {}
	self[key] = t
	return t
end}

local data = setmetatable(oldLib and oldLib.data or {}, mt) -- stores the maximum health of mobs that will actually be shown to the user
lib.data = data

local accumulatedHP = setmetatable({}, mt) -- Keeps Damage-taken data for mobs that we've actually poked during this session
local accumulatedPercent = setmetatable({}, mt) -- Keeps Percentage-taken data for mobs that we've actually poked during this session
local calculationUnneeded = setmetatable({}, mt) -- Keeps a list of things that don't need calculation (e.g. Beast Lore'd mobs)

local currentAccumulatedHP = { target = nil, focus = nil }
local currentAccumulatedPercent = { target = nil, focus = nil }
local currentName = { target = nil, focus = nil }
local currentLevel = { target = nil, focus = nil }
local recentDamage = { target = nil, focus = nil }
local lastPercent = { target = nil, focus = nil }


_G.hash_SlashCmdList["LIBMOBHEALTHFOUR"] = nil
_G.SlashCmdList["LIBMOBHEALTHFOUR"] = nil

function frame:ADDON_LOADED(name)
	if name == MAJOR_VERSION then
		-- if we're not an embedded library, then use a saved variable
		frame:RegisterEvent("PLAYER_LOGOUT")
		if type(_G.LibMobHealth40DB) == "table" then
			data = setmetatable(_G.LibMobHealth40DB, mt)
			lib.data = data
		else
			_G.LibMobHealth40DB = data
		end
		
		local options = _G.LibMobHealth40Opt
		if type(options) ~= "table" then
			options = {
				save = true,
				prune = 1000,
			}
			_G.LibMobHealth40Opt = options
		end
		if type(options.save) ~= "boolean" then
			options.save = true
		end
		if type(options.prune) ~= "number" then
			options.prune = 1000
		end
		
		_G.hash_SlashCmdList["LIBMOBHEALTHFOUR"] = nil
		_G.SlashCmdList["LIBMOBHEALTHFOUR"] = function(text)
			text = text:lower():trim()
			local alpha, bravo = text:match("^([^%s]+)%s+(.*)$")
			if not alpha then
				alpha = text
			end
			if alpha == "" or alpha == "help" then
				DEFAULT_CHAT_FRAME:AddMessage(("|cffffff7f%s|r"):format(MAJOR_VERSION))
				DEFAULT_CHAT_FRAME:AddMessage((" - |cffffff7f%s|r [%s] - %s"):format("save", options.save and "|cff00ff00On|r" or "|cffff0000Off|r", "whether to save mob health data"))
				DEFAULT_CHAT_FRAME:AddMessage((" - |cffffff7f%s|r [%s] - %s"):format("prune", options.prune == 0 and "|cffff0000Off|r" or "|cff00ff00" .. options.prune .. "|r", "how many data points until data is pruned, 0 means no pruning"))
			elseif alpha == "save" then
				options.save = not options.save
				DEFAULT_CHAT_FRAME:AddMessage(("|cffffff7f%s|r"):format(MAJOR_VERSION))
				DEFAULT_CHAT_FRAME:AddMessage((" - |cffffff7f%s|r [%s]"):format("save", options.save and "|cff00ff00On|r" or "|cffff0000Off|r"))
			elseif alpha == "prune" then
				local bravo_num = tonumber(bravo)
				if bravo_num then
					options.prune = math.floor(bravo_num+0.5)
					DEFAULT_CHAT_FRAME:AddMessage(("|cffffff7f%s|r"):format(MAJOR_VERSION))
					DEFAULT_CHAT_FRAME:AddMessage((" - |cffffff7f%s|r [%s]"):format("prune", options.prune == 0 and "|cffff0000Off|r" or "|cff00ff00" .. options.prune .. "|r"))
				else
					DEFAULT_CHAT_FRAME:AddMessage(("|cffffff7f%s|r - prune must take a number, %q is not a number"):format(MAJOR_VERSION, bravo or ""))
				end
			else
				DEFAULT_CHAT_FRAME:AddMessage(("|cffffff7f%s|r - unknown command %q"):format(MAJOR_VERSION, alpha))
			end
		end
		
		_G.SLASH_LIBMOBHEALTHFOUR1 = "/lmh4"
		_G.SLASH_LIBMOBHEALTHFOUR2 = "/lmh"
		_G.SLASH_LIBMOBHEALTHFOUR3 = "/libmobhealth4"
		_G.SLASH_LIBMOBHEALTHFOUR4 = "/libmobhealth"
		
		function frame:PLAYER_LOGOUT()
			if not options.save then
				_G.LibMobHealth40DB = nil
				return
			end
			local count = 0
			setmetatable(data, nil)
			for k,v in pairs(data) do
				if not next(v) then
					data[k] = nil
				else
					for _ in pairs(v) do
						count = count + 1
					end
				end
			end
			local prune = options.prune
			if not prune or prune <= 0 then
				return
			end
			if count <= prune then
				return
			end
			-- let's try to only have one mob-level, don't have duplicates for each level, since they can be estimated, and for players/pets, this will get rid of old data
			local mobs = {}
			for level, d in pairs(data) do
				for mob, health in pairs(d) do
					if mobs[mob] then
						d[mob] = nil
						count = count - 1
					end
					mobs[mob] = level
				end	
				if next(d) == nil then
					data[level] = nil
				end
			end
			mobs = nil
			if count <= prune then
				return
			end
			-- still too much data, let's get rid of low-level non-bosses until we're at `prune`
			for level = 1, UnitLevel("player")*3/4 do
				local d = data[level]
				if d then
					for mob, health in pairs(d) do
						d[mob] = nil
						count = count - 1
					end
					data[level] = nil
					if count <= prune then
						return
					end
				end
			end
		end
	end
	frame:UnregisterEvent("ADDON_LOADED")
	frame.ADDON_LOADED = nil
	if IsLoggedIn() then
		frame.PLAYER_LOGIN(self)
	end
end

function frame:PLAYER_LOGIN()
	if type(_G.MobHealth3DB) == "table" then
		for index, value in pairs(_G.MobHealth3DB) do
			if type(index) == "string" and type(value) == "number" then
				local name, level = index:match("^(.+):(%-?%d+)$")
				if name then
					level = level+0
					if not data[level][name] then
						data[level][name] = value
					end
				end
			end
		end
	end
	frame:UnregisterEvent("PLAYER_LOGIN")
	frame.PLAYER_LOGIN = nil
end

function frame:UNIT_COMBAT(unit, _, _, damage)
	if (unit ~= "target" and unit ~= "focus") or not currentAccumulatedHP[unit] then
		return
	end
	recentDamage[unit] = recentDamage[unit] + damage
end

local function PLAYER_unit_CHANGED(unit)
	if not UnitCanAttack("player", unit) or UnitIsDead(unit) or UnitIsFriend("player", unit) then
		-- don't store data on friends and dead men tell no tales
		currentAccumulatedHP[unit] = nil
		currentAccumulatedPercent[unit] = nil
		return
	end
	
	local name, server = UnitName(unit)
	if server and server ~= "" then
		name = name .. "-" .. server
	end
	if UnitPlayerControlled(unit) and not UnitIsPlayer(unit) then
		-- some owners name their pets the same name as other people, because they're think they're funny. They're not.
		name = name .. ";pet"
	end
	currentName[unit] = name
	local level = UnitLevel(unit)
	currentLevel[unit] = level
	
	recentDamage[unit] = 0
	lastPercent[unit] = UnitHealth(unit)
	
	currentAccumulatedHP[unit] = accumulatedHP[level][name]
	currentAccumulatedPercent[unit] = accumulatedPercent[level][name]
	
	if not UnitIsPlayer(unit) then
		-- Mob
		if not currentAccumulatedHP[unit] then
			local saved = data[level][name]
			if saved then
				-- We claim that the saved value is worth 100%
				accumulatedHP[level][name] = saved
				accumulatedPercent[level][name] = 100
			else
				-- Nothing previously known. Start fresh.
				accumulatedHP[level][name] = 0
				accumulatedPercent[level][name] = 0
			end
			currentAccumulatedHP[unit] = accumulatedHP[level][name]
			currentAccumulatedPercent[unit] = accumulatedPercent[level][name]
		end

		if currentAccumulatedPercent[unit] > 200 then
			-- keep accumulated percentage below 200% in case we hit mobs with different hp
			currentAccumulatedHP[unit] = currentAccumulatedHP[unit] / currentAccumulatedPercent[unit] * 100
			currentAccumulatedPercent[unit] = 100
		end
	else
		-- Player health can change a lot. Different gear, buffs, etc.. we only assume that we've seen 10% knocked off players previously
		if not currentAccumulatedHP[unit] then
			local saved = data[level][name]
			if saved then
				-- We claim that the saved value is worth 10%
				accumulatedHP[level][name] = saved/10
				accumulatedPercent[level][name] = 10
			else
				accumulatedHP[level][name] = 0
				accumulatedPercent[level][name] = 0
			end
			currentAccumulatedHP[unit] = accumulatedHP[level][name]
			currentAccumulatedPercent[unit] = accumulatedPercent[level][name]
		end

		if currentAccumulatedPercent[unit] > 10 then
			currentAccumulatedHP[unit] = currentAccumulatedHP[unit] / currentAccumulatedPercent[unit] * 10
			currentAccumulatedPercent[unit] = 10
		end
	end
end

function frame:PLAYER_TARGET_CHANGED()
	PLAYER_unit_CHANGED("target")
end

function frame:PLAYER_FOCUS_CHANGED()
	PLAYER_unit_CHANGED("focus")
end

function frame:UNIT_HEALTH(unit)
	if (unit ~= "target" and unit ~= "focus") or not currentAccumulatedHP[unit] then
		return
	end
	
	local current = UnitHealth(unit)
	
	if unit == "focus" and UnitIsUnit("target", "focus") then
		-- don't want to double-accumulate
		recentDamage[unit] = 0
		lastPercent[unit] = current
		return
	end
	
	local max = UnitHealthMax(unit)
	local name = currentName[unit]
	local level = currentLevel[unit]
	
	if calculationUnneeded[level][name] then
		return
	elseif current == 0 then
		-- possibly targetting/focusing a dead person
	elseif max ~= 100 then
		-- beast lore, don't need to calculate.
		data[level][name] = max
		calculationUnneeded[level][name] = true
	elseif current > lastPercent[unit] or lastPercent[unit] > 100 then
		-- it healed, so let's reset our ephemeral calculations
		lastPercent[unit] = current
		recentDamage[unit] = 0
	elseif recentDamage[unit] > 0 then
		if current ~= lastPercent[unit] then
			currentAccumulatedHP[unit] = currentAccumulatedHP[unit] + recentDamage[unit]
			currentAccumulatedPercent[unit] = currentAccumulatedPercent[unit] + (lastPercent[unit] - current)
			recentDamage[unit] = 0
			lastPercent[unit] = current
			
			if currentAccumulatedPercent[unit] >= 10 then
				data[level][name] = currentAccumulatedHP[unit] / currentAccumulatedPercent[unit] * 100
			end
		end
	end
end

local function guessAtMaxHealth(name, level)
	-- if we have data on a mob of the same name but a different level, check within two levels and guess from there.
	local value = data[level][name]
	if value or level <= 0 then
		return value
	end
	if level > 1 then
		value = data[level - 1][name]
		if value then
			return value * level/(level - 1)
		end
	end
	value = data[level + 1][name]
	if value then
		return value * level/(level + 1)
	end
	if level > 2 then
		value = data[level - 2][name]
		if value then
			return value * level/(level - 2)
		end
	end
	value = data[level + 2][name]
	if value then
		return value * level/(level + 2)
	end
	return nil
end

--[[
Arguments:
	string - name of the unit in question in the form of "Someguy", "Someguy-Some Realm", "Somepet;pet", or "Somepet-Some Realm;pet"
	number - level of the unit in question
Returns:
	number or nil - the maximum health of the unit or nil if unknown
Example:
	local hp = LibStub("LibMobHealth-4.0"):GetMaxHP("Young Wolf", 2)
]]
function lib:GetMaxHP(name, level)
	local value = guessAtMaxHealth(name, level)
	if value then
		return math_floor(value + 0.5)
	else
		return nil
	end
end

--[[
Arguments:
	string - a unit ID
Returns:
	number, boolean - the maximum health of the unit, whether the health is known or not
Example:
	local maxhp, found = LibStub("LibMobHealth-4.0"):GetUnitMaxHP("target")
]]
function lib:GetUnitMaxHP(unit)
	local max = UnitHealthMax(unit)
	if max ~= 100 then
		return max, true
	end
	local name, server = UnitName(unit)
	if server and server ~= "" then
		name = name .. "-" .. server
	end
	if UnitPlayerControlled(unit) and not UnitIsPlayer(unit) then
		name = name .. ";pet"
	end
	local level = UnitLevel(unit)
	
	local value = guessAtMaxHealth(name, level)
	if value then
		return math_floor(value + 0.5), true
	else
		return max, false
	end
end

--[[
Arguments:
	string - a unit ID
Returns:
	number, boolean - the current health of the unit, whether the health is known or not
Example:
	local curhp, found = LibStub("LibMobHealth-4.0"):GetUnitCurrentHP("target")
]]
function lib:GetUnitCurrentHP(unit)
	local current, max = UnitHealth(unit), UnitHealthMax(unit)
	if max ~= 100 then
		return current, true
	end
	
	local name, server = UnitName(unit)
	if server and server ~= "" then
		name = name .. "-" .. server
	end
	if UnitPlayerControlled(unit) and not UnitIsPlayer(unit) then
		name = name .. ";pet"
	end
	local level = UnitLevel(unit)
	
	local value = guessAtMaxHealth(name, level)
	if value then
		return math_floor(current/max * value + 0.5), true
	else
		return current, false
	end
end

--[[
Arguments:
	string - a unit ID
Returns:
	number, number, boolean - the current health of the unit, the maximum health of the unit, whether the health is known or not
Example:
	local curhp, maxhp, found = LibStub("LibMobHealth-4.0"):GetUnitHealth("target")
]]
function lib:GetUnitHealth(unit)
	local current, max = UnitHealth(unit), UnitHealthMax(unit)
	if max ~= 100 then
		return current, max, true
	end

	local name, server = UnitName(unit)
	if server and server ~= "" then
		name = name .. "-" .. server
	end
	if UnitPlayerControlled(unit) and not UnitIsPlayer(unit) then
		name = name .. ";pet"
	end
	local level = UnitLevel(unit)
	
	local value = guessAtMaxHealth(name, level)
	if value then
		return math_floor(current/max * value + 0.5), math_floor(value + 0.5), true
	else
		return current, max, false
	end
end
