local moduleName = "Basics"
local core = SimpleUnitFrames

local Basics = core:NewModule(moduleName, "AceEvent-2.0")
local LMH4 = LibStub("LibMobHealth-4.0")
local DEFAULT_FONT = "GameFontHighlightSmall"
local DEFAULT_FONT_SIZE = 10
local creatureTypeMap = core.loc.UnitCreatureType

-- List of fonts to be shown in the configuration.
local fonts = {
	"GameFontHighlight",
	"GameFontHighlightSmall",
	"GameFontNormal",
	"GameFontNormalSmall",
	"TextStatusBarText",
	"TextStatusBarTextSmall",
}


-- List of text value functions.
local styles = {

-- Health point

	["HPdeficit"] = function(unit)
		local curr, max
		curr, max = LMH4:GetUnitHealth(unit)
		return curr - max
	end,

	["HPpercent"] = function(unit) 
		local curr, max = UnitHealth(unit), UnitHealthMax(unit)
		return string.format( "%d%%", curr * 100 / ( max > 0 and max or 1 ) ) 
	end,
	
	["HPcurrmax"] = function(unit)
		local curr, max
		curr, max = LMH4:GetUnitHealth(unit)
		return curr .. '/' .. max
	end,
	
	["HPcomplete"] = function(unit)
		local curr, max
		curr, max = LMH4:GetUnitHealth(unit)
		local percent = curr*100/( max > 0 and max or 1 )
		if max > 100000 then
			max = string.format("%dK", max/1000)
		end
		if curr > 100000 then
			curr = string.format("%dK", curr/1000)
		end
		if max == 100 then -- Only return % when the exact HP is not know.
			return string.format( "(%d%%)", percent )
		else
			return string.format( "%s/%s (%d%%)", curr, max, percent)
		end
	end,

-- Mana point
	["MPdeficit"] = function(unit)
		return UnitMana(unit) - UnitManaMax(unit)
	end,

	["MPpercent"] = function(unit) 
		local curr, max = UnitMana(unit), UnitManaMax(unit)
		return string.format( "%d%%", curr*100 / ( max > 0 and max or 1 ) ) 
	end,

	["MPcurrmax"] = function(unit) 
		return UnitMana(unit) .. '/' .. UnitManaMax(unit) 
	end,
	
	["MPcomplete"] = function(unit) 
		local curr, max = UnitMana(unit), UnitManaMax(unit)
		local deno = max > 0 and max or 1
		return string.format( "%d/%d (%d%%)", curr, max, curr*100/deno)
	end,
	
-- None

	["none"] = function() return "" end,
	
}


local unitMap = {}

local icons = {
	DRUID = "DRUID",
	HUNTER = "HUNTER",
	MAGE = "MAGE",
	PALADIN = "PALADIN",
	PRIEST = "PRIEST",
	ROGUE = "ROGUE",
	SHAMAN = "SHAMAN",
	UNKNOWN = "UNKNOWN",
	WARLOCK = "WARLOCK",
	WARRIOR = "WARRIOR",
}

local frames = {}

Basics.formatList = {}
for k in pairs(styles) do
	table.insert(Basics.formatList, k)
	table.sort(Basics.formatList)
end

Basics.unitMap = unitMap

Basics.consoleOptions = {}
Basics.defaultOptions = {} 
Basics.frameSettings = {}


function Basics:OnInitialize()
	self.db = core:AcquireDBNamespace(moduleName)

	core:RegisterDefaults(moduleName, 'profile', self.defaultOptions)
	
	for english, locale in pairs(creatureTypeMap) do
		icons[locale] = english
	end
	creatureTypeMap = nil
	
	local args = self.consoleOptions
	
	local settings = self.frameSettings

	self.frames = frames
	for unit, unitSettings in pairs(settings) do
		local frame = CreateFrame("Frame", nil, unitSettings.parent)
		frame:SetFrameLevel(unitSettings.parent:GetFrameLevel()+2)
		frame.fontStrings = {}
		frames[unit] = frame
		for text, textSettings in pairs(unitSettings.texts) do
			local mappedUnit = unitMap[unit] or unit
			local fontString = frame:CreateFontString(nil, "OVERLAY", (self.db.profile[mappedUnit] and self.db.profile[mappedUnit].font) or DEFAULT_FONT )
			fontString:SetPoint(unpack(textSettings.point))
			frame.fontStrings[text] = fontString
		end
		if unitSettings.classIcon then
			local texture = frame:CreateTexture("OVERLAY")
			texture:SetHeight(unitSettings.classIcon.size)
			texture:SetWidth(unitSettings.classIcon.size)
			texture:SetPoint( unpack(unitSettings.classIcon.point ) )
			frame.classIcon = texture
		end
	end

	for unit, options in pairs(args) do
		core.consoleOptions.args[unit] = options
		options.args.font = {
			order = 1000,
			type = 'text',
			name = 'Font',
			desc = 'Set the font for this frame.',
			get = function() return self.db.profile[unit].font or DEFAULT_FONT end,
			set = function(v)
				self.db.profile[unit].font = v
				self:RefreshFont(unit)
			end,
			validate = fonts
		}
	end


	self.consoleOptions = nil
	self.defaultOptions = nil
	self.frameSettings = nil
	
end

function Basics:OnEnable()

	self:RegisterEvent("UNIT_HEALTH", "Refresh")
	self:RegisterEvent("UNIT_MAXHEALTH", "Refresh")
	self:RegisterEvent("UNIT_MANA", "Refresh")
	self:RegisterEvent("UNIT_MAXMANA", "Refresh")
	self:RegisterEvent("UNIT_RAGE", "Refresh")
	self:RegisterEvent("UNIT_MAXRAGE", "Refresh")
	self:RegisterEvent("UNIT_ENERGY", "Refresh")
	self:RegisterEvent("UNIT_MAXENERGY", "Refresh")
	self:RegisterEvent("UNIT_DISPLAYPOWER", "Refresh")
	

	self:Refresh()
end

function Basics:OnDisable()
	self:UnregisterAllEvents()
	for unit, unitFrame in pairs(frames) do
		if unitFrame.classIcon then
			unitFrame.classIcon:Hide()
		end
		if unitFrame.fontStrings then
			for text, fontString in pairs(unitFrame.fontStrings) do
				fontString:SetText("")
			end
		end
	end
end

function Basics:Refresh(unit)
	if unit then
		if UnitExists(unit) then
			local mappedUnit = unitMap[unit] or unit
			if mappedUnit and frames[unit] then
				if frames[unit].classIcon then
					self:UpdateClassIcon(unit)
				end
				for text, fontString in pairs(frames[unit].fontStrings) do
					self:UpdateValue(unit, self.db.profile[mappedUnit][text], fontString )
				end
			end
		end
	else
		for unit in pairs(frames) do
			self:Refresh(unit)
		end
	end
end

function Basics:RefreshFont(unit)
	if unit then
		for frameUnit in pairs(frames) do
			if( frameUnit:find('^'..unit) ) then
				for text, fontString in pairs(frames[frameUnit].fontStrings) do
					local font = self.db.profile[unit].font or DEFAULT_FONT
					fontString:SetFontObject(font)
				end
			end
		end
	end
end

function Basics:UpdateValue(unit, style, fontString)
	fontString:SetText( self:GetValueText(unit, style) )
end

function Basics:UpdateClassIcon(unit)
	
	local classIcon = frames[unit] and frames[unit].classIcon
	local mappedUnit = unitMap[unit] or unit
	
	if classIcon and mappedUnit and UnitExists(unit) then
	
		if self.db.profile[mappedUnit].class then
		
			local icon = nil
			if UnitIsPlayer(unit) then
				_, icon = UnitClass(unit)
			else
				icon = UnitCreatureType(unit)
			end
			
			icon = (icon and icons[icon]) or "UNKNOWN"
			
			classIcon:SetTexture("Interface\\AddOns\\SimpleUnitFrames\\Icons\\" .. icon .. ".tga")
			classIcon:Show()
		
		else
			
			classIcon:Hide()
			
		end
	end
	
end


function Basics:GetValueText(unit, style)
	if not styles[style] then return "ERROR" end
	return styles[style](unit)
end



