--[[ LibRingCache.lua ]--
	LibRingCache is a fast implementation of a cache that selectively lets the
	garbage collector collect old items and tries to keep the newest X (amount
	can be freely chosen via the size parameter in LibRingCache:create(size))
	
	Old items are discarded by number, if size is N and the N+1th item was just
	inserted, then the 1st item is free to collect from the GC, IF the item is
	not in use anywhere else. This way any cached data can be safely used, but 
	you should be aware of global variables holding cached data and so 
	preventing the GC from collecting them.
	
	Also note that the Cache will only work correctly if either the key or the
	value (or both) is a table.

	========================================================================
	This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
	========================================================================
]]

local LibRingCache = LibStub:NewLibrary("LibRingCache-1.0", 2);
if not LibRingCache then
	return;
end

local meta = {__mode = "kv"};
local prototype = {};
local kReorder = true;

--[[ LibRingCache:create(size, feed) ]--
	Create a new ring cache of size 'size' and returns a reference.
	
	@arg  size  the desired size of the cache, actual size may vary depending
	            on how much of the data is being held by other variables
	@arg  feed  feed an existing table to use in the cache
	@example
	do
		local LibRingCache = LibStub:GetLibrary("LibRingCache-1.0");
		local cache = LibRingCache:create(10);
		cache:insert("foo", {"bar"});
		print(cache:get("foo"));
	end
]]
function LibRingCache:create(size, feed)
	local new  = {};
	new.size   = size or 100;
	new.ring   = {};
	new.data   = setmetatable(feed or {}, meta);
	new.start  = 1;
	new.insert = prototype.insert;
	new.get    = prototype.get;
	new.remove = prototype.remove;
	new.clear  = prototype.clear;
	
	for key,data in pairs(new.data) do
		new.ring[new.start] = data;
		new.start = (new.start%new.size)+1; -- stay in the 1..n range to make use
		                                    -- of array tables
	end
	
	return new;
end

--[[ LibRingCache:createBlackbox(size, feed) ]--
	Create a new blackbox ring cache of size 'size' and returns a reference.
	
	A blackbox is something that you don't know what's inside, that's the basic
	idea of this cache. You can however feed it with a table (feed argument),
	but use this table only read-only.
	
	One possible use of the blackbox cache type is shown in the addon ShortTerm.
	
	@arg  size  the desired size of the cache, actual size may vary depending
	            on how much of the data is being held by other variables
	@arg  feed  feed an existing table to use in the cache
	@example
	do
		local LibRingCache = LibStub:GetLibrary("LibRingCache-1.0");
		local cache = LibRingCache:create(10);
		cache:insert("foo", {"bar"});
		print(cache:get("foo"));
	end
]]
function LibRingCache:createBlackbox(size, feed)
	local new  = {};
	new.size   = size or 100;
	new.ring   = {};
	new.data   = setmetatable(feed or {}, meta);
	new.start  = 1;
	new.app    = kReorder(new.data, new.size)-1; -- kReorder just looks simple here, scroll down!
	new.append = prototype.append;
	new.clear  = prototype.clear;
	
	for key,data in pairs(new.data) do
		new.ring[new.start] = data;
		new.start = (new.start%new.size)+1; -- stay in the 1..n range to make use
		                                    -- of array tables
	end
	
	return new;
end

--[[ Cache:insert(key, data) ]--
	insert 'data' with key 'key' into this buffer.
	This method is not available in blackbox caches.
	
	@arg     key   a (unique) key by wich the data can be identified
	@arg     data  the data to be cached
	@return  returns the inserted data (equal to the argument data)
	@note    note that either key or data (or both) must be a table or object
	         for the cache to work properly, otherwise the parameters can be of
	         any type.
	@example @see LibRingCache:create(size)
]]
function prototype:insert(key, data)
	self.data[key] = data;
	self.ring[self.start] = data;
	self.start = (self.start%self.size)+1; -- stay in the 1..n range to make use
	                                       -- of array tables
	return data;
end

--[[ Cache:append(data) ]--
	append 'data' to this buffer.
	This method is only available in blackbox caches.
	
	@arg     data  the data to be cached
	@return  returns nil
	@note    note that data must be a table or object for the cache to work
	         properly, otherwise the parameters can be of any type.
	@example @see LibRingCache:create(size)
]]
function prototype:append(data)
	self.ring[self.start] = data;
	self.start = (self.start%self.size)+1; -- stay in the 1..n range to make use
	                                       -- of array tables
	local app = self.app+1;
	if app > self.size*4 and self.data[1] == nil then -- 1st index nil indicates
		app = kReorder(self.data);                    -- that gc collected sth
	end
	self.data[app] = data;
	self.app = app;
end

--[[ Cache:get(key) ]--
	retrieve a cached value by key from the ringcache.
	This method is not available in blackbox caches.
	
	@arg     key   a (unique) key by wich the value can be identified
	@note    note that either key or data (or both) must be a table or object
	         for the cache to work properly, otherwise the parameters can be of
	         any type.
	@example @see LibRingCache:create(size)
]]
function prototype:get(key)
	return self.data[key];
end

--[[ Cache:remove(key) ]--
	remove a cached value by key from the ringcache.
	This method is not available in blackbox caches.
	
	@arg     key    a (unique) key by wich the value can be identified
	@return  returns the removed data
	@note    note that either key or data (or both) must be a table or object
	         for the cache to work properly, otherwise the parameters can be of
	         any type.
	@example @see LibRingCache:create(size)
]]
function prototype:remove(key)
	local k = self.data[key];
	self.data[key] = nil;
	return k;
end

--[[ Cache:clear() ]--
	Completely clears the cache.
]]
function prototype:clear()
	wipe(self.data);
end

--[[ private kReorder(table) ]--
	Sorts table entries with numeric keys by their keys and makes the table
	gapless. Keys will change, but entries will retain their relative position.
	
	@return first unused (numeric) key
]]
kReorder = function (tbl)
	-- get a list of keys
	local keys = {};
	for k,v in pairs(tbl) do
		if type(k) == "number" then
			tinsert(keys, k);
		end
	end
	sort(keys);
	
	local j = 1;
	local v;
	-- reorder the elements gapless
	for i=1,#keys do
		--print(i, keys[i], "to", j);
		v = tbl[keys[i]];
		tbl[keys[i]] = nil;
		tbl[j] = v
		j = j+1;
	end
	
	-- returns the first unused index
	return j;
end

