local SN, L, F, T, M, _ = unpack(select(2,...))
local LSM = LibStub("LibSharedMedia-3.0")
local MODNAME = "raidbuffs"
local RB = SN:NewModule(MODNAME, "AceEvent-3.0", "AceTimer-3.0", "AceConsole-3.0")

local setmetatable = setmetatable
local pairs = pairs
local select = select
local wipe = wipe
local tostring = tostring
local unpack = unpack
local type = type
local collectgarbage = collectgarbage
local string = string
local strjoin = strjoin
local SendChatMessage = SendChatMessage
local NORMAL_FONT_COLOR = NORMAL_FONT_COLOR
local GameFontGreenLarge = GameFontGreenLarge
local GetSpellInfo = GetSpellInfo
local UnitBuff = UnitBuff
local IsInGroup = IsInGroup
local GetNumGroupMembers = GetNumGroupMembers
local GetRaidRosterInfo = GetRaidRosterInfo
local InCombatLockdown = InCombatLockdown
local UnitInRaid = UnitInRaid
local UnitFactionGroup = UnitFactionGroup
local GetRealZoneText = GetRealZoneText
local IsInInstance = IsInInstance
local GetUnitName = GetUnitName
local UnitIsDeadOrGhost = UnitIsDeadOrGhost
local UnitIsGroupLeader = UnitIsGroupLeader
local UnitIsGroupAssistant = UnitIsGroupAssistant
local GetSpellLink = GetSpellLink

local raidMembers = {}
local UpdateDelay = 0
local raidbuffs
local missingFood = {}
local missingFlask = {}

local RBIconSize = 32
			
local function SpellName(spellID)
	local name = GetSpellInfo(spellID)
	return name
end

local function SpellIcon(spellID)
	local _, _, icon = GetSpellInfo(spellID)
	return icon
end

local player = {}
player.__index = player
function player:new()
	self = {}
	setmetatable(self, player)
	self.name = ""
	self.zone = ""
	self.flask = false
	self.food = false
	self.fortitude = false
	self.statbuff = false
	self.masterybuff = false
	self.spbuff = false
	self.critbuff = false
	
	return self
end

local buffs = {
	flask = 0,
	food = 0,
	fortitude = 0,
	statbuff = 0,
	masterybuff = 0,
	spbuff = 0,
	critbuff = 0,
}
local missingBuffs = {
	flask = {},
	food = {},
	fortitude = {},
	statbuff = {},
	masterybuff = {},
	spbuff = {},
	critbuff = {},
}

local bufftext = {
	["flask"] =       "{triangle}Flasks or Elixirs{triangle}",
	["food"] =        "{moon}Well Fed Buff{moon}",
	["fortitude"] =   "{square}Stamina Buff{square}",
	["statbuff"] =    "{diamond}Stat Buff{diamond}",
	["masterybuff"] = "{star}Mastery Buff{star}",
	["spbuff"] =      "{circle}Spell Power Buff{circle}",
	["critbuff"] =    "{skull}Crit Buff{skull}",
}

local buffspells = {
	["flask"] = {
		SpellName(105689),  -- Flask of Spring Blossoms
		SpellName(105691),  -- Flask of the Warm Sun
		SpellName(105693),  -- Flask of Falling Leaves
		SpellName(105694),  -- Flask of the Earth
		SpellName(105696),  -- Flask of Winter\'s Bite
		SpellName(105682),	-- Mad Hozen Elixir
		SpellName(105683),	-- Elixir of Weaponry
		SpellName(105684),	-- Elixir of the Rapids
		SpellName(105685),	-- Elixir of Peace
		SpellName(105686),	-- Elixir of Perfection
		SpellName(105688),	-- Monk\'s Elixir
		SpellName(105681),	-- Mantid Elixir
		SpellName(105687),	-- Elixir of Mirrors
		SpellName(127230),  -- Vision of Insanity
	}, 
	["food"] = {
		SpellName(35272),	-- Well Fed
	},
	["fortitude"] = {
		SpellName(21562), 	-- Prayer of Fortitude
		SpellName(109773), 	-- Dark Intent
		SpellName(469),     -- Commanding Shout
		SpellName(90364),   -- Qiraji Fortitude
		SpellName(111923),  -- Runescroll of Fortitude III
	},
	["statbuff"] = {
		SpellName(1126), 	-- Mark of the Wild
		SpellName(20217), 	-- Blessing of Kings
		SpellName(117666), 	-- Legacy of the Emperor
		SpellName(90363),   -- Embrace of the Shale Spider
	},
	["masterybuff"] = { 
		SpellName(19740),   -- Blessing of Might
		SpellName(116956),  -- Grace of Air
		SpellName(93435),   -- Roar of Courage
		SpellName(128997),  -- Spirit Beast Blessing        
	},
	["spbuff"] = { 
		SpellName(1459), 	-- Arcane Intellect
		SpellName(61316), 	-- Dalaran Brilliance
		SpellName(109773), 	-- Dark Intent
		SpellName(77747), 	-- Burning Wrath
		SpellName(126309), 	-- Still Water
	},
	["critbuff"] = {
		SpellName(116781),	-- Legacy of the White Tiger
		SpellName(17007), 	-- Leader of the Pack
		SpellName(1459), 	-- Arcane Brilliance
		SpellName(61316),	-- Dalaran Brilliance
		SpellName(24604), 	-- Furious Howl
		SpellName(90309), 	-- Terrifying Roar
		SpellName(97229), 	-- Bellowing Roar
		SpellName(126373),	-- Fearless Roar
		SpellName(126309),	-- Still Water        
	},
}

local specials = {
	[126503] = { category="Feast", itemid=87246 },   -- Banquet of the Brew
	[126504] = { category="Feast", itemid=87248 },   -- Great Banquet of the Brew
	[126492] = { category="Feast", itemid=87226, bonus="STRENGTH" },  -- Banquet of the Grill
	[126494] = { category="Feast", itemid=87228, bonus="STRENGTH" },  -- Great Banquet of the Grill
	[126501] = { category="Feast", itemid=87242, bonus="STAMINA" },   -- Banquet of the Oven
	[126502] = { category="Feast", itemid=87244, bonus="STAMINA" },   -- Great Banquet of the Oven
	[126497] = { category="Feast", itemid=87234, bonus="INTELLECT" }, -- Banquet of the Pot
	[126498] = { category="Feast", itemid=87236, bonus="INTELLECT" }, -- Great Banquet of the Pot
	[126499] = { category="Feast", itemid=87238, bonus="SPIRIT" },    -- Banquet of the Steamer
	[126500] = { category="Feast", itemid=87240, bonus="SPIRIT" },    -- Great Banquet of the Steamer
	[126495] = { category="Feast", itemid=87230, bonus="AGILITY" },   -- Banquet of the Wok
	[126496] = { category="Feast", itemid=87232, bonus="AGILITY" },   -- Great Banquet of the Wok
	[104958] = { category="Feast", itemid=74919 },   -- Pandaren Banquet
	[105193] = { category="Feast", itemid=75016 },   -- Great Pandaren Banquet

	[145166] = { category="Cart", itemid=101630, ungrouped=true },   -- Noodle Cart (duration 180 but can be cancelled)
	[145169] = { category="Cart", itemid=101661, ungrouped=true },   -- Deluxe Noodle Cart
	[145196] = { category="Cart", itemid=101662, ungrouped=true },   -- Pandaren Treasure Noodle Cart
	
	[43987]  = { category="Table", label="Refreshment Table", slow=true }, -- Ritual of Refreshment 

	[29893]  = { category="Soulwell", itemid=5512, label="Soul Well", slow=true },  -- Create Soulwell 

	[126459] = { category="Blingtron", itemid=87214, label="Blingtron", ungrouped=true }, -- Blingtron 4000

	[54710]  = { category="Mailbox", item=40768, label="Mailbox", ungrouped=true }, 

	[22700]  = { category="Repair", itemid=18232, label="Repair Bot", ungrouped=true }, -- Field Repair Bot 74A
	[44389]  = { category="Repair", itemid=34113, label="Repair Bot", ungrouped=true }, -- Field Repair Bot 110G
	[54711]  = { category="Repair", itemid=40769, label="Repair Bot", ungrouped=true }, -- Scrapbot
	[67826]  = { category="Repair", itemid=49040, label="Repair Bot", ungrouped=true }, -- Jeeves

	[53142]  = { category="Portal", slow=true }, -- Dalaran
	[11419]  = { category="Portal", slow=true }, -- Darnassus
	[32266]  = { category="Portal", slow=true }, -- Exodar
	[11416]  = { category="Portal", slow=true }, -- Ironforge
	[11417]  = { category="Portal", slow=true }, -- Orgrimmar
	[35717]  = { category="Portal", slow=true }, -- Shattrath
	[33691]  = { category="Portal", slow=true }, -- Shattrath
	[32267]  = { category="Portal", slow=true }, -- Silvermoon
	[49361]  = { category="Portal", slow=true }, -- Stonard
	[10059]  = { category="Portal", slow=true }, -- Stormwind
	[49360]  = { category="Portal", slow=true }, -- Theramore
	[11420]  = { category="Portal", slow=true }, -- Thunder Bluff
	[88346]  = { category="Portal", slow=true }, -- Tol Barad
	[88345]  = { category="Portal", slow=true }, -- Tol Barad
	[11418]  = { category="Portal", slow=true }, -- Undercity
	[132626] = { category="Portal", slow=true }, -- Vale of Eternal Blossoms
	[132620] = { category="Portal", slow=true }, -- Vale of Eternal Blossoms
	[120146] = { category="Portal", slow=true }, -- Ancient Dalaran
}

local spellIcons = {
	flask = SpellIcon(105691),
	food = SpellIcon(35272),
	fortitude = SpellIcon(21562),
	statbuff = SpellIcon(20217),
	masterybuff = SpellIcon(19740),
	spbuff = SpellIcon(1459),
	critbuff = SpellIcon(17007),
}

local function getOptions()
	raidbuffs = {
		type = "group",
		name = L["Raid Buffs"],
		args = {
			enable = {
				type = "toggle",
				name = L["Enable Raid Buff System"],
				order = 0,
				get = function(info)
					return SN.data.profile.raidbuffs[info[#info]]
				end,
				set = function(info, value)
					SN.data.profile.raidbuffs[info[#info]] = value
					if value then
						SN:EnableModule(MODNAME)
					else
						SN:DisableModule(MODNAME)
					end
				end,
			},
			orientation = {
				order = 1,
				type = "select",
				name = L["Direction"],
				values = {
					["HORIZONTAL"] = "HORIZONTAL",
					["VERTICAL"] = "VERTICAL",
				},
				hidden = function() return not SN.data.profile.raidbuffs.enable end,
				get = function(info)
					return SN.data.profile.raidbuffs[info[#info]]
				end,
				set = function(info, value)
					SN.data.profile.raidbuffs[info[#info]] = value
					RB:ApplySettings()
				end,
			},
			whisper = {
				type = "toggle",
				name = L["Whisper"],
				desc = L["Whisper people with missing Flask/Food"],
				hidden = function() return not SN.data.profile.raidbuffs.enable end,
				order = 2,
				get = function(info)
					return SN.data.profile.raidbuffs[info[#info]]
				end,
				set = function(info, value)
					SN.data.profile.raidbuffs[info[#info]] = value
				end,
			},
			readycheck = {
				type = "toggle",
				name = L["Announce"],
				desc = L["Announce people with missing Flask/Food when a Ready Check is issued"],
				hidden = function() return not SN.data.profile.raidbuffs.enable end,
				order = 3,
				get = function(info)
					return SN.data.profile.raidbuffs[info[#info]]
				end,
				set = function(info, value)
					SN.data.profile.raidbuffs[info[#info]] = value
				end,
			},
			scale = {
				type = "range",
				name = L["Scale of Raid Buffs"],
				hidden = function() return not SN.data.profile.raidbuffs.enable end,
				order = 4,
				min = 0.5, max = 5, step = 0.1,
				get = function(info)
					return SN.data.profile.raidbuffs[info[#info]] 
				end,
				set = function(info, value)
					SN.data.profile.raidbuffs[info[#info]] = value
					RB:ApplySettings()
				end,
			},
			instance = {
				type = "toggle",
				name = L["Instance Only"],
				desc = L["Announce only if you are within an instance"],
				hidden = function() return not SN.data.profile.raidbuffs.enable end,
				order = 5,
				get = function(info)
					return SN.data.profile.raidbuffs[info[#info]]
				end,
				set = function(info, value)
					SN.data.profile.raidbuffs[info[#info]] = value
				end,
			},
		},
	}
	return raidbuffs
end

function player:CheckBuffs(buff)
	--Check for buffs
	for k, v in pairs(buffspells[buff]) do
		local cachename = self.name:gsub("^(.*)%-.*$", "%1")
		if UnitBuff(cachename, buffspells[buff][k]) then
			self[buff] = true
			return
		end
	end
end

function RB:OnInitialize()
	SN:RegisterModuleOptions(MODNAME, getOptions, L["Raid Buffs"])

	--Raid Buff Window
	--
	RB.Buffs = SN:CreateFrame("Raid Buff Frame", MODNAME, UIParent)
	SN:CreateBackDrop(RB.Buffs)
	
	local size = RBIconSize * SN.data.profile.raidbuffs.scale
	local width = (size * 7) + 16
	local height = size + 4

	if SN.data.profile.raidbuffs.orientation == "VERTICAL" then
		RB.Buffs:SetSize(height, width)
	else
		RB.Buffs:SetSize(width, height)
	end
	
	
	local counter = ""
	for k, v in pairs(buffs) do		
		RB.Buffs[k] = CreateFrame("Button", nil, RB.Buffs)
		RB.Buffs[k]:SetSize(size, size)
		RB.Buffs[k]:SetHighlightTexture("Interface\\Buttons\\UI-Common-MouseHilight")
		RB.Buffs[k]:SetNormalTexture(spellIcons[k])
		RB.Buffs[k]:SetParent(RB.Buffs)
		if counter == "" then
			if SN.data.profile.raidbuffs.orientation == "VERTICAL" then
				RB.Buffs[k]:SetPoint("TOPLEFT", RB.Buffs, 2, -2)
			else
				RB.Buffs[k]:SetPoint("LEFT", RB.Buffs, 2, 0)
			end
		else
			if SN.data.profile.raidbuffs.orientation == "VERTICAL" then
				RB.Buffs[k]:SetPoint("TOPLEFT", RB.Buffs[counter], "BOTTOMLEFT", 0, -2)
			else
				RB.Buffs[k]:SetPoint("LEFT", RB.Buffs[counter], "RIGHT", 2, 0)
			end
		end
		RB.Buffs[k]:SetScript("OnEnter", function(self, motion)
			GameTooltip:SetOwner(self, "ANCHOR_BOTTOMLEFT")
			GameTooltip:SetText("Missing Buff:")
			if not missingBuffs[k] then
				GameTooltip:AddLine("none")
			else
				for a, b in pairs(missingBuffs[k]) do
					local cachename = b:gsub("^(.*)%-.*$", "%1")
					GameTooltip:AddLine(cachename)
				end
			end
			GameTooltip:Show()
		end)
		RB.Buffs[k]:SetScript("OnLeave", function()
			GameTooltip:ClearLines()
			GameTooltip:Hide()
		end)
		RB.Buffs[k]:SetScript("OnClick", function(self, button, down)
			RB:AnnounceBuffs()
		end)
		local color = NORMAL_FONT_COLOR
		RB.Buffs[k].text = RB.Buffs[k]:CreateFontString(nil, "ARTWORK")
		RB.Buffs[k].text:SetFontObject(GameFontGreenLarge )
		RB.Buffs[k].text:SetPoint("CENTER", RB.Buffs[k], "CENTER", 0, 0)
		RB.Buffs[k].text:SetText("0")
		RB.Buffs[k].text:Show()
		counter = k
		RB.Buffs[k].name = k
	end
	
	RB.Buffs:SetScript("OnEvent", function(self, event, ...)
		if event == "PLAYER_REGEN_ENABLED" or event == "PET_BATTLE_CLOSE" then
			if IsInGroup() then
				RB:ToggleActive(true)
				RB:RefreshBuffs()
			end
		elseif event == "PLAYER_REGEN_DISABLED" or event == "PET_BATTLE_OPENING_START" then
			RB:ToggleActive(false)
		elseif event == "READY_CHECK" then
			if SN.data.profile.raidbuffs.readycheck then
				RB:AnnounceBuffs()
			end
		elseif event == "GROUP_ROSTER_UPDATE" then
			if GetNumGroupMembers() > 1 and not self:IsVisible() then
				RB:ToggleActive(true)
			end
			wipe(raidMembers)
			for i = 1, GetNumGroupMembers() do
				local name, _, _, _, _, _, inzone = GetRaidRosterInfo(i)
				if name then
					--local cachename = name:gsub("^(.*)%-.*$", "%1")
					if not raidMembers[name] then
						raidMembers[name] = player:new()
						raidMembers[name].name = name
						raidMembers[name].zone = inzone
					end
				end
			end
		elseif event == "COMBAT_LOG_EVENT_UNFILTERED" then
			local _, type, _, sourceGUID, sourceName, _, _, destGUID, destName = ...
			local spellId, spellName = select(12, ...)

			if not type then return	end

			if not InCombatLockdown() then
				if not sourceName then return end
			
				if type == "SPELL_CAST_START" or type == "SPELL_CAST_SUCCESS" then
					local special = specials[spellId]
					if special then
						--if special.slow and type == "SPELL_CAST_START" then return end -- wait for completion of long casts

						if UnitInRaid(sourceName) or (special.ungrouped and UnitFactionGroup(sourceName) == UnitFactionGroup("player")) then
							RB:AnnounceSpecials(special.category, sourceName, spellId)
						end
					end
				end
			end
		end
	end)
	RB.Buffs:Hide()
end

function RB:OnEnable()
	if not IsInGroup() then
		RB:ToggleActive(false)
		for k, v in pairs(buffs) do
			RB.Buffs[k].text:SetText("0")
			buffs[k] = 0
			if missingBuffs[k] then
				wipe(missingBuffs[k])
			end
		end
	else
		wipe(raidMembers)
		for i = 1, GetNumGroupMembers() do
			local name, _, _, _, _, _, inzone = GetRaidRosterInfo(i)
			if name then
				--local cachename = name:gsub("^(.*)%-.*$", "%1")
				if not raidMembers[name] then
					raidMembers[name] = player:new()
					raidMembers[name].name = name
					raidMembers[name].zone = inzone
				end
			end
		end
		RB:ToggleActive(true)
		RB:RefreshBuffs()
		for k, v in pairs(buffs) do
			RB.Buffs[k].text:SetText(tostring(buffs[k]))
		end
	end
	RB.Buffs:RegisterEvent("PLAYER_REGEN_DISABLED")
	RB.Buffs:RegisterEvent("PLAYER_REGEN_ENABLED")
	RB.Buffs:RegisterEvent("GROUP_ROSTER_UPDATE")
	RB.Buffs:RegisterEvent("READY_CHECK")
	RB.Buffs:RegisterEvent("PET_BATTLE_OPENING_START")
	RB.Buffs:RegisterEvent("PET_BATTLE_CLOSE")
	RB.Buffs:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
end

function RB:OnDisable()
	RB.Buffs:UnregisterAllEvents()
	RB:ToggleActive(false)
end

function RB:Lock()
	SN:LockFrame(RB.Buffs)
	for k, v in pairs(buffs) do		
		RB.Buffs[k]:Show()
	end
	SN:CreateBackDrop(RB.Buffs)
end

function RB:Unlock()
	SN.data.profile.raidbuffs.width = RB.Buffs:GetWidth()
	SN.data.profile.raidbuffs.height = RB.Buffs:GetHeight()

	for k, v in pairs(buffs) do		
		RB.Buffs[k]:Hide()
	end
	SN:UnlockFrame(RB.Buffs)
end

function RB:Reset()
	local static = SN.DB.profile.frames
	local temp = SN.data.profile.frames
	RB.Buffs:ClearAllPoints()

	RB.Buffs:SetPoint(unpack(static.raidbuffs))
	wipe(temp.raidbuffs)
end

function RB:ApplySettings()
	RB.Buffs:ClearAllPoints()
	RB.Buffs:SetPoint(unpack(SN.data.profile.frames.raidbuffs))
	
	local size = RBIconSize * SN.data.profile.raidbuffs.scale
	local width = (size * 7) + 16
	local height = size + 4

	if SN.data.profile.raidbuffs.orientation == "VERTICAL" then
		RB.Buffs:SetSize(height, width)
	else
		RB.Buffs:SetSize(width, height)
	end
	local counter = ""
	for k, v in pairs(buffs) do
		RB.Buffs[k]:ClearAllPoints()
		RB.Buffs[k]:SetSize(size, size)
		if counter == "" then
			if SN.data.profile.raidbuffs.orientation == "VERTICAL" then
				RB.Buffs[k]:SetPoint("TOPLEFT", RB.Buffs, 2, -2)
			else
				RB.Buffs[k]:SetPoint("LEFT", RB.Buffs, 2, 0)
			end
		else
			if SN.data.profile.raidbuffs.orientation == "VERTICAL" then
				RB.Buffs[k]:SetPoint("TOPLEFT", RB.Buffs[counter], "BOTTOMLEFT", 0, -2)
			else
				RB.Buffs[k]:SetPoint("LEFT", RB.Buffs[counter], "RIGHT", 2, 0)
			end
		end
		counter = k
	end
end

function RB:RefreshBuffs()
	local ZoneName = GetRealZoneText()

	for k, v in pairs(buffs) do
		buffs[k] = 0
	end
	for k, v in pairs(missingBuffs) do
		wipe(missingBuffs[k])
	end
	
	for k, v in pairs(raidMembers) do
		for i, j in pairs(raidMembers[k]) do
			if type(v) == "boolean" then
				raidMembers[k][i] = false
			end
		end
	end

	for k, v in pairs(raidMembers) do
		for i, j in pairs(buffs) do
			raidMembers[k]:CheckBuffs(i)
		end
	end

	for i, j in pairs(raidMembers) do
		for k, v in pairs(buffs) do
			if not raidMembers[i][k] then 
				buffs[k] = buffs[k] + 1
				missingBuffs[k][i] = i
			end
		end
	end
end

function RB:DelayToggle(args)
	RB:ToggleActive(unpack(args))
end

function RB:ToggleActive(bool)
	if bool then
		UpdateDelay = 0
		RB.Buffs:SetScript("OnUpdate", function(self, elapsed)
			UpdateDelay = UpdateDelay + elapsed
			if UpdateDelay > 6 then
				if not IsInGroup() then
					for k, v in pairs(buffs) do
						self[k].text:SetText("0")
						buffs[k] = 0
						if missingBuffs[k] then
							wipe(missingBuffs[k])
						end
					end 
				else
					RB:RefreshBuffs()
					for k, v in pairs(buffs) do
						self[k].text:SetText(tostring(buffs[k]))
					end
				end
				UpdateDelay = 0
			end
		end)
		if not IsInGroup() then
			for k, v in pairs(buffs) do
				RB.Buffs[k].text:SetText("0")
				buffs[k] = 0
				if missingBuffs[k] then
					wipe(missingBuffs[k])
				end
			end 
		else
			RB:RefreshBuffs()
			for k, v in pairs(buffs) do
				RB.Buffs[k].text:SetText(tostring(buffs[k]))
			end
		end
		RB.Buffs:Show()
	else
		RB.Buffs:SetScript("OnUpdate", nil)
		RB.Buffs:Hide()
		collectgarbage("collect")
	end
end

local function Annouce(group)
	local report = {}
	local allgood = true

	SendChatMessage("{star}Serenity Now{star}", group)

	for a, b in pairs(buffs) do
		for c, d in pairs (missingBuffs[a]) do
			if not report[a] then
				local cachename = c:gsub("^(.*)%-.*$", "%1")
				report[a] = cachename
			else
				local cachename = c:gsub("^(.*)%-.*$", "%1")
				report[a] = strjoin(", ", report[a], cachename)
			end
				
			if a == "flask" then
				table.insert(missingFlask, c)
			elseif a == "food" then
				table.insert(missingFood, c)
			end
		end
		if report[a] then
			SendChatMessage("People missing "..bufftext[a]..": "..report[a], group)
		end
	end

	if SN.data.profile.raidbuffs.whisper then
		RB:ScheduleTimer("WhisperWarning", 5)
	end

	for a, b in pairs(report) do
		if b then
			allgood = false
			return
		end
	end

	if allgood then
		SendChatMessage("Buffs are all good", group)
	end
end

function RB:AnnounceBuffs()
	local group
	
	if SN:GroupType() == 1 then
		group = "PARTY"
	elseif SN:GroupType() == 2 then
		group = "RAID"
	end
	
	local inInstance, instanceType = IsInInstance()
	local instance = SN.data.profile.raidbuffs.instance

	if group then
		if instance then
			if inInstance then
				Annouce(group)
			else
				return
			end
		else
			Annouce(group)
		end
	end
end

function RB:WhisperWarning()
	--Msg the people missing flasks
	if #missingFlask > 0 then
		for k, v in pairs(missingFlask) do
			if v ~= GetUnitName("player", true) then
				SendChatMessage(bufftext["flask"], "WHISPER", nil, v)
			end
		end
	end
	--Msg the people missing food
	if #missingFood > 0 then
		for k, v in pairs(missingFood) do
			if v ~= GetUnitName("player", true) then
				SendChatMessage(bufftext["food"], "WHISPER", nil, v)
			end
		end
	end
	wipe(missingFlask)
	wipe(missingFood)
end

function RB:AnnounceSpecials(message, caster, spellId)
	local special = spellId and specials[spellId]

	if not message or not caster or not spellId or not special then return end

	
	local group = nil

	if SN:GroupType() == 1 then
		group = "PARTY"
	elseif SN:GroupType() == 2 then
		group = "RAID"
	end
	
	if not group then return end

	local isdead = UnitIsDeadOrGhost("player") or false

	local msg = ""
    local canSpeak = UnitIsGroupLeader("player") or UnitIsGroupAssistant("player")

	caster  = caster:gsub("%-[^%-]+$","")
	local label
	local link = spellId and GetSpellLink(spellId)
	local linksuffix = (link and " ("..link..")") or ""

	if special.label then
		label = special.label
	elseif link then
		label = link
		linksuffix = ""
	else
		label = special.name
	end
	
	if message == "Feast" then
		msg = caster .. " " .. ("prepares a %s!"):format(label)
		if special.bonus then
			local btext = _G["ITEM_MOD_"..special.bonus.."_SHORT"]
			if btext then
			msg = msg .. " (".."Bonus".." "..btext..")" 
			end
		end
	else
		msg = caster.." "..string.format("has set us up a %s", label)..linksuffix
	end

	if not msg or msg == "" then return	end

	if not canSpeak then return	end

	msg = "{star}Serenity Now{star} "..msg
	SendChatMessage(msg, "RAID_WARNING")
end