------------------------------------------------------------------------------------------
--Cast Bar--------------------------------------------------------------------------------
------------------------------------------------------------------------------------------
local SN, L, F, T, M, _ = unpack(select(2,...))
local LSM = LibStub("LibSharedMedia-3.0")
local MODNAME = "castbar"
local CB = SN:NewModule(MODNAME, "AceEvent-3.0")

local IsAddOnLoaded = IsAddOnLoaded
local GetTime = GetTime
local GetNetStats = GetNetStats
local unpack = unpack
local wipe = wipe
local RAID_CLASS_COLORS = RAID_CLASS_COLORS
local UnitCastingInfo = UnitCastingInfo
local UnitChannelInfo = UnitChannelInfo
local UnitClass = UnitClass

local channelTicks = {
	-- warlock
	[GetSpellInfo(1120)] = 6, -- drain soul
	[GetSpellInfo(689)] = 6, -- drain life
	[GetSpellInfo(103103)] = 4, -- malefic grasp
	[GetSpellInfo(108371)] = 6, -- harvest life
	-- druid
	[GetSpellInfo(740)] = 4, -- Tranquility
	[GetSpellInfo(16914)] = 10, -- Hurricane
	[GetSpellInfo(106996)] = 10, -- Astral Storm
	-- priest
	[GetSpellInfo(15407)] = 3, -- mind flay
	[GetSpellInfo(48045)] = 5, -- mind sear
	[GetSpellInfo(47540)] = 2, -- penance
	-- mage
	[GetSpellInfo(5143)] = 5, -- arcane missiles
	[GetSpellInfo(10)] = 8, -- blizzard
	[GetSpellInfo(12051)] = 3, -- evocation
}
local castbar
local isCastbarShown = false

local function getOptions()
	castbar = {
		type = "group",
		name = L["Castbar"],
		args = {
			enable = {
				order = 0,
				type = "toggle",
				name = L["Enable Castbar"],
				get = function(info)
					return SN.data.profile.castbar[info[#info]]
				end,
				set = function(info, value)
					SN.data.profile.castbar[info[#info]] = value
					if value then
						SN:EnableModule(MODNAME)
					else
						SN:DisableModule(MODNAME)
					end
					if (IsAddOnLoaded("ElvUI")) then
						SN:ElvOverride()
					end
				end,
			},
			lag = {
				order = 1,
				type = "toggle",
				name = L["Enable Latency"],
				hidden = function() return not SN.data.profile.castbar.enable end,
				get = function(info)
					return SN.data.profile.castbar[info[#info]]
				end,
				set = function(info, value)
					SN.data.profile.castbar[info[#info]] = value
				end,
			},
			h10 = {
				order = 2,
				type = "header",
				name = "Dimensions",
				hidden = function() return not SN.data.profile.castbar.enable end,
			},
			width = {
				type = "range",
				name = L["Width of Castbar"],
				hidden = function() return not SN.data.profile.castbar.enable end,
				order = 4,
				min = 50, max = 500, step = 1,
				get = function(info)
					return SN.data.profile.castbar[info[#info]] 
				end,
				set = function(info, value)
					SN.data.profile.castbar[info[#info]] = value
					CB:ApplySettings()
				end,
			},
			height = {
				type = "range",
				name = L["Height of Castbar"],
				hidden = function() return not SN.data.profile.castbar.enable end,
				order = 5,
				min = 12, max = 30, step = 1,
				get = function(info)
					return SN.data.profile.castbar[info[#info]] 
				end,
				set = function(info, value)
					SN.data.profile.castbar[info[#info]] = value
					CB:ApplySettings()
				end,
			},
			h12 = {
				order = 10,
				type = "header",
				name = "Texture",
				hidden = function() return not SN.data.profile.castbar.enable end,
			},
			texture = {
				type = "select",
				dialogControl = 'LSM30_Statusbar',
				name = L["Texture"],
				hidden = function() return not SN.data.profile.castbar.enable end,
				order = 12,
				values = AceGUIWidgetLSMlists.statusbar,
				get = function(info) return SN.data.profile.castbar[info[#info]] end,
				set = function(info, value)
					SN.data.profile.castbar[info[#info]] = value
					CB:ApplySettings()
				end,
			},
			show = {
				type = "execute",
				name = L["Show Castbar"],
				func = function()
					if not isCastbarShown then 
						F.castbar:Show()
						isCastbarShown = true
					else
						F.castbar:Hide()
						isCastbarShown = false
					end
				end,
			},
		},
	}
	return castbar
end

function CB:OnInitialize()
	SN:RegisterModuleOptions(MODNAME, getOptions, L["Castbar"])
	local db = SN.data.profile

	F.castframe = SN:CreateFrame("Cast Bar", MODNAME, UIParent)

	--The castbar statusbar holder
	F.castbar = CreateFrame("Frame", nil, F.castframe)
	F.castbar:SetPoint("CENTER", F.castframe)
	F.castbar:SetSize(db.castbar.width, db.castbar.height)

	SN:CreateBackDrop(F.castbar)

	local class, classFileName, classID = UnitClass("player")
	local colortmp = RAID_CLASS_COLORS[classFileName]
	local color = { colortmp.r, colortmp.g, colortmp.b }

	--The castbar statusbar
	F.cast = CreateFrame("StatusBar", nil, F.castbar)
	F.cast:SetPoint("CENTER", F.castbar)

	if (IsAddOnLoaded("ElvUI")) then
		F.cast:SetSize(db.castbar.width - 2, db.castbar.height - 2)
	else
		F.cast:SetSize(db.castbar.width - 2, db.castbar.height - 2)
	end
	F.cast:SetStatusBarTexture(LSM:Fetch("statusbar", db.castbar.texture))
	F.cast:SetStatusBarColor(unpack(color))


	F.cast:SetScript("OnEvent", function(self, event, ...)
		CB:CastbarEvents(self, event, ...)
	end)
	
	--The castbar icon
	F.cast.icon = F.castbar:CreateTexture(nil, "OVERLAY")
	F.cast.icon:SetPoint("TOPLEFT")
	F.cast.icon:SetPoint("BOTTOMLEFT")
	F.cast.icon:SetTexCoord(0.07, 0.93, 0.07, 0.93)

	--The castbar statusbar time text
	F.cast.timeText = F.cast:CreateFontString(nil, "OVERLAY")
	F.cast.timeText:SetPoint("RIGHT", F.cast)
	F.cast.timeText:SetFont(LSM:Fetch("font", db.castbar.font), db.castbar.fontsize, "OUTLINE")
	F.cast.timeText:SetJustifyH("RIGHT")
	F.cast.timeText:SetShadowOffset(1, -1)
	
	--The castbar statusbar spell text
	F.cast.castText = F.cast:CreateFontString(nil, "OVERLAY")
	F.cast.castText:SetPoint("LEFT", F.cast)
	F.cast.castText:SetFont(LSM:Fetch("font", db.castbar.font), db.castbar.fontsize, "OUTLINE")
	F.cast.castText:SetJustifyH("LEFT")
	F.cast.castText:SetShadowOffset(1, -1)
	
	--The latency frame
	F.cast.lag = F.cast:CreateTexture(nil, "OVERLAY")
	F.cast.lag:SetTexture(M.media.blankTex)
	
	F.cast:SetMinMaxValues(0, 1)
	F.cast:SetValue(0)
	F.castbar:Hide()
	F.cast.lag:Hide()
end

function CB:OnEnable()
	F.castframe:Show()
	CastingBarFrame:UnregisterAllEvents()
	CastingBarFrame:Hide()
	F.cast:SetScript("OnUpdate", function(self, elapsed)
		if (self.isCasting) or (self.isChanneling) then
			if self.isChanneling then
				self.value = self.value - elapsed
				if (self.value <= 0) then
					self:SetValue(self.maxValue)
					self.isCasting = nil
					self.isChanneling = nil
					self.timeText:SetText("")
					self.castText:SetText("")
					self.icon:SetTexture("")
					self:SetValue(0)
					F.castbar:Hide()
					if SN.data.profile.castbar.lag then
						self.lag:Hide()
					end
					return
				end
				self:SetValue(self.value)
			else
				self.value = self.value + elapsed
				if (self.value >= self.maxValue) then
					self:SetValue(self.maxValue)
					self.isCasting = nil
					self.isChanneling = nil
					self.timeText:SetText("")
					self.castText:SetText("")
					self.icon:SetTexture("")
					self:SetValue(0)
					F.castbar:Hide()
					if SN.data.profile.castbar.lag then
						self.lag:Hide()
					end
					return
				end
				self:SetValue(self.value)
			end
			--self.timeText:SetFormattedText("%.1f", self.value)
			self.timeText:SetText(("%.1f / %.1f"):format(self.value, self.maxValue))
		end
	end)
	--F.cast:RegisterEvent("UNIT_SPELLCAST_SENT")
	F.cast:RegisterUnitEvent("UNIT_SPELLCAST_START", "player", "vehicle")
	F.cast:RegisterUnitEvent("UNIT_SPELLCAST_STOP", "player", "vehicle")
	F.cast:RegisterUnitEvent("UNIT_SPELLCAST_FAILED", "player", "vehicle")
	F.cast:RegisterUnitEvent("UNIT_SPELLCAST_INTERRUPTED", "player", "vehicle")
	F.cast:RegisterUnitEvent("UNIT_SPELLCAST_DELAYED", "player", "vehicle")
	F.cast:RegisterUnitEvent("UNIT_SPELLCAST_CHANNEL_START", "player", "vehicle")
	F.cast:RegisterUnitEvent("UNIT_SPELLCAST_CHANNEL_STOP", "player", "vehicle")
	F.cast:RegisterUnitEvent("UNIT_SPELLCAST_CHANNEL_UPDATE", "player", "vehicle")
end

function CB:OnDisable()
	F.cast:UnregisterAllEvents()
	F.cast:SetScript("OnUpdate", nil)
	F.castframe:Hide()
	CastingBarFrame:GetScript("OnLoad")(CastingBarFrame)
	if (IsAddOnLoaded("ElvUI")) then
		SN:ElvOverride()
	end
end

function CB:Lock()
	if SN.data.profile.castbar.enable then
		SN:LockFrame(F.castframe)
	end
end

function CB:Unlock()
	if SN.data.profile.castbar.enable then
		F.castbar:Hide()	
		SN:UnlockFrame(F.castframe)
	end
end

function CB:Reset()
	local static = SN.DB.profile.frames
	local temp = SN.data.profile.frames
	if SN.data.profile.castbar.enable then
		F.castframe:ClearAllPoints()

		F.castframe:SetPoint(unpack(static.castbar))
	end
	wipe(temp.castbar)
end

function CB:ApplySettings()
	if SN.data.profile.castbar.enable then
		F.castframe:ClearAllPoints()
		F.castframe:SetPoint(unpack(SN.data.profile.frames.castbar))
		F.castframe:SetSize(SN.data.profile.castbar.width, SN.data.profile.castbar.height)
		F.castbar:SetSize(SN.data.profile.castbar.width, SN.data.profile.castbar.height)
		F.cast:SetSize(SN.data.profile.castbar.width - 2, SN.data.profile.castbar.height - 2)
		F.cast:SetStatusBarTexture(LSM:Fetch("statusbar", SN.data.profile.castbar.texture))
		F.cast.castText:SetFont(LSM:Fetch("font", SN.data.profile.castbar.font), SN.data.profile.castbar.fontsize, "OUTLINE")
		F.cast.timeText:SetFont(LSM:Fetch("font", SN.data.profile.castbar.font), SN.data.profile.castbar.fontsize, "OUTLINE")
		if (IsAddOnLoaded("ElvUI")) then
			SN:ScheduleTimer("ElvOverride", 5)
		end
	end
end

local sparkfactory = {
	__index = function(t,k)
		local spark = F.cast:CreateTexture(nil, 'OVERLAY')
		t[k] = spark
		spark:SetTexture("Interface\\CastingBar\\UI-CastingBar-Spark")
		spark:SetVertexColor(1, 1, 1, 0.5)
		spark:SetBlendMode('ADD')
		spark:SetWidth(16)
		spark:SetHeight(SN.data.profile.castbar.height)
		return spark
	end
}

local barticks = setmetatable({}, sparkfactory)

local function setBarTicks(ticknum)
	if( ticknum and ticknum > 0) then
		local delta = ( F.cast:GetWidth() / ticknum )
		for k = 1, ticknum do
			local t = barticks[k]
			t:ClearAllPoints()
			t:SetPoint("CENTER", F.cast, "LEFT", delta * (k - 1), 0 )
			t:Show()
		end
		for k = ticknum + 1, #barticks do
			barticks[k]:Hide()
		end
	else
		barticks[1].Hide = nil
		for i = 1, #barticks do
			barticks[i]:Hide()
		end
	end
end

local function getChannelingTicks(spell)
	return channelTicks[spell] or 0
end

function CB:CastbarEvents(self, event, ...)
	local unitID = ...
	if event == "UNIT_SPELLCAST_START" then
		F.castbar:Show()
		local spell, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitCastingInfo(unitID)
		if (spell) then
			self.isCasting = true
			self.value = (GetTime() - (startTime / 1000))
			self.maxValue = (endTime - startTime) / 1000
			self:SetMinMaxValues(0, self.maxValue)
			self:SetValue(self.value)
			self.timeText:SetText(("%.1f / %.1f"):format(self.value, self.maxValue))
			self.icon:SetTexture(icon)

			local size = F.castbar:GetHeight()
			self.icon:ClearAllPoints()
			self.icon:SetWidth(size - 4)
			self.icon:SetPoint("TOPLEFT", F.castbar, "TOPLEFT", 1, -1)
			self.icon:SetPoint("BOTTOMLEFT", F.castbar, "BOTTOMLEFT", 0, 1)

			self:SetWidth(F.castbar:GetWidth() - (size + 2))
			self:SetPoint("TOPLEFT", F.castbar, "TOPLEFT", size, - 1)
			self:SetPoint("BOTTOMLEFT", F.castbar, "BOTTOMLEFT", size, 1)

			setBarTicks(0)
			self.castText:SetText(displayName)
		end
		
		if SN.data.profile.castbar.lag then
			local _, _, _, ms = GetNetStats()
			if(ms ~= 0) then
				local perc = (self:GetWidth() / self.maxValue) * (ms / 1e5)
				if(perc > 1) then perc = 1 end
							
				self.lag:ClearAllPoints()
				local side
				if self.isCasting then
					side = "RIGHT"
					self.lag:SetTexCoord(1-perc,1,0,1)
				else -- channeling
					side = "LEFT"
					self.lag:SetTexCoord(perc,1,0,1)
				end
				self.lag:SetDrawLayer("OVERLAY")
				self.lag:SetVertexColor(0.69, 0.31, 0.31, 0.75)
				self.lag:SetPoint(side, self, side)
				self.lag:SetWidth(self:GetWidth() * perc)
				self.lag:SetHeight(self:GetHeight())
				self.lag:Show()
			else
				self.lag:Hide()
			end
		end
	end

	if event == "UNIT_SPELLCAST_CHANNEL_START" then
		F.castbar:Show()
		local spell, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitChannelInfo(unitID)	
		if (spell) then
			self.isChanneling = true
			self.value = ((endTime / 1000) - GetTime())
			self.maxValue = (endTime - startTime) / 1000
			self:SetMinMaxValues(0, self.maxValue)
			self:SetValue(self.value)
			self.timeText:SetText(self.maxValue)
			self.icon:SetTexture(icon)

			local size = F.castbar:GetHeight()
			self.icon:ClearAllPoints()
			self.icon:SetWidth(size - 4)
			self.icon:SetPoint("TOPLEFT", F.castbar, "TOPLEFT", 1, -1)
			self.icon:SetPoint("BOTTOMLEFT", F.castbar, "BOTTOMLEFT", 0, 1)

			self:SetWidth(F.castbar:GetWidth() - (size + 2))
			self:SetPoint("TOPLEFT", F.castbar, "TOPLEFT", size, - 1)
			self:SetPoint("BOTTOMLEFT", F.castbar, "BOTTOMLEFT", size, 1)

			self.castText:SetText(spell)
			
			setBarTicks(getChannelingTicks(spell))

			if SN.data.profile.castbar.lag and self.sendTime ~= nil then
				local _, _, _, ms = GetNetStats()
				if(ms ~= 0) then
					local perc = (self:GetWidth() / self.maxValue) * (ms / 1e5)
					if(perc > 1) then perc = 1 end
							
					self.lag:ClearAllPoints()
					local side
					if self.isCasting then
						side = "RIGHT"
						self.lag:SetTexCoord(1-perc,1,0,1)
					else -- channeling
						side = "LEFT"
						self.lag:SetTexCoord(perc,1,0,1)
					end
					self.lag:SetDrawLayer("OVERLAY")
					self.lag:SetVertexColor(0.69, 0.31, 0.31, 0.75)
					self.lag:SetPoint(side, self, side)
					self.lag:SetWidth(self:GetWidth() * perc)
					self.lag:SetHeight(self:GetHeight())
					self.lag:Show()
				else
					self.lag:Hide()
				end
			end
		end
	end

	if event == "UNIT_SPELLCAST_DELAYED" then
		local spell, rank, displayName, icon, startTime, endTime, isTradeSkill
		if (event == "UNIT_SPELLCAST_DELAYED") then
			spell, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitCastingInfo(unitID)
		else
			spell, rank, displayName, icon, startTime, endTime, isTradeSkill = UnitChannelInfo(unitID)
		end
   
		if (startTime == nil) then return end
   
		self.value = (GetTime() - (startTime / 1000))
		self.maxValue = (endTime - startTime) / 1000
		self:SetMinMaxValues(0, self.maxValue)
	end

	if event == "UNIT_SPELLCAST_STOP" then 
		self.isCasting = nil
		self.isChanneling = nil
		self.timeText:SetText("")
		self.castText:SetText("")
		self.icon:SetTexture("")
		self:SetValue(0)
		F.castbar:Hide()
		if SN.data.profile.castbar.lag then
			self.lag:Hide()
		end
	end

	if event == "UNIT_SPELLCAST_CHANNEL_STOP" then
		self.isCasting = nil
		self.isChanneling = nil
		self.timeText:SetText("")
		self.castText:SetText("")
		self.icon:SetTexture("")
		self:SetValue(0)
		F.castbar:Hide()
		if SN.data.profile.castbar.lag then
			self.lag:Hide()
		end
	end
end