
--[[
	type:
		none: (item) or (spell) of said name
		item: (item) partial name, item id
		type: (item) Quest, Herb, Metal & Stone, Gem, Leather, Cloth
		spell: (spell) partial name, spell id
		mount: (spell) flying, land, any, fflying, fland
		profession: (spell) primary, secondary, any
		pet: (macro) name, any, favorite
		toy: (item) favorite, any, partial
]]

local _,s = ...

s.rtable = {} -- reusable table where flyout button attributes are accumulated
local rtable = s.rtable

s.filter = {} -- table of search:keyword search functions (s.filter.item(arg))

-- adds a type/value attribute pair to rtable if it's not already there
local function addToTable(actionType,actionValue)
	for i=1,#rtable,2 do
		if rtable[i]==actionType and rtable[i+1]==actionValue then
			return
		end
	end
	tinsert(rtable,actionType)
	tinsert(rtable,actionValue)
end

-- returns true if arg and compareTo match (arg will likely be a case-insensitive pattern)
local function compare(arg,compareTo)
	return compareTo:match(arg) and true
end

--[[ Item Cache ]]

s.itemCache = {}
s.bagsToCache = {[0]=true,[1]=true,[2]=true,[3]=true,[4]=true,["Worn"]=true}

function s.BAG_UPDATE(self,bag)
	if bag>=0 and bag<=4 then
		s.bagsToCache[bag] = true
		s.StartTimer(0.05,s.CacheBags)
	end
end

function s.PLAYER_EQUIPMENT_CHANGED(self,slot,equipped)
	if equipped then
		s.bagsToCache.Worn = true
		s.StartTimer(0.05,s.CacheBags)
	end
end

local function addToCache(itemID)
	if itemID then
		local name = GetItemInfo(itemID)
		if name then
			s.itemCache[format("item:%d",itemID)] = name
		else
			s.StartTimer(0.05,s.CacheBags)
			return true
		end
	end
end

function s.CacheBags()
	local cacheComplete = true
	if not s.cacheTimeout or s.cacheTimeout < 10 then
		for bag in pairs(s.bagsToCache) do
			if bag=="Worn" then
				for slot=1,19 do
					local itemID = GetInventoryItemID("player",slot)
					if addToCache(itemID) then
						cacheComplete = false
					end
				end
			else
				for slot=1,GetContainerNumSlots(bag) do
					local itemID = GetContainerItemID(bag,slot)
					if addToCache(itemID) then
						cacheComplete = false
					end
				end
			end
		end
	end
	if cacheComplete then
		s.flyoutsNeedFilled = true
		wipe(s.bagsToCache)
		if s.firstLogin then
			s.firstLogin = nil
			s.FillAttributes()
		end
	else
		s.cacheTimeout = (s.cacheTimeout or 0)+1
	end
end

--[[ Toy Cache ]]

-- toy cache is backwards due to bugs with secure action buttons' inability to
-- cast a toy by item:id (and inability to SetMacroItem from a name /sigh)
-- cache is indexed by the toyName and equals the itemID
-- the attribValue for toys will be the toyName, and unsecure stuff can pull
-- the itemID from toyCache where needed
s.toyCache = {}
function s.TOYS_UPDATED()
	wipe(s.toyCache)
	-- note filter settings
	local filterCollected = C_ToyBox.GetFilterCollected()
	local filterUncollected = C_ToyBox.GetFilterUncollected()
	local sources = {}
	for i=1,10 do
		sources[i] = C_ToyBox.IsSourceTypeFiltered(i)
	end
	-- set filters to all toys
	C_ToyBox.SetFilterCollected(true)
	C_ToyBox.SetFilterUncollected(false) -- we don't need to uncollected toys
	C_ToyBox.ClearAllSourceTypesFiltered()
	C_ToyBox.SetFilterString("")

	-- fill cache with itemIDs = name
	for i=1,C_ToyBox.GetNumFilteredToys() do
		local itemID = C_ToyBox.GetToyFromIndex(i)
		local name = GetItemInfo(itemID) or "UNKNOWN"
		s.toyCache[name] = itemID
	end

	-- restore filters
	C_ToyBox.SetFilterCollected(filterCollected)
	C_ToyBox.SetFilterUncollected(filterUncollected)
	for i=1,10 do
		C_ToyBox.SetFilterSourceType(i,not sources[i])
	end
end

--[[ Filters ]]

-- for arguments without a search, look for items or spells by that name
function s.filter.none(arg)
	-- if a regular item in bags/on person
	if GetItemCount(arg)>0 then
		local _, link = GetItemInfo(arg)
		if link then
			addToTable("item",(link:match("(item:%d+)")))
			return
		end
	end
	-- if a spell
	local spellName,subName = GetSpellInfo(arg)
	if spellName and spellName~="" then
		if subName and subName~="" then
			addToTable("spell",format("%s(%s)",spellName,subName)) -- for Polymorph(Turtle)
		else
			addToTable("spell",spellName)
		end
		return
	end
	-- if a toy
	local toyName = GetItemInfo(arg)
	if toyName and s.toyCache[toyName] then
		addToTable("item",toyName)
	end
end

-- item:id will get all items of that itemID
-- item:name will get all items that contain "name" in its name
function s.filter.item(arg)
	local itemID = tonumber(arg)
	if itemID and GetItemCount(itemID)>0 then
		addToTable("item",format("item:%d",itemID))
		return
	end
	-- look for arg in itemCache
	for itemID,name in pairs(s.itemCache) do
		if name:match(arg) and GetItemCount(name)>0 then
			addToTable("item",itemID)
		end
	end
end
s.filter.i = s.filter.item

-- spell:id will get all spells of that spellID
-- spell:name will get all spells that contain "name" in its name or its flyout parent
function s.filter.spell(arg)
	if type(arg)=="number" and IsSpellKnown(arg) then
		local name = GetSpellInfo(arg)
		if name then
			addToTable("spell",name)
			return
		end
	end
	-- look for arg in the spellbook
	for i=1,2 do
		local _,_,offset,numSpells = GetSpellTabInfo(i)
		for j=offset+1, offset+numSpells do
			local spellType,spellID = GetSpellBookItemInfo(j,"spell")
			local name = GetSpellBookItemName(j,"spell")
			if name and name:match(arg) then
				if spellType=="SPELL" and IsSpellKnown(spellID) then
					addToTable("spell",name)
				elseif spellType=="FLYOUT" then
					local _, _, numFlyoutSlots, isFlyoutKnown = GetFlyoutInfo(spellID)
					if isFlyoutKnown then
						for k=1,numFlyoutSlots do
							local _,_,flyoutSpellKnown,flyoutSpellName = GetFlyoutSlotInfo(spellID,k)
							if flyoutSpellKnown then
								addToTable("spell",flyoutSpellName)
							end
						end
					end
				end
			end
		end
	end
end
s.filter.s = s.filter.spell

-- type:quest will get all quest items in bags, or those on person with Quest in a type field
-- type:name will get all items that have "name" in its type, subtype or slot name
function s.filter.type(arg)
	if ("quest"):match(arg) then
		-- many quest items don't have "Quest" in a type field, but GetContainerItemQuestInfo
		-- has them flagged as quests.  check those first
		for i=0,4 do
			for j=1,GetContainerNumSlots(i) do
				local isQuestItem, questID, isActive = GetContainerItemQuestInfo(i,j)
				if isQuestItem or questID or isActive then
					addToTable("item",format("item:%d",GetContainerItemID(i,j)))
				end
			end
		end
	end
	-- some quest items can be marked quest as an item type also
	for itemID,name in pairs(s.itemCache) do
		if GetItemCount(name)>0 then
			local _, _, _, _, _, itemType, itemSubType, _, itemSlot = GetItemInfo(itemID)
			if itemType and (itemType:match(arg) or itemSubType:match(arg) or itemSlot:match(arg)) then
				addToTable("item",itemID)
			end
		end
	end
end
s.filter.t = s.filter.type

-- mount:any, mount:flying, mount:land, mount:favorite, mount:fflying, mount:fland

if select(4,GetBuildInfo())==50400 then -- if on MoP client

	function s.filter.mount(arg)
		local any = compare(arg,"Any")
		local flying = compare(arg,"Flying")
		local land = compare(arg,"Land")
		local fflying, fland, favorite
		local canCheckFavorites = IsAddOnLoaded("Select Favorite Mounts")
		if canCheckFavorites then
			fflying = compare(arg,"FFlying") or compare(arg,"FavFlying")
			fland = compare(arg,"FLand") or compare(arg,"FavLand")
			favorite = compare(arg,"Favorite") or fflying or fland
		end
		for i=1,GetNumCompanions("mount") do
			local _, _, spellID, _, _, mountType = GetCompanionInfo("mount",i)
			local mountName = GetSpellInfo(spellID) -- mount name isn't necessarily spell name
			if mountName and mountType then
				local flyingMount = bit.band(mountType,0x2)==0x2
				local landMount = bit.band(mountType,0x1)==0x1 and bit.band(mountType,0x2)==0
				if favorite and SelectFavoriteMounts:IsFavorite(mountName) then
					if fflying then
						if flyingMount then
							addToTable("spell",mountName)
						end
					elseif fland then
						if landMount then
							addToTable("spell",mountName)
						end
					else
						addToTable("spell",mountName)
					end
				elseif flyingMount and flying then
					addToTable("spell",mountName)
				elseif landMount and land then
					addToTable("spell",mountName)
				elseif any or mountName:match(arg) then
					addToTable("spell",mountName)
				end
			end
		end
	end
	s.filter.m = s.filter.mount

else -- WoD client

	-- mount:arg filters mounts that include arg in the name or arg="flying" or arg="land" or arg=="any"
	function s.filter.mount(arg)

		local any = compare(arg,"Any")
		local flying = compare(arg,"Flying")
		local land = compare(arg,"Land")
		local fflying, fland, favorite
		local	fflying = compare(arg,"FFlying") or compare(arg,"FavFlying")
		local	fland = compare(arg,"FLand") or compare(arg,"FavLand")
		local	favorite = compare(arg,"Favorite") or fflying or fland

		for i=1,C_MountJournal.GetNumMounts() do
			local mountName, mountSpellId, mountTexture, _, canSummon, _, isFavorite = C_MountJournal.GetMountInfo(i)
			if mountName and canSummon then
				local _,_,_,_,mountType = C_MountJournal.GetMountInfoExtra(i)
				local canFly = mountType==247 or mountType==248
				if favorite and isFavorite then
					if (fflying and canFly) or (fland and not canFly) or (not fflying and not fland) then
						addToTable("spell",mountName)
					end
				elseif (lowerarg=="flying" and canFly) or (lowerarg=="land" and not canFly) then
					addToTable("spell",mountName)
				elseif any or mountName:match(arg) then
					addToTable("spell",mountName)
				end
			end
		end
	end
	s.filter.m = s.filter.mount

end

-- profession:arg filters professions that include arg in the name or arg="primary" or arg="secondary" or arg="all"
function s.filter.profession(arg)
	s.professions = s.professions or {}
	wipe(s.professions)
	s.RunForEach(function(entry) tinsert(s.professions,entry or false) end,GetProfessions())
	local any = compare(arg,"Any")
	local primaryOnly = compare(arg,"Primary")
	local secondaryOnly = compare(arg,"Secondary")

	for index,profession in pairs(s.professions) do
		if profession then
			local name, _, _, _, numSpells, offset = GetProfessionInfo(profession)
			if (index<3 and primaryOnly) or (index>2 and secondaryOnly) or any or name:match(arg) then
				for i=1,numSpells do
					local _, spellID = GetSpellBookItemInfo(offset+i,"professions")
					addToTable("spell",(GetSpellInfo(spellID)))
				end
			end
		end
	end
end


-- pet:arg filters companion pets that include arg in the name or arg="any" or arg="favorite(s)"
function s.filter.pet(arg,rtable)
	local any = compare(arg,"Any")
	local favorite = compare(arg,"Favorite")
	if not any and not favorite then
		local name = arg:gsub("%[[A-Z][a-z]%]",function(c) return c:sub(2,2) end) -- convert arg to a regular string
		local speciesID,petID = C_PetJournal.FindPetIDByName(name)
		if petID then
			local _,customName,_,_,_,_,_,realName = C_PetJournal.GetPetInfoByPetID(petID)
			addToTable("macro",format("/summonpet %s",customName or realName))
			return
		end
	end
	-- the following can create 150-200k of garbage...why? pets are officially unsupported so this is permitted to stay
	for i=1,C_PetJournal.GetNumPets() do
		local petID,_,owned,customName,_,isFavorite,_,realName = C_PetJournal.GetPetInfoByIndex(i)
		if petID and owned then
			if any or (favorite and isFavorite) or (customName and customName:match(arg)) or (realName and realName:match(arg)) then
				addToTable("macro",format("/summonpet %s",customName or realName))
			end
		end
	end
end
s.filter.p = s.filter.pet

if s.WoDBuild then
	-- toy:arg filters items from the toybox; arg="favorite" "any" or partial name
	function s.filter.toy(arg)
		local any = compare(arg,"Any")
		local favorite = compare(arg,"Favorite")
		if favorite then -- toy:favorite
			for toyName,itemID in pairs(s.toyCache) do
				if C_ToyBox.GetIsFavorite(itemID) then
					addToTable("item",toyName)
				end
			end
		elseif any then -- toy:any
			for toyName in pairs(s.toyCache) do
				addToTable("item",toyName)
			end
		else -- toy:name
			for toyName in pairs(s.toyCache) do
				if toyName:match(arg) then
					addToTable("item",toyName)
				end
			end
		end
	end
end
