local Abacus = Rock("LibAbacus-3.0", false, true)
local Tablet = Rock("Tablet-2.0", false, true)

RaidSaveFu = Rock:NewAddon("RaidSaveFu", "LibRockDB-1.0", "LibRockEvent-1.0", "LibRockConfig-1.0", "LibFuBarPlugin-3.0")
local RaidSaveFu = RaidSaveFu
RaidSaveFu.title = "RaidSaveFu"
RaidSaveFu:SetDatabase("RaidSaveFuDB");

RaidSaveFu:SetDatabaseDefaults('profile', {
	textwhenzero = false,
	shorttext = false,
	showid = true,
	showcooldown = true,
	formatcooldown = "Full",
	showreset = true,
	formatdate = "Chinese",
	formattime = "Show24"
} )
RaidSaveFu:SetFuBarOption('tooltipType', "Tablet-2.0")
RaidSaveFu:SetFuBarOption('iconPath', "Interface\\Addons\\FuBar_RaidSaveFu\\icon.tga")
RaidSaveFu:SetFuBarOption('hasNoColor', true)
RaidSaveFu:SetFuBarOption('cannotHideText', false)
RaidSaveFu:SetFuBarOption('clickableTooltip', true)

function RaidSaveFu:OnInitialize()
	local optionsTable = {
		name = "FuBar_RaidSaveFu",
		desc = self.notes,
		handler = RaidSaveFu,
		type = 'group',
		args = {
			instance = {
				type = 'group',
				name = "Instance information",
				desc = "Select instance information to be displayed",
				order = 20,
				args = {
					showID = {
						type = 'toggle',
						name = "Show instance IDs",
						desc = "Show or hide the numeric instance IDs",
						order = 10,
						get = function(value)
							return self.db.profile.showid
						end,
						set = function(value)
							self.db.profile.showid = not self.db.profile.showid
							self:UpdateFuBarTooltip()
							return self.db.profile.showid
						end,
					},
					showCooldown = {
						type = 'toggle',
						name = "Show time remaining",
						desc = "Display the time the instance ID is still valid",
						order = 20,
						get = function(value)
							return self.db.profile.showcooldown
						end,
						set = function(value)
							self.db.profile.showcooldown = not self.db.profile.showcooldown
							self:UpdateFuBarTooltip()
							return self.db.profile.showcooldown
						end,
					},
					formatCooldown = {
						type = 'choice',
						name = "Remaining time format",
						desc = "Display style for the time until reset",
						order = 30,
						get = function(value)
								return self.db.profile.formatcooldown
						end,
						set = function(value)
								self.db.profile.formatcooldown = value
						end,
						choiceType = "dict",
						choices = { 
							Extended = "Extended",
							Full = "Full",
							Short = "Short",
							Condensed = "Condensed",
						},
						choiceDescs = {
							Extended = "Yrs, Mnths, Days, Hrs, Mins, Secs",
							Full = "First two of Yrs, Mnths, Days, Hrs, Mins, Secs",
							Short = "First of Yrs, Mnths, Days, Hrs, Mins, Secs",
							Condensed = "Digits only",
						},
						choiceOrder = { "Extended", "Full", "Short", "Condensed", },
					},
					showReset = {
						type = 'toggle',
						name = "Show reset date and time",
						desc = "Display the date (and/or time) the instance ID expires",
						order = 40,
						get = function(value)
							return self.db.profile.showreset
						end,
						set = function(value)
							self.db.profile.showreset = not self.db.profile.showreset
							self:UpdateFuBarTooltip()
							return self.db.profile.showreset
						end,
					},
					formatDate = {
						type = 'choice',
						name = "Expiry date format",
						desc = "Display style for the date instance IDs expire",
						order = 50,
						get = function(value)
								return self.db.profile.formatdate
						end,
						set = function(value)
								self.db.profile.formatdate = value
						end,
						choiceType = "dict",
						choices = { 
							Chinese = "Chinese",
							USAmerican = "USAmerican",
							USAmericanS = "USAmerican (Short)",
							European = "European",
							EuropeanS = "European (Short)",
						},
						choiceDescs = {
							Chinese = "YYYY-MM-DD",
							USAmerican = "MM/DD/YYYY",
							USAmericanS = "MM/DD/YY",
							European = "DD/MM/YYYY",
							EuropeanS = "DD/MM/YY",
						},
						choiceOrder = { "Chinese", "USAmerican", "USAmericanS", "European", "EuropeanS", },
					},
					formatTime = {
						type = 'choice',
						name = "Expiry time format",
						desc = "Display style for the time instance IDs expire",
						order = 60,
						get = function(value)
								return self.db.profile.formattime
						end,
						set = function(value)
								self.db.profile.formattime = value
						end,
						choiceType = "dict",
						choices = { 
							Hide = "Hide",
							Show12 = "12-Hour",
							Show24 = "24-Hour",
						},
						choiceDescs = {
							Hide = "Time not shown",
							Show12 = "H:MM TT",
							Show24 = "HH:MM",
						},
						choiceOrder = { "Hide", "Show12", "Show24", },
					},
				}
			},
			textWhenZero = {
				type = 'toggle',
				name = "Hide text for no saved instances",
				desc = "Show or hide the FuBar text when the character has no saved instances",
				order = 40,
				get = function(value)
					return self.db.profile.textwhenzero
				end,
				set = function(value)
					self.db.profile.textwhenzero = not self.db.profile.textwhenzero
					self:UpdateFuBarText()
					return self.db.profile.textwhenzero
				end,
			},
			shortText = {
				type = 'toggle',
				name = "Short FuBar text",
				desc = "Only show the number of saved instances (no extra text)",
				order = 30,
				get = function(value)
					return self.db.profile.shorttext
				end,
				set = function(value)
					self.db.profile.shorttext = not self.db.profile.shorttext
					self:UpdateFuBarText()
					return self.db.profile.shorttext
				end,
			},
		}
	}
	RaidSaveFu:SetConfigTable(optionsTable)
	RaidSaveFu:SetConfigSlashCommand("/raidsavefu")
	RaidSaveFu.OnMenuRequest = optionsTable
	if self.db.profile.realm == nil then
		self.db.profile.realm = { }
	end
	if self.db.profile.realm[GetRealmName()] == nil then
		self.db.profile.realm[GetRealmName()] = { }
	end
	if self.db.profile.realm[GetRealmName()].char == nil then
		self.db.profile.realm[GetRealmName()].char = { }
	end
	if self.db.profile.realm[GetRealmName()].name == nil then
		self.db.profile.realm[GetRealmName()].name = GetRealmName()
	end
	if self.db.profile.realm[GetRealmName()].char[GetUnitName("player")] == nil then
		self.db.profile.realm[GetRealmName()].char[GetUnitName("player")] = { }	
	end
	local player = self.db.profile.realm[GetRealmName()].char[GetUnitName("player")]
	if player.numsaved == nil then
		player.numsaved = 0
	end
	if player.instance == nil then
		player.instance = { }
	end
	player.name = GetUnitName("player")
	player.class = strupper(UnitClass("player"))
end

function RaidSaveFu:OnEnable()
	self:AddEventListener("Blizzard", "UPDATE_INSTANCE_INFO", "Refresh")
	self:AddEventListener("Blizzard", "RAID_INSTANCE_WELCOME", "RequestRefresh")
	self:AddEventListener("Blizzard", "CHAT_MSG_SYSTEM", "CheckSystemMessage")
end

function RaidSaveFu:OnDisable()
	self:RemoveEventListener("Blizzard", "UPDATE_INSTANCE_INFO")
	self:RemoveEventListener("Blizzard", "RAID_INSTANCE_WELCOME")
	self:RemoveEventListener("Blizzard", "CHAT_MSG_SYSTEM")
end

function RaidSaveFu:OnUpdateFuBarText()
	local player = self.db.profile.realm[GetRealmName()].char[GetUnitName("player")]
	local fubartext
	if self.db.profile.shorttext then
		fubartext = string.format("%d", player.numsaved)
	elseif player.numsaved == 1 then
		fubartext = string.format("%d Saved Instance", player.numsaved)
	else
		fubartext = string.format("%d Saved Instances", player.numsaved)
	end
	
	if player.numsaved == 0 and self.db.profile.textwhenzero then
		self:SetFuBarText("");
	else
		self:SetFuBarText(fubartext)
	end
end

function RaidSaveFu:OnUpdateFuBarTooltip()
	local player = self.db.profile.realm[GetRealmName()].char[GetUnitName("player")]
	local numcolumns = 1
	if self.db.profile.showid then
		numcolumns = numcolumns + 1
	end
	if self.db.profile.showcooldown then
		numcolumns = numcolumns + 1
	end
	if self.db.profile.showreset then
		numcolumns = numcolumns + 1
	end
	local headings = { }
	if self.db.profile.showid then
		headings[1] = HIGHLIGHT_FONT_COLOR_CODE.."Instance ID|r"
		headings[2] = HIGHLIGHT_FONT_COLOR_CODE.."Instance|r"
	else
		headings[1] = HIGHLIGHT_FONT_COLOR_CODE.."Instance|r"
	end
	if self.db.profile.showcooldown then
		if self.db.profile.showid then
			headings[3] = HIGHLIGHT_FONT_COLOR_CODE.."Remaining|r"
		else
			headings[2] = HIGHLIGHT_FONT_COLOR_CODE.."Remaining|r"
		end
	end
	if self.db.profile.showreset then
		if self.db.profile.showid and self.db.profile.showcooldown then
			headings[4] = HIGHLIGHT_FONT_COLOR_CODE.."Expires|r"
		elseif self.db.profile.showid or self.db.profile.showcooldown then
			headings[3] = HIGHLIGHT_FONT_COLOR_CODE.."Expires|r"
		else
			headings[2] = HIGHLIGHT_FONT_COLOR_CODE.."Expires|r"
		end
	end			
	local cat = Tablet:AddCategory(
		'columns', numcolumns
	)
	if player.numsaved > 0 then
		local displaytable = { }
		local emptytooltip = true -- track the emptiness of the tooltip for inserting spacers between toons
		local i
		for i = 1, player.numsaved do
			if player.instance[i].expires > time() then
				local display = { "", "", "", "" }
				local namestring = player.instance[i].name
				local idstring = GRAY_FONT_COLOR_CODE..player.instance[i].id..FONT_COLOR_CODE_CLOSE
				local cooldownstring = ""
				if self.db.profile.formatcooldown == "Extended" then
					cooldownstring = Abacus:FormatDurationExtended(player.instance[i].expires - time())
				elseif self.db.profile.formatcooldown == "Condensed" then
					cooldownstring = Abacus:FormatDurationCondensed(player.instance[i].expires - time())
				elseif self.db.profile.formatcooldown == "Short" then
					cooldownstring = Abacus:FormatDurationShort(player.instance[i].expires - time())
				else -- Full style
					cooldownstring = SecondsToTime(player.instance[i].expires - time())
				end
				if player.instance[i].expires - time() <= 86400 then --less than a day
					cooldownstring = RED_FONT_COLOR_CODE..cooldownstring..FONT_COLOR_CODE_CLOSE
				else -- more than a day
					cooldownstring = GREEN_FONT_COLOR_CODE..cooldownstring..FONT_COLOR_CODE_CLOSE
				end
				local expirestring = self:FormatDate(player.instance[i].expires)
				expirestring = expirestring.." "..self:FormatTime(player.instance[i].expires)
				expirestring = strtrim(expirestring)
				if self.db.profile.showid then
					display[1] = idstring
					display[2] = namestring
				else
					display[1] = namestring
				end
				if self.db.profile.showcooldown then
					if self.db.profile.showid then
						display[3] = cooldownstring
					else
						display[2] = cooldownstring
					end
				end
				if self.db.profile.showreset then
					if self.db.profile.showid and self.db.profile.showcooldown then
						display[4] = expirestring
					elseif self.db.profile.showid or self.db.profile.showcooldown then
						display[3] = expirestring
					else
						display[2] = expirestring
					end
				end			
				table.insert(displaytable, display)
			end
		end
		if # displaytable > 0 then
			emptytooltip = false
			cat:AddLine(
				'text', self:ClassColorise(player.class, player.name)
			)
			if numcolumns > 1 then
				cat:AddLine(
					'text', headings[1],
					'text2', headings[2],
					'text3', headings[3],
					'text4', headings[4]
				)
				local i
				for i = 1, # displaytable do
					cat:AddLine(
						'text', displaytable[i][1],
						'text2', displaytable[i][2],
						'text3', displaytable[i][3],
						'text4', displaytable[i][4]
					)
				end
			else -- numcolumns == 1
				local i
				for i = 1, # displaytable do
					cat:AddLine(
						'text', displaytable[i][1]
					)
				end
			end
		end
	end
	local char = self.db.profile.realm[GetRealmName()].char
	local k, v
	for k, v in pairs(char) do
		if v.numsaved > 0 and v.name ~= player.name then
			local displaytable = { }
			local i
			for i = 1, v.numsaved do
				if v.instance[i].expires > time() then
					local display = { "", "", "", "" }
					local namestring = v.instance[i].name
					local idstring = GRAY_FONT_COLOR_CODE..v.instance[i].id..FONT_COLOR_CODE_CLOSE
					local cooldownstring = ""
					if self.db.profile.formatcooldown == "Extended" then
						cooldownstring = Abacus:FormatDurationExtended(v.instance[i].expires - time())
					elseif self.db.profile.formatcooldown == "Condensed" then
						cooldownstring = Abacus:FormatDurationCondensed(v.instance[i].expires - time())
					elseif self.db.profile.formatcooldown == "Short" then
						cooldownstring = Abacus:FormatDurationShort(v.instance[i].expires - time())
					else -- Full style
						cooldownstring = SecondsToTime(v.instance[i].expires - time())
					end
					if v.instance[i].expires - time() <= 86400 then --less than a day
						cooldownstring = RED_FONT_COLOR_CODE..cooldownstring..FONT_COLOR_CODE_CLOSE
					else -- more than a day
						cooldownstring = GREEN_FONT_COLOR_CODE..cooldownstring..FONT_COLOR_CODE_CLOSE
					end
					local expirestring = self:FormatDate(v.instance[i].expires)
					expirestring = expirestring.." "..self:FormatTime(v.instance[i].expires)
					expirestring = strtrim(expirestring) 
					if self.db.profile.showid then
						display[1] = idstring
						display[2] = namestring
					else
						display[1] = namestring
					end
					if self.db.profile.showcooldown then
						if self.db.profile.showid then
							display[3] = cooldownstring
						else
							display[2] = cooldownstring
						end
					end
					if self.db.profile.showreset then
						if self.db.profile.showid and self.db.profile.showcooldown then
							display[4] = expirestring
						elseif self.db.profile.showid or self.db.profile.showcooldown then
							display[3] = expirestring
						else
							display[2] = expirestring
						end
					end			
					table.insert(displaytable, display)
				end
			end
			if # displaytable > 0 then
				if not emptytooltip then
					cat:AddLine(
						'text', ""
					)
				end
				emptytooltip = false
				cat:AddLine(
					'text', self:ClassColorise(v.class, v.name)
				)
				if numcolumns > 1 then
					cat:AddLine(
						'text', headings[1],
						'text2', headings[2],
						'text3', headings[3],
						'text4', headings[4]
					)
					local i
					for i = 1, # displaytable do
						cat:AddLine(
							'text', displaytable[i][1],
							'text2', displaytable[i][2],
							'text3', displaytable[i][3],
							'text4', displaytable[i][4]
						)
					end
				else -- numcolumns == 1
					local i
					for i = 1, # displaytable do
						cat:AddLine(
							'text', displaytable[i][1]
						)
					end
				end
			end
		end
	end
end

function RaidSaveFu:CheckSystemMessage()
	if tostring(arg1) == INSTANCE_SAVED then
		self:RequestRefresh()
	end
end

function RaidSaveFu:RequestRefresh()
	RequestRaidInfo()
end

function RaidSaveFu:Refresh()
	local player = self.db.profile.realm[GetRealmName()].char[GetUnitName("player")]
	player.numsaved = GetNumSavedInstances()
	player.instance = { }
	if player.numsaved == 0 then
	else
		local iname, iid, iexpires, temp
		for i = 1, player.numsaved do
			iname, iid, iexpires = GetSavedInstanceInfo(i)
			iexpires = iexpires + time()
			temp = { name = iname, id = iid, expires = iexpires }
			table.insert(player.instance, temp)
		end
	end
	self:UpdateFuBarPlugin()
end

function RaidSaveFu:GetDaysInMonth(year, month)
	local function is_leap_year(year)
		return year % 4 == 0 and (year % 100 ~= 0 or year % 400 == 0)
	end
	return month == 2 and is_leap_year(year) and 29 or ("\31\28\31\30\31\30\31\31\30\31\30\31"):byte(month)
end

function RaidSaveFu:ClassColorise(class, targetstring)
	local r = string.format("%x", math.floor(RAID_CLASS_COLORS[gsub(class, " ", "")].r * 255))
	if strlen(r) == 1 then
		r = "0"..r
	end
	local g = string.format("%x", math.floor(RAID_CLASS_COLORS[gsub(class, " ", "")].g * 255))
	if strlen(g) == 1 then
		g = "0"..g
	end
	local b = string.format("%x", math.floor(RAID_CLASS_COLORS[gsub(class, " ", "")].b * 255))
	if strlen(b) == 1 then
		b = "0"..b
	end
	return "|cff"..r..g..b..targetstring..FONT_COLOR_CODE_CLOSE
end

function RaidSaveFu:FormatDate(seconds)
	local datestring = ""
	local tempdate = date("*t", seconds)
	if self.db.profile.formatdate == "USAmerican" then
		datestring = datestring..tostring(tempdate.month).."/"
		if tempdate.day < 10 then
			datestring = datestring.."0"
		end
		datestring = datestring..tostring(tempdate.day).."/"
		datestring = datestring..tostring(tempdate.year)
	elseif self.db.profile.formatdate == "USAmericanS" then
		datestring = datestring..tostring(tempdate.month).."/"
		if tempdate.day < 10 then
			datestring = datestring.."0"
		end
		datestring = datestring..tostring(tempdate.day).."/"
		local yearstring = tostring(tempdate.year)
		datestring = datestring..strsub(yearstring, strlen(yearstring) - 1, strlen(yearstring))
	elseif self.db.profile.formatdate == "European" then
		datestring = datestring..tostring(tempdate.day).."/"
		if tempdate.month < 10 then
			datestring = datestring.."0"
		end
		datestring = datestring..tostring(tempdate.month).."/"
		datestring = datestring..tostring(tempdate.year)
	elseif self.db.profile.formatdate == "EuropeanS" then
		datestring = datestring..tostring(tempdate.day).."/"
		if tempdate.month < 10 then
			datestring = datestring.."0"
		end
		datestring = datestring..tostring(tempdate.month).."/"
		local yearstring = tostring(tempdate.year)
		datestring = datestring..strsub(yearstring, strlen(yearstring) - 1, strlen(yearstring))
	else -- Chinese
		datestring = datestring..tostring(tempdate.year).."-"
		if tempdate.month < 10 then
			datestring = datestring.."0"
		end
		datestring = datestring..tostring(tempdate.month).."-"
		if tempdate.day < 10 then
			datestring = datestring.."0"
		end
		datestring = datestring..tostring(tempdate.day)
	end
	return datestring
end

function RaidSaveFu:FormatTime(seconds)
	local timestring = ""
	if self.db.profile.formattime == "Show12" then
		local tempdate = date("*t", seconds)
		local am = true
		local hour = tempdate.hour
		if hour > 11 then
			am = false
		elseif hour > 12 then
			hour = hour - 12
		elseif hour == 0 then
			hour = 12
		end
		timestring = timestring..tostring(hour)..":"
		if tempdate.min < 10 then
			timestring = timestring.."0"
		end
		timestring = timestring..tostring(tempdate.min)
		if am then
			timestring = timestring.." AM"
		else -- pm
			timestring = timestring.." PM"
		end
	elseif self.db.profile.formattime == "Show24" then
		local tempdate = date("*t", seconds)
		timestring = timestring..tostring(tempdate.hour)..":"
		if tempdate.min < 10 then
			timestring = timestring.."0"
		end
		timestring = timestring..tostring(tempdate.min)
	end
	return timestring
end