local ADDON, data = ...

local MAX_LEVEL = 90
local GEAR = 'Gear'
local AURA = 'Aura'

local tblOops = {}              -- Primary warnings: bad stats, missing auras
local tblOops2 = {}             -- Secondary warnings: missing gems, missing enchants
local tblOops3 = {}             -- Tertiary warnings: missing buffs
local tblOops4 = {}             -- <What word comes next...?> warnings: consumables
local tblShoppingListGems = {}
local tblShoppingListEnch = {}
local tooltip = ''

local PLAIN_LETTER = 8383   -- For use in generating simple tooltips
local LibWeagleTooltip = LibStub("LibWeagleTooltip-2.1")

local THROTTLE = 1
local inCombat = false
local class

local _G = _G

------------------------------------------------------------------
--      iter_cycle
do
    local function next (self)
        local ret = self.values[self.cur]
        self.cur = self.cur + 1
        if self.cur>#self.values then
            self.cur = 1
        end
        return ret
    end
   
    function iter_cycle (...)
        local obj = {
            values = {...},
            cur = 1,
            next = next,
        }
      
        return obj
    end   
end

------------------------------------------------------------------

local tblProfessions = {}
local function InitProfessions ()
    local prof1, prof2 = GetProfessions()

    for i,v in ipairs ({prof1,prof2}) do
        local name, _, rank, maxRank, numSpells, _, skillLine = GetProfessionInfo(v)
        if rank==maxRank then
            tblProfessions[skillLine] = true
        end
    end
    
    -- Enchanters: rings are enchantable
    if tblProfessions[data.professionIds.Enchanting] then
        data.slots_Enchantable[INVSLOT_FINGER1] = true
        data.slots_Enchantable[INVSLOT_FINGER2] = true
    end
end
------------------------------------------------------------------

local currentRaidDifficulty = 0
local function UpdateRaidDifficulty ()
    local _,_, difficultyID,_,_,_,_, instanceMapID = GetInstanceInfo()
    currentRaidDifficulty = data.difficultyFromID[difficultyID]
end

------------------------------------------------------------------

 tblWrongStats = {}    
 tblSlightlyWrongStats = {}
-- These will be set to point at the correct sets of wrong stats
local function UpdateWrongStats ()
    local spec = _G.GetSpecialization('player')

    if data.classification[class]==nil  or  spec==nil then 
        tblWrongStats = {}
        tblSlightlyWrongStats = {}
    else
        tblWrongStats = data.wrongStats[data.classification[class][spec]] or {}
        tblSlightlyWrongStats = data.slightlyWrongStats[data.classification[class][spec]] or {}
    end
end
------------------------------------------------------------------

local function VerifyItemStats (itemLink)
    -- Check against the bad stats table, return false for a bad item.
    local tblStats = _G.GetItemStats (itemLink)  or  {}     -- The {} is just to prevent a stupid login error, really
    
    for stat,statVal in _G.pairs(tblStats) do
        -- Check wrong stats
        for i,wrongStat in _G.pairs(tblWrongStats) do
            if stat==wrongStat then
                return false
            end
        end
        
        -- Check slightly wrong stats
        if g_SanityCheck_config.StrictStats then
            for i,wrongStat in _G.pairs(tblSlightlyWrongStats) do
                if stat==wrongStat then
                    return false
                end
            end
        end
    end
    
    return true
end

------------------------------------------------------------------

local function VerifyItemType (itemLink)
    -- For high level toons, also check ilvl and quality. The ilvl threshold is pretty arbitrary.
    -- Return false for a bad item.
    if _G.UnitLevel('player') == MAX_LEVEL then
        local _,_,qual, ilvl = _G.GetItemInfo (itemLink)
        if qual==nil or ilvl==nil then      -- This seems to happen occasionally... must be a bug of some sort of while logging on...
            return true
        end
        if qual<=1 or ilvl<=300 then
            return false
        end
    end
    
    return true
end

------------------------------------------------------------------

local function ShoppingList_CountSockets (itemLink)
    if not itemLink then return end
    
    local _,_,_, ilvl,_,_,_,_,equipSlot = _G.GetItemInfo(itemLink)
    if ilvl==nil or ilvl < g_SanityCheck_config['GemMinLvl'] then return end

    
    local tmp
    local i = 1
    while true do
        tmp = LibWeagleTooltip:GetTooltipLine(itemLink, i)
        if not tmp then 
            break
        else 
            if tmp:match ('Socket') and not tmp:match('Bonus') then 
                tblShoppingListGems[tmp] = (tblShoppingListGems[tmp] or 0) + 1
            end            
            i = i+1
        end
    end
end

------------------------------------------------------------------

local function CountSockets_Total (itemLink)
    if itemLink==nil then return 0 end
    local tblStats = _G.GetItemStats (itemLink)
    local ret = 0

    if tblStats==nil then tblStats={} end -- Just to prevent a stupid login error, really
    for k,v in _G.pairs(tblStats) do
        if k:match('SOCKET') then 
            ret = ret+v
        end
    end
    
    return ret
end

------------------------------------------------------------------

local function CountSockets_Gemmed (itemLink)
    if itemLink==nil then return 0 end
    local itemId, enchantId, gem1, gem2, gem3, gem4 = itemLink:match("item:(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)")
    
    local ret = 0
    for i,v in _G.ipairs({gem1,gem2,gem3,gem4}) do
        if v~='0' then
            ret = ret+1
        end
    end
    
    return ret
end

------------------------------------------------------------------

local function StringHasBadStat (str)
    if str==nil then 
        return false
    end
    
    for i,stat in _G.pairs(tblWrongStats) do
        if str:match (data.statNames[stat]) then
            return true
        end
    end
    
    return false
end

------------------------------------------------------------------

local function VerifySocketAndEnchantStats (itemLink)
    -- Generate a tooltip of a plain letter (statless, cached item) with the same enchant and gems as this item
    local itemId, enchantId, gem1, gem2, gem3, gem4 = itemLink:match("item:(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)")
    local tooltip = ("item:%i:%i:%i:%i:%i:%i"):format(PLAIN_LETTER, enchantId, gem1, gem2, gem3, gem4)

    -- Get the enchant from line 2 of the tooltip, and up to 4 gems from the next 4 lines
    local tmp = {}  -- This table will contain the enchant text, followed by up to 4 gem texts
    for i=2,6 do
        tmp[i-1] =  LibWeagleTooltip:GetTooltipLine(tooltip, i)
    end
    
    for k,v in _G.pairs(tmp) do
        if StringHasBadStat(v) then
            return false
        end
    end

    return true
end

------------------------------------------------------------------

local function VerifyItemSockets (itemLink)
    local _,_,_, ilvl,_,_,_,_,equipSlot = _G.GetItemInfo(itemLink)
    
    if ilvl==nil or ilvl < g_SanityCheck_config['GemMinLvl'] then
        return true
    else
        return (CountSockets_Total(itemLink) <= CountSockets_Gemmed(itemLink))
    end
end

------------------------------------------------------------------

local function VerifyExtraSocket (slot)
    -- Only if the item is gemmed to begin with
    local itemLink = _G.GetInventoryItemLink ('player', slot)
    if not itemLink then return true end
    
    local _,_,_, ilvl = _G.GetItemInfo(itemLink)
    
    if ilvl==nil or ilvl < g_SanityCheck_config['EnchMinLvl'] then
        return true
    end
    
    return (CountSockets_Total(itemLink)+1 == CountSockets_Gemmed(itemLink))
end

local function VerifyBeltBuckle ()
    return VerifyExtraSocket (INVSLOT_WAIST)
end


------------------------------------------------------------------

local function VerifyItemEnchant (itemLink)
    local _,_,_, ilvl = _G.GetItemInfo(itemLink)
    
    if ilvl==nil or ilvl < g_SanityCheck_config['EnchMinLvl'] then
        return true
    else
        local itemId, enchantId = itemLink:match("item:(%d+):(%d+)")
        return enchantId~='0'
    end
end

------------------------------------------------------------------

local function VerifyRoleAndSpec ()
    local spec = _G.GetSpecialization('player')
    local role = _G.UnitGroupRolesAssigned('player')
    
    if data.roles[class]==nil or spec==nil or _G.UnitGroupRolesAssigned('player')=='NONE' then return true,'' end
    
    if data.roles[class][spec] == role then
        return true, ''
    else
        local _, spec = _G.GetSpecializationInfo(_G.GetSpecialization())
        return false, 'Are you sure you want to '..data.verbs[role]..' in '..spec..' spec?'
    end
end

------------------------------------------------------------------

local function VerifyProfessions_Blacksmithing ()
    return VerifyExtraSocket (INVSLOT_HAND) and VerifyExtraSocket (INVSLOT_WRIST)
end

------------------------------------------------------------------

local function VerifyProfessions_Engineering ()
    -- Currently not verifying the belt, because, let's face it, all existing belt tinkers are bullshit :p
    return GetItemSpell(GetInventoryItemLink('player',INVSLOT_HAND)) and GetItemSpell(GetInventoryItemLink('player',INVSLOT_BACK))
end

------------------------------------------------------------------

local function VerifyFlask ()
    if (not g_SanityCheck_config.ReadyToRaid.Flask[currentRaidDifficulty]) then
        return true,''
    end
    
    for _,name in _G.ipairs(data.flasks) do
        if HasAura ('player', name) then
            return true,''
        end
    end
    return false, 'Don\'t forget to eat a flask!'
end

------------------------------------------------------------------

local function VerifyFood ()
    -- Verify that either the player has the Well Fed buff, or does not wish to be reminded of it in the current difficulty
    if (not g_SanityCheck_config.ReadyToRaid.Food[currentRaidDifficulty]) or HasAura ('player', 'Well Fed') then
        return true,''
    end
    return false, 'You look hungry!'
end

------------------------------------------------------------------

local function VerifySymbiosis ()
    if _G.GetNumGroupMembers()==0 then
        return true,''
    end
    
    if class~='Druid' or HasAura('player', 'Symbiosis') then
        return true,''
    end
    return false, 'A true druid would have used Symbiosis!'
end

------------------------------------------------------------------

local function CallVerifyInv (func, link, tbl, msg)            -- A wrapper for inventory verification functions
    if not func(link) then
        _G.table.insert (tbl, msg)
    end
end
------------------------------------------------------------------

local function CheckInventory ()
    for slot,slotName in _G.pairs(data.slots) do
        local link = _G.GetInventoryItemLink ('player', slot)
        if link~=nil then
            CallVerifyInv (VerifyItemType, link,                tblOops[GEAR],      'Low level item: '..slotName)
            CallVerifyInv (VerifyItemStats, link,               tblOops[GEAR],      'Bad item: '..slotName)
            CallVerifyInv (VerifyItemSockets, link,             tblOops2[GEAR],     'Missing gems: '..slotName)
            CallVerifyInv (VerifySocketAndEnchantStats, link,   tblOops2[GEAR],     'Bad gems/ enchant: '..slotName)
            
            if data.slots_Enchantable[slot] then
                CallVerifyInv (VerifyItemEnchant, link,             tblOops2[GEAR],     'Missing enchant: '..slotName)
            end
        else
            -- Mention if the item is missing - unless it's an offhand, and there's a 2H equipped
            -- Low level toons can get away with empty slots
            if (slot==INVSLOT_OFFHAND and Has2HanderEquipped()) or (_G.UnitLevel('player')<60) then
                -- All is well
            else
                _G.table.insert (tblOops[GEAR], 'Missing item: '..slotName)
            end
        end
    end
    
    CallVerifyInv (VerifyBeltBuckle, nil, tblOops2[GEAR], 'Hey, how about a belt buckle and an extra gem?')
    
    if tblProfessions[data.professionIds.Blacksmithing] then
        CallVerifyInv (VerifyProfessions_Blacksmithing, nil, tblOops2[GEAR], 'Don\'t forget the extra sockets you can get as a Blacksmith!')
    end
    
    if tblProfessions[data.professionIds.Engineering] then
        CallVerifyInv (VerifyProfessions_Engineering, nil, tblOops2[GEAR], 'Don\'t forget to put some engineering gizmos on your gear!')
    end
end

------------------------------------------------------------------

local function CallVerifyAura (func, tbl)      -- A wrapper for aura verification functions
    local ret, msg = func()
    if not ret then
        _G.table.insert (tbl, msg)
    end
end

------------------------------------------------------------------

local function CheckAuras ()
    local spec = _G.GetSpecialization('player')
    
    -- Personal auras and stuff
    if data.verifyAuras[class]==nil or spec==nil then return end
    
    local funcs
    local tmp = data.verifyAuras[class][spec]
    if _G.type(tmp)=='function' then
        funcs = {tmp}
    else
        funcs = tmp
    end
    
    for i,func in _G.ipairs(funcs) do
        CallVerifyAura (func, tblOops[AURA])
    end
    
    -- Raid buffs
    if data.verifyRaidBuffs[class] then
        CallVerifyAura (data.verifyRaidBuffs[class], tblOops3[AURA])
    end
 
    -- Role
    CallVerifyAura (VerifyRoleAndSpec,  tblOops[AURA])
    
    -- Ready to raid
    CallVerifyAura (VerifyFlask,        tblOops4[AURA])
    CallVerifyAura (VerifyFood,         tblOops4[AURA])
    CallVerifyAura (VerifySymbiosis,    tblOops3[AURA])     -- Well OBVIOUSLY this is only relevant for druids...

end

------------------------------------------------------------------

local function Init ()
    -- Main frame thing. This controls all the lights but is not visible in itself, to allow for skins etc.
    do
        local f = CreateFrame ("Frame", 'frmSanityCheck', UIParent)
        
        f.needGearScan = false
        f.needAuraScan = false
        
        f.elapsed2 = 0
        
        function f:GetTooltipText_Light1 ()
            local ret = {}
            for key,subtbl in _G.pairs(tblOops) do
                if #subtbl>0 then
                    for k,v in pairs(subtbl) do
                        table.insert (ret, v)
                    end
                end
            end
            return ret
        end


        function f:GetTooltipText_Light2 ()
            local ret = {}
            for key,subtbl in _G.pairs(tblOops2) do
                for k,v in pairs(subtbl) do
                    table.insert (ret, v)
                end
            end
            return ret
        end


        function f:GetTooltipText_Light3 ()
            local ret = {}
            
            for key,subtbl in _G.pairs(tblOops3) do      -- Iterating over the GEAR and AURA sub-tables
                for k,v in pairs(subtbl) do
                        table.insert (ret, v)
                end
            end
            return ret
        end


        function f:GetTooltipText_Light4 ()
            local ret = {}
            for key,subtbl in _G.pairs(tblOops4) do      -- Iterating over the consumables table
                for k,v in pairs(subtbl) do
                    table.insert (ret, v)
                end
            end
            return ret
        end
        
    end
end

------------------------------------------------------------------
--  Events and such
local function RegisterEvents ()
    local f = frmSanityCheck
    function f:Scan ()
        -- Aura scanning is much lighter than gear scanning, so we always do it
        
        -- Clear tables
        tblOops[AURA] = {}
        tblOops2[AURA] = {}
        tblOops3[AURA] = {}
        tblOops4[AURA] = {}
        if self.needGearScan then
            tblOops[GEAR] = {}
            tblOops2[GEAR] = {}
            tblOops3[GEAR] = {}
            tblOops4[GEAR] = {}
            tblShoppingListGems = {}
            tblShoppingListEnch = {}
        end

        -- Now do stuff
        if self.needGearScan and (not inCombat) then
            CheckInventory ()
            self.needGearScan = false
        end
            
        CheckAuras ()
        self.needAuraScan = false
        
        if #tblOops[GEAR] > 0 or #tblOops[AURA] > 0 then
            frmSanityCheck_Light1:Light ()
        else
            frmSanityCheck_Light1:Dark ()
        end

        if #tblOops2[GEAR] > 0 or #tblOops2[AURA] > 0 then
            frmSanityCheck_Light2:Light ()
        else
            frmSanityCheck_Light2:Dark ()
        end

        if #tblOops3[GEAR] > 0 or #tblOops3[AURA] > 0 then
            frmSanityCheck_Light3:Light ()
        else
            frmSanityCheck_Light3:Dark ()
        end        
        
        if #tblOops4[GEAR] > 0 or #tblOops4[AURA] > 0 then
            frmSanityCheck_Light4:Light ()
        else
            frmSanityCheck_Light4:Dark ()
        end
        
        if g_SanityCheck_config.HideWhenAwesome and (not frmSanityCheck_Light1.isOn) and (not frmSanityCheck_Light2.isOn) and (not frmSanityCheck_Light3.isOn) and (not frmSanityCheck_Light4.isOn) then
            frmSanityCheck_Parent:Hide ()
        else
            frmSanityCheck_Parent:Show ()
        end
    end
    
    f.needGearScan = true
    f.needAuraScan = true
    UpdateRaidDifficulty ()
    inCombat = _G.UnitAffectingCombat('player')
    
    --f:RegisterUnitEvent ("UNIT_AURA", 'player')
    f:RegisterEvent ("UNIT_AURA")
    f:RegisterUnitEvent ("UNIT_PET", 'player')
    f:RegisterEvent ("UNIT_INVENTORY_CHANGED")
    f:RegisterEvent ("SOCKET_INFO_UPDATE")
    f:RegisterEvent ("PLAYER_SPECIALIZATION_CHANGED")
    f:RegisterEvent ("GROUP_ROSTER_UPDATE")   -- To detect role change!
    f:RegisterEvent ("PLAYER_REGEN_ENABLED")
    f:RegisterEvent ("PLAYER_REGEN_DISABLED")
    f:RegisterEvent ("ZONE_CHANGED")
    f:RegisterEvent ("ZONE_CHANGED_NEW_AREA")
    f:RegisterEvent ("PLAYER_DIFFICULTY_CHANGED")
    
    f:SetScript ("OnEvent", function  (self, event, GUID, ...)
        if event=='PLAYER_REGEN_DISABLED' then
            inCombat = true
        elseif event=='PLAYER_REGEN_ENABLED' then
            inCombat = false
        elseif event=='UNIT_INVENTORY_CHANGED' then
            self.needGearScan = true
        elseif event=='UNIT_PET' or event=='UNIT_AURA' then
            self.needAuraScan = true
        elseif event=='PLAYER_SPECIALIZATION_CHANGED' then
            UpdateWrongStats ()
            self.needGearScan = true
            self.needAuraScan = true
        elseif event=='ZONE_CHANGED' or event=='ZONE_CHANGED_NEW_AREA' or event=='PLAYER_DIFFICULTY_CHANGED' then
            UpdateRaidDifficulty ()
        end
    end)
    function f:OnUpdate (elapsed, ...)
        self.elapsed2 = self.elapsed2 + elapsed
        
        --if self.elapsed2 > THROTTLE and (self.needGearScan or self.needAuraScan) then
        if ((not self.needGearScan) and self.needAuraScan and self.elapsed2 > THROTTLE)  or  (self.needGearScan and self.elapsed2 > THROTTLE * 3) then
            self:Scan ()
            self.elapsed2 = 0
            if (not self.isOn) and (not self.frmLight2.isOn) and (not self.frmLight3.isOn) and (not self.frmLight4.isOn) and g_SanityCheck_config.HideWhenAwesome then
                self:Hide ()
            else
                self:Show ()
            end
        end
    end

    local updater = CreateFrame ("Frame", '', UIParent)
    updater:SetScript ("OnUpdate", function (self, elapsed, ...)
        frmSanityCheck:OnUpdate (elapsed)
    end)
end


------------------------------------------------------------------

function CreateShoppingList ()
    -- A point!
    -- Because the frame gets anchored to the fontstring, and fontstrings apparently can't be moved (?)
    local point = CreateFrame ("Frame", 'frmShoppingList_point', UIParent)
    point:SetSize (1,1)
    point:SetPoint ("CENTER", UIParent, "CENTER")
    
    local f = CreateFrame ("Frame", 'frmShoppingList', UIParent)
    f:SetBackdrop ({
        bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
        edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Gold-Border",
        edgeSize = 8,
    })

    f.lbl = f:CreateFontString(nil, "BACKGROUND", "GameFontNormal")
    f.lbl:SetPoint ("CENTER", point, "CENTER")
    f.lbl:SetTextColor (1.0,1.0,1.0)
    f.lbl:SetJustifyH ("LEFT")
    
    f:SetPoint ("TOPLEFT", f.lbl, "TOPLEFT", -12, 12)
    f:SetPoint ("BOTTOMRIGHT", f.lbl, "BOTTOMRIGHT", 12, -12)
    f:Hide ()
    
    -- Make movable
    point:SetMovable (true)
    point:RegisterForDrag ("LeftButton")
    point:SetUserPlaced (true)
    point:SetClampedToScreen (true)
    point:EnableMouse (true)
    f:SetScript("OnMouseDown", function(self, button)
        if button=='LeftButton' then
            frmShoppingList_point:StartMoving ()
        elseif button=='RightButton' then
            frmShoppingList_point:StopMovingOrSizing ()
            self:Hide ()
        end
    end)
    f:SetScript("OnMouseUp", function(self, button)
        frmShoppingList_point:StopMovingOrSizing ()
    end)

    function f:Generate ()
        tblShoppingListGems = {}
        tblShoppingListEnch = {}

        for slot,slotName in _G.pairs(data.slots) do
            local link = _G.GetInventoryItemLink ('player', slot)
            if link~=nil then
                if data.slots_Enchantable[slot] and (not VerifyItemEnchant(link)) then
                    tblShoppingListEnch[slotName] = 1
                end
                
                ShoppingList_CountSockets (link)
            end
        end    
    
        local txt = 'Empty sockets:\n'
        for k,v in pairs(tblShoppingListGems) do
            txt = txt..v..'x  '..k..'\n'
        end
        txt = txt..'\nMissing enchants:\n'
        for k,v in pairs(tblShoppingListEnch) do
            txt = txt..k..'\n'
        end
        self.lbl:SetText (txt)
        
        self:Show ()
    end
end

------------------------------------------------------------------
------------------------------------------------------------------

g_SanityCheck_config = g_SanityCheck_config or {}

Init ()

local starter = CreateFrame ("Frame")
starter:RegisterEvent ("PLAYER_LOGIN")
starter:RegisterEvent ("ADDON_LOADED")
starter:SetScript ("OnEvent", function  (self, event, ...)
    if event=="PLAYER_LOGIN" then
        RegisterEvents ()
        InitProfessions ()
        frmSanityCheck.needGearScan = true
        frmSanityCheck.needAuraScan = true
        UpdateRaidDifficulty ()
        inCombat = _G.UnitAffectingCombat('player')
        class = _G.UnitClass('player')
        UpdateWrongStats ()
    elseif event=="ADDON_LOADED" then
        local arg1 = ...
        if arg1==ADDON then
            --BuildGui ()
            data.skins[g_SanityCheck_config.skin or 'Classic'].build ()
            CreateShoppingList ()
            SanityCheck_CreateOptions ()
            UpdateWrongStats () -- Yes, I need to call it on both events because who knows which one happens first, and I need both, and ugh.
        end
    end
end)