--[[
Copyright (c) 2013-2014 Simon Ward

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
--]]

local addon_name = "RoleCall"
local addon_title = GetAddOnInfo(addon_name, "Title")
local addon_version = GetAddOnMetadata(addon_name, "X-Curse-Packaged-Version") or GetAddOnMetadata(addon_name, "Version")

RoleCall = LibStub("AceAddon-3.0"):NewAddon(addon_name, "AceConsole-3.0", "AceEvent-3.0", "AceBucket-3.0")

local L = LibStub("AceLocale-3.0"):GetLocale(addon_name, true)

local LDBIcon = LibStub("LibDBIcon-1.0", true)
local libGroupInSpecT = LibStub("LibGroupInSpecT-1.0")
local LibQTip = LibStub("LibQTip-1.0")

local UnitName, UnitGUID, UnitExists, UnitClass, UnitGroupRolesAssigned, IsInGroup, IsInRaid =
      UnitName, UnitGUID, UnitExists, UnitClass, UnitGroupRolesAssigned, IsInGroup, IsInRaid

local CreateFrame, UIParent =
      CreateFrame, UIParent

local ROW_HEIGHT = 20
local MAX_ROWS = 10

local BUTTON_WIDTH = 120
local BUTTON_HEIGHT = 22

local COLUMN_GAP = 5

local NAME_LABEL_WIDTH = 140
local CLASS_LABEL_WIDTH = 100
local SPEC_NAME_LABEL_WIDTH = 100
local SPEC_ROLE_LABEL_WIDTH = 80
local ROLE_ICON_WIDTH = 19

RoleCall.options = {
    name = addon_title,
    handler = RoleCall,
    type = "group",
    childGroups = "tab",
    args = {
        main = {
            name = L["Options"],
            desc = L["Main Options"],
            type = "group",
            args = {
                notifyRoleChange = {
                    name = L["Notify on role changes"],
                    desc = L["Notify you when someone in your group changes roles"],
                    type = "toggle",
                    width = "double",
                    get = function() return RoleCall.db.profile.notifyRoleChange end,
                    set = function(_, v)
                        RoleCall.db.profile.notifyRoleChange = v and true or false
                    end,
                },
                notifySpecChange = {
                    name = L["Notify on specialization changes"],
                    desc = L["Notify you when someone in your group changes specializations"],
                    type = "toggle",
                    width = "double",
                    get = function() return RoleCall.db.profile.notifySpecChange end,
                    set = function(_, v)
                        RoleCall.db.profile.notifySpecChange = v and true or false
                    end,
                },
                reportReadyCheck = {
                    name = L["Report on Ready Check"],
                    desc = L["Report role mismatches when a Ready Check is performed"],
                    type = "toggle",
                    width = "double",
                    get = function() return RoleCall.db.profile.reportReadyCheck end,
                    set = function(_, v)
                        RoleCall.db.profile.reportReadyCheck = v and true or false
                        if v then
                            RoleCall:RegisterEvent("READY_CHECK")
                        else
                            RoleCall:UnregisterEvent("READY_CHECK")
                        end
                    end,
                },
            },
        },
    },
}

local defaults = {
    profile = {
        notifyRoleChange = true,
        notifySpecChange = true,
        reportReadyCheck = true,
    },
}

local roster_unitid = {}
local roster_name = {}
local roster_localized_class = {}
local roster_class = {}
local roster_level = {}
local roster_assigned_role = {}
local roster_spec_icon = {}
local roster_spec_name = {}
local roster_spec_role = {}
local roster_spec_blizz_role = {}
local roster_talents = {}
local roster_glyphs = {}

local roster_remove_unitid = {}
local roster_add_unitid = {}

local party_units = {}
local raid_units = {}

local main_window

local ldb_tooltip
local talents_tooltip

local unit_frames = {}

local sort_buttons = {}

local roster_sorted_guids = {}
local guids_sorted

local in_combat

local role_count

do
    tinsert(party_units, "player")

    for i = 1, MAX_PARTY_MEMBERS do
        tinsert(party_units, string.format("party%d", i))
    end

    for i = 1, MAX_RAID_MEMBERS do
        tinsert(raid_units, string.format("raid%d", i))
    end
end

local localized_blizz_role = {
    ["DAMAGER"] = L["DPS"],
    ["HEALER"] = L["Healer"],
    ["TANK"] = L["Tank"],
}

local localized_spec_role = {
    ["healer"] = L["Healer"],
    ["melee"] = L["Melee"],
    ["ranged"] = L["Ranged"],
    ["tank"] = L["Tank"],
}

local role_texture_coord = {
    ["DAMAGER"] = {20/64, 39/64, 22/64, 41/64},
    ["HEALER"] =  {20/64, 39/64,  1/64, 20/64},
    ["TANK"] =    { 0/64, 19/64, 22/64, 41/64},
}

local slash_handlers = {
    [L["assign"]] = function() RoleCall:AssignAllRoles() end,
    [L["report"]] = function() RoleCall:ReportRoleMismatches() end,
    [L["rescan"]] = function() RoleCall:Rescan() end,
}

local glyph_levels = {25, 50, 75}

local class_colors = setmetatable({}, {__index = function(t, k)
    if type(k) == "nil" then k = "" end
    local color = (CUSTOM_CLASS_COLORS and CUSTOM_CLASS_COLORS[k] or RAID_CLASS_COLORS[k]) or GRAY_FONT_COLOR
    rawset(t, k, color)
    return color
end })

local class_color_formatters = setmetatable({}, {__index = function(t, k)
    if type(k) == "nil" then k = "" end
    local color = class_colors[k]
    local formatter = string.format("|cff%02x%02x%02x%%s|r", color.r * 255, color.g * 255, color.b * 255)
    rawset(t, k, formatter)
    return formatter
end })

local function CompareGUIDsByName(guid1, guid2, reverse)
    local name1, name2 = roster_name[guid1], roster_name[guid2]
    if name1 and name2 then
        if name1 < name2 then
            return not reverse
        elseif name1 > name2 then
            return reverse
        end
    elseif name1 then
        return true
    end
end

local function CompareGUIDsByRoster(guid1, guid2, roster, reverse)
    local roster1, roster2 = roster[guid1], roster[guid2]
    if roster1 and roster2 then
        if roster1 < roster2 then
            return not reverse
        elseif roster1 > roster2 then
            return reverse
        end
    elseif roster1 then
        return true
    elseif roster2 then
        return false
    end
    return CompareGUIDsByName(guid1, guid2)
end

local function CompareGUIDsBySpecRole(guid1, guid2, reverse)
    local specRole1, specRole2 = roster_spec_role[guid1], roster_spec_role[guid2]
    specRole1 = specRole1 and localized_spec_role[specRole1]
    specRole2 = specRole2 and localized_spec_role[specRole2]
    if specRole1 and specRole2 then
        if specRole1 < specRole2 then
            return not reverse
        elseif specRole1 > specRole2 then
            return reverse
        end
    elseif specRole1 then
        return true
    elseif specRole2 then
        return false
    end
    return CompareGUIDsByName(guid1, guid2)
end

local sort_method = {
    ["name"] = CompareGUIDsByName,
    ["class"] = function (guid1, guid2, reverse) return CompareGUIDsByRoster(guid1, guid2, roster_localized_class, reverse) end,
    ["specName"] = function (guid1, guid2, reverse) return CompareGUIDsByRoster(guid1, guid2, roster_spec_name, reverse) end,
    ["specRole"] = CompareGUIDsBySpecRole,
}

local function SortButtonAction(frame)
    PlaySound("gsTitleOptionExit")
    RoleCall:ToggleSortMethod(frame.sortMethod)
end

local function TalentTierCount(class, level)
    local tierCount = 0
    local talentLevels = CLASS_TALENT_LEVELS[class] or CLASS_TALENT_LEVELS["DEFAULT"]
    for i = MAX_NUM_TALENT_TIERS, 1, -1 do
        if level >= talentLevels[i] then
            tierCount = i
            break
        end
    end
    return tierCount
end

local function MajorGlyphSlotCount(level)
    local slotCount = 0
    for i = 3, 1, -1 do
        if level >= glyph_levels[i] then
            slotCount = i
            break
        end
    end
    return slotCount
end

local function TableCount(t)
    local tableCount = 0
    for k in pairs(t) do tableCount = tableCount + 1 end
    return tableCount
end

local function TableShallowCopy(src, dst)
    if dst then wipe(dst) else dst = {} end
    for k, v in pairs(src) do dst[k] = v end
    return dst
end

local function TableKeysDiffer(t1, t2)
    if t1 ~= t2 then
        if t1 and t2 then
            for k in pairs(t1) do
                if not t2[k] then
                    return true
                end
            end
            for k in pairs(t2) do
                if not t1[k] then
                    return true
                end
            end
        else
            return true
        end
    end
end

local function BuildLDBTooltip()
    ldb_tooltip:AddHeader(addon_title)

    ldb_tooltip:SetCell(1, 2, addon_version, GameFontDisable)

    ldb_tooltip:SetFont(GameFontNormal)

    ldb_tooltip:SetCell(ldb_tooltip:AddLine(), 1, L["Click to toggle the group roster"], 2)
    ldb_tooltip:SetCell(ldb_tooltip:AddLine(), 1, L["Right-click for options"], 2)

    if not role_count then
    	role_count = {}

        for _, role in pairs(roster_spec_role) do
            role_count[role] = (role_count[role] or 0) + 1
        end

        for guid in pairs(roster_unitid) do
            local assignedRole = roster_assigned_role[guid]
            local specBlizzRole = roster_spec_blizz_role[guid]
            if assignedRole and specBlizzRole and assignedRole ~= specBlizzRole then
                role_count.mismatch_count = (role_count.mismatch_count or 0) + 1
            end

            if roster_spec_name[guid] then
				local talents = roster_talents[guid]
				local level = roster_level[guid]
				local talentCount = talents and TableCount(talents) or 0
				local tierCount = level and TalentTierCount(class, level) or 0

				if talentCount < tierCount then
					role_count.talent_missing = (role_count.talent_missing or 0) + 1
				end

				local glyphs = roster_glyphs[guid]
				local majorGlyphCount = 0
				if glyphs then
					for _, glyphInfo in pairs(glyphs) do
						if glyphInfo.glyph_type == GLYPH_TYPE_MAJOR then
							majorGlyphCount = majorGlyphCount + 1
						end
					end
				end
				if majorGlyphCount < MajorGlyphSlotCount(level) then
					role_count.glyph_missing = (role_count.glyph_missing or 0) + 1
				end
			end
        end
    end

    local healerCount = role_count.healer or 0
    local meleeCount = role_count.melee or 0
    local rangedCount = role_count.ranged or 0
    local tankCount = role_count.tank or 0
    local unknownCount = #roster_sorted_guids - (healerCount + meleeCount + rangedCount + tankCount)

    ldb_tooltip:SetFont(GameFontHighlight)

    ldb_tooltip:AddLine(" ")

    if healerCount > 0 then
        ldb_tooltip:AddLine(L["Healers:"], tostring(healerCount))
    end
    if meleeCount > 0 then
        ldb_tooltip:AddLine(L["Melee:"], tostring(meleeCount))
    end
    if rangedCount > 0 then
        ldb_tooltip:AddLine(L["Ranged:"], tostring(rangedCount))
    end
    if tankCount > 0 then
        ldb_tooltip:AddLine(L["Tanks:"], tostring(tankCount))
    end
    if unknownCount > 0 then
        ldb_tooltip:AddLine(L["Unknown:"], tostring(unknownCount))
    end

    local mismatchCount = role_count.mismatch_count or 0
    local talentMissingCount = role_count.talent_missing or 0
    local glyphMissingCount = role_count.glyph_missing or 0

    if mismatchCount > 0 or talentMissingCount > 0 or glyphMissingCount > 0 then
    	ldb_tooltip:AddLine(" ")
    	if mismatchCount > 0 then
			ldb_tooltip:SetFont(GameFontRed)
			ldb_tooltip:AddLine(L["Role Mismatches:"], tostring(mismatchCount))
		end
		if talentMissingCount > 0 then
			ldb_tooltip:SetFont(RoleCall_FontNormalOrange)
			ldb_tooltip:AddLine(L["Missing Talents:"], tostring(talentMissingCount))
		end
		if glyphMissingCount > 0 then
			ldb_tooltip:SetFont(GameFontNormal)
			ldb_tooltip:AddLine(L["Missing Major Glyphs:"], tostring(glyphMissingCount))
		end
    end
end

local function ShowLDBTooltip(frame)
    if ldb_tooltip then
        ldb_tooltip:Clear()
    else
        ldb_tooltip = LibQTip:Acquire("RoleCall_LDBTooltip", 2, "LEFT", "RIGHT")
    end

    BuildLDBTooltip()

    ldb_tooltip:SmartAnchorTo(frame)
    ldb_tooltip:Show()
end

local function ReleaseLDBTooltip(frame)
    if ldb_tooltip then
        LibQTip:Release(ldb_tooltip)
        ldb_tooltip = nil
    end
end

function RoleCall:OnInitialize()
    self.db = LibStub("AceDB-3.0"):New("RoleCallDB", defaults, true)

    self.db.RegisterCallback(self, "OnProfileChanged", "RefreshConfig")
    self.db.RegisterCallback(self, "OnProfileCopied", "RefreshConfig")
    self.db.RegisterCallback(self, "OnProfileReset", "RefreshConfig")

    self.options.args.profile = LibStub("AceDBOptions-3.0"):GetOptionsTable(self.db)

    LibStub("AceConfig-3.0"):RegisterOptionsTable(addon_name, self.options)

    local configDialog = LibStub("AceConfigDialog-3.0")

    local DataBroker = LibStub("LibDataBroker-1.1", true)
    if DataBroker then
        self.Broker = DataBroker:NewDataObject(addon_name, {
            type = "launcher",
            label = addon_title,
            icon = "Interface\\Icons\\Inv_inscription_runescrolloffortitude_blue",
            OnClick = function(self, button)
                if button == "RightButton" then
                    if configDialog.OpenFrames[addon_name] then
                        configDialog:Close(addon_name)
                    else
                        configDialog:Open(addon_name)
                    end
                else
                    RoleCall:ToggleMainWindow()
                end
            end,
            OnEnter = function(self)
                ShowLDBTooltip(self)
            end,
            OnLeave = function(self)
                ReleaseLDBTooltip(self)
            end,
        })
    end

    if LDBIcon then
        self.db.profile.minimap = self.db.profile.minimap or {}
        LDBIcon:Register(addon_name, self.Broker, self.db.profile.minimap)

        self.options.args.minimap = {
            name = L["Hide minimap icon"],
            desc = L["Hide minimap icon"],
            width = "double",
            type = "toggle",
            get = function() return self.db.profile.minimap.hide end,
            set = function(_, v)
                if v then
                    LDBIcon:Hide(addon_name)
                    self.db.profile.minimap.hide = true
                else
                    LDBIcon:Show(addon_name)
                    self.db.profile.minimap.hide = nil
                end
            end,
        }

        if self.db.profile.minimap.hide then
            LDBIcon:Hide(addon_name)
        else
            LDBIcon:Show(addon_name)
        end
    end

    self:RegisterChatCommand("rolecall", function(input)
                                            if not input or strtrim(input) == "" then
                                                RoleCall:ToggleMainWindow()
                                            else
                                                local argument = self:GetArgs(input)
                                                local handler = argument and slash_handlers[string.lower(argument)]

                                                if handler then
                                                    handler()
                                                else
                                                    configDialog:Open(addon_name)
                                                end
                                            end
                                         end)
end

function RoleCall:OnEnable()
    self:RegisterEvent("PLAYER_ENTERING_WORLD")
    self:RegisterEvent("GROUP_ROSTER_UPDATE")
    self:RegisterEvent("PLAYER_REGEN_DISABLED")
    self:RegisterEvent("PLAYER_REGEN_ENABLED")
    self:RegisterEvent("UNIT_NAME_UPDATE")
    self:RegisterEvent("UNIT_LEVEL")
    self:RegisterEvent("PLAYER_ROLES_ASSIGNED")

    if self.db.profile.reportReadyCheck then
        self:RegisterEvent("READY_CHECK")
    end

    if CUSTOM_CLASS_COLORS then
        CUSTOM_CLASS_COLORS:RegisterCallback("ClassColorsChanged", self)
    end

    libGroupInSpecT.RegisterCallback(self, "GroupInSpecT_Update")

    self:RegisterBucketMessage("RoleCall_UpdateLists", 0.2, "UpdateLists")
end

function RoleCall:OnDisable()
    libGroupInSpecT.UnregisterCallback(self, "GroupInSpecT_Update")

    if CUSTOM_CLASS_COLORS then
        CUSTOM_CLASS_COLORS:UnregisterCallback("ClassColorsChanged", self)
    end
end

function RoleCall:UpdateLists()
    if main_window and main_window:IsShown() then
        self:UpdateGroupList()
    end

    if ldb_tooltip then
        self:UpdateLDBTooltip()
    end
end

function RoleCall:ClassColorsChanged()
    wipe(class_colors)
    wipe(class_color_formatters)

    self:SendMessage("RoleCall_UpdateLists")
end

function RoleCall:RefreshConfig()
    if LDBIcon then
        self.db.profile.minimap = self.db.profile.minimap or {}

        LDBIcon:Refresh(addon_name, self.db.profile.minimap)

        if self.db.profile.minimap.hide then
            LDBIcon:Hide(addon_name)
        else
            LDBIcon:Show(addon_name)
        end
    end

    if self.db.profile.reportReadyCheck then
        self:RegisterEvent("READY_CHECK")
    else
        self:UnregisterEvent("READY_CHECK")
    end

    self:RestoreMainWindowPosition()

    guids_sorted = nil
    self:UpdateSortButtons()
    self:UpdateGroupList()
end

function RoleCall:UpdateRoster()
    roster_unitid, roster_remove_unitid = roster_remove_unitid, roster_unitid

    local changed

    local group_units = IsInRaid() and raid_units or party_units
    for i = 1, #group_units do
        local unitid = group_units[i]
        if UnitExists(unitid) then
            local name = GetUnitName(unitid, true)

            local guid = UnitGUID(unitid)
            local localizedClass, class = UnitClass(unitid)

            local level = UnitLevel(unitid)
            if level == -1 then level = nil end

            local assignedRole = UnitGroupRolesAssigned(unitid)
            if assignedRole == "NONE" then assignedRole = nil end

            local specName, specIcon, specRole, blizzRole, talents, glyphs
            local info = libGroupInSpecT:GetCachedInfo(guid)
            if info then
                specName, specIcon, specRole, blizzRole, talents, glyphs = info.spec_name_localized, info.spec_icon, info.spec_role_detailed, info.spec_role, info.talents, info.glyphs
                if blizzRole == "NONE" then blizzRole = nil end
            end

            local oldSpecName = roster_spec_name[guid]
            local oldSpecRole = roster_spec_role[guid]

            if roster_remove_unitid[guid] ~= unitid or
               roster_name[guid] ~= name or
               roster_localized_class[guid] ~= localizedClass or
               roster_class[guid] ~= class or
               roster_level[guid] ~= level or
               roster_assigned_role[guid] ~= assignedRole or
               roster_spec_icon[guid] ~= specIcon or
               oldSpecName ~= specName or
               oldSpecRole ~= specRole or
               roster_spec_blizz_role[guid] ~= blizzRole or
               TableKeysDiffer(roster_talents[guid], talents) or
               TableKeysDiffer(roster_glyphs[guid], glyphs) then
                changed = true
            end

            roster_unitid[guid] = unitid
            roster_name[guid] = name
            roster_localized_class[guid] = localizedClass
            roster_class[guid] = class
            roster_level[guid] = level

            roster_assigned_role[guid] = assignedRole

            roster_spec_icon[guid] = specIcon
            roster_spec_name[guid] = specName
            roster_spec_role[guid] = specRole
            roster_spec_blizz_role[guid] = blizzRole
            roster_talents[guid] = talents and TableShallowCopy(talents, roster_talents[guid]) or nil
            roster_glyphs[guid] = glyphs and TableShallowCopy(glyphs, roster_glyphs[guid]) or nil

            if oldSpecRole and oldSpecRole ~= specRole and self.db.profile.notifyRoleChange then
                self:NotifyRoleChange(guid, oldSpecRole, specRole, oldSpecName, specName)
            elseif oldSpecName and oldSpecName ~= specName and self.db.profile.notifySpecChange then
                self:NotifySpecChange(guid, oldSpecName, specName)
            end

            if roster_remove_unitid[guid] then
                roster_remove_unitid[guid] = nil
                changed = true
            else
                roster_add_unitid[guid] = unitid
                changed = true
            end
        end
    end

    for i = #roster_sorted_guids, 1, -1 do
        if roster_remove_unitid[roster_sorted_guids[i]] then
            tremove(roster_sorted_guids, i)
        end
    end

    for guid, unitid in pairs(roster_remove_unitid) do
        roster_name[guid] = nil
        roster_localized_class[guid] = nil
        roster_class[guid] = nil
        roster_level[guid] = nil
        roster_assigned_role[guid] = nil
        roster_spec_icon[guid] = nil
        roster_spec_name[guid] = nil
        roster_spec_role[guid] = nil
        roster_spec_blizz_role[guid] = nil
        roster_talents[guid] = nil
        roster_glyphs[guid] = nil

        roster_remove_unitid[guid] = nil
    end

    for guid, unitid in pairs(roster_add_unitid) do
        tinsert(roster_sorted_guids, guid)

        roster_add_unitid[guid] = nil
    end

    if changed then
        guids_sorted = nil
        role_count = nil

        self:SendMessage("RoleCall_UpdateLists")
    end

    if main_window and main_window:IsShown() then
        self:UpdateRoleButtons()
    end

--[===[@debug@
    self:TestRoster()
--@end-debug@]===]
end

--[===[@debug@
function RoleCall:TestRoster()
    local rosterUnitCount = 0
    for guid, unitid in pairs(roster_unitid) do
        rosterUnitCount = rosterUnitCount + 1

        if not roster_name[guid] then
            self:Printf("Missing entry in roster_name for %s (%s)!", unitid, guid)
        end

        if guid ~= UnitGUID(unitid) then
            self:Printf("GUID mismatch for %s (%s)!", unitid, guid)
        end
    end

    local groupCount = IsInGroup() and GetNumGroupMembers() or 1
    if rosterUnitCount ~= groupCount then
        self:Printf("Incorrect roster_unitid count: %d!", rosterUnitCount)
    end

    if rosterUnitCount ~= #roster_sorted_guids then
        self:Printf("Incorrect roster_sorted_guids count: %d!", #roster_sorted_guids)
    end

    for guid, name in pairs(roster_name) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_name: %s (%s)!", name, guid)
        end
    end

    for guid, class in pairs(roster_localized_class) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_localized_class: %s (%s)!", class, guid)
        end
    end

    for guid, class in pairs(roster_class) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_class: %s (%s)!", class, guid)
        end
    end

    for guid, level in pairs(roster_level) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_level: %d (%s)!", level, guid)
        end
    end

    for guid, role in pairs(roster_assigned_role) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_assigned_role: %s (%s)!", role, guid)
        end
    end

    for guid, icon in pairs(roster_spec_icon) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_spec_icon: %s (%s)!", icon, guid)
        end
    end

    for guid, spec in pairs(roster_spec_name) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_spec_name: %s (%s)!", spec, guid)
        end
    end

    for guid, role in pairs(roster_spec_role) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_spec_role: %s (%s)!", role, guid)
        end
    end

    for guid, role in pairs(roster_spec_blizz_role) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_spec_blizz_role: %s (%s)!", role, guid)
        end
    end

    for guid in pairs(roster_talents) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_talents (%s)!", guid)
        end
    end

    for guid in pairs(roster_glyphs) do
        if not roster_unitid[guid] then
            self:Printf("Extra entry in roster_glyphs (%s)!", guid)
        end
    end

    if role_count then
        local totalRoles = 0
        totalRoles = totalRoles + (role_count.healer or 0)
		totalRoles = totalRoles + (role_count.melee or 0)
		totalRoles = totalRoles + (role_count.ranged or 0)
		totalRoles = totalRoles + (role_count.tank or 0)

        if totalRoles > groupCount then
            self:Printf("Too many roles (%d/%d)", totalRoles, groupCount)
        end
    end

    if next(roster_remove_unitid) then
        self:Print("Remove roster not empty!")
    end

    if next(roster_add_unitid) then
        self:Print("Add roster not empty!")
    end
end

function RoleCall:DumpRoster()
    for guid, unitid in pairs(roster_unitid) do
        self:Printf("name: %s class: %s level: %d role: %s spec: %s specrole: %s blizzrole: %s unitid: %s", roster_name[guid] or "?", roster_class[guid] or "?", roster_level[guid] or 0, roster_assigned_role[guid] or "?", roster_spec_name[guid] or "?", roster_spec_role[guid] or "?", roster_spec_blizz_role[guid] or "?", unitid or "?")
    end
end
--@end-debug@]===]

function RoleCall:PLAYER_ENTERING_WORLD()
    in_combat = nil
    self:UpdateRoster()
end

function RoleCall:GROUP_ROSTER_UPDATE()
    self:UpdateRoster()
end

function RoleCall:PLAYER_REGEN_DISABLED()
    in_combat = true
    if main_window and main_window:IsShown() then
        self:UpdateRoleButtons()
    end
end

function RoleCall:PLAYER_REGEN_ENABLED()
    in_combat = nil
    if main_window and main_window:IsShown() then
        self:UpdateRoleButtons()
    end
end

function RoleCall:UNIT_NAME_UPDATE(_, unitid)
    local guid = UnitGUID(unitid)
    if guid and roster_unitid[guid] then
        self:UpdateRoster()
    end
end

function RoleCall:UNIT_LEVEL(_, unitid)
    local guid = UnitGUID(unitid)
    if guid and roster_unitid[guid] then
        local level = UnitLevel(unitid)
        if level == -1 then level = nil end

        if roster_level[guid] ~= level then
            roster_level[guid] = level
            role_count = nil

            self:SendMessage("RoleCall_UpdateLists")
        end
    end
end

function RoleCall:PLAYER_ROLES_ASSIGNED()
    local changed

    for guid, unitid in pairs(roster_unitid) do
        local assignedRole = UnitGroupRolesAssigned(unitid)
        if assignedRole == "NONE" then assignedRole = nil end

        if roster_assigned_role[guid] ~= assignedRole then
            roster_assigned_role[guid] = assignedRole
            changed = true
        end
    end

    if changed then
        guids_sorted = nil
        role_count = nil

        self:SendMessage("RoleCall_UpdateLists")
    end
end

function RoleCall:READY_CHECK()
    if self.db.profile.reportReadyCheck then
        self:ReportRoleMismatches()
    end
end

function RoleCall:NotifyRoleChange(guid, oldSpecRole, newSpecRole, oldSpecName, newSpecName)
    local newRoleName = newSpecRole and localized_spec_role[newSpecRole]
    if newRoleName then
        local assignedRole = roster_assigned_role[guid]
        local specBlizzRole = roster_spec_blizz_role[guid]
        if assignedRole and specBlizzRole and assignedRole ~= specBlizzRole then
            newRoleName = string.format("|cffff0000%s|r", newRoleName)
        end
    end

    self:Printf(L["%s changed role from %s (%s) to %s (%s)."], string.format(class_color_formatters[roster_class[guid]], roster_name[guid] or L["Unknown"]),
                                                                  oldSpecRole and localized_spec_role[oldSpecRole] or L["Unknown"],
                                                                  oldSpecName or L["Unknown"],
                                                                  newRoleName or L["Unknown"],
                                                                  newSpecName or L["Unknown"])
end

function RoleCall:NotifySpecChange(guid, oldSpecName, newSpecName)
    self:Printf(L["%s changed specialization from %s to %s."], string.format(class_color_formatters[roster_class[guid]], roster_name[guid] or L["Unknown"]),
                                                                  oldSpecName or L["Unknown"],
                                                                  newSpecName or L["Unknown"])
end

function RoleCall:GroupInSpecT_Update(_, guid, unitid, info)
    if guid and roster_unitid[guid] then

        local changed

        local specName, specIcon, specRole, blizzRole, talents, glyphs
        if info then
            specName, specIcon, specRole, blizzRole, talents, glyphs = info.spec_name_localized, info.spec_icon, info.spec_role_detailed, info.spec_role, info.talents, info.glyphs
            if blizzRole == "NONE" then blizzRole = nil end
        end

        local oldSpecName = roster_spec_name[guid]
        local oldSpecRole = roster_spec_role[guid]

        if oldSpecName ~= specName or
           oldSpecRole ~= specRole or
           roster_spec_blizz_role[guid] ~= blizzRole or
           roster_spec_icon[guid] ~= specIcon or
           TableKeysDiffer(roster_talents[guid], talents) or
           TableKeysDiffer(roster_glyphs[guid], glyphs) then
            changed = true
        end

        roster_spec_icon[guid] = specIcon
        roster_spec_name[guid] = specName
        roster_spec_role[guid] = specRole
        roster_spec_blizz_role[guid] = blizzRole
        roster_talents[guid] = talents and TableShallowCopy(talents, roster_talents[guid]) or nil
        roster_glyphs[guid] = glyphs and TableShallowCopy(glyphs, roster_glyphs[guid]) or nil

        if oldSpecRole and oldSpecRole ~= specRole and self.db.profile.notifyRoleChange then
            self:NotifyRoleChange(guid, oldSpecRole, specRole, oldSpecName, specName)
        elseif oldSpecName and oldSpecName ~= specName and self.db.profile.notifySpecChange then
            self:NotifySpecChange(guid, oldSpecName, specName)
        end

        if changed then
            guids_sorted = nil
            role_count = nil

            self:SendMessage("RoleCall_UpdateLists")
        end
    end
end

function RoleCall:UpdateGroupList()
    if main_window then
        local rosterCount = #roster_sorted_guids
        local scrollFrame = main_window.scrollFrame
        FauxScrollFrame_Update(scrollFrame, rosterCount, MAX_ROWS, ROW_HEIGHT)
        local offset = FauxScrollFrame_GetOffset(scrollFrame)

        if not guids_sorted then
            local sortMethod = self.db.profile.sortMethod
            local sortReverse = self.db.profile.sortReverse

            local sortFunction = sortMethod and sort_method[sortMethod] or CompareGUIDsByName

            sort(roster_sorted_guids, function (guid1, guid2) return sortFunction(guid1, guid2, sortReverse) end)
            guids_sorted = true
        end

        for i = 1, MAX_ROWS do
            local unitFrame = unit_frames[i]

            local offsetIndex = i + offset
            if offsetIndex <= rosterCount then
                local guid = roster_sorted_guids[offsetIndex]

                local class = roster_class[guid]

                local nameLabel = unitFrame.nameLabel
                local color = class_colors[class]
                nameLabel:SetTextColor(color.r, color.g, color.b)
                nameLabel:SetText(roster_name[guid] or "")

                local classIconFrame = unitFrame.classIconFrame
                if class then
                    classIconFrame:SetTexCoord(unpack(CLASS_ICON_TCOORDS[class]))
                    classIconFrame:Show()
                else
                    classIconFrame:Hide()
                end

                unitFrame.classLabel:SetText(roster_localized_class[guid] or "")

                local specIconFrame = unitFrame.specIconFrame
                local specIcon = roster_spec_icon[guid]
                if specIcon then
                    specIconFrame:SetTexture(specIcon)
                    specIconFrame:Show()
                else
                    specIconFrame:Hide()
                end

                local specNameLabel = unitFrame.specNameLabel
                local specName = roster_spec_name[guid]
                if specName then
                    local talents = roster_talents[guid]
                    local level = roster_level[guid]
                    local talentCount = talents and TableCount(talents) or 0
                    local tierCount = level and TalentTierCount(class, level) or 0

                    if talentCount < tierCount then
                        specNameLabel:SetFontObject(RoleCall_FontNormalOrange)
                    else
                        local glyphs = roster_glyphs[guid]
                        local majorGlyphCount = 0
                        if glyphs then
                            for _, glyphInfo in pairs(glyphs) do
                                if glyphInfo.glyph_type == GLYPH_TYPE_MAJOR then
                                    majorGlyphCount = majorGlyphCount + 1
                                end
                            end
                        end
                        if majorGlyphCount < MajorGlyphSlotCount(level) then
                            specNameLabel:SetFontObject(GameFontNormal)
                        else
                            specNameLabel:SetFontObject(GameFontHighlight)
                        end
                    end

                    specNameLabel:SetText(specName)
                else
                    specNameLabel:SetFontObject(GameFontDisable)
                    specNameLabel:SetText(L["Unknown"])
                end

                local assignedRole = roster_assigned_role[guid]

                local specRoleLabel = unitFrame.specRoleLabel
                local specRole = roster_spec_role[guid]
                if specRole then
                    local specBlizzRole = roster_spec_blizz_role[guid]
                    if assignedRole and specBlizzRole and assignedRole ~= specBlizzRole then
                        specRoleLabel:SetFontObject(GameFontRed)
                    else
                        specRoleLabel:SetFontObject(GameFontHighlight)
                    end
                else
                    specRoleLabel:SetFontObject(GameFontDisable)
                end
                specRoleLabel:SetText(specRole and localized_spec_role[specRole] or L["Unknown"])

                local roleIcon = unitFrame.roleIcon
                local textureCoord = assignedRole and role_texture_coord[assignedRole]
                if textureCoord then
                    roleIcon:SetTexCoord(unpack(textureCoord))
                    roleIcon:Show()
                else
                    roleIcon:Hide()
                end

                unitFrame.guid = guid
                unitFrame:Show()
            else
                unitFrame:Hide()
                unitFrame.guid = nil
            end
        end
    end
end

function RoleCall:SaveMainWindowPosition()
    if main_window then
        local opts = self.db.profile.frameopts
        if not opts then
            opts = {}
            self.db.profile.frameopts = opts
        end

        local anchorFrom, _, anchorTo, offsetX, offsetY = main_window:GetPoint()
        opts.anchorFrom = anchorFrom
        opts.anchorTo = anchorTo
        opts.offsetX = offsetX
        opts.offsetY = offsetY
    end
end

function RoleCall:RestoreMainWindowPosition()
    if main_window then
        main_window:ClearAllPoints()

        local opts = self.db.profile.frameopts
        if opts then
            main_window:SetPoint(opts.anchorFrom or "TOPLEFT", UIParent, opts.anchorTo or "TOPLEFT", opts.offsetX or 0, opts.offsetY or 0)
        else
            main_window:SetPoint("CENTER", UIParent)
        end
    end
end

function RoleCall:UpdateSortButtons()
    local sortMethod = self.db.profile.sortMethod
    if not (sortMethod and sort_method[sortMethod]) then
        sortMethod = "name"
    end
    local sortReverse = self.db.profile.sortReverse

    for i = 1, #sort_buttons do
        local button = sort_buttons[i]
        local arrow = button.arrow
        if button.sortMethod == sortMethod then
            button:SetNormalFontObject(GameFontHighlight)
            if not arrow then
                arrow = button:CreateTexture(nil, "ARTWORK")
                arrow:SetWidth(9)
                arrow:SetHeight(8)
                arrow:SetPoint("LEFT", button:GetFontString(), "RIGHT", 3, -2)
                arrow:SetTexture("Interface\\Buttons\\UI-SortArrow")
                button.arrow = arrow
            end
            if sortReverse then
                arrow:SetTexCoord(0, 0.5625, 1.0, 0)
            else
                arrow:SetTexCoord(0, 0.5625, 0, 1.0)
            end
            arrow:Show()
        else
            button:SetNormalFontObject(GameFontNormal)
            if arrow then
                arrow:Hide()
            end
        end
    end
end

function RoleCall:ToggleSortMethod(sortMethod)
    if not (sortMethod and sort_method[sortMethod]) then
        sortMethod = "name"
    end

    if sortMethod == (self.db.profile.sortMethod or "name") then
        self.db.profile.sortReverse = not self.db.profile.sortReverse
    else
        self.db.profile.sortMethod = sortMethod
        self.db.profile.sortReverse = nil
    end

    guids_sorted = nil
    self:UpdateSortButtons()
    self:UpdateGroupList()
end

function RoleCall:ToggleMainWindow()
    if not main_window then
        main_window = CreateFrame("Frame", "RoleCall_MainWindow", UIParent, "UIPanelDialogTemplate")
        main_window:Hide()

        main_window.title:SetText(addon_title)

        main_window:EnableMouse(true)
        main_window:SetMovable(true)
        main_window:SetClampedToScreen(true)

        main_window:SetFrameStrata("MEDIUM")
        main_window:SetToplevel(true)

        main_window:SetWidth(565)
        main_window:SetHeight(310)

        main_window:SetScript("OnShow", function() PlaySound("igCharacterInfoOpen")
                                                   RoleCall:RestoreMainWindowPosition()
                                                   RoleCall:UpdateSortButtons()
                                                   RoleCall:UpdateRoleButtons()
                                                   RoleCall:UpdateGroupList() end)
        main_window:SetScript("OnHide", function() PlaySound("igCharacterInfoClose") end)

        local titleButton = CreateFrame("Frame", nil, main_window)
        titleButton:SetPoint("TOPLEFT", RoleCall_MainWindowTitleBG)
        titleButton:SetPoint("BOTTOMRIGHT", RoleCall_MainWindowTitleBG)
        titleButton:SetScript("OnMouseDown", function() main_window:StartMoving() end)
        titleButton:SetScript("OnMouseUp", function() main_window:StopMovingOrSizing() RoleCall:SaveMainWindowPosition() end)
        titleButton:SetScript("OnHide", function() main_window:StopMovingOrSizing() RoleCall:SaveMainWindowPosition() end)

        local sortNameButton = CreateFrame("Button", nil, main_window, "RoleCall_HeaderButtonTemplate")
        sortNameButton:SetSize(NAME_LABEL_WIDTH, BUTTON_HEIGHT)
        sortNameButton:SetPoint("TOPLEFT", 20, -26)
        sortNameButton:SetText(L["Name"])
        sortNameButton.sortMethod = "name"
        sortNameButton:SetScript("OnClick", SortButtonAction)
        tinsert(sort_buttons, sortNameButton)

        local sortClassButton = CreateFrame("Button", nil, main_window, "RoleCall_HeaderButtonTemplate")
        sortClassButton:SetSize(CLASS_LABEL_WIDTH + ROW_HEIGHT + COLUMN_GAP, BUTTON_HEIGHT)
        sortClassButton:SetPoint("TOPLEFT", sortNameButton, "TOPRIGHT", COLUMN_GAP, 0)
        sortClassButton:SetText(L["Class"])
        sortClassButton.sortMethod = "class"
        sortClassButton:SetScript("OnClick", SortButtonAction)
        tinsert(sort_buttons, sortClassButton)

        local sortSpecButton = CreateFrame("Button", nil, main_window, "RoleCall_HeaderButtonTemplate")
        sortSpecButton:SetSize(SPEC_NAME_LABEL_WIDTH + ROW_HEIGHT + COLUMN_GAP, BUTTON_HEIGHT)
        sortSpecButton:SetPoint("TOPLEFT", sortClassButton, "TOPRIGHT", COLUMN_GAP, 0)
        sortSpecButton:SetText(L["Specialization"])
        sortSpecButton.sortMethod = "specName"
        sortSpecButton:SetScript("OnClick", SortButtonAction)
        tinsert(sort_buttons, sortSpecButton)

        local sortRoleButton = CreateFrame("Button", nil, main_window, "RoleCall_HeaderButtonTemplate")
        sortRoleButton:SetSize(SPEC_ROLE_LABEL_WIDTH + ROLE_ICON_WIDTH + COLUMN_GAP, BUTTON_HEIGHT)
        sortRoleButton:SetPoint("TOPLEFT", sortSpecButton, "TOPRIGHT", COLUMN_GAP, 0)
        sortRoleButton:SetText(L["Role"])
        sortRoleButton.sortMethod = "specRole"
        sortRoleButton:SetScript("OnClick", SortButtonAction)
        tinsert(sort_buttons, sortRoleButton)

        local assignButton = CreateFrame("Button", nil, main_window, "UIPanelButtonTemplate")
        assignButton:SetSize(BUTTON_WIDTH, BUTTON_HEIGHT)
        assignButton:SetText(L["Assign Roles"])
        assignButton:SetPoint("BOTTOMRIGHT", -15, 20)
        assignButton:SetScript("OnClick", function() RoleCall:AssignAllRoles() end)
        main_window.assignButton = assignButton

        local reportButton = CreateFrame("Button", nil, main_window, "UIPanelButtonTemplate")
        reportButton:SetSize(BUTTON_WIDTH, BUTTON_HEIGHT)
        reportButton:SetText(L["Report Roles"])
        reportButton:SetPoint("BOTTOMRIGHT", assignButton, "BOTTOMLEFT", -10, 0)
        reportButton:SetScript("OnClick", function() RoleCall:ReportRoleMismatches() end)
        main_window.reportButton = reportButton

        local rescanButton = CreateFrame("Button", nil, main_window, "UIPanelButtonTemplate")
        rescanButton:SetSize(BUTTON_WIDTH, BUTTON_HEIGHT)
        rescanButton:SetText(L["Rescan"])
        rescanButton:SetPoint("BOTTOMLEFT", 20, 20)
        rescanButton:SetScript("OnClick", function() RoleCall:Rescan() end)
        main_window.rescanButton = rescanButton

        local scrollBGFrame = CreateFrame("Frame", nil, main_window, "InsetFrameTemplate3")
        scrollBGFrame:SetPoint("TOPLEFT", 10, -49)
        scrollBGFrame:SetPoint("BOTTOMRIGHT", main_window, "TOPRIGHT", -8, -(49 + 8 + ROW_HEIGHT * MAX_ROWS))

        local scrollFrame = CreateFrame("ScrollFrame", "RoleCall_ScrollFrame", scrollBGFrame, "FauxScrollFrameTemplate")
        scrollFrame:SetPoint("TOPLEFT", 0, -4)
        scrollFrame:SetPoint("BOTTOMRIGHT", -23, 4)

        --local scrollBackground = main_window:CreateTexture(nil, "OVERLAY")
        --scrollBackground:SetAllPoints(scrollFrame)
        --scrollBackground:SetTexture(1,1,1,0.2)

        scrollFrame:SetScript("OnVerticalScroll", function(self, offset) FauxScrollFrame_OnVerticalScroll(self, offset, ROW_HEIGHT, function() RoleCall:UpdateGroupList() end) end)
        main_window.scrollFrame = scrollFrame

        tinsert(UISpecialFrames, "RoleCall_MainWindow")

        main_window:Show()
    else
        if main_window:IsShown() then
            main_window:Hide()
        else
            main_window:Show()
        end
    end
end

function RoleCall:UpdateRoleButtons()
    if main_window then
        local isInGroup = IsInGroup()

        if isInGroup and not in_combat and not HasLFGRestrictions() then
            main_window.assignButton:Enable()
        else
            main_window.assignButton:Disable()
        end

        if isInGroup then
            main_window.reportButton:Enable()
            main_window.rescanButton:Enable()
        else
            main_window.reportButton:Disable()
            main_window.rescanButton:Disable()
        end
    end
end

-- C_Scenario.IsInScenario() or IsInScenarioGroup()
function RoleCall:AssignAllRoles()
    if IsInGroup() then
        if not HasLFGRestrictions() then
            if not InCombatLockdown() then
                local changed
                if UnitIsGroupLeader("player") or UnitIsGroupAssistant("player") then
                    for guid, unitid in pairs(roster_unitid) do
                        local specBlizzRole = roster_spec_blizz_role[guid]
                        if specBlizzRole and specBlizzRole ~= roster_assigned_role[guid] then
                            UnitSetRole(unitid, specBlizzRole)
                            changed = true
                        end
                    end
                else
                    local guid = UnitGUID("player")
                    local specBlizzRole = guid and roster_spec_blizz_role[guid]
                    if specBlizzRole and specBlizzRole ~= roster_assigned_role[guid] then
                        UnitSetRole("player", specBlizzRole)
                        changed = true
                    end
                end
                if not changed then
                    self:Print(L["No roles changed."])
                end
            else
                self:Print(L["Cannot assign roles in combat."])
            end
        else
            self:Print(L["Cannot assign roles."])
        end
    else
        self:Print(L["You are not in a group."])
    end
end

function RoleCall:ReportRoleMismatches()
    local chatType = (IsInGroup(LE_PARTY_CATEGORY_INSTANCE) and "INSTANCE_CHAT") or (IsInRaid() and "RAID") or (IsInGroup(LE_PARTY_CATEGORY_HOME) and "PARTY")
    if chatType then
        local mismatched
        for guid in pairs(roster_unitid) do
            local assignedRole = roster_assigned_role[guid]
            local specBlizzRole = roster_spec_blizz_role[guid]
            if assignedRole and specBlizzRole and assignedRole ~= specBlizzRole then
                mismatched = mismatched or {}
                tinsert(mismatched, guid)
            end
        end

        if mismatched then
            sort(mismatched, CompareGUIDsByName)
            for i = 1, #mismatched do
                local guid = mismatched[i]
                local assignedRole = roster_assigned_role[guid]
                local specBlizzRole = roster_spec_blizz_role[guid]
                SendChatMessage(string.format(L["[RoleCall] %s is assigned as %s, but is specialized as %s (%s)."], roster_name[guid] or "", localized_blizz_role[assignedRole] or "", localized_blizz_role[specBlizzRole] or "", roster_spec_name[guid] or ""), chatType)
            end
        else
            self:Print(L["No role mismatches."])
        end
    else
        self:Print(L["You are not in a group."])
    end
end

function RoleCall:Rescan()
    if IsInGroup() then
        libGroupInSpecT:Rescan()
    else
        self:Print(L["You are not in a group."])
    end
end

function RoleCall:UpdateLDBTooltip()
    if ldb_tooltip then
        ldb_tooltip:Clear()
        BuildLDBTooltip()
        ldb_tooltip:UpdateScrolling()
    end
end

local talentArray = {}
local majorGlyphs = {}
local minorGlyphs = {}

local function ShowTalentsTooltip(frame)
    local guid = frame.guid
    if guid then
        if talents_tooltip then
            talents_tooltip:Clear()
        else
            talents_tooltip = LibQTip:Acquire("RoleCall_TalentsTooltip", 3)
        end

        talents_tooltip.owner = frame

        talents_tooltip:AddHeader(TALENTS, MAJOR_GLYPHS, MINOR_GLYPHS)

        local talents = roster_talents[guid]
        local glyphs = roster_glyphs[guid]

        talents_tooltip:SetFont(GameFontNormal)

        if talents then
            for _, talentInfo in pairs(talents) do
                local talentTier = talentInfo.tier
                if talentTier then
                    talentArray[talentTier] = talentInfo.name_localized
                end
            end
        end

        if glyphs then
            for _, glyphInfo in pairs(glyphs) do
                local glyphName = glyphInfo.name_localized
                if glyphName then
                    local glyphType = glyphInfo.glyph_type
                    if glyphType == GLYPH_TYPE_MAJOR then
                        tinsert(majorGlyphs, glyphName)
                    elseif glyphType == GLYPH_TYPE_MINOR then
                        tinsert(minorGlyphs, glyphName)
                    end
                end
            end

            sort(majorGlyphs)
            sort(minorGlyphs)
        end

        for i = 1, max(MAX_NUM_TALENT_TIERS, 3) do
            local talentName = talentArray[i]
            local majorName = majorGlyphs[i]
            local minorName = minorGlyphs[i]

            local line = talents_tooltip:AddLine(talentName, majorName, minorName)

            if not talentName and (i <= MAX_NUM_TALENT_TIERS) then
                talents_tooltip:SetCell(line, 1, L["None"], GameFontDisable)
            end

            if not majorName and (i <= 3) then
                talents_tooltip:SetCell(line, 2, L["None"], GameFontDisable)
            end

            if not minorName and (i <= 3) then
                talents_tooltip:SetCell(line, 3, L["None"], GameFontDisable)
            end
        end

        wipe(talentArray)
        wipe(majorGlyphs)
        wipe(minorGlyphs)

        talents_tooltip:SmartAnchorTo(frame)
        talents_tooltip:Show()
    end
end

local function ReleaseTalentsTooltip(frame)
    if talents_tooltip and (talents_tooltip.owner == frame) then
        talents_tooltip.owner = nil
        LibQTip:Release(talents_tooltip)
        talents_tooltip = nil
    end
end

setmetatable(unit_frames, {__index = function(t, i)
    local unitFrame = CreateFrame("Frame", nil, main_window)
    unitFrame:Hide()
    unitFrame:SetWidth(600)
    unitFrame:SetHeight(ROW_HEIGHT)

    unitFrame:SetScript("OnEnter", function(self) ShowTalentsTooltip(self) end)
    unitFrame:SetScript("OnLeave", function(self) ReleaseTalentsTooltip(self) end)

    local nameLabel = unitFrame:CreateFontString(nil, "ARTWORK", "GameFontHighlight")
    nameLabel:SetWidth(NAME_LABEL_WIDTH)
    nameLabel:SetHeight(ROW_HEIGHT)
    nameLabel:SetPoint("TOPLEFT", unitFrame, "TOPLEFT", 10, 0)
    nameLabel:SetJustifyH("LEFT")
    nameLabel:SetJustifyV("MIDDLE")
    unitFrame.nameLabel = nameLabel

    local classIconFrame = unitFrame:CreateTexture(nil, "ARTWORK")
    classIconFrame:SetWidth(ROW_HEIGHT)
    classIconFrame:SetHeight(ROW_HEIGHT)
    classIconFrame:SetPoint("TOPLEFT", nameLabel, "TOPRIGHT", COLUMN_GAP, 0)
    classIconFrame:SetTexture("Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes")
    classIconFrame:Hide()
    unitFrame.classIconFrame = classIconFrame

    local classLabel = unitFrame:CreateFontString(nil, "ARTWORK", "GameFontHighlight")
    classLabel:SetWidth(CLASS_LABEL_WIDTH)
    classLabel:SetHeight(ROW_HEIGHT)
    classLabel:SetPoint("TOPLEFT", classIconFrame, "TOPRIGHT", COLUMN_GAP, 0)
    classLabel:SetJustifyH("LEFT")
    classLabel:SetJustifyV("MIDDLE")
    unitFrame.classLabel = classLabel

    local specIconFrame = unitFrame:CreateTexture(nil, "ARTWORK")
    specIconFrame:SetWidth(ROW_HEIGHT)
    specIconFrame:SetHeight(ROW_HEIGHT)
    specIconFrame:SetPoint("TOPLEFT", classLabel, "TOPRIGHT", COLUMN_GAP, 0)
    specIconFrame:Hide()
    unitFrame.specIconFrame = specIconFrame

    local specNameLabel = unitFrame:CreateFontString(nil, "ARTWORK", "GameFontHighlight")
    specNameLabel:SetWidth(SPEC_NAME_LABEL_WIDTH)
    specNameLabel:SetHeight(ROW_HEIGHT)
    specNameLabel:SetPoint("TOPLEFT", specIconFrame, "TOPRIGHT", COLUMN_GAP, 0)
    specNameLabel:SetJustifyH("LEFT")
    specNameLabel:SetJustifyV("MIDDLE")
    unitFrame.specNameLabel = specNameLabel

    local specRoleLabel = unitFrame:CreateFontString(nil, "ARTWORK", "GameFontHighlight")
    specRoleLabel:SetWidth(SPEC_ROLE_LABEL_WIDTH)
    specRoleLabel:SetHeight(ROW_HEIGHT)
    specRoleLabel:SetPoint("TOPLEFT", specNameLabel, "TOPRIGHT", COLUMN_GAP, 0)
    specRoleLabel:SetJustifyH("LEFT")
    specRoleLabel:SetJustifyV("MIDDLE")
    unitFrame.specRoleLabel = specRoleLabel

    local roleIcon = unitFrame:CreateTexture(nil, "ARTWORK")
    roleIcon:SetWidth(ROLE_ICON_WIDTH)
    roleIcon:SetHeight(ROW_HEIGHT)
    roleIcon:SetPoint("TOPLEFT", specRoleLabel, "TOPRIGHT", COLUMN_GAP, 0)
    roleIcon:SetTexture("Interface\\LFGFrame\\UI-LFG-ICON-PORTRAITROLES")
    roleIcon:Hide()
    unitFrame.roleIcon = roleIcon

    if i == 1 then
        unitFrame:SetPoint("TOPLEFT", main_window.scrollFrame, "TOPLEFT")
    else
        unitFrame:SetPoint("TOPLEFT", unit_frames[i-1], "BOTTOMLEFT")
    end

    rawset(t, i, unitFrame)
    return unitFrame
end })
