-- AddOn by Dezyne, Argent Dawn, Europe. Please give me credit when you want to redistribute or modify this addon! Contact: coenvdwel@planet.nl
-- Coen van der Wel, Eindhoven, the Netherlands.
													 local rewatch_version = 4.0; -- current revision
--// FUNCTIONS //----------------------------------------------------------------------------------------------------------------------------------------------------------------

-- display a message to the user in the chat pane
-- msg: the message to pass onto the user
-- return: void
function rewatch_Message(msg)
	-- send the message to the chat pane, in yellow-ish
	DEFAULT_CHAT_FRAME:AddMessage(rewatch_loc["prefix"]..msg, 0.95, 0.85, 0.15);
	
	-- return
	return;
end;

-- (re)sets the global saved variable table
-- return: void
function rewatch_SetData()
	-- set default scalings
	rewatch_data = {
		SpellBarWidth = 80; SpellBarHeight = 17; SpellBarMargin = 0; NumFramesWide = 5;
		SmallButtonWidth = 20; SmallButtonHeight = 20; SmallButtonMargin = 3;
		FrameWidth = 1; FrameHeight = 1;
	};
	-- deducted scalings
	rewatch_data["FrameWidth"] = rewatch_data["SpellBarWidth"]+4;
	rewatch_data["FrameHeight"] = (rewatch_data["SpellBarMargin"] + rewatch_data["SpellBarHeight"])*4 + rewatch_data["SmallButtonHeight"] + rewatch_data["SpellBarMargin"];
	
	-- return
	return;
end;

-- pops up the tooltip bar
-- data: the data to put in the tooltip. either a spell name or player name.
-- return: void
function rewatch_SetTooltip(data)
	-- empty tooltip
	rewatch_tooltip:ClearLines();
	-- is it a spell?
	local md = rewatch_GetSpellId(data);
	if(md < 0) then
		-- if not, then is it a player?
		md = rewatch_GetPlayer(data);
		if(md >= 0) then
			rewatch_tooltip:SetUnit(rewatch_bars[md]["Player"]);
			rewatch_tooltip:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMRIGHT", -25, 25);
		end; -- do nothing with the tooltip if not
	else
		rewatch_tooltip:SetSpell(md, BOOKTYPE_SPELL);
		rewatch_tooltip:SetPoint("BOTTOMRIGHT", UIParent, "BOTTOMRIGHT", -25, 25);
	end;
	
	-- return
	return;
end;

-- gets the spell ID of the highest rank of the specified spell
-- spellName: the name of the spell to get the highest ranked spellId from
-- return: the corresponding spellId
function rewatch_GetSpellId(spellName)
	-- get spell info and highest rank, return if the user can't cast it (not learned, etc)
	local name, rank, icon = GetSpellInfo(spellName);
	if(name == nil) then return -1; end;
	-- loop through all book spells, return the number if it matches above data
	local i, ispell, irank = 1, GetSpellName(1, BOOKTYPE_SPELL);
	repeat
		if ((ispell == name) and ((rank == irank) or (not irank))) then return i; end;
		i, ispell, irank = i+1, GetSpellName(i+1, BOOKTYPE_SPELL);
	until (not ispell);
	
	-- return default -1
	return -1;
end;

-- clears the entire list and resets it
-- return: void
function rewatch_Clear()
	-- call each playerframe's Hide method
	for i=1,rewatch_i-1 do local val = rewatch_bars[i]; if(val) then rewatch_HidePlayer(i); end; end;
	rewatch_bars = nil; rewatch_bars = {}; rewatch_i = 1;
	
	-- return
	return;
end;

-- get the number of the supplied player's place in the player table, or -1
-- player: name of the player to search for
-- return: the supplied player's table index, or -1 if not found
function rewatch_GetPlayer(player)
	-- for every seen player; return if the name matches the supplied name
	for i=1,rewatch_i-1 do local val = rewatch_bars[i]; if(val) then
		if(val["Player"] == player) then return i; end;
	end; end;
	
	-- return -1 if not found
	return -1;
end;

-- checks if the player is in the group
-- player: name of the player to check for
-- return: true, if the player is the user, or in the user's party or raid; false elsewise
function rewatch_InGroup(player)
	-- catch a self-check; return true if searching for the user itself
	if(UnitName("player") == player) then return true;
	-- process by environment (party or raid)
	elseif(GetNumRaidMembers() > 0) then if(UnitPlayerOrPetInRaid(player)) then return true; end;
	elseif(GetNumPartyMembers() > 0) then if(UnitPlayerOrPetInParty(player)) then return true; end;
	end;
	
	-- return
	return false;
end;

-- performs a poison check on given unit, will return true if a poison debuff was found
-- playerId: the index number of the player in the player table
-- return: true if the player with the supplied id has a poison, false if not
function rewatch_IsPoisoned(playerId)
	-- go through all possible debuff spots
	local debuffType; for i=1,40 do
		-- get the debuff type, return true if it's poison
		debuffType = select(5, UnitDebuff(rewatch_bars[playerId]["Player"], i, 1));
		if(debuffType == "Poison") then return true;
		-- or, if it's empty, we reached the end of the debuff queue, return false
		elseif((debuffType == "") or (not debuffType)) then return false; end;
	end;
	
	-- return
	return false;
end;

-- performs a curse check on given unit, will return true if a curse debuff was found
-- playerId: the index number of the player in the player table
-- return: true if the player with the supplied id has a curse, false if not
function rewatch_IsCursed(playerId)
	-- go through all possible debuff spots
	local debuffType; for i=1,40 do
		-- get the debuff type, return true if it's a curse
		debuffType = select(5, UnitDebuff(rewatch_bars[playerId]["Player"], i, 1));
		if(debuffType == "Curse") then return true;
		-- or, if it's empty, we reached the end of the debuff queue, return false
		elseif((debuffType == "") or (not debuffType)) then return false; end;
	end;
	
	-- return
	return false;
end;

-- checks the druid buffs on players in your list
-- return: void
function rewatch_BuffCheck()
	-- check all listed players
	local thorns, motw = false, {};
	for i,val in ipairs(rewatch_bars) do if(val) then
		local thisMotw = false;
		for j=1,40 do
			local name = UnitBuff(val["Player"], j, 0);
			if((name == nil) or (thorns and thisMotw)) then break; elseif(name == rewatch_loc["thorns"]) then thorns = true;
			elseif((name == rewatch_loc["markofthewild"]) or (name == rewatch_loc["giftofthewild"])) then thisMotw = true; end;
		end;
		if(not thisMotw) then table.insert(motw, val["Player"]); end;
	end; end;
	-- build report
	local msg = rewatch_loc["buffresults"].." ";
	if(not thorns) then msg = msg..rewatch_loc["nothorns"].." "; end;
	if(table.getn(motw) > 0) then
		msg = msg..rewatch_loc["missingmotw"].." "..motw[1];
		for i=2,table.getn(motw) do	msg = msg..", "..motw[i]; end;
	end;
	if((thorns) and (table.getn(motw) == 0)) then msg = msg.."Ok."; end;
	-- message the user
	rewatch_Message(msg);
	
	-- return
	return;
end;

-- loads the internal vars from the savedvariables
-- return: void
function rewatch_OnLoad()
	-- has been loaded before, get vars
	if(rewatch_load) then
		-- if the savedvariable is complete
		if((rewatch_load["GcdAlpha"]) and (rewatch_load["HideSolo"]) and (rewatch_load["Hide"]) and (rewatch_load["AutoGroup"])) then
			rewatch_loadInt["Loaded"] = true;
			rewatch_loadInt["GcdAlpha"] = rewatch_load["GcdAlpha"];
			rewatch_loadInt["HideSolo"] = rewatch_load["HideSolo"];
			rewatch_loadInt["Hide"] = rewatch_load["Hide"];
			rewatch_loadInt["AutoGroup"] = rewatch_load["AutoGroup"];
			-- apply these possible changes
			for _, cd in ipairs(rewatch_gcds) do cd:SetAlpha(rewatch_loadInt["GcdAlpha"]); end;
			if(((rewatch_i == 2) and (rewatch_loadInt["HideSolo"] == 1)) or (rewatch_loadInt["Hide"] == 1)) then rewatch_f[1]:Hide(); else rewatch_f[1]:Show(); end;
			rewatch_OptionsFromData(true);
		-- or if it's modified/outdated, reset it
		else rewatch_load = nil; end;
	-- not loaded before, initialise and welcome new user
	else
		rewatch_load = {};
		rewatch_load["GcdAlpha"], rewatch_load["HideSolo"], rewatch_load["Hide"], rewatch_load["AutoGroup"] = 1, 0, 0, 1;
		rewatch_Message(rewatch_loc["welcome"]);
	end;
end;

-- trigger the cooldown overlays
-- return: void
function rewatch_TriggerCooldown()
	-- get global cooldown, and trigger it on all frames
	local start, duration, enabled = GetSpellCooldown(rewatch_loc["rejuvenation"]); -- some non-cd spell
	for _, cd in ipairs(rewatch_gcds) do
		CooldownFrame_SetTimer(cd, start, duration, enabled);
	end;
	
	-- return
	return;
end;

-- adjusts the parent frame container's height
-- parentFrame: the frame to adjust
-- return: void
function rewatch_AlterFrame(parentFrame)
	-- get the index of the frame
	local i = 1; if(parentFrame) then
		for j, frame in ipairs(rewatch_f) do
			if(frame == parentFrame) then i = j; break; end;
		end;
	end;
	-- set height and width according to number of frames
	local num = rewatch_f[i]:GetNumChildren()-1;
	local width = math.min(rewatch_data["NumFramesWide"],  math.max(num, 1)) * rewatch_data["FrameWidth"];
	local height = math.ceil(num/rewatch_data["NumFramesWide"]) * rewatch_data["FrameHeight"];
	rewatch_f[i]:SetWidth(width); rewatch_f[i]:SetHeight(height+10);
	rewatch_gcds[i]:SetWidth(rewatch_f[i]:GetWidth()); rewatch_gcds[i]:SetHeight(rewatch_f[i]:GetHeight());
	-- hide/show on solo
	if(((num == 1) and (rewatch_loadInt["HideSolo"] == 1)) or (rewatch_loadInt["Hide"] == 1)) then rewatch_f[i]:Hide(); else rewatch_f[i]:Show(); end;
	-- make sure frames have a solid height and width (bugfix)
	for j=1,rewatch_i-1 do local val = rewatch_bars[j]; if(val) then
		if(not((val["Frame"]:GetWidth() == rewatch_data["FrameWidth"]) and (val["Frame"]:GetHeight() == rewatch_data["FrameHeight"]))) then
			val["Frame"]:SetWidth(rewatch_data["FrameWidth"]); val["Frame"]:SetHeight(rewatch_data["FrameHeight"]);
		end;
	end; end;
	
	-- return
	return;
end;

-- snap the supplied frame to the grid when it's placed on a rewatch_f frame
-- frame: the frame to snap to a grid
-- return: void
function rewatch_SnapToGrid(frame)
	-- return if in combat
	if(InCombatLockdown() == 1) then return -1; end;
	
	-- get parent frame
	local parent = frame:GetParent();
	-- get frame's location relative to it's parent's
	local dx, dy = frame:GetLeft()-parent:GetLeft(), frame:GetTop()-parent:GetTop();
	-- make it snap (make dx a number closest to frame:GetWidth*n...)
	dx = math.floor((dx/(rewatch_data["FrameWidth"] + rewatch_data["SpellBarMargin"]))+0.5) * (rewatch_data["FrameWidth"] + rewatch_data["SpellBarMargin"]);
	dy = math.floor((dy/(rewatch_data["FrameHeight"] + rewatch_data["SpellBarMargin"]))+0.5) * (rewatch_data["FrameHeight"] + rewatch_data["SpellBarMargin"]);
	-- apply the location
	frame:ClearAllPoints();
	frame:SetPoint("TOPLEFT", parent, "TOPLEFT", dx, dy);
	frame:SetPoint("BOTTOMRIGHT", parent, "TOPLEFT", dx+rewatch_data["FrameWidth"], dy-rewatch_data["FrameHeight"]);
	
	-- return
	return;
end;

-- return the first available empty spot in the frame
-- frame: the outline (parent) frame in which the player frame should be positioned
-- return: position coordinates; { x, y }
function rewatch_GetFramePos(frame)
	-- assume: there is at least one free position in the specified parent frame
	local children = { frame:GetChildren() }; local x, y, found;
	-- walk through the available spots, left to right, top to bottom
	for dy=0, -7, -1 do for dx=0, 4 do
		found, x, y = false, (rewatch_data["FrameWidth"] + rewatch_data["SpellBarMargin"])*dx, (rewatch_data["FrameHeight"] + rewatch_data["SpellBarMargin"])*dy;
		-- check if there's a frame here already
		for i, child in ipairs(children) do
			if((child:GetLeft() and (i>1))) then
				if((math.abs(x - (child:GetLeft()-frame:GetLeft())) < 1) and (math.abs(y - (child:GetTop()-frame:GetTop())) < 1)) then
					found = true; break; --[[ break for children loop ]] end;
			end;
		end;
		-- if not, we found a spot and we should break!
		if(not found) then break; --[[ break for dxloop ]] end;
	end; if(not found) then break; --[[ break for dy loop ]] end; end;
	
	-- return either the found spot, or a formula based on array positioning (fallback)
	if(found) then return frame:GetWidth()*((rewatch_i-1)%rewatch_data["NumFramesWide"]), math.floor((rewatch_i-1)/rewatch_data["NumFramesWide"]) * -frame:GetHeight();
	else return x, y; end;
end;

-- process a swiftmend cast by the player, on specified playerId
-- playerId: the index number of the player in the player table
-- return: void
function rewatch_ProcessSwiftmend(playerId)
	-- assume it consumed a rejuvenation
	local consumed = rewatch_loc["rejuvenation"];
	-- check if this assumption was wrong, and if so, alter it
	if((rewatch_bars[playerId][consumed] == 0) or (((rewatch_bars[playerId][rewatch_loc["regrowth"]] > 0)) and (rewatch_bars[playerId][rewatch_loc["regrowth"]] < rewatch_bars[playerId][consumed]))) then
		consumed = rewatch_loc["regrowth"];
	end;
	-- clear the bar that was consumed and unset it's end time
	rewatch_bars[playerId][consumed] = 0; rewatch_bars[playerId][consumed.."Bar"]:SetValue(0);
	-- trigger all cooldown overlays of every player's swiftmend button
	for i=1,rewatch_i-1 do local val = rewatch_bars[i]; if(val) then
		CooldownFrame_SetTimer(val["SwiftmendButton"].cooldown, GetTime(), rewatch_abs[rewatch_loc["swiftmend"]]["Duration"], 1);
	end; end;
	
	-- return
	return;
end;

-- compares the current player table to the party/raid schedule
-- return: void
function rewatch_ProcessGroup()
	-- return if in combat
	if(InCombatLockdown() == 1) then return; end;
	
	rewatch_changed = nil;
	-- remove non-grouped players
	for i=1,rewatch_i-1 do if(rewatch_bars[i]) then
		if(not rewatch_InGroup(rewatch_bars[i]["Player"])) then rewatch_HidePlayer(i); end;
	end; end;
	-- set group environment
	local env, n;
	if(GetNumRaidMembers() > 0) then env, n = "raid", GetNumRaidMembers();
	elseif(GetNumPartyMembers() > 0) then env, n = "party", GetNumPartyMembers();
	else
		rewatch_i = 2; return;
	end;
	-- for each group member, if he's not in the list, add him
	for i=1, n do local name = UnitName(env..i); if(rewatch_GetPlayer(name) == -1) then rewatch_AddPlayer(name); end; end;
	
	-- return
	return;
end;

-- create a spell bar with text and add it to the global player table
-- spellName: the name of the spell to create a bar for (must be in rewatch_abs and have Offset)
-- playerId: the index number of the player in the player table
-- return: the created spell bar reference
function rewatch_CreateBar(spellName, playerId)
	-- create the bar
	local b = CreateFrame("STATUSBAR", spellName..playerId, rewatch_bars[playerId]["Frame"], "TextStatusBar"); b:SetWidth(rewatch_data["SpellBarWidth"]); b:SetHeight(rewatch_data["SpellBarHeight"]);
	b:SetPoint("TOPLEFT", rewatch_bars[playerId]["Frame"], "TOPLEFT", rewatch_abs[spellName]["Offset"], rewatch_abs[spellName]["Offset_Y"]);
	b:SetBackdrop({bgFile = nil, edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", tile = true, tileSize = 5, edgeSize = 5, insets = { left = 0, right = 0, top = 0, bottom = 0 }});
	b:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar"); b:SetStatusBarColor(0.0, 0.0, 0.6, 0.8);
	b:SetMinMaxValues(0, rewatch_abs[spellName]["Duration"]); b:SetValue(0);
	-- put text in status bar
	b.text = b:CreateFontString("$parentText", "ARTWORK", "GameFontHighlightSmall");
	b.text:SetPoint("RIGHT", b); b.text:SetAllPoints(); b.text:SetText(spellName);
	-- overlay cast button
	local bc = CreateFrame("BUTTON", nil, b, "SecureActionButtonTemplate");
	bc:SetWidth(rewatch_data["SpellBarWidth"]); bc:SetHeight(rewatch_data["SpellBarHeight"]); bc:SetPoint("TOPLEFT", b, "TOPLEFT", 0, 0);
	bc:RegisterForClicks("LeftButtonDown"); bc:SetAttribute("type1", "spell"); bc:SetAttribute("unit", rewatch_bars[playerId]["Player"]);
	bc:SetAttribute("spell1", spellName); bc:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square.blp");
	
	-- return the reference to the spell bar
	return b;
end;

-- update a bar by resetting spell duration
-- spellName: the name of the spell to reset it's duration from
-- playerId: the index number of the player in the player table
-- return: void
function rewatch_UpdateBar(spellName, playerId)	
	-- return if playerId (returned from AddPlayer) is invalid
	if(playerId < 0) then return; end;
	
	-- refresh the end time and bar value
	local lag = select(3, GetNetStats());
	rewatch_bars[playerId][spellName] = GetTime() +  rewatch_abs[spellName]["Duration"] - lag/1000;
	rewatch_bars[playerId][spellName.."Bar"]:SetValue(rewatch_abs[spellName]["Duration"]);
	-- update lifebloom stack counter
	if(spellName == rewatch_loc["lifebloom"]) then
		rewatch_bars[playerId]["LifebloomStacks"] = math.min(rewatch_bars[playerId]["LifebloomStacks"]+1, 3);
		rewatch_bars[playerId][spellName.."Bar"].text:SetText("Lifebloom x"..rewatch_bars[playerId]["LifebloomStacks"]);
	end;
	
	-- return
	return;
end;

-- add a player to the players table and create his bars and button
-- player: the name of the player
-- return: the index number the player has been assigned
function rewatch_AddPlayer(player)
	-- return if in combat
	if(InCombatLockdown() == 1) then return -1; end;
	
	-- set vars
	rewatch_bars[rewatch_i] = {};
	-- build frame
	local x, y = rewatch_GetFramePos(rewatch_f[1]);
	local frame = CreateFrame("FRAME", nil, rewatch_f[1]); frame:SetWidth(rewatch_data["FrameWidth"]); frame:SetHeight(rewatch_data["FrameHeight"]);
	frame:SetPoint("TOPLEFT", rewatch_f[1], "TOPLEFT", x, y); frame:EnableMouse(true); frame:SetMovable(true);
	frame:SetBackdrop({bgFile = "Interface/Tooltips/UI-Tooltip-Background", edgeFile = nil, tile = 1, tileSize = 5, edgeSize = 5, insets = { left = 0, right = 0, top = 0, bottom = 0 }});
	frame:SetBackdropColor(0.1, 0.1, 0.1, 0.4);
	frame:SetScript("OnMouseDown", function() frame:StartMoving(); end); frame:SetScript("OnMouseUp", function() frame:StopMovingOrSizing(); rewatch_SnapToGrid(frame); end);
	-- create player HP bar
	local statusbar = CreateFrame("STATUSBAR", nil, frame, "TextStatusBar"); statusbar:SetWidth(rewatch_data["SpellBarWidth"]); statusbar:SetHeight(rewatch_data["SpellBarHeight"]-5);
	statusbar:SetPoint("TOPLEFT", frame, "TOPLEFT", 0, -1);
	statusbar:SetBackdrop({bgFile = nil, edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", tile = 1, tileSize = 5, edgeSize = 5, insets = { left = 0, right = 0, top = 0, bottom = 0 }});
	statusbar:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar"); statusbar:SetStatusBarColor(0.0, 0.6, 0.0, 0.8);
	statusbar:SetMinMaxValues(0, 1); statusbar:SetValue(0);
	-- put text in HP bar
	statusbar.text = statusbar:CreateFontString("$parentText", "ARTWORK", "GameFontHighlightSmall");
	statusbar.text:SetPoint("LEFT", statusbar); statusbar.text:SetAllPoints(); statusbar.text:SetText(player);
	-- energy/mana/rage bar
	local statusbar2 = CreateFrame("STATUSBAR", nil, frame, "TextStatusBar"); statusbar2:SetWidth(rewatch_data["SpellBarWidth"]); statusbar2:SetHeight(5);
	statusbar2:SetPoint("TOPLEFT", frame, "TOPLEFT", 0, 5-rewatch_data["SpellBarHeight"]);
	statusbar2:SetBackdrop({bgFile = "Interface/Tooltips/UI-Tooltip-Background", edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", tile = 1, tileSize = 5, edgeSize = 5, insets = { left = 0, right = 0, top = 0, bottom = 0 }});
	statusbar2:SetStatusBarTexture("Interface\\TargetingFrame\\UI-StatusBar"); statusbar2:SetStatusBarColor(0.0, 0.0, 0.8, 0.8);
	statusbar2:SetMinMaxValues(0, 1); statusbar2:SetValue(0);
	-- overlay target/remove button
	local tgb = CreateFrame("BUTTON", nil, statusbar, "SecureActionButtonTemplate");
	tgb:SetWidth(statusbar:GetWidth()); tgb:SetHeight(statusbar:GetHeight()); tgb:SetPoint("TOPLEFT", statusbar, "TOPLEFT", 0, 0);
	tgb:RegisterForClicks("LeftButtonDown"); tgb:SetAttribute("type1", "target"); tgb:SetAttribute("unit", player);
	tgb:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square.blp");
	tgb:SetScript("OnMouseDown", function() if(arg1 == "RightButton") then rewatch_HidePlayerByName(player); end; end);
	tgb:SetScript("OnEnter", function() rewatch_SetTooltip(player); end);
	tgb:SetScript("OnLeave", function() rewatch_tooltip:SetOwner(WorldFrame, "ANCHOR_NONE"); end);
	-- add a swiftmend button
	local smb = CreateFrame("BUTTON", nil, frame, "SecureActionButtonTemplate");
	smb:SetWidth(rewatch_data["SmallButtonWidth"]); smb:SetHeight(rewatch_data["SmallButtonHeight"]);
	smb:SetPoint("TOPLEFT", frame, "TOPLEFT", rewatch_abs[rewatch_loc["swiftmend"]]["Offset"], rewatch_abs[rewatch_loc["swiftmend"]]["Offset_Y"]);
	smb:RegisterForClicks("LeftButtonDown"); smb:SetAttribute("type1", "spell"); smb:SetAttribute("unit", player);
	smb:SetAttribute("spell1", rewatch_loc["swiftmend"]); smb:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square.blp");
	smb:SetNormalTexture("Interface\\Icons\\INV_Relics_IdolofRejuvenation.blp");
	smb:SetScript("OnEnter", function() rewatch_SetTooltip(rewatch_loc["swiftmend"]); end);
	smb:SetScript("OnLeave", function() rewatch_tooltip:SetOwner(WorldFrame, "ANCHOR_NONE"); end);
	-- create the cooldown overlay
	local smbcd = CreateFrame("Cooldown", nil, smb, "CooldownFrameTemplate");
	smb.cooldown = smbcd; smbcd:SetPoint("CENTER", 0, -1);
	smbcd:SetWidth(rewatch_data["SmallButtonWidth"]); smbcd:SetHeight(rewatch_data["SmallButtonHeight"]); smbcd:Hide();
	-- add abolish poison button
	local apb = CreateFrame("BUTTON", nil, frame, "SecureActionButtonTemplate");
	apb:SetWidth(rewatch_data["SmallButtonWidth"]); apb:SetHeight(rewatch_data["SmallButtonHeight"]);
	apb:SetPoint("TOPLEFT", frame, "TOPLEFT", rewatch_abs[rewatch_loc["abolishpoison"]]["Offset"], rewatch_abs[rewatch_loc["abolishpoison"]]["Offset_Y"]);
	apb:RegisterForClicks("LeftButtonDown"); apb:SetAttribute("type1", "spell"); apb:SetAttribute("unit", player);
	apb:SetAttribute("spell1", rewatch_loc["abolishpoison"]); apb:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square.blp");
	apb:SetNormalTexture("Interface\\Icons\\Spell_Nature_NullifyPoison_02.blp");
	apb:SetScript("OnEnter", function() rewatch_SetTooltip(rewatch_loc["abolishpoison"]); end);
	apb:SetScript("OnLeave", function() rewatch_tooltip:SetOwner(WorldFrame, "ANCHOR_NONE"); end);
	-- add decurse button
	local dec = CreateFrame("BUTTON", nil, frame, "SecureActionButtonTemplate");
	dec:SetWidth(rewatch_data["SmallButtonWidth"]); dec:SetHeight(rewatch_data["SmallButtonHeight"]);
	dec:SetPoint("TOPLEFT", frame, "TOPLEFT", rewatch_abs[rewatch_loc["removecurse"]]["Offset"], rewatch_abs[rewatch_loc["removecurse"]]["Offset_Y"]);
	dec:RegisterForClicks("LeftButtonDown"); dec:SetAttribute("type1", "spell"); dec:SetAttribute("unit", player);
	dec:SetAttribute("spell1", rewatch_loc["removecurse"]); dec:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square.blp");
	dec:SetNormalTexture("Interface\\Icons\\Spell_Shadow_Curse.blp");
	dec:SetScript("OnEnter", function() rewatch_SetTooltip(rewatch_loc["removecurse"]); end);
	dec:SetScript("OnLeave", function() rewatch_tooltip:SetOwner(WorldFrame, "ANCHOR_NONE"); end);
	-- save player data
	rewatch_bars[rewatch_i]["Frame"] = frame; rewatch_bars[rewatch_i]["Player"] = player;
	rewatch_bars[rewatch_i]["PlayerBar"] = statusbar; rewatch_bars[rewatch_i]["ManaBar"] = statusbar2;
	rewatch_bars[rewatch_i][rewatch_loc["lifebloom"]] = 0; rewatch_bars[rewatch_i]["LifebloomStacks"] = 0;
	rewatch_bars[rewatch_i][rewatch_loc["rejuvenation"]] = 0; rewatch_bars[rewatch_i][rewatch_loc["regrowth"]] = 0;
	rewatch_bars[rewatch_i][rewatch_loc["lifebloom"].."Bar"] = rewatch_CreateBar(rewatch_loc["lifebloom"], rewatch_i);
	rewatch_bars[rewatch_i][rewatch_loc["rejuvenation"].."Bar"] = rewatch_CreateBar(rewatch_loc["rejuvenation"], rewatch_i);
	rewatch_bars[rewatch_i][rewatch_loc["regrowth"].."Bar"] = rewatch_CreateBar(rewatch_loc["regrowth"], rewatch_i);
	rewatch_bars[rewatch_i]["SwiftmendButton"] = smb; rewatch_bars[rewatch_i]["AbolishPoisonButton"] = apb; rewatch_bars[rewatch_i]["DecurseButton"] = dec;
	-- increment the global index
	rewatch_i = rewatch_i+1; rewatch_AlterFrame(); rewatch_SnapToGrid(frame);
	
	-- return the inserted player's player table index
	return rewatch_GetPlayer(player);
end;

-- hide all bars and buttons from - and the player himself, by name
-- player: the name of the player to hide
-- return: void
-- PRE: Called by specific user request
function rewatch_HidePlayerByName(player)
	if(InCombatLockdown() == 1) then rewatch_Message(rewatch_loc["combatfailed"]);
	else
		-- get the index of this player
		local playerId = rewatch_GetPlayer(player);
		-- if this player exists, hide all bars and buttons from - and the player himself
		if(playerId > 0) then rewatch_HidePlayer(playerId); end;
	end;
	
	-- return
	return;
end;

-- hide all bars and buttons from - and the player himself
-- playerId: the table index of the player to hide
-- return: void
function rewatch_HidePlayer(playerId)
	-- return if in combat
	if(InCombatLockdown() == 1) then return; end;
	
	-- remove the bar
	local parent = rewatch_bars[playerId]["Frame"]:GetParent();
	rewatch_bars[playerId]["PlayerBar"]:Hide();
	rewatch_bars[playerId][rewatch_loc["lifebloom"].."Bar"]:Hide();
	rewatch_bars[playerId][rewatch_loc["rejuvenation"].."Bar"]:Hide();
	rewatch_bars[playerId][rewatch_loc["regrowth"].."Bar"]:Hide();
	rewatch_bars[playerId]["SwiftmendButton"]:Hide();
	rewatch_bars[playerId]["AbolishPoisonButton"]:Hide();
	rewatch_bars[playerId]["DecurseButton"]:Hide();
	rewatch_bars[playerId]["Frame"]:Hide();
	rewatch_bars[playerId]["Frame"]:SetParent(nil);
	rewatch_bars[playerId] = nil;
	
	-- update the frame width/height
	rewatch_AlterFrame(parent);
	
	-- return
	return;
end;

-- build a frame
-- return: void
function rewatch_BuildFrame()
	-- create it
	local frame = CreateFrame("FRAME", "Rewatch_Frame"..table.getn(rewatch_f), UIParent);
	-- set proper dimentions and location
	frame:SetWidth(100); frame:SetHeight(100);
	frame:SetPoint("TOPLEFT", UIParent, "TOPLEFT"); frame:EnableMouse(true); frame:SetMovable(true);
	-- set looks
	frame:SetBackdrop({bgFile = nil, edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border", tile = 1, tileSize = 10, edgeSize = 10, insets = { left = 3, right = 3, top = 3, bottom = 3 }});
	--local frameText = frame:CreateFontString("$parentText", "ARTWORK", "GameFontHighlightSmall");
	--frameText:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -3, 3); frameText:SetText("Rewatch");
	-- make sure it will not only move itself on dragging, but also all it's children
	frame:SetScript("OnMouseDown", function()
		frame:StartMoving();
		local children = { frame:GetChildren() };
		for _, child in ipairs(children) do
			local dx, dy = child:GetLeft()-frame:GetLeft(), child:GetTop()-frame:GetTop();
			local dxi, dyi = child:GetRight()-frame:GetRight(), child:GetBottom()-frame:GetBottom();
			child:ClearAllPoints();
			child:SetPoint("TOPLEFT", frame, "TOPLEFT", dx, dy);
			child:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", dxi, dyi);
		end;
	end);
	frame:SetScript("OnMouseUp", function() frame:StopMovingOrSizing(); end);
	-- add the frame to the table
	table.insert(rewatch_f, frame);
	-- create cooldown overlay and add it to it's own table
	local gcd = CreateFrame("Cooldown", nil, frame, "CooldownFrameTemplate"); gcd:SetAlpha(1);
	gcd:SetPoint("CENTER", 0, -1); gcd:SetWidth(frame:GetWidth()); gcd:SetHeight(frame:GetHeight()); gcd:Hide();
	table.insert(rewatch_gcds, gcd);
	
	-- return
	return;
end;

-- function set set the options frame values
-- get: boolean if the function will get data (true) or set data (false) from the options frame
-- return: void
function rewatch_OptionsFromData(get)
	-- get the childeren elements
	local children = { rewatch_options:GetChildren() };
	for _, child in ipairs(children) do
		-- if it's the slider, set or get his data
		if(child:GetName() == "Rewatch_AlphaSlider") then
			if(get) then child:SetValue(rewatch_loadInt["GcdAlpha"]);
			else rewatch_load["GcdAlpha"], rewatch_loadInt["GcdAlpha"] = child:GetValue(), child:GetValue(); end;
		-- if it's the autogroup checkbutton, set or get his data
		elseif(child:GetName() == "Rewatch_AutoGroupCB") then
			if(get) then if(rewatch_loadInt["AutoGroup"] == 1) then child:SetChecked(true); else child:SetChecked(false); end;
			else if(child:GetChecked()) then rewatch_load["AutoGroup"], rewatch_loadInt["AutoGroup"] = 1, 1;
				else rewatch_load["AutoGroup"], rewatch_loadInt["AutoGroup"] = 0, 0; end; end;
		-- if it's the hidesolo checkbutton, set or get his data
		elseif(child:GetName() == "Rewatch_SoloHideCB") then
			if(get) then if(rewatch_loadInt["HideSolo"] == 1) then child:SetChecked(true); else child:SetChecked(false); end;
			else if(child:GetChecked()) then rewatch_load["HideSolo"], rewatch_loadInt["HideSolo"] = 1, 1;
				else rewatch_load["HideSolo"], rewatch_loadInt["HideSolo"] = 0, 0; end; end;
		-- if it's the hide checkbutton, set or get his data
		elseif(child:GetName() == "Rewatch_HideCB") then
			if(get) then if(rewatch_loadInt["Hide"] == 1) then child:SetChecked(true); else child:SetChecked(false); end;
			else if(child:GetChecked()) then rewatch_load["Hide"], rewatch_loadInt["Hide"] = 1, 1;
				else rewatch_load["Hide"], rewatch_loadInt["Hide"] = 0, 0; end; end;
		end;
	end;
	-- apply changes
	if(not get) then
		for _, cd in ipairs(rewatch_gcds) do cd:SetAlpha(rewatch_loadInt["GcdAlpha"]); end;
		if(((rewatch_i == 2) and (rewatch_loadInt["HideSolo"] == 1)) or (rewatch_loadInt["Hide"] == 1)) then rewatch_f[1]:Hide(); else rewatch_f[1]:Show(); end;
	end;
end;

-- process the sent commands
-- cmd: the command that has to be executed
-- return: void
function rewatch_SlashCommandHandler(cmd)
	-- when there's a command typed
	if(cmd) then
		-- declaration and initialisation
		local pos, commands = 0, {};
		for st, sp in function() return string.find(cmd, " ", pos, true) end do
			table.insert(commands, string.sub(cmd, pos, st-1));
			pos = sp + 1;
		end; table.insert(commands, string.sub(cmd, pos));
		-- on a help request, reply with the localisation help table
		if(string.lower(commands[1]) == "help") then
			for _,val in ipairs(rewatch_loc["help"]) do
				rewatch_Message(val);
			end;
		-- if the user wants to add a player manually
		elseif(string.lower(commands[1]) == "add") then
			if(InCombatLockdown() == 1) then rewatch_Message(rewatch_loc["combatfailed"]);
			elseif(commands[2]) then if(rewatch_GetPlayer(commands[2]) < 0) then rewatch_AddPlayer(commands[2]); end;
			elseif(UnitName("target")) then if(rewatch_GetPlayer(UnitName("target")) < 0) then rewatch_AddPlayer(UnitName("target")); end;
			else rewatch_Message(rewatch_loc["noplayer"]); end;
		-- if the user wants to resort the list (clear and processgroup)
		elseif(string.lower(commands[1]) == "sort") then
			if(rewatch_loadInt["AutoGroup"] == 0) then
				rewatch_Message(rewatch_loc["nosort"]);
			else
				rewatch_Clear(); rewatch_changed = true;
				rewatch_Message(rewatch_loc["sorted"]);
			end;
		-- if the user wants to clear the player list
		elseif(string.lower(commands[1]) == "clear") then
			rewatch_Clear();
			rewatch_Message(rewatch_loc["cleared"]);
		-- if the user wants to set the gcd alpha
		elseif(string.lower(commands[1]) == "gcdalpha") then
			if(not tonumber(commands[2])) then rewatch_Message(rewatch_loc["nonumber"]);
			elseif((commands[2] < 0) or (commands[2] > 1)) then rewatch_Message(rewatch_loc["nonumber"]);
			else
				rewatch_load["GcdAlpha"] = tonumber(commands[2]); rewatch_loadInt["GcdAlpha"] = rewatch_load["GcdAlpha"];
				for _, cd in ipairs(rewatch_gcds) do cd:SetAlpha(rewatch_loadInt["GcdAlpha"]); end;
				rewatch_OptionsFromData(true);
				rewatch_Message(rewatch_loc["setalpha"]..commands[2]);
			end;
		-- if the user wants to set the hide solo feature
		elseif(string.lower(commands[1]) == "hidesolo") then
			if(not((commands[2] == "0") or (commands[2] == "1"))) then rewatch_Message(rewatch_loc["nonumber"]);
			else
				rewatch_load["HideSolo"] = tonumber(commands[2]); rewatch_loadInt["HideSolo"] = rewatch_load["HideSolo"];
				if(((rewatch_i == 2) and (rewatch_load["HideSolo"] == 1)) or (rewatch_["Hide"] == 1)) then rewatch_f[1]:Hide(); else rewatch_f[1]:Show(); end;
				rewatch_OptionsFromData(true);
				rewatch_Message(rewatch_loc["sethidesolo"]..commands[2]);
			end;
		-- if the user wants to set the hide solo feature
		elseif(string.lower(commands[1]) == "hide") then
			if(not((commands[2] == "0") or (commands[2] == "1"))) then rewatch_Message(rewatch_loc["nonumber"]);
			else
				rewatch_load["Hide"] = tonumber(commands[2]); rewatch_loadInt["Hide"] = rewatch_load["Hide"];
				if(((rewatch_i == 2) and (rewatch_load["HideSolo"] == 1)) or (rewatch_load["Hide"] == 1)) then rewatch_f[1]:Hide(); else rewatch_f[1]:Show(); end;
				rewatch_OptionsFromData(true);
				rewatch_Message(rewatch_loc["sethide"]..commands[2]);
			end;
		-- if the user wants to set the autoadjust list to group feature
		elseif(string.lower(commands[1]) == "autogroup") then
			if(not((commands[2] == "0") or (commands[2] == "1"))) then rewatch_Message(rewatch_loc["nonumber"]);
			else
				rewatch_load["AutoGroup"] = tonumber(commands[2]); rewatch_loadInt["AutoGroup"] = rewatch_load["AutoGroup"];
				rewatch_OptionsFromData(true);
				rewatch_Message(rewatch_loc["setautogroup"]..commands[2]);
			end;
		-- if the user demanded a druid buff check
		elseif(string.lower(commands[1]) == "check") then
			rewatch_BuffCheck();
		-- if the user wants to toggle the settings GUI
		elseif(string.lower(commands[1]) == "options") then
			InterfaceOptionsFrame_OpenToFrame("Rewatch");
		-- if the user wants something else (unsupported)
		elseif(string.len(commands[1]) > 0) then rewatch_Message(rewatch_loc["invalid_command"]);
		else rewatch_Message(rewatch_loc["credits"]); end;
	-- if there's no command typed
	else rewatch_Message(rewatch_loc["credits"]); end;
	
	-- return
	return;
end;


--// INITIALISE //----------------------------------------------------------------------------------------------------------------------------------------------------------------

-- make the addon stop here if the user isn't a druid
if((select(2, UnitClass("player"))) ~= "DRUID") then return; end;
-- build event logger
rewatch_events = CreateFrame("FRAME", nil, UIParent); rewatch_events:SetWidth(0); rewatch_events:SetHeight(0);
rewatch_events:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED"); rewatch_events:RegisterEvent("PARTY_MEMBERS_CHANGED");
-- initialise all vars
rewatch_f, rewatch_bars, rewatch_i, rewatch_loadInt = {},  {}, 1, { Loaded = false; GcdAlpha = 1; HideSolo = 0; AutoGroup = 1};
rewatch_tooltip, rewatch_gcds, rewatch_changed, rewatch_options = nil, {}, true, nil;
-- add the slash command handler
SLASH_REWATCH1 = "/Rewatch";
SLASH_REWATCH2 = "/rewatch";
SLASH_REWATCH3 = "/ReWatch";
SLASH_REWATCH4 = "/reWatch";
SLASH_REWATCH5 = "/Rew";
SLASH_REWATCH6 = "/rew";
SLASH_REWATCH7 = "/ReW";
SLASH_REWATCH8 = "/reW";
SlashCmdList["REWATCH"] = function(cmd)
	rewatch_SlashCommandHandler(cmd);
end;
-- initialise the data table
rewatch_SetData();
-- initialise the abilities table
rewatch_abs = {
	[rewatch_loc["lifebloom"]] = {
		Duration = 7;
		Offset = 0;
		Offset_Y = (rewatch_data["SpellBarHeight"] + rewatch_data["SpellBarMargin"])*-1;
	};
	[rewatch_loc["rejuvenation"]] = {
		Duration = 12;
		Offset = 0;
		Offset_Y = (rewatch_data["SpellBarHeight"] + rewatch_data["SpellBarMargin"])*-2;
	};
	[rewatch_loc["regrowth"]] = {
		Duration = 21;
		Offset = 0;
		Offset_Y = (rewatch_data["SpellBarHeight"] + rewatch_data["SpellBarMargin"])*-3;
	};
	[rewatch_loc["swiftmend"]] = {
		Duration = 15;
		Offset = 2;
		Offset_Y = (rewatch_data["SpellBarHeight"] + rewatch_data["SpellBarMargin"])*-4;
	};
	[rewatch_loc["abolishpoison"]] = {
		Duration = 1.5;
		Offset = (rewatch_data["SmallButtonMargin"]+rewatch_data["SmallButtonWidth"])*1;
		Offset_Y = (rewatch_data["SpellBarHeight"] + rewatch_data["SpellBarMargin"])*-4;
	};
	[rewatch_loc["removecurse"]] = {
		Duration = 1.5;
		Offset = (rewatch_data["SmallButtonMargin"]+rewatch_data["SmallButtonWidth"])*2;
		Offset_Y = (rewatch_data["SpellBarHeight"] + rewatch_data["SpellBarMargin"])*-4;
	};
};

--// SCRIPT //-----------------------------------------------------------------------------------------------------------------------------------------------------------------------

-- create the outline frame
rewatch_BuildFrame();

-- create the options frame
rewatch_options = CreateFrame("FRAME", "Rewatch_Options", UIParent); rewatch_options.name = "Rewatch";
local alphaSliderT = rewatch_options:CreateFontString("$parentText", "ARTWORK", "GameFontHighlightSmall");
alphaSliderT:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 10, -10); alphaSliderT:SetText(rewatch_loc["gcdText"]);
local alphaSlider = CreateFrame("SLIDER", "Rewatch_AlphaSlider", rewatch_options, "OptionsSliderTemplate");
alphaSlider:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 150, -25); alphaSlider:SetMinMaxValues(0, 1); alphaSlider:SetValueStep(0.1);
getglobal("Rewatch_AlphaSliderLow"):SetText(rewatch_loc["invisible"]); getglobal("Rewatch_AlphaSliderHigh"):SetText(rewatch_loc["visible"]); 

local hideCBT = rewatch_options:CreateFontString("$parentText", "ARTWORK", "GameFontHighlightSmall");
hideCBT:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 50, -60); hideCBT:SetText(rewatch_loc["hide"]);
local hideCB = CreateFrame("CHECKBUTTON", "Rewatch_HideCB", rewatch_options, "OptionsCheckButtonTemplate");
hideCB:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 10, -50);

local soloHideCBT = rewatch_options:CreateFontString("$parentText", "ARTWORK", "GameFontHighlightSmall");
soloHideCBT:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 50, -80); soloHideCBT:SetText(rewatch_loc["hideSolo"]);
local soloHideCB = CreateFrame("CHECKBUTTON", "Rewatch_SoloHideCB", rewatch_options, "OptionsCheckButtonTemplate");
soloHideCB:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 10, -70);

local autoGroupCBT = rewatch_options:CreateFontString("$parentText", "ARTWORK", "GameFontHighlightSmall");
autoGroupCBT:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 50, -100); autoGroupCBT:SetText(rewatch_loc["autoAdjust"]);
local autoGroupCB = CreateFrame("CHECKBUTTON", "Rewatch_AutoGroupCB", rewatch_options, "OptionsCheckButtonTemplate");
autoGroupCB:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 10, -90);
local buffCheckBTN = CreateFrame("BUTTON", "Rewatch_BuffCheckBTN", rewatch_options, "OptionsButtonTemplate"); buffCheckBTN:SetText(rewatch_loc["buffCheck"]);
buffCheckBTN:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 10, -130); buffCheckBTN:SetScript("OnClick", rewatch_BuffCheck);
local sortBTN = CreateFrame("BUTTON", "Rewatch_BuffCheckBTN", rewatch_options, "OptionsButtonTemplate"); sortBTN:SetText(rewatch_loc["sortList"]);
sortBTN:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 100, -130); sortBTN:SetScript("OnClick", function() if(rewatch_loadInt["AutoGroup"] == 0) then rewatch_Message(rewatch_loc["nosort"]); else rewatch_Clear(); rewatch_changed = true; rewatch_Message(rewatch_loc["sorted"]); end; end);
local clearBTN = CreateFrame("BUTTON", "Rewatch_BuffCheckBTN", rewatch_options, "OptionsButtonTemplate"); clearBTN:SetText(rewatch_loc["clearList"]);
clearBTN:SetPoint("TOPLEFT", rewatch_options, "TOPLEFT", 190, -130); clearBTN:SetScript("OnClick", function() rewatch_Clear(); rewatch_Message(rewatch_loc["cleared"]); end);
rewatch_options.okay = function(self) rewatch_OptionsFromData(false) end;
rewatch_options.cancel = function(self) rewatch_OptionsFromData(true); end;
rewatch_options.default = function(self)
	rewatch_load["GcdAlpha"], rewatch_load["HideSolo"], rewatch_load["Hide"], rewatch_load["AutoGroup"] = 1, 0, 0, 1;
	rewatch_loadInt["GcdAlpha"], rewatch_loadInt["HideSolo"], rewatch_loadInt["Hide"], rewatch_loadInt["AutoGroup"] = 1, 0, 0, 1;
	rewatch_OptionsFromData(true);
end;
InterfaceOptions_AddCategory(rewatch_options); 

-- make a tooltip object
rewatch_tooltip = CreateFrame("GameTooltip", "rewatch_tooltip", WorldFrame, "GameTooltipTemplate");
rewatch_tooltip:SetOwner(WorldFrame, "ANCHOR_NONE");
rewatch_tooltip:AddFontStrings(
    rewatch_tooltip:CreateFontString("$parentTextLeft1", nil, "GameTooltipText"),
	rewatch_tooltip:CreateFontString("$parentTextRight1", nil, "GameTooltipText")
);

-- make sure we catch events and process them
rewatch_events:SetScript("OnEvent", function(timestamp, event, ...)
	-- if the group/raid the player is in has changed
	if(event == "PARTY_MEMBERS_CHANGED") then		
		rewatch_changed = true;
	-- if a spell was cast successfull by the user or a heal has been received
	elseif((arg2 == "SPELL_CAST_SUCCESS") or (arg2 == "SPELL_HEAL")) then
		-- if it was your spell/heal
		if(arg3 == UnitGUID("player")) then
			rewatch_TriggerCooldown();
			-- if this was a regrowth or rejuvenation or LB HoT effect
			if((arg10 == rewatch_loc["regrowth"]) or (arg10 == rewatch_loc["rejuvenation"]) or ((arg10 == rewatch_loc["lifebloom"]) and (arg2 == "SPELL_CAST_SUCCESS"))) then
				--  ignore heals on non-party-/raidmembers
				if(not rewatch_InGroup(arg7)) then return; end;
				-- get the player position, or if -1, create a new frame
				local playerId = rewatch_GetPlayer(arg7);
				if(playerId < 0) then
					if(rewatch_loadInt["AutoGroup"] == 0) then return;
					else playerId = rewatch_AddPlayer(UnitName(arg7)); end;
				end;
				-- update the spellbar
				rewatch_UpdateBar(arg10, playerId);
			-- if a swiftmend was received
			elseif((arg10 == rewatch_loc["swiftmend"]) and (arg2 == "SPELL_HEAL")) then
				--  ignore heals on non-party-/raidmembers
				if(not rewatch_InGroup(arg7)) then return; end;
				-- get the player position, return if -1
				local playerId = rewatch_GetPlayer(UnitName(arg7));
				if(playerId < 0) then return;
				else rewatch_ProcessSwiftmend(playerId); end;
			end;
		end;
	end;
	
	-- return
	return;
end);

-- update everything
local targets = {}, val, a, b, left, i;
rewatch_events:SetScript("OnUpdate", function()
	-- load saved vars
	if(not rewatch_loadInt["Loaded"]) then
		rewatch_OnLoad();
	else
		-- if the group formation has been changed, add new group members to the list
		if(rewatch_changed) then
			if(rewatch_loadInt["AutoGroup"] == 1) then
				if(rewatch_i == 1) then rewatch_AddPlayer(UnitName("player")); end;
				rewatch_ProcessGroup();
			end;
		end;
		-- process updates
		for i=1,rewatch_i-1 do val = rewatch_bars[i]; if(val) then
			-- update player health bars if possible
			a, b = val["PlayerBar"]:GetMinMaxValues(); if(true or (b > 0)) then
				-- get and set mana data
				a, b = UnitManaMax(val["Player"]), UnitMana(val["Player"]);
				val["ManaBar"]:SetMinMaxValues(0, a); val["ManaBar"]:SetValue(b);
				-- get and set health data
				a, b = UnitHealthMax(val["Player"]), UnitHealth(val["Player"]);
				val["PlayerBar"]:SetMinMaxValues(0, a); val["PlayerBar"]:SetValue(b);
				-- clear buffs if the player died (set it to 1 so it will be seen as expired)
				if((b <= 0) and (a > 0)) then val[rewatch_loc["rejuvenation"]], val[rewatch_loc["regrowth"]], val[rewatch_loc["lifebloom"]] = 1, 1, 1;
				else
					-- set healthbar color accordingly
					if(b/a < .25) then val["PlayerBar"]:SetStatusBarColor(0.6, 0.0, 0.0, 0.9);
					elseif(a == b) then val["PlayerBar"]:SetStatusBarColor(0.0, 0.8, 0.0, 0.9);
					else val["PlayerBar"]:SetStatusBarColor(0.0, 0.6, 0.0, 0.8); end;
					-- poison/curse checks
					if(rewatch_IsPoisoned(i)) then val["AbolishPoisonButton"]:SetAlpha(1); else val["AbolishPoisonButton"]:SetAlpha(0.15); end;
					if(rewatch_IsCursed(i)) then val["DecurseButton"]:SetAlpha(1); else val["DecurseButton"]:SetAlpha(0.15); end;
					-- set manabar accordingly power type (energy, rage, mana)
					a = UnitPowerType(val["Player"]);
					if(a == 0) then val["ManaBar"]:SetStatusBarColor(0.0, 0.0, 0.6, 0.8);
					elseif(a == 1) then val["ManaBar"]:SetStatusBarColor(0.6, 0.0, 0.0, 0.8);
					else val["ManaBar"]:SetStatusBarColor(0.95, 0.85, 0.15, 0.8); end;
				end;
			end;
			-- check if this unit's target is targetting one of the listed players
			val["PlayerBar"].text:SetTextColor(1, 1, 1, 1);
			if(InCombatLockdown() == 1) then if(UnitCanAttack(val["Player"], val["Player"].."-target")) then
				i = rewatch_GetPlayer(UnitName(val["Player"].."-target-target"));
				if(i > 0) then table.insert(targets, i); end;
			end; end;
			-- fade when out of range
			if(IsSpellInRange(rewatch_loc["rejuvenation"], val["Player"]) == 1) then val["Frame"]:SetAlpha(1); else val["Frame"]:SetAlpha(0.3); end;
			-- if no rejuvenation was cast on this player
			if(val[rewatch_loc["rejuvenation"]] == 0) then -- nuffin!
			-- else, if it has just expired
			elseif(GetTime() >= val[rewatch_loc["rejuvenation"]]) then
				-- clear the end time and bar value
				val[rewatch_loc["rejuvenation"].."Bar"]:SetValue(0); val[rewatch_loc["rejuvenation"]] = 0;
			-- else update the bar's value
			else
				left = val[rewatch_loc["rejuvenation"]]-GetTime();
				val[rewatch_loc["rejuvenation"].."Bar"]:SetValue(left);
				if(left > 2) then val[rewatch_loc["rejuvenation"].."Bar"]:SetStatusBarColor(0.0, 0.0, 0.6, 0.8);
				else val[rewatch_loc["rejuvenation"].."Bar"]:SetStatusBarColor(0.6, 0.0, 0.0, 0.8); end;
			end;
			-- if no regrowth was cast on this player
			if(val[rewatch_loc["regrowth"]] == 0) then --nuffin!
			-- else, if it has just expired
			elseif(GetTime() >= val[rewatch_loc["regrowth"]]) then
				-- clear the end time and bar value
				val[rewatch_loc["regrowth"].."Bar"]:SetValue(0); val[rewatch_loc["regrowth"]] = 0;
			-- else update the bar's value
			else
				left = val[rewatch_loc["regrowth"]]-GetTime();
				val[rewatch_loc["regrowth"].."Bar"]:SetValue(left);
				if(left > 2) then val[rewatch_loc["regrowth"].."Bar"]:SetStatusBarColor(0.0, 0.0, 0.6, 0.8);
				else val[rewatch_loc["regrowth"].."Bar"]:SetStatusBarColor(0.6, 0.0, 0.0, 0.8); end;
			end;
			-- if no regrowth was cast on this player
			if(val[rewatch_loc["lifebloom"]] == 0) then --nuffin!
			-- else, if it has just expired
			elseif(GetTime() >= val[rewatch_loc["lifebloom"]]) then
				-- clear the end time and bar value
				val[rewatch_loc["lifebloom"].."Bar"]:SetValue(0); val[rewatch_loc["lifebloom"]] = 0;
				val["LifebloomStacks"] = 0; val[rewatch_loc["lifebloom"].."Bar"].text:SetText(rewatch_loc["lifebloom"]);
			-- else update the bar's value
			else
				left = val[rewatch_loc["lifebloom"]]-GetTime(); val[rewatch_loc["lifebloom"].."Bar"]:SetValue(left);
				-- check if lifebloom stack counter is valid
				if(left>(rewatch_abs[rewatch_loc["lifebloom"]].Duration/2)) then for i=1,40 do
					local name, rank, iconTexture, count, duration = UnitBuff(val["Player"], i, 1);
					if(not name) then break; elseif((name == rewatch_loc["lifebloom"]) and (duration)) then val["LifebloomStacks"] = count; break; end;
				end; end;
				val[rewatch_loc["lifebloom"].."Bar"].text:SetText("Lifebloom x"..val["LifebloomStacks"]);
				if(left > 2) then val[rewatch_loc["lifebloom"].."Bar"]:SetStatusBarColor(0.0, 0.0, 0.6, 0.8);
				else val[rewatch_loc["lifebloom"].."Bar"]:SetStatusBarColor(0.6, 0.0, 0.0, 0.8); end;
			end;
		end; end;
		-- color targetted
		for _, i in ipairs(targets) do if(rewatch_bars[i]) then rewatch_bars[i]["PlayerBar"].text:SetTextColor(1, 0, 0, 1); end; end;
		targets = {};
	end;
	
	-- return
	return;
end);