-- QuestNoise v1.3

-- v1.3
-- * Fixed bug with saving settings
-- v1.2
-- * Added option to enable/disable "Quest Complete" message
-- v1.1
-- * Added Ace3 config menu (under Interface > Addons) to enable/disable/change sounds for events

-- our Ace3 addon
QuestNoise = LibStub("AceAddon-3.0"):NewAddon("QuestNoise", "AceConsole-3.0")
QuestNoise.version = "1.3"
QuestNoise.date = "2010-05-11"

-- our frame
QuestNoiseFrame = CreateFrame("Frame", "QuestNoise", UIParent)
local f = QuestNoiseFrame

-- AceConfig options tree
local options = {
	name = "QuestNoise",
	type = "group",
	get = function(info) return QuestNoise.db.profile[info[#info]] end,
	set = function(info, value) QuestNoise.db.profile[info[#info]] = value end,

	args = {
		General = {
			order = 1,
			type = "group",
			name = "General Settings",
			desc = "General Settings",
			args = {
				desc = {
					type = "description",
					order = 1,
					name = "Addon that plays sounds upon making progress on quest objectives",
				},
				hdr1 = {
					type = "header",
					name = "Objective Progress",
					order = 2,
				},
				enableObjProgress = {
					type = "toggle",
					order = 3,
--					width = "double",
					name = "Enabled",
					desc = "Enables/disables playing a sound when objective progress has been made",
				},
				objProgressSound = {
					type = "select",
					style = "dropdown",
					order = 5,
					width = "double",
					name = "Pick Sound",
					desc = "Pick Sound",
					values = {"AuctionWindowOpen","AuctionWindowClose"}
				},
				hdr2 = {
					type = "header",
					name = "Objective Complete",
					order = 6,
				},
				enableObjComplete = {
					type = "toggle",
					order = 7,
--					width = "double",
					name = "Enabled",
					desc = "Enables/disables playing a sound when an objective has been completed",
				},
				objCompleteSound = {
					type = "select",
					style = "dropdown",
					order = 8,
					width = "double",
					name = "Pick Sound",
					desc = "Pick Sound",
					values = {"AuctionWindowOpen","AuctionWindowClose"}
				},
				hdr3 = {
					type = "header",
					name = "Quest Complete",
					order = 9,
				},
				enableQuestComplete = {
					type = "toggle",
					order = 10,
--					width = "double",
					name = "Enabled",
					desc = "Enables/disables playing a sound when a quest has been completed",
				},
				questCompleteSound = {
					type = "select",
					style = "dropdown",
					order = 11,
					width = "double",
					name = "Pick Sound",
					desc = "Pick Sound",
					values = {"ReadyCheck","Sound\\Creature\\Peon\\PeonBuildingComplete1.wav"}
				},
				hdr4 = {
					type = "header",
					name = "\"Quest Complete\" Message",
					order = 12,
				},
				enableQuestCompleteMsg = {
					type = "toggle",
					order = 13,
--					width = "double",
					name = "Enabled",
					desc = "Enables/disables a center-screen message indicating which quest you have just completed",
				},
			},
		},
	},
}
QuestNoise.Options = options
local defaults = {
	profile =  {
		enableObjProgress = true,
		objProgressSound = 1,
		enableObjComplete = true,
		objCompleteSound = 2,
		enableQuestComplete = true,
		questCompleteSound = 2,
		enableQuestCompleteMsg = true,
	},
}

-- note this is the Ace3 addon frame, not the one we originally had
-- eventually i'll merge them together but meh
function QuestNoise:OnInitialize()
	QuestNoise.OptionsFrames = {}

	QuestNoise.db = LibStub("AceDB-3.0"):New("QuestNoiseDB", defaults, "Default")
	options.args.Profiles = LibStub("AceDBOptions-3.0"):GetOptionsTable(QuestNoise.db)
	options.args.Profiles.cmdHidden = true
	options.args.Profiles.order = -1

	LibStub("AceConfigRegistry-3.0"):RegisterOptionsTable("QuestNoise", options)

	QuestNoise.OptionsFrames.General = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("QuestNoise", nil, nil, "General")
	QuestNoise.OptionsFrames.Profiles = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("QuestNoise", "Profiles", "QuestNoise", "Profiles")
end


-- events and main event handler (simply forwards to a member function of the same name)
f:RegisterEvent("QUEST_LOG_UPDATE")
f:RegisterEvent("UI_INFO_MESSAGE")
f:SetScript("OnEvent", function(this, event, ...)
	this[event](this, ...)
end)

-- a few constants for clarity in our sound function
local QUESTNOISE_OBJECTIVEPROGRESS = 1
local QUESTNOISE_OBJECTIVECOMPLETE = 2
local QUESTNOISE_QUESTCOMPLETE = 3

-- holds stored messages from UI_INFO_MESSAGE
local QuestNoise_ObjectivesBuffer = {}

-- When a message is output to the screen, the quest info obtained from GetQuestLogTitle/GetQuestLogLeaderBoard isn't updated yet,
-- so we save it here and then look for it later in QUEST_LOG_UPDATE. This is slightly better than just checking for any changes
-- to previously-saved objectives in QUEST_LOG_UPDATE since it will only check if there has been recent info messages displayed,
-- and it only checks for exact string matches of the displayed message.
function f:UI_INFO_MESSAGE(arg)
	tinsert(QuestNoise_ObjectivesBuffer, arg)	
end

-- This is the only quest-related function that I can find that is guaranteed called AFTER any objective change are reported via
-- GetNumQuestLeaderBoards. UNIT_QUEST_LOG_CHANGED seemingly isn't called on item-based objectives, and QUEST_WATCH_UPDATE is called
-- before the objective changes are reported via GetNumQuestLeaderBoards. This function simply iterates through the list of queued
-- UI messages (if any) and calls the EvalMsg function. This function should have a fairly low overhead cost since it does nothing
-- if there are no saved messages in the queue. I can't imagine a scenario where 2 messages will be added to the queue at once,
-- excepting the few quests which both have a "kill X of this specific guy" objective.
function f:QUEST_LOG_UPDATE()
	local text = QuestNoise_ObjectivesBuffer[1]
	while text do
		self:EvalMsg(tremove(QuestNoise_ObjectivesBuffer, 1))
		text = QuestNoise_ObjectivesBuffer[1]
	end
end

-- This function takes a message (that originated from UI_INFO_MESSAGE) and scans all quests for a matching objective string.
-- If it finds one, it determines the current status of quest (quest completed, objective completed, objective progress)
-- and plays a sound. If quest is completed, a message is also output stating this.
function f:EvalMsg(arg)

	-- begin looping through every quest
	local numQuests = GetNumQuestLogEntries()
	for qindex = 1, numQuests do
		local title, _, _, _, isHeader, _, isComplete, _, questID =  GetQuestLogTitle(qindex) -- title, level, questTag,  suggestedGroup, isHeader, isCollapsed, isComplete, isDaily, questID
		
		-- GetQuestLogTitle returns EVERY line in the Quest Log, including the zone headers, and we don't care about them
		if (not isHeader) then

			-- begin checking each of this quest's objectives
			local numObjectives = GetNumQuestLeaderBoards(qindex) or 0
			for obj = 1, numObjectives do
				local text, _, finished = GetQuestLogLeaderBoard(obj, qindex) -- text, type, finished
					
				-- check if this objective matches what was displayed
				if ((arg == text) or (arg == text.." ("..COMPLETE..")")) then
					-- quest complete has higher precedence
					if (isComplete) then
						self:MakeSound(QUESTNOISE_QUESTCOMPLETE)
						if (QuestNoise.db.profile.enableQuestCompleteMsg) then
							UIErrorsFrame:AddMessage("Quest complete: "..title, 1, 1, 0, 1, 5);
						end
					-- then we see if the objective we just made progress on is complete
					elseif (finished) then
						self:MakeSound(QUESTNOISE_OBJECTIVECOMPLETE)
					-- otherwise we just made some progress
					else
						self:MakeSound(QUESTNOISE_OBJECTIVEPROGRESS)
					end
				end
			end
		end
	end
end

-- This is just a helper function to play sounds based on a specific event constant
function f:MakeSound(event)
	local sound = nil
	
	if (event == QUESTNOISE_QUESTCOMPLETE) then
		if (QuestNoise.db.profile.enableQuestComplete) then
			sound = QuestNoise.Options.args.General.args.questCompleteSound.values[QuestNoise.db.profile.questCompleteSound]
		end
	elseif (event == QUESTNOISE_OBJECTIVECOMPLETE) then
		if (QuestNoise.db.profile.enableObjComplete) then
			sound = QuestNoise.Options.args.General.args.objCompleteSound.values[QuestNoise.db.profile.objCompleteSound]
		end
	elseif (event == QUESTNOISE_OBJECTIVEPROGRESS) then
		if (QuestNoise.db.profile.enableObjProgress) then
			sound = QuestNoise.Options.args.General.args.objProgressSound.values[QuestNoise.db.profile.objProgressSound]
		end
	end
	
	if (not sound) then
		return
	end
	
	sound = strlower(sound)
	
	if ((string.sub(sound, -4) == ".wav") or (string.sub(sound, -4) == ".mp3")) then
		PlaySoundFile(sound)
	else
		PlaySound(sound)
	end


end
