----------------------------------------------------------------------------------------------------
--[[				SCROLL TIP
----------------------------------------------------------------------------------------------------

title:		ScrollTip
category:	Library
notes:	Displays a scrollable help tip.
author:	Shaktar of Stonemaul, Incindarella of Drak'Tharon
date:		Sept 15, 2010
version:	1.1

]]

local SCROLLTIP_VERSION = 1
if SCROLLTIP_INSTALLED and SCROLLTIP_VERSION < SCROLLTIP_INSTALLED then
	return end   -- newer version already installed
SCROLLTIP_INSTALLED = SCROLLTIP_VERSION

SCROLLTIP_INSTRUCTIONS =

[[
<big>ScrollTip</big> <i>by Shaktar of Stonemaul</i>

This library contains two functions. The main function is ScrollTip, a function that displays a scrollable tooltip. The bonus function is dumplist, a function used during debugging to display variables.

<b>function ScrollTip(message, size)</b>

ScrollTip is a simple function which displays a large movable scrollable tooltip given a multi-line string. This string is usually instructions for a ui mod. If kept in a separate file, these instructions are known as a "read me" or help file. ScrollTip lets ui mod authors include the "read me" within the ui mod itself.

The user can scroll the text within the tooltip by using the mouse wheel. Since ScrollTip is rather simple, there is no scroll bar. The text can be quite long but if it's enormously long, the user would get tired of rolling the mouse wheel to scroll to the bottom of the text. If your text is so long that scrolling with the mouse wheel takes far too much time, then you would have to break up the text into smaller chunks, or use a more advanced function than ScrollTip.

The text can include three very simple html tags: big, b, i. Text within these tags are not displayed bigger, bolder or italicized, however. Since ScrollTip is rather simple, those tags are just turned into appropriately colorized text. For example, big is displayed in <big>bright yellow</big>, b in <b>pastel yellow</b>, and i in <i>translucent yellow</i>.

The size of the scrollable tooltip can be 'small', 'medium', or 'large'. If the size is omitted, medium is assumed. If the size is small, a small font is used because the normal font does not scroll well in a small tooltip. The normal font is used in medium and large tooltips.

ScrollTip is really easy to add to your ui mod.
1) Put ScrollTip.lua in your ui mod directory
2) Add ScrollTip.lua to your .toc file.
3) In your lua code, call ScrollTip(YourHelpString)

Once ScrollTip has been added to your ui mod and your ui mod is running, you can type /scrolltip to display these instructions and information on using ScrollTip. You can also type /scrolltip small to view a small tooltip and /scrolltip large to view a large tooltip. This way you can compare the sizes and see which size works best for you.

The ScrollTip function returns the frame containing the scrollable tooltip but it can be ignored since the user can close the tooltip themselves.

Example function calls:
ScrollTip('Hello World!', 'small')
ScrollTip('<big</b>>big</big</b>>')
tip = ScrollTip('save return value to close tip later')

ScrollTip is just a simple function to easily display a help file and there are no plans to add any features. I hope ScrollTip fits your needs.

<b>function dumplist(...)</b>

dumplist is a debugging function which displays the values of arguments passed to it. You can either pass it a list of arguments or pass it a table. The result is displayed in the Chat frame.

Passing it a list of arguments: The values of the arguments are displayed in one line separated by commas and ending with a double colon. The list can be any length. Usually, the list contains pairs, each pair containing the name of a variable as a string and the actual variable. A sample list of pairs would be dumplist('class', class, 'level', level). The list can also just be a single argument such as dumplist('entering SomeFunc').

Example for a list:
one = 1; two = one + one
dumplist('one', one, 'two', two)
one, 1, two, 2::

Passing it a table: The table is enumerated and displayed.

Example for a table:
table = { one = 1, two = 2 }
dumplist(table)
one = 1, two = 2::
]]

----------------------------------------------------------------------------------------------------
function dumplist(...)
-- two ways to use dumplist() function
-- first way is to pass it a list of arguments, it displays the values of all arguments
-- second way is to pass it a single table, it displays all the keys and values in the table
----------------------------------------------------------------------------------------------------

if (false) then return end
local msg = ''
local isTable = false
if (select('#', ...) == 1 and type(select(1, ...)) == 'table') then
	isTable = true end
if (isTable) then
	for key, value in pairs(select(1, ...)) do
		msg = msg .. tostring(key) .. '=' .. tostring(value)
		if (next(select(1, ...), key)) then msg = msg .. ', ' end
		end
	end
if (not isTable) then
	for ix = 1, select('#', ...) do
		if (ix > 1) then msg = msg .. ', ' end
		msg = msg .. tostring(select(ix, ...))
		end
	end
msg = msg .. '::'
DEFAULT_CHAT_FRAME:AddMessage(msg)
end

----------------------------------------------------------------------------------------------------
function ScrollTip(message, size)
-- display a relatively large tooltip which is scrollable with the mouse wheel
-- displays a second small tooltip informing the user to use the mouse wheel to scroll
-- this function is useful for displaying instructions or help
-- message is a multi-line string which can include 3 very simple html tags: big, b, i
-- html tags are turned into colorized text instead of altering the font
-- size is 'small', 'medium', or 'large'
-- size is optional; if omitted, size is assumed to be 'medium'
----------------------------------------------------------------------------------------------------

local NPB_PASTEL = '|cffffff80'		-- pastel yellow
local NPB_BRIGHT = '|cffffff00'		-- bright yellow
local NPB_PLAIN = '|r'			-- reset color

local SCROLL_TIP = 'Use the mouse wheel to scroll up or down'
local SCROLL_TIP_INFO = 'Scroll down to see more text in the big tooltip.'
local WIDTHS = { small = 300, medium = 400, large = 600, }
local HEIGHTS = { small = 225, medium = 300, large = 450}

local BACKDROP  = {
	bgFile = 'Interface/Tooltips/UI-Tooltip-Background', 
	edgeFile = 'Interface/Tooltips/UI-Tooltip-Border', 
	tile = true, tileSize = 16, edgeSize = 16,
	insets = { left = 3, right = 3, top = 3, bottom = 3 }
}

----------------------------------------------------------------------------------------------------

function ScrollUpToTop0()
scrollframe:ScrollToBottom()
while not scrollframe:AtTop() do
	scrollframe:ScrollUp() end
scrollframe:SetScript('OnUpdate', nil)
end

function ScrollUpToTop(self)
self:ScrollToBottom()
while not self:AtTop() do
	self:ScrollUp() end
self:SetScript('OnUpdate', nil)
end

function ScrollWithWheel(self, delta)
if delta > 0 then
	self:ScrollUp() end
if delta < 0 then
	self:ScrollDown() end
end

function ShowMouseScrollTip()
GameTooltip_SetDefaultAnchor(GameTooltip, UIParent)
GameTooltip:AddLine(SCROLL_TIP)
GameTooltip:AddLine(SCROLL_TIP_INFO, .85, .85, .85, 1)
GameTooltip:Show()
end

function HideMouseScrollTip()
GameTooltip:Hide()
end
	
----------------------------------------------------------------------------------------------------

message = string.gsub(message, '<big>', ' |cffFFFF00')
message = string.gsub(message, '<b>', ' |cffFFFFA0')
message = string.gsub(message, '<i>', ' |cff808040')
message = string.gsub(message, '</big>', '|r')
message = string.gsub(message, '</b>', '|r')
message = string.gsub(message, '</i>', '|r')

if not size then size = 'medium' end
size = string.lower(size)
local width = WIDTHS[size]
local height = HEIGHTS[size]

local font = GameFontHighlight
if size == 'small' then font = GameFontHighlightSmall end
	
----------------------------------------------------------------------------------------------------

local tipframe = CreateFrame('Frame', nil, UIParent)
tipframe:SetMovable(1)
tipframe:RegisterForDrag('LeftButton')
tipframe:SetScript('OnDragStart', function(self, button) self:StartMoving() end)
tipframe:SetScript('OnDragStop', function(self, button) self:StopMovingOrSizing() end)
tipframe:SetBackdrop(BACKDROP)
tipframe:SetBackdropColor(.1, .1, .1)
tipframe:SetWidth(width)
tipframe:SetHeight(height)
tipframe:SetPoint('CENTER', UIParent, 'CENTER', 0, 0);
tipframe:EnableMouse(1)
tipframe:SetClampedToScreen(true)
tipframe:Show()

local closebutton = CreateFrame('Button', nil, tipframe, 'UIPanelCloseButton')
closebutton:SetParent(tipframe)
closebutton:SetScript('OnClick', function(self, button, down) self:GetParent():Hide() end)
closebutton:SetPoint('TOPRIGHT', tipframe, 'TOPRIGHT', 0, 0)

local scrollframe = CreateFrame('ScrollingMessageFrame', nil, tipframe)
scrollframe:SetPoint('TOPLEFT', tipframe, 'TOPLEFT', 10, -10)
scrollframe:SetPoint('BOTTOMRIGHT', tipframe, 'BOTTOMRIGHT', -25, 10)
scrollframe:SetFontObject(font)
scrollframe:SetJustifyH('LEFT')
scrollframe:SetMaxLines(9999)
scrollframe:SetFading(false)
scrollframe:RegisterForDrag('LeftButton')
scrollframe:SetScript('OnDragStart', function(self, button) self:GetParent():StartMoving() end)
scrollframe:SetScript('OnDragStop', function(self, button) self:GetParent():StopMovingOrSizing() end)
scrollframe:SetScript('OnEnter', ShowMouseScrollTip)
scrollframe:SetScript('OnLeave', HideMouseScrollTip)
scrollframe:SetScript('OnUpdate', ScrollUpToTop)
scrollframe:SetScript('OnMouseWheel', ScrollWithWheel)
scrollframe:EnableMouse(1)
scrollframe:EnableMouseWheel(1)

message = string.gsub(message, '\n', ' \n')
for line in message:gmatch("[^\n]+") do
	scrollframe:AddMessage(line) end
return tipframe
end

----------------------------------------------------------------------------------------------------
function ScrollTip_DoSlash(size)
----------------------------------------------------------------------------------------------------

if size ~= 'small' and size ~= 'medium' and size ~= 'large' then
	size = 'medium' end
ScrollTip(SCROLLTIP_INSTRUCTIONS, size)
end

----------------------------------------------------------------------------------------------------
-- initialization
----------------------------------------------------------------------------------------------------

SlashCmdList['ScrollTip'] = ScrollTip_DoSlash
SLASH_ScrollTip1 = '/scrolltip'